/*
 *  File:        math_write.h
 *  Purpose:     Write math paragraphs in LaTeX
 *  Author:      Alejandro Aguilar Sierra <asierra@servidor.unam.mx> 
 *  Created:     January 1996
 *  Description: 
 *
 *  Dependencies: Xlib, XForms
 *
 *  Copyright: (c) 1996, Alejandro Aguilar Sierra
 *
 *   Version: 0.5beta, Mathed & Lyx project.
 *
 *   You are free to use and modify this code under the terms of
 *   the GNU General Public Licence version 2 or later.
 */

#include "config.h"
#include "math_draw.h"
#include "math_parser.h"


extern const char *latex_mathenv[];
extern char *latex_mathspace;

char *math_font_name[] = {
   "mathrm",
   "mathcal",
   "mathbf",
   "mathsf",
   "mathtt",
   "mathit"
};

void
MathSpaceInset::Write(FILE *outf)
{ 
   if (space>=-1 && space<4)
     fprintf(outf, "\\%c ", latex_mathspace[space+1]);
}

void MathSqrtInset::Write(FILE *outf)
{ 
   fprintf(outf, "\\%s{", name);
   MathParInset::Write(outf);  
   fprintf(outf, "}");
}

void MathDelimInset::Write(FILE *outf)
{ 
   latexkeys* l = lm_get_key_by_id(left, LM_TK_SYM);
   latexkeys* r = lm_get_key_by_id(right, LM_TK_SYM);
   fprintf(outf, "\\left");
   if (l) 
     fprintf(outf, "\\%s ", l->name);
   else
     fprintf(outf, "%c ", left);
   
   MathParInset::Write(outf);  
   fprintf(outf, "\\right");
   if (r) 
     fprintf(outf, "\\%s ", r->name);
   else
     fprintf(outf, "%c ", right);
}

void MathFracInset::Write(FILE *outf)
{ 
   fprintf(outf, "\\%s{", name);
   num->Write(outf);  
   fprintf(outf, "}{");
   den->Write(outf);  
   fprintf(outf, "}");
}

void MathParInset::Write(FILE *outf)
{
   if (!array) return;
   LyxMathIter data(array);
       
   data.Reset();
   while (data.OK()) {
      byte cx = data.GetChar();
      if (cx>=' ') {
//	 fprintf(outf, "OUT<%d %d>", data.FCode(), cx);
	 int ls;
	 byte *s = data.GetString(ls);
       
	 if (data.FCode()>=LM_TC_RM && data.FCode()<=LM_TC_TT)
	   fprintf(outf, "\\%s{", math_font_name[data.FCode()-LM_TC_RM]);
	 while (ls>0) {
	    if (data.FCode()==LM_TC_BSYM || data.FCode()==LM_TC_SYMB) {
	       latexkeys *l = 
		 lm_get_key_by_id(*s,(data.FCode()==LM_TC_SYMB)?LM_TK_SYM: LM_TK_BIGSYM);
	       if (l)
		 fprintf(outf, "\\%s ", l->name);
	       else 
		 fprintf(stderr, "Illegal symbol code[%u %d %d]", *s, ls, data.FCode());
	    } else {
	       // Is there a standard logical XOR?
	       if ((*s=='{' || *s=='}') && (data.FCode()!=LM_TC_TEX) ||
		   !(*s=='{' || *s=='}') && (data.FCode()==LM_TC_TEX))
		 fprintf(outf, "\\");
	       fprintf(outf, "%c", *s);
	    }
	    s++; ls--;
	 }
	 if (data.FCode()>=LM_TC_RM && data.FCode()<=LM_TC_TT)
	   fprintf(outf, "}");
	 
      } else     
      if (MathIsInset(cx)) {
	 LyxMathInset *p = data.GetInset();
	 if (cx==LM_TC_UP)
	   fprintf(outf, "^{");
	 if (cx==LM_TC_DOWN)
	   fprintf(outf, "_{");
	 p->Write(outf);
	 if (cx==LM_TC_UP || cx==LM_TC_DOWN)
	   fprintf(outf, "}");
	 data.Next();
      } else
      	switch(cx) {
	 case LM_TC_TAB:
	    {
	       fprintf(outf, " & ");
	       data.Next();
	       break;
	    }
	 case LM_TC_CR:
	    { 
	       fprintf(outf, "\\\\\n");
	       data.Next();
	       break;
	    }
	 default:
	   fprintf(stderr, "WMath Error: unrecognized code[%d]", cx);
	   return;
	}     
   }
}

void MathMatrixInset::Write(FILE *outf)
{
   rowst *r = row;
   //char s[80]; // unused

   if (GetType() == LM_OT_MATRIX)
     fprintf(outf, "\n\\begin{%s}{%s}\n", name, h_align);
   while (r) {
      array = r->array;
      MathParInset::Write(outf);
      if (r->next) 
	fprintf(outf,"\\\\\n");
      r = r->next;      
   }
   if (GetType() == LM_OT_MATRIX)
     fprintf(outf, "\n\\end{%s}\n", name);  
}

void mathed_write(MathParInset* p, FILE* outf, const char* label)
{  
   short mathed_env = p->GetType();

   if (mathed_env==LM_EN_INTEXT)  
     fprintf(outf," \\( ");
   else if (mathed_env==LM_EN_DISPLAY)
     fprintf(outf,"\n\\[\n");
   else 
     fprintf(outf,"\n\\begin{%s}\n", latex_mathenv[mathed_env]);
   
   if (label && mathed_env > LM_EN_DISPLAY)
     fprintf(outf,"\\label{%s}\n", label);

   p->Write(outf);
   
   if (mathed_env==LM_EN_INTEXT)
     fprintf(outf," \\)");
   else if (mathed_env==LM_EN_DISPLAY)
     fprintf(outf,"\n\\]\n");
   else 
   fprintf(outf,"\n\\end{%s}\n", latex_mathenv[mathed_env]);
   fflush(outf);
}
