/*
 *  File:        formula.h
 *  Purpose:     Implementation of formula inset
 *  Author:      Alejandro Aguilar Sierra <asierra@servidor.unam.mx> 
 *  Created:     January 1996
 *  Description: Allows the edition of math paragraphs inside Lyx. 
 *
 *  Copyright: (c) 1996, Alejandro Aguilar Sierra
 *
 *  Version: 0.4, Lyx project.
 *
 *   You are free to use and modify this code under the terms of
 *   the GNU General Public Licence version 2 or later.
 */

#include "config.h"
#include "formula.h"
#include "keybind.h"
#include "xdefinitions.h"
#include "math_draw.h"
#include "math_parser.h"
#include "file.h"
#include "buffer.h"
#include "lyx_cb.h"
#include "minibuffer.h"

extern void UpdateInset(Inset*);
extern void ToggleLockedInsetCursor(long, long, int, int);
extern void FitLockedInsetCursor(long, long, int, int);
extern int LockInset(UpdatableInset*);

extern GC canvasGC, mathGC, latexGC, cursorGC;
extern char *mathed_label;
extern GC LyXGetRedGC();
extern GC LyXGetBlueGC();

Bool greek_kb_flag = False;

//   variables, constants, symbol, bold, cal, tt, text
LyXFont Math_Fonts[] = {
    { LYX_ROMAN_FAMILY, LYX_MEDIUM_SERIES, LYX_ITALIC_SHAPE,},
//    { LYX_ROMAN_FAMILY, LYX_MEDIUM_SERIES, LYX_UP_SHAPE,},
    { LYX_SYMBOL_FAMILY, LYX_MEDIUM_SERIES, LYX_UP_SHAPE,},
    { LYX_SYMBOL_FAMILY, LYX_MEDIUM_SERIES, LYX_ITALIC_SHAPE,},
    { LYX_ROMAN_FAMILY, LYX_BOLD_SERIES, LYX_UP_SHAPE,},
    { LYX_SANS_FAMILY, LYX_MEDIUM_SERIES, LYX_ITALIC_SHAPE,},
    { LYX_TYPEWRITER_FAMILY, LYX_MEDIUM_SERIES, LYX_UP_SHAPE,},
    { LYX_ROMAN_FAMILY, LYX_MEDIUM_SERIES, LYX_UP_SHAPE,},
};

LyxMathCursor* mathcursor = NULL;

static int lfont_size = LYX_SIZE_NORMAL;


LyXFont WhichFont(short type, int size)
{
   LyXFont f;

   switch (size) {
    case LM_ST_DISPLAY:     
      if (type==LM_TC_BSYM) {
	 size = (lfont_size<LYX_SIZE_GIANT-1) ? lfont_size+2: LYX_SIZE_GIANT;
	 break;
      }
    case LM_ST_TEXT:
      size = lfont_size;
      break;
    case LM_ST_SCRIPT:
      size = (lfont_size>0) ? lfont_size-1: 0;
      break;
    case LM_ST_SCRIPTSCRIPT:
      size = (lfont_size>1) ? lfont_size-2: 0;
      break;
    default:
      fprintf(stderr, "Mathed Error: wrong font size: %d\n", size);
      size = lfont_size;
      break;
   }
   
   switch (type) {
    case LM_TC_SYMB:	     
      f = Math_Fonts[2];
      break;
    case LM_TC_BSYM:	     
      f = Math_Fonts[2];
      break;
    case LM_TC_VAR:
      f = Math_Fonts[0];
      break;
    case LM_TC_BF:
      f = Math_Fonts[3];
      break;
    case LM_TC_CAL:
      f = Math_Fonts[4];
      break;
    case LM_TC_TT:
      f = Math_Fonts[5];
      break;
//    case LM_TC_TEX: Math_Fonts[2]; break;
    case LM_TC_TEXTRM:
      f = Math_Fonts[6];
      break;
    default:
      f = Math_Fonts[1];
      break;
   }
   f.latex = LYX_FORMULA_LATEX;
   f.size = size;
   return f;
}


void mathed_set_font(short type, int size)
{
   if (!canvasGC) {
      cursorGC = LyXGetGreenGC();    //LyXGetSelectGC();
      canvasGC = LyXGetLightedGC();  //LyXGetClearGC();
      latexGC =  LyXGetRedGC();
      //mathGC = LyXGetBlueGC();
   }  
   mathGC = LyXGetGC(WhichFont(type, size));
}

int mathed_char_width(short type, int size, byte c)
{
   int t=0;
   t = LyXTextWidth(WhichFont(type, size), (char*)&c, 1);
   return t;
}

int mathed_string_width(short type, int size, byte* s, int ls)
{
   int t = 0;
   t = LyXTextWidth(WhichFont(type, size), (char*)s, ls);
   return t;
}

int mathed_string_height(short type, int size, byte* s, int ls, int& asc, int& des)
{
   LyXFont font = WhichFont(type, size);
   asc = des = 0;
   for (int i=0; i<ls; i++) {
      if (LyXDescent(font, s[i]) > des)
	des = LyXDescent(font, s[i]);
      if (LyXAscent(font, s[i]) > asc)
	asc = LyXAscent(font, s[i]);
   }
   return asc+des;
}

int mathed_char_height(short type, int size, byte c, int& asc, int& des)
{
   LyXFont font = WhichFont(type, size);
   asc = des = 0;
   des = LyXDescent(font, c);
   asc = LyXAscent(font, c);
   return asc+des;
}

static int left_delim='(', right_delim=')'; 

void mathed_define_delimiter(int left, int right)
{
   left_delim = left;
   right_delim = right;
}

InsetFormula::InsetFormula(bool display)
{
   par = new MathParInset;
   disp_flag = display;
   label = NULL;
   if (disp_flag) {
      par->SetType(LM_OT_PAR);
      par->SetStyle(LM_ST_DISPLAY);
   }
}

InsetFormula::InsetFormula(MathParInset *p)
{
   par = (p->GetType()>=LM_OT_MPAR) ? 
         new MathMatrixInset((MathMatrixInset*)p): 
         new MathParInset(p);
   
   disp_flag = (par->GetType()>0);
   label = NULL;
}

InsetFormula::~InsetFormula()
{
   delete par;
   if (label) delete label;
}

Inset* InsetFormula::Clone(void)
{
   InsetFormula* f = new InsetFormula(par);
   if (label) f->label = StringCopy(label);   
   return (Inset*)f;
}

void InsetFormula::Write(FILE *file)
{
   fprintf(file, "Formula ");
   Latex(file);
}

int InsetFormula::Latex(FILE *file)
{
   mathed_write(par, file, label);
   return 4; // temporary hardcoded number of \n  (Matthias)
}

void InsetFormula::Read(FILE *file)
{
   mathed_parser_file(file);   
   
   // Silly hack to read labels. 
   mathed_label = NULL;
   mathed_parse(0, NULL, &par);
   par->Metrics();
   disp_flag = (par->GetType()>0);
   
   if (mathed_label) {
      label = mathed_label;
      mathed_label = NULL;
   }
   
#ifdef DEBUG
   Write(stdout);
   fflush(stdout);
#endif
}

int InsetFormula::Ascent(LyXFont)
{
   return par->Ascent() + ((disp_flag) ? 8: 0);
}

int InsetFormula::Descent(LyXFont)
{
   return par->Descent() + ((disp_flag) ? 8: 0);
}

int InsetFormula::Width(LyXFont)
{
   return par->Width();
}

void InsetFormula::Draw(LyXFont f, unsigned long pm, int baseline, float &x)
{
   lfont_size = f.size;
   mathed_set_font(LM_TC_TEXTRM, lfont_size); // otherwise a segfault could occur
								// in some XDrawRectangles (i.e. matrix) (Matthias)
   if (mathcursor && mathcursor->GetPar()==par)   
     mathcursor->Draw(pm, (int)x, baseline);
   else
     par->Draw(pm, (int)x, baseline);   
   x += (float)par->Width();
   cursor_visible = false;
}

void InsetFormula::Edit(int x, int y)
{
   mathcursor = new LyxMathCursor(par);
   height = 5*par->Height()/4;
   width = par->Width();
   LockInset(this);
   UpdateInset(this);
   x += par->xo; // a stupid hack. Would be better to change all the mathed code (Matthias)
   y += par->yo; // a stupid hack. Would be better to change all the mathed code (Matthias)
   mathcursor->SetPos(x, y);
}
				       
void InsetFormula::InsetUnlock()
{
   if (mathcursor)
     delete mathcursor;
   mathcursor = NULL;
   UpdateInset(this);
}

// Now a symbol can be inserted only if the inset is locked
void InsetFormula::InsertSymbol(const char* s)
{ 
   if (!s || !mathcursor) return;
   if (!par->array) {
      par->array = new LyxArrayBase;
   }   
   latexkeys *l = in_word_set (s, strlen(s));
   if (!l) {
      fprintf(stderr, "Math warning: unrecognized command '%s'\n", s);
      mathcursor->Insert(new MathLatexInset(StringCopy(s)));
   } else {
      LyxMathTextCodes t = (l->token==LM_TK_BIGSYM) ? LM_TC_BSYM: LM_TC_SYMB;
      if (l->id<255 && (l->token==LM_TK_BIGSYM || l->token==LM_TK_SYM)) {
	 mathcursor->Insert((byte)l->id, t);
      } else {
	 if (l->token!=LM_TK_BIGSYM && l->token!=LM_TK_SYM)
	   LocalDispatch(LFUN_INSERT_MATH, l->name); 
	 else 
	   mathcursor->Insert(new MathLatexInset(l->name));
      }  
   }
   UpdateLocal();
}
   
void InsetFormula::ToggleInsetCursor()
{
  if (!mathcursor)
    return;
  int x, y, asc, desc;
  mathcursor->GetPos(x, y);
//  x -= par->xo; // a stupid hack. Would be better to change all the mathed code (Matthias)
  y -= par->yo; // a stupid hack. Would be better to change all the mathed code (Matthias)
  asc = LyXMaxAscent(WhichFont(LM_TC_TEXTRM, lfont_size));
  desc = LyXMaxDescent(WhichFont(LM_TC_TEXTRM, lfont_size));
  
  ToggleLockedInsetCursor(x, y, asc, desc);
  cursor_visible = !cursor_visible;
}

void InsetFormula::ShowInsetCursor(){
  if (!cursor_visible){
    int x, y, asc, desc;
    if (mathcursor){
      mathcursor->GetPos(x, y);
      //  x -= par->xo; // a stupid hack. Would be better to change all the mathed code (Matthias)
      y -= par->yo; // a stupid hack. Would be better to change all the mathed code (Matthias)
      asc = LyXMaxAscent(WhichFont(LM_TC_TEXTRM, lfont_size));
      desc = LyXMaxDescent(WhichFont(LM_TC_TEXTRM, lfont_size));
      FitLockedInsetCursor(x, y, asc, desc);
    }
    ToggleInsetCursor();
  }
}

void InsetFormula::HideInsetCursor(){
  if (cursor_visible)
    ToggleInsetCursor();
}



void InsetFormula::SetDisplay(bool dspf)
{
   fprintf(stderr, "To disp %d %d\n", dspf, disp_flag); 
   if (dspf!=disp_flag) {
      if (dspf && par->GetType()<LM_OT_PAR) {
	    par->SetType(LM_OT_PAR);
	    par->SetStyle(LM_ST_DISPLAY);
      } else 
	if (!dspf && par->GetType()>=LM_OT_MPAR) { 
	   MathParInset *p = new MathParInset(par);
	   delete par;
	   par = p;
	   par->SetType(LM_OT_MIN);
	   par->SetStyle(LM_ST_TEXT);
	   if (mathcursor) {
	      fprintf(stderr, "Warning: Changing the display mode of "
		      "a locked inset.\n");
	      mathcursor->SetPar(par);
	      UpdateInset(this);
	   }
	}
      disp_flag = dspf;
   }   
}

void InsetFormula::UpdateLocal()
{
   par->Metrics();  // To inform lyx kernel the exact size 
                  // (there were problems with arrays).
   UpdateInset(this);
   //     mathcursor->Redraw();
}
   


void InsetFormula::InsetButtonRelease(int x, int y, int /*button*/)
{
   HideInsetCursor();
   x += par->xo; // a stupid hack. Would be better to change all the mathed code (Matthias)
   y += par->yo; // a stupid hack. Would be better to change all the mathed code (Matthias)
   mathcursor->SetPos(x, y);
   ShowInsetCursor();
}

void InsetFormula::InsetButtonPress(int /*x*/, int /*y*/, int /*button*/)
{

}

void InsetFormula::InsetKeyPress(XKeyEvent *)
{
   fprintf(stderr, "Used InsetFormula::InsetKeyPress.\n");
}

// Special Mathed functions
bool InsetFormula::SetNumber(bool numbf)
{
   if (disp_flag) {
      short type = par->GetType();
      bool oldf = (type==LM_OT_PARN || type==LM_OT_MPARN);
      if (numbf && !oldf) type++;
      if (!numbf && oldf) type--;
      par->SetType(type);
      return oldf;
   } else
     return false;
}


void LyxMathCursor::Hide()
{
   if (!cursor) 
      return; 
   // obsolete
}

void LyxMathCursor::Show()
{
   if (!cursor) 
      return; 
   // obsolete
}


char *InsetFormula::LocalDispatch(int action, char *arg)
{
   char *dispatch_result=NULL;
   static char macrobf[80];
   static int macroln = 0;
   static MathLatexInset* imacro=NULL;
   static LyxMathTextCodes varcode = LM_TC_MIN;       
   static short accent = 0;
   bool was_macro = false;
   
   HideInsetCursor();
   switch (action) {
    // --- Cursor Movements ---------------------------------------------
    case LFUN_RIGHT:
      {
	 mathcursor->Right();
	 break;
      }
    case LFUN_LEFT:
      {
	 mathcursor->Left();
	 break;
      }
    case LFUN_UP:
      mathcursor->Up();
      break;
    case LFUN_DOWN:
      mathcursor->Down();
      break;
    case LFUN_HOME:
      mathcursor->Home();
      break;
    case LFUN_END:
      mathcursor->End();
      break;
    case LFUN_DELETE_LINE_FORWARD:
      mathcursor->DelLine();
      UpdateInset(this);
      break;
    case LFUN_NEWLINE:
      if (par->GetType()>0)
	fprintf(stderr, "math new line\n");
      break;
    case LFUN_PROTECTEDNEWLINE:
      mathcursor->Insert(' ', LM_TC_CR);
      par = mathcursor->GetPar();
      UpdateLocal();     
      break;
    case LFUN_TAB:
      mathcursor->Insert(' ', LM_TC_TAB);
      UpdateInset(this);
      break;     
    case LFUN_BACKSPACE:
      if (imacro && macroln>0) {
	 macrobf[--macroln] = '\0';
	 imacro->Metrics();
	 was_macro = true;
	 UpdateInset(this);
	 break;
      } else
	if (!mathcursor->Left()) break;
    case LFUN_DELETE:
      mathcursor->Delete();       
      UpdateInset(this);
      break;    
    case LFUN_GETXY:
//      sprintf(dispatch_buffer, "%d %d",);
//      dispatch_result = dispatch_buffer;
      break;
    case LFUN_SETXY:
      {
	 int x, y, x1, y1;
         sscanf(arg, "%d %d", &x, &y);
	 par->GetXY(x1, y1);
	 mathcursor->SetPos(x1+x, y1+y);
      }
      break;
      
    // --- accented characters ------------------------------

    case LFUN_UMLAUT: accent = LM_ddot; break;
    case LFUN_CIRCUMFLEX: accent = LM_hat; break;
    case LFUN_GRAVE: accent = LM_grave; break;
    case LFUN_ACUTE: accent = LM_acute; break;
    case LFUN_TILDE: accent = LM_tilde; break;
    case LFUN_MACRON: accent = LM_bar; break;
    case LFUN_DOT: accent = LM_dot; break;
    case LFUN_CARON: accent = LM_check; break;
    case LFUN_BREVE: accent = LM_breve; break;
    case LFUN_VECTOR: accent = LM_vec; break; 
      
    // Greek keyboard      
    case LFUN_GREEK_TOGGLE:
    {
       greek_kb_flag = (greek_kb_flag) ? False: True;
       if (greek_kb_flag)
	 minibuffer.Set("Math greek keyboard");
       else
	 minibuffer.Set("Normal keyboard");
       break;
    }  
   
      //  Math fonts 
    case LFUN_BOLD:  varcode = LM_TC_BF; break;
    case LFUN_SANS:  varcode = LM_TC_SF; break;
    case LFUN_EMPH:  varcode = LM_TC_CAL; break;
    case LFUN_CODE: varcode = LM_TC_TT; break;
    case LFUN_DEFAULT:  varcode = LM_TC_VAR; break;
    case LFUN_TEX: varcode = LM_TC_TEX; break;

    case LFUN_MATH_NUMBER:
    {
       if (disp_flag) {
	  short type = par->GetType();
	  bool oldf = (type==LM_OT_PARN || type==LM_OT_MPARN);
	  if (oldf) {
	     type--;
	     if (label) {
		delete label;
		label = NULL;
	     }
	     minibuffer.Set("No number");  
	  } else {
	     type++;
             minibuffer.Set("Number");
	  }
	  par->SetType(type);
	  
       }
       break;
    }
      
    case LFUN_INSERT_SYMBOL:
    {
       InsertSymbol(arg);
       break;
    }
    
    case LFUN_INSERT_MATRIX:
    { 
       int m, n;
       char s[80];
       sscanf(arg, "%d %d %s", &m, &n, s);
       LyxMathInset *p = new MathMatrixInset(m, n);      
       if (mathcursor && p) {
	  if ((int)strlen(s)>m)
	    p->SetAlign(s[0], &s[1]);
	  mathcursor->Insert(p, LM_TC_ACTIVE_INSET);
	  UpdateLocal();     
       }
       break;
    }
      
    case LFUN_INSERT_MATH:
     {
	LyxMathInset *p = NULL;
	latexkeys *l = in_word_set (arg, strlen(arg));
	if (!l) {
	   if (strcmp(arg, "delim")==0) {
	      p = new MathDelimInset(left_delim, right_delim);
	      p->SetData(new LyxArrayBase);
	   } else
	     fprintf(stderr, "Math warning: unrecognized command '%s'\n", arg);
	} else {
	   switch (l->token) {
	    case LM_TK_FRAC: 
	      p = new MathFracInset; 
	      ((MathFracInset*)p)->SetData(new LyxArrayBase, new LyxArrayBase);
	      break;
	    case LM_TK_SQRT: 
	      p = new MathSqrtInset; 
	      p->SetData(new LyxArrayBase);
	      break;
	   } 
	}
	if (mathcursor && p) {
	   mathcursor->Insert(p, LM_TC_ACTIVE_INSET);
	   UpdateLocal();
	}
	break;
     }

    case LFUN_INSERT_LABEL:
    {
       if (par->GetType()>=LM_OT_PARN)  
	 label = StringCopy(arg);
       break;
    }
    default:
      if (action==-1 && arg)  {
	 char c = arg[0];	
	 if ('A' <= c  && c<='Z' || 'a' <= c  && c<='z') {
	    
	    if (accent) {
	       latexkeys *l = lm_get_key_by_id(accent, LM_TK_ACCENT);
	       if (l) 
		 mathcursor->Insert(new MathLatexInset(StringCopy(l->name)));
	       accent = 0;
	    }
	    
	    if (varcode==LM_TC_TEX) {	      
	       macroln = 0;
	       macrobf[0] = '\0';
	       was_macro =  true;
	       minibuffer.Set("Macro mode");
	       imacro = new MathLatexInset(&macrobf[0]);
	       mathcursor->Insert (imacro);
	       varcode = LM_TC_MIN;
	    }	    
	    if (imacro) {
	       macrobf[macroln+1] = macrobf[macroln];
	       macrobf[macroln++] = c;
	       was_macro = true;
	       imacro->Metrics();
	    } else {
	       if (!varcode) {
		  short f = mathcursor->GetFCode();
		  varcode = MathIsAlphaFont(f) ? (LyxMathTextCodes)f:LM_TC_VAR;
	       }
	       mathcursor->Insert(c, (greek_kb_flag) ? LM_TC_SYMB: varcode);
	       varcode = LM_TC_MIN; 
	    }
	 } else 
	   if (strchr("!,:;{}", c) && (varcode==LM_TC_TEX || imacro)) {
	      mathcursor->Insert(c, LM_TC_TEX);
	      varcode = LM_TC_MIN;
	   } else
	   if (('0'<=c && c<='9') || strchr(":!|[]().,{}", c)) 
	      mathcursor->Insert(c, LM_TC_CONST);
	 else
	   if (strchr("+/-*<>=", c))
	   mathcursor->Insert(c, LM_TC_BOP);
	 else   
	   if (c=='_' || c=='^') {
	      MathParInset *p = new MathParInset(LM_ST_SCRIPT, "", LM_OT_SCRIPT);
	      LyxArrayBase* a = new LyxArrayBase();
	      p->SetData(a);
	      mathcursor->Insert (p, (c=='_') ? LM_TC_DOWN: LM_TC_UP);
	   } else
	   if (c==' ') {
	      mathcursor->Insert (new MathSpaceInset(3));
	   } else
	   if (c=='\'') {
	      mathcursor->Insert (c, LM_TC_VAR);
	   } else
	   if (c=='\\') {
	      varcode = LM_TC_TEX;
	   } 
	 	   
	 UpdateLocal();
      } else 
	fprintf(stderr, "ACT2[%d] ", action);
   }
   if (imacro && !was_macro) {
      if (macroln==0) {
	 mathcursor->Left();
	 mathcursor->Delete();       
      } else
	imacro->SetName(StringCopy(macrobf));
      imacro = NULL;
   }
   ShowInsetCursor();
   return dispatch_result;
}


void
MathLatexInset::Draw(long unsigned int pm, int x, int y)
{ 
   LyXFont  font = WhichFont(LM_TC_TEXTRM, size);
   font.latex = LYX_LATEX;
   XDrawString(fl_display, pm, LyXGetGC(font), x, y, name, strlen(name));
}


void MathLatexInset::Metrics() 
{
      ln = strlen(name);
      LyXFont  font = WhichFont(LM_TC_TEXTRM, size);
      font.latex = LYX_LATEX;
      width = LyXTextWidth(font, name, ln);
      mathed_string_height(LM_TC_TEXTRM, size, (byte*)name, strlen(name), ascent, descent);
}


// If a mathinset exist at cursor pos, just lock it.
// Otherwise create a new one and lock it.
// This function should not be here, but... 
bool OpenMathInset()
{
   if (!bufferlist.current()->screen)
     return false;
  
   extern bool selection_possible;

   LyXCursor cursor = bufferlist.current()->text->cursor;
   if (cursor.pos < cursor.par->Last() 
       && cursor.par->GetChar(cursor.pos) == LYX_META_INSET
       && cursor.par->GetInset(cursor.pos)
       && cursor.par->GetInset(cursor.pos)->Editable()) {

      Inset* tmpinset = cursor.par->GetInset(cursor.pos);
      if (tmpinset->LyxCode()==LYX_MATH_CODE) {
	 selection_possible = false;
	 tmpinset->Edit(0, 0);
	 return true;
      } 
   } else {
      bufferlist.open_new_inset(new InsetFormula);
      return true;
   }

   return false;
}

