/* buffer.h
   Declarations of classes for holding files.

   Queries, and work-to-do: see "??"s.

    Author: John Collins, collins@phys.psu.edu.
    22 Jan 96

    (C) John Collins & Penn State University.

*/

#define BUFFER_H
//#include <iostream.h>
#include <stdio.h>
#include "errors.h"
#ifndef SOS_H
#include "sos.h"
#endif


#define BUFSIZE 15000
   /* Size of input buffer when reading files.  This is a round number
      less than half the maximum value of a signed integer, so that
      all character arrays will be addressable by a signed integer
      index, and two indices to an array may be added without overflow.
   */



// Null string.  Use pointer to it to avoid NULL pointers to strings.
extern char Blank[];
// Similarly an unattached line, and a pointer to it:
class Posn;
class Line;
extern Line DummyLine;
extern Posn Nowhere;
#define MYNULL &Blank[0]


class Line {
   /* Designed to hold one of a linked set of lines of text.
   */


   public:
      Line *prev, *next;
      unsigned int alloc, len, LineNum;
      char *data;
      /* alloc   = number of characters allocated, and
         len     = number actually used for text.
         LineNum = Line number in file.  Usually set when reading into
                   a sequence of lines from a file.
         data    = pointer to the text contained.  Possibly it is NULL.
         In normal use, the text, pointed to by data, will be terminated
         by a 0, and there must be room to do this.  Thus always
         data[len] == 0, and alloc >= len+1.
         In normal conditions this object is to contain one line, complete
         with the terminating \n (if it exists in the corresponding
         file). Excessively long lines in a file will be split into a
         sequence of these objects, and only the last will contain a '\n'.
         The count of characters, len, does not include the 0, but does
         include the '\n', if any.

         data should always point to something; it should never be
         NULL.  So there's an empty string in the class definition.
         If no data is allocated then data points to it (i.e., data==MYNULL),
         and alloc==0.

      */


   private:

      void Init();

   public:

      Line();
      /* Constructs an empty unattached line.*/
      Line(Line *&last, char *s);
      /* Constructs a line attached to the line last, if it exists.
         Adjusts all the pointers, and updates last to point to this.
         Initializes the data to be equal to the string.
      */
      Line(Line *&last, unsigned int size, char *s);
      /*  The same except the number of characters is defined, so that
          the string is not necessarily zero terminated.
      */
      virtual ~Line();
      virtual void Remove(); //Remove from list
      virtual void Clear();  //Clear the data and delete its space
      virtual void Attach(Line *&last);
      /*  Attach to a list after the element last.
          Set last to point to this.
          Ignore values of prev and next.
      */
      inline char operator[](int i) const {return data[i];}
};


class Buffer {
   /* To hold a sequence of lines, such as would be read in from a file.
   */

   public:
      Line *first;
      /* Pointer to the first line of the text.
      */
      unsigned int NumLastRead;
      /* Number of lines read in last read operation.
      */
      int ErrorCode;
      // Error code from last read. Zero => no error.


   private:
      inline void Init(){
         /* To be called only when setting up the Buffer, to set the elements
            to sensible defaults.
         */
         first = NULL;
         NumLastRead = 0;
         ErrorCode = 0;
      }


   public:
      Buffer();

//      Buffer(istream &source);
      Buffer(FILE *source);
      /* Create the Buffer by reading from the file, which is assumed to be
         open for reading.  Leave the file open after reading it.
         See method Read for specifications of the reading.
      */
      ~Buffer();

//      virtual int Read (istream &source = cin);
      virtual int Read (FILE *source = stdin);
      /* Fill the Buffer from the file source, which is assumed to be open
         for reading.  Append the results to any previous contents of the
         Buffer.  Start at the current position of the file, and leave the
         file open after the read, positioned at the end.

         Put one line of the file in each line of the Buffer.
         But if a line of the file is too long, then it is to span two or
         more lines of the Buffer.  New line characters are included at the
         end of each line of the Buffer when they are in the file.
         A future version will handle all of the UNIX, MSDOS and Macintosh
         conventions for the end of line character(s).  Note the MSDOS files
         will be OK on UNIX, but there will be an extra ^M character at the
         end of each line.

         Put a NULL character at the end of each Line.

         Return 0 for success or a non-zero error code for failure.
      */

      virtual int Write (sos &dest = lite_cout) const;
      /* Write the Buffer into the file dest, which is assumed to be open
         for writing.  Put one line of the file in each line of the Buffer.
         Start at the current position of the file, and leave the
         file open after the write, positioned at the end.
         Just put the characters in the Buffer into the file.

         Assumes the lines are null-terminated.

         A future version will handle all of the UNIX, MSDOS and Macintosh
         conventions for the end of line character(s).

         Return 0 for success or a non-zero error code for failure.
      */

      virtual void Clear();
      // Delete all the data in the Buffer.

      unsigned int NumLines() const;
      /* Number of lines in the Buffer.
         A future version will take into account the vagaries of incomplete
         lines.
      */

};



struct Posn {
   Line *l;
   unsigned int index;
/* Points to a character in a Buffer of Lines.
   Assume index is within range, always.

   ??Ultimately, remove need to test l!=NULL,
   and require this to be always true.
*/

   inline Posn() {l = NULL; index = 0;}
   inline Posn(Posn &p) {l = p.l;  index = p.index;};
   inline Posn(Line *Pl, unsigned int i = 0) {l = Pl;  index = i;};

   char c() const { if (l) return l->data[index];
            else return 0;}

   char operator[](int offset) const;
   /* Return the character offset positions beyond the current position.
      Assume offset >=0, as for array.  (We can relax that later.)
   */

   inline int eof() {return (index >= l->len) && (l->next == NULL); }
      // End-of-file
   inline int eol() {return (index >= l->len) ; }
      // End-of-line
   inline void NextLine() { if (l->next) {l=l->next; index = 0;}}
      // Advance to first character of next line.
   void inc();
      // Skip to next character.  Ultimately, this should be inline.??
   void dec();
      /* Skip to previous character.
         If I am at the beginning of the file, stay put.
         Ultimately, this should be inline.??
      */

   void Show(sos &f = lite_cerr) const;
   // Show line # and position
   void ShowContext(sos &f = lite_cerr) const;
   int LineNum() const;

};

void Error (const char *s, const Posn &where);
   // Show error message and context on stderr

void Error (const char *s);
   // Show error message on stderr

void ShowContext (const Posn &where, FILE *f = stderr);
   // Show line contents

void ShowWhere (const Posn &where, FILE *f = stderr);
   // Show line number and 1-based index.
