/* This software is Copyright 1995 by Karl-Johan Johnsson
 *
 * Permission is hereby granted to copy, reproduce, redistribute or otherwise
 * use this software as long as: there is no monetary profit gained
 * specifically from the use or reproduction of this software, it is not
 * sold, rented, traded or otherwise marketed, and this copyright notice is
 * included prominently in any copy made. 
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. ANY USE OF THIS
 * SOFTWARE IS AT THE USER'S OWN RISK.
 */
#include "global.h"
#include <X11/Shell.h>
#include "ahead.h"
#include "kill.h"
#include "misc.h"
#include "newsrc.h"
#include "pixmaps.h"
#include "procs.h"
#include "psetup.h"
#include "tag.h"
#include "thread.h"
#include "util.h"
#include "viewer.h"
#include "widgets.h"
#include "xutil.h"
#include "../Widgets/ArtText.h"
#include "../Widgets/ArtTree.h"
#include "../Widgets/Knapp.h"
#include "../Widgets/Layout.h"
#include "../Widgets/Manager.h"
#include "../Widgets/MenuKnapp.h"
#include "../Widgets/Message.h"
#include "../Widgets/Sash.h"
#include "../Widgets/ScrBar.h"
#include "../Widgets/ScrList.h"
#include "../Widgets/Util.h"

#if HAVE_XMU && XtSpecificationRelease > 4
#include <X11/Xmu/Editres.h>
#endif

void set_message(char *message, int beep)
{
    MessageSetAndRedraw(main_widgets.message_line,
			message, beep && global.bell);
}

#define DESCRIPTION(desc)	((desc) ? (desc) : "")
#define ARTTREE_STEPSIZE	16
#define LINE_LEN		128

struct main_widgets main_widgets;

static void print_group_info_subscr(char *buffer, GROUP *group)
{
    sprintf(buffer, "%c %-42.42s %6ld %-64.64s",
	    group->ahead_flag ? thread_ahead_char(group) : ' ',
	    group->name, group->no_unread,
	    DESCRIPTION(group->description));
}

static void print_group_info_all(char *buffer, GROUP *group)
{
    long	num;

    if (group->last_art == 0)
	num = 0;
    else {
	num = group->last_art - group->first_art + 1;
	if (num < 0)
	    num = 0;
    }

    sprintf(buffer, "%3s %c %-42.42s %6ld %-64.64s",
	    group->subscribed ? "[s]" : "   ",
	    group->ahead_flag ? thread_ahead_char(group) : ' ',
	    group->name,
	    num < 0 ? 0 : num,
	    DESCRIPTION(group->description));
}

static void print_subj_info(char *buffer, SUBJECT *subj)
{
    Boolean	indent = False;
    SUBJECT	*loop = subj->prev;
    ARTICLE	*first = first_unread_article_with_subject(subj);
    char	*author;

    if (subj->no_unread == 0)
	sprintf(buffer, "     ");
    else
	sprintf(buffer, "%3ld  ", subj->no_unread);
    if (subj->has_tagged)
	buffer[3] = '*';

    while (loop && loop->thread == subj->thread) {
	if (loop->disp >= 0) {
	    indent = True;
	    break;
	}
	loop = loop->prev;
    }

    if (first && first->tree_data.label)
	author = first->tree_data.label;
    else
	author = "";

    if (first && A_PARENT(first) && A_PARENT(first)->subject == subj) {
	if (indent)
	    sprintf(buffer + 5, "  Re: %-50.50s %-64.64s",
		    subj->subject, author);
	else
	    sprintf(buffer + 5, "Re: %-52.52s %-64.64s",
		    subj->subject, author);
    } else {
	if (indent)
	    sprintf(buffer + 5, "  %-54.54s %-64.64s",
		    subj->subject, author);
	else
	    sprintf(buffer + 5, "%-56.56s %-64.64s",
		    subj->subject, author);
    }
}

static void knapp_set_mode(NewsMode mode)
{
    Arg	arg;

    switch (mode) {
    case NewsModeDisconnected:
	XtSetMappedWhenManaged(main_widgets.knapp[2], False);
	XtSetMappedWhenManaged(main_widgets.knapp[6], False);
	XtSetMappedWhenManaged(main_widgets.knapp[7], False);
	XtSetMappedWhenManaged(main_widgets.knapp[8], False);

	KnappSetLabelNo(main_widgets.knapp[0], 0, True);
	KnappSetLabelNo(main_widgets.knapp[1], 0, True);

	KnappSetSensitive(main_widgets.knapp[3], False);
	KnappSetSensitive(main_widgets.knapp[5], False);
	KnappSetSensitive(main_widgets.knapp[10], False);
	KnappSetSensitive(main_widgets.knapp[11], False);

	XtSetArg(arg, XtNmenuName, "miscshell2");
	break;
    case NewsModeConnected:
	XtSetMappedWhenManaged(main_widgets.knapp[2], True);
	XtSetMappedWhenManaged(main_widgets.knapp[6], True);
	XtSetMappedWhenManaged(main_widgets.knapp[7], True);
	XtSetMappedWhenManaged(main_widgets.knapp[8], True);

	KnappSetLabelNo(main_widgets.knapp[0], 0, True);
	KnappSetLabelNo(main_widgets.knapp[1], 1, True);
	KnappSetLabelNo(main_widgets.knapp[2], 0, True);
	KnappSetLabelNo(main_widgets.knapp[6], 0, True);
	KnappSetLabelNo(main_widgets.knapp[7], 0, True);
	KnappSetLabelNo(main_widgets.knapp[8], 0, True);

	KnappSetSensitive(main_widgets.knapp[3], True);
	KnappSetSensitive(main_widgets.knapp[5], True);
	KnappSetSensitive(main_widgets.knapp[10], False);
	KnappSetSensitive(main_widgets.knapp[11], False);

	XtSetArg(arg, XtNmenuName, "miscshell2");
	break;
    case NewsModeGroup:
	XtSetMappedWhenManaged(main_widgets.knapp[2], True);
	XtSetMappedWhenManaged(main_widgets.knapp[6], True);
	XtSetMappedWhenManaged(main_widgets.knapp[7], True);
	XtSetMappedWhenManaged(main_widgets.knapp[8], True);

	KnappSetLabelNo(main_widgets.knapp[0], 1, True);
	KnappSetLabelNo(main_widgets.knapp[1], 2, True);
	KnappSetLabelNo(main_widgets.knapp[2], 1, True);
	KnappSetLabelNo(main_widgets.knapp[6], 1, True);
	KnappSetLabelNo(main_widgets.knapp[7], 1, True);
	KnappSetLabelNo(main_widgets.knapp[8], 1, True);

	KnappSetSensitive(main_widgets.knapp[3], True);
	KnappSetSensitive(main_widgets.knapp[5], True);
	KnappSetSensitive(main_widgets.knapp[10], True);
	KnappSetSensitive(main_widgets.knapp[11], True);

	XtSetArg(arg, XtNmenuName, "miscshell1");
	break;
    case NewsModeThread:
	XtSetMappedWhenManaged(main_widgets.knapp[2], True);
	XtSetMappedWhenManaged(main_widgets.knapp[6], True);
	XtSetMappedWhenManaged(main_widgets.knapp[7], True);
	XtSetMappedWhenManaged(main_widgets.knapp[8], True);

	KnappSetLabelNo(main_widgets.knapp[0], 1, False);
	KnappSetLabelNo(main_widgets.knapp[1], 3, True);
	KnappSetLabelNo(main_widgets.knapp[2], 1, False);
	KnappSetLabelNo(main_widgets.knapp[6], 1, False);
	KnappSetLabelNo(main_widgets.knapp[7], 1, True);
	KnappSetLabelNo(main_widgets.knapp[8], 1, True);

	KnappSetSensitive(main_widgets.knapp[3], True);
	KnappSetSensitive(main_widgets.knapp[5], True);
	KnappSetSensitive(main_widgets.knapp[10], True);
	KnappSetSensitive(main_widgets.knapp[11], True);

	XtSetArg(arg, XtNmenuName, "miscshell1");
	break;
    case NewsModeAllgroups:
	XtSetMappedWhenManaged(main_widgets.knapp[2], True);
	XtSetMappedWhenManaged(main_widgets.knapp[6], True);
	XtSetMappedWhenManaged(main_widgets.knapp[7], True);
	XtSetMappedWhenManaged(main_widgets.knapp[8], False);

	KnappSetLabelNo(main_widgets.knapp[0], 1, True);
	KnappSetLabelNo(main_widgets.knapp[1], 4, True);
	KnappSetLabelNo(main_widgets.knapp[2], 2, True);
	KnappSetLabelNo(main_widgets.knapp[6], 0, True);
	KnappSetLabelNo(main_widgets.knapp[7], 2, True);

	KnappSetSensitive(main_widgets.knapp[3], True);
	KnappSetSensitive(main_widgets.knapp[5], True);
	KnappSetSensitive(main_widgets.knapp[10], False);
	KnappSetSensitive(main_widgets.knapp[11], True);

	XtSetArg(arg, XtNmenuName, "miscshell2");
	break;
    case NewsModeNewgroups:
	XtSetMappedWhenManaged(main_widgets.knapp[2], True);
	XtSetMappedWhenManaged(main_widgets.knapp[6], False);
	XtSetMappedWhenManaged(main_widgets.knapp[7], False);
	XtSetMappedWhenManaged(main_widgets.knapp[8], False);

	KnappSetLabelNo(main_widgets.knapp[0], 1, True);
	KnappSetLabelNo(main_widgets.knapp[1], 4, True);
	KnappSetLabelNo(main_widgets.knapp[2], 2, True);

	KnappSetSensitive(main_widgets.knapp[3], False);
	KnappSetSensitive(main_widgets.knapp[5], True);
	KnappSetSensitive(main_widgets.knapp[10], False);
	KnappSetSensitive(main_widgets.knapp[11], False);

	XtSetArg(arg, XtNmenuName, "miscshell2");
	break;
    }

    XtSetValues(main_widgets.knapp[3], &arg, 1);
}

void setNewsModeDisconnected(void)
{
    global.mode = NewsModeDisconnected;
    XtUnmanageChild(main_widgets.arttree);
    XtUnmanageChild(main_widgets.thread_list);
    XtUnmanageChild(main_widgets.group_list);
    knapp_set_mode(NewsModeDisconnected);
    ArtTextClearLines(main_widgets.text);
    XtSetSensitive(main_widgets.top_vbar, False);
    XtSetSensitive(main_widgets.top_hbar, False);
    XtSetSensitive(main_widgets.text_scrbar, False);
    ScrBarSetLengthsAndPos(main_widgets.top_vbar, 0, 0, 0);
    ScrBarSetLengthsAndPos(main_widgets.top_hbar, 0, 0, 0);
    ScrBarSetLengthsAndPos(main_widgets.text_scrbar, 0, 0, 0);
}

void update_group_entry(GROUP *group)
{
    char	buffer[LINE_LEN];

    switch (global.mode) {
    case NewsModeConnected:
	if (group->disp >= 0) {
	    print_group_info_subscr(buffer, group);
	    ScrListSetLine(main_widgets.group_list, group->disp,
			   buffer, None);
	}
	break;
    case NewsModeAllgroups:
	if (group->disp >= 0) {
	    print_group_info_all(buffer, group);
	    ScrListSetLine(main_widgets.group_list, group->disp,
			   buffer, None);
	}
	break;
    case NewsModeDisconnected:
    case NewsModeNewgroups:
    case NewsModeGroup:
    case NewsModeThread:
	break;
    }
}

void setNewsModeConnected(void)
{
    Arg		args[4];
    long	i, n, sel = -1;
    long	shown, first;
    Boolean	flag = False;

    global.mode = NewsModeConnected;

    XtUnmanageChild(main_widgets.arttree);
    XtUnmanageChild(main_widgets.thread_list);
    ArtTextClearLines(main_widgets.text);
    ScrListClearLines(main_widgets.thread_list);
    ScrListClearLines(main_widgets.group_list);
    XtSetSensitive(main_widgets.top_vbar, True);
    XtSetSensitive(main_widgets.top_hbar, False);
    XtSetSensitive(main_widgets.text_scrbar, False);
    ScrBarSetLengthsAndPos(main_widgets.text_scrbar, 0, 0, 0);
    ScrBarSetLengthsAndPos(main_widgets.top_hbar, 0, 0, 0);
    XtSetArg(args[0], XtNstepSize, 1);
    XtSetValues(main_widgets.top_vbar, args, 1);
    ScrBarSetLengthsAndPos(main_widgets.top_vbar, 0, 0, 0);
    knapp_set_mode(NewsModeConnected);
    update_misc_menu(NewsModeConnected);
    XtSetArg(args[0], XtNatMostOne, True);
    XtSetArg(args[1], XtNatLeastOne, True);
    XtSetArg(args[2], XtNallowDnd, False);
    XtSetValues(main_widgets.group_list, args, 3);
    XtManageChild(main_widgets.group_list);

    for (n = 0 ; n < global.no_groups ; n++) {
	GROUP	*group = global.groups[n];
	char	buffer[LINE_LEN];

	if (!group->subscribed)
	    break;

	calc_no_unread(group);
	if (group == global.curr_group)
	    flag = True;
	if (group->no_unread > 0) {
	    print_group_info_subscr(buffer, group);
	    group->disp = ScrListAddLine(main_widgets.group_list,
					 buffer, None);
	    if (flag) {
		sel = group->disp;
		flag = False;
	    }
	} else {
	    group->disp = -1;
	}
    }

    while (n < global.no_groups)
	global.groups[n++]->disp = -1;

    if (sel < 0) {
	global.curr_group = NULL;
	for (n = 0 ; n < global.no_groups ; n++) {
	    if (!global.groups[n]->subscribed) {
		sel = 0;
		break;
	    } else if (global.groups[n]->disp == 0) {
		global.curr_group = global.groups[n];
		sel = 0;
		break;
	    }
	}
    }

    if (sel >= 0)
	ScrListSetSelected(main_widgets.group_list, sel, True);
    ScrListGetFirstShownSize(main_widgets.group_list, NULL, &shown, &i);
    first = 0;
    if (sel >= shown) {
	first = sel;
	if (first + shown >= i) {
	    first = i - shown;
	    if (first < 0)
		first = 0;
	}
	ScrListSetFirst(main_widgets.group_list, first);
    }
    ScrBarSetLengthsAndPos(main_widgets.top_vbar, i, shown, first);

    set_standard_message();
}

void update_subj_entry(SUBJECT *subj)
{
    long	row = subj->disp;
    char	buffer[LINE_LEN];

    if (row >= 0) {
	print_subj_info(buffer, subj);
	ScrListSetLine(main_widgets.thread_list, row, buffer,
		       global.hot_pixmaps[subj->hot]);
    }
}

void setNewsModeGroup(int show_all)
{
    Arg		args[8];
    SUBJECT	*loop;
    long	n, shown;

    XtSetSensitive(main_widgets.top_vbar, True);
    XtSetSensitive(main_widgets.top_hbar, False);
    XtSetSensitive(main_widgets.text_scrbar, True);

    if (global.mode != NewsModeThread) {
	global.mode = NewsModeGroup;
	XtUnmanageChild(main_widgets.arttree);
	XtUnmanageChild(main_widgets.group_list);
	ScrListClearLines(main_widgets.group_list);
	ScrListClearLines(main_widgets.thread_list);
	XtManageChild(main_widgets.thread_list);
	ArtTreeSetTree(main_widgets.arttree, NULL);
	knapp_set_mode(NewsModeGroup);
	update_misc_menu(NewsModeGroup);
	ScrBarSetLengthsAndPos(main_widgets.top_hbar, 0, 0, 0);

	for (n = 0, loop = get_subjects(main_thr) ; loop ; loop = loop->next) {
	    char	buffer[LINE_LEN];

	    if (!show_all && loop->no_unread == 0) {
		loop->disp = -1;
		continue;
	    }

	    print_subj_info(buffer, loop);
	    loop->disp =
		ScrListAddLine(main_widgets.thread_list, buffer,
			       global.hot_pixmaps[loop->hot]);
	    if (n == 0)
		global.curr_subj = loop;

	    n++;
	}

	ScrListGetFirstShownSize(main_widgets.thread_list, NULL, &shown, &n);
	XtSetArg(args[0], XtNstepSize, 1);
	XtSetValues(main_widgets.top_vbar, args, 1);
	ScrBarSetLengthsAndPos(main_widgets.top_vbar, n, shown, 0);

	/* message already set by get_articles */
    } else {
	long	first;

	global.mode = NewsModeGroup;
	XtUnmanageChild(main_widgets.arttree);
	ArtTreeSetTree(main_widgets.arttree, NULL);
	XtManageChild(main_widgets.thread_list);
	set_curr_subj(global.curr_subj);
	ScrListGetFirstShownSize(main_widgets.thread_list, &first, &shown, &n);
	ScrBarSetLengthsAndPos(main_widgets.top_hbar, 0, 0, 0);
	ScrBarSetLengthsAndPos(main_widgets.top_vbar, n, shown, first);
	XtSetArg(args[0], XtNstepSize, 1);
	XtSetValues(main_widgets.top_vbar, args, 1);
	knapp_set_mode(NewsModeGroup);
	set_standard_message();
    }
}

void setNewsModeThread(void)
{
    Arg		args[8];
    Dimension	s_w, s_h, w, h;

    global.mode = NewsModeThread;
    XtUnmanageChild(main_widgets.group_list);
    XtUnmanageChild(main_widgets.thread_list);
    ArtTreeSetTree(main_widgets.arttree, NULL);
    XtSetArg(args[0], XtNtree, global.curr_subj->thread);
    XtSetArg(args[1], XtNsuspend, True);
    XtSetArg(args[2], XtNx, 0);
    XtSetArg(args[3], XtNy, 0);
    XtSetValues(main_widgets.arttree, args, 4);
    set_tree_stuff(global.curr_subj->thread);
    mark_tagged_articles(global.curr_subj->thread);
    XtSetArg(args[0], XtNwidth, &w);
    XtSetArg(args[1], XtNheight, &h);
    XtGetValues(main_widgets.arttree, args, 2);
    XtSetArg(args[0], XtNwidth, &s_w);
    XtSetArg(args[1], XtNheight, &s_h);
    XtGetValues(main_widgets.top_manager, args, 2);
    XtSetSensitive(main_widgets.top_vbar, True);
    XtSetSensitive(main_widgets.top_hbar, True);
    XtSetSensitive(main_widgets.text_scrbar, True);
    ScrBarSetLengthsAndPos(main_widgets.top_hbar, w, s_w, 0);
    ScrBarSetLengthsAndPos(main_widgets.top_vbar, h, s_h, 0);
    XtSetArg(args[0], XtNstepSize, ARTTREE_STEPSIZE);
    XtSetValues(main_widgets.top_hbar, args, 1);
    XtSetValues(main_widgets.top_vbar, args, 1);
    if (global.curr_art) {
	ArtTreeNodeMakeVisible(main_widgets.arttree,
			       (ART_TREE_NODE *)global.curr_art);
	ArtTreeNodeSetSelected(main_widgets.arttree,
			       (ART_TREE_NODE *)global.curr_art, True);
    }
    XtManageChild(main_widgets.arttree);
    XtSetArg(args[0], XtNsuspend, False);
    XtSetValues(main_widgets.arttree, args, 1);
    knapp_set_mode(NewsModeThread);
    set_standard_message();
}

void setNewsModeAllgroups(void)
{
    Arg		args[8];
    long	n = global.no_groups;
    long	shown, first = -1;

    global.mode = NewsModeAllgroups;

    XtSetArg(args[0], XtNnAlloc, global.no_groups + 4);
    XtSetValues(main_widgets.group_list, args, 1);

    XtSetSensitive(main_widgets.top_vbar, True);
    XtSetSensitive(main_widgets.top_hbar, False);
    XtSetSensitive(main_widgets.text_scrbar, False);

    XtUnmanageChild(main_widgets.arttree);
    XtUnmanageChild(main_widgets.thread_list);
    ScrListClearLines(main_widgets.thread_list);
    ArtTextClearLines(main_widgets.text);
    ScrBarSetLengthsAndPos(main_widgets.text_scrbar, 0, 0, 0);
    knapp_set_mode(NewsModeAllgroups);
    update_misc_menu(NewsModeAllgroups);
    ScrListClearLines(main_widgets.group_list);
    XtSetArg(args[0], XtNatMostOne, False);
    XtSetArg(args[1], XtNatLeastOne, False);
    XtSetArg(args[2], XtNallowDnd, True);
    XtSetValues(main_widgets.group_list, args, 3);
    XtManageChild(main_widgets.group_list);
    XtSetArg(args[0], XtNstepSize, 1);
    XtSetValues(main_widgets.top_vbar, args, 1);
    ScrBarSetLengthsAndPos(main_widgets.top_hbar, 0, 0, 0);

    for (n = 0 ; n < global.no_groups ; n++) {
	GROUP	*group = global.groups[n];
	char	buffer[LINE_LEN];

	if (group == global.curr_group)
	    first = n;
	print_group_info_all(buffer, group);
	group->disp =
	    ScrListAddLine(main_widgets.group_list, buffer, None);
    }

    ScrListGetFirstShownSize(main_widgets.group_list, NULL, &shown, &n);
    if (first < 0)
	first = 0;
    else {
	ScrListSetSelected(main_widgets.group_list, first, True);
	if (first > shown && first > n - shown)
	    first = n - shown;
	ScrListSetFirst(main_widgets.group_list, first);
    }
    ScrBarSetLengthsAndPos(main_widgets.top_vbar, n, shown, first);

    set_standard_message();
}

void setNewsModeNewgroups(void)
{
    Arg		args[8];
    long	n;
    long	shown;

    global.mode = NewsModeNewgroups;

    for (n = 0 ; n < global.no_groups ; n++) {
	global.groups[n]->disp = -1;
	calc_no_unread(global.groups[n]);
    }

    XtSetArg(args[0], XtNnAlloc, global.no_new_groups + 4);
    XtSetValues(main_widgets.group_list, args, 1);

    XtSetSensitive(main_widgets.top_vbar, True);
    XtSetSensitive(main_widgets.top_hbar, False);
    XtSetSensitive(main_widgets.text_scrbar, False);

    XtUnmanageChild(main_widgets.arttree);
    XtUnmanageChild(main_widgets.thread_list);
    ScrListClearLines(main_widgets.thread_list);
    ScrListClearLines(main_widgets.group_list);
    ArtTextClearLines(main_widgets.text);
    ScrBarSetLengthsAndPos(main_widgets.text_scrbar, 0, 0, 0);
    knapp_set_mode(NewsModeNewgroups);
    update_misc_menu(NewsModeNewgroups);
    XtSetArg(args[0], XtNatMostOne, False);
    XtSetArg(args[1], XtNatLeastOne, False);
    XtSetArg(args[2], XtNallowDnd, False);
    XtSetValues(main_widgets.group_list, args, 3);
    XtManageChild(main_widgets.group_list);
    XtSetArg(args[0], XtNstepSize, 1);
    XtSetValues(main_widgets.top_vbar, args, 1);
    ScrBarSetLengthsAndPos(main_widgets.top_hbar, 0, 0, 0);

    for (n = 0 ; n < global.no_new_groups ; n++) {
	GROUP	*group = global.new_groups[n];
	char	buffer[LINE_LEN];

	print_group_info_all(buffer, group);
	group->disp = ScrListAddLine(main_widgets.group_list, buffer, None);
    }

    ScrListGetFirstShownSize(main_widgets.group_list, NULL, &shown, NULL);
    ScrBarSetLengthsAndPos(main_widgets.top_vbar,
			   global.no_new_groups, shown, 0);

    set_message("New groups found.", False);
}

void create_main_widgets(void)
{
    static char	*knapp_name[] = {
	"knapp0",	"knapp1",	"knapp2",	"misc",
	"post",		"kill",		"knapp6",	"knapp7",
	"knapp8",	"abort",	"save",		"search",
    };
    Widget	main_layout, knapp_layout, text_layout;
    Widget	sash;
    Arg		args[10];
    int		one_win = !global.separate_windows;
    int		i;

    if (one_win) {
	main_widgets.second_shell = NULL;
	main_layout =
	    XtVaCreateManagedWidget("singlelayout", layoutWidgetClass,
				    main_widgets.shell,
				    XtVaTypedArg, XtNlayout, XtRString,
#include "layouts/single.l"
				    (int)sizeof(String), NULL);
    } else {
	XtSetArg(args[0], XtNinput, True);
	XtSetArg(args[1], XtNcolormap, global.cmap);
	XtSetArg(args[2], XtNvisual, global.visual);
	XtSetArg(args[3], XtNdepth, global.depth);
	main_widgets.second_shell =
	    XtCreatePopupShell("second", topLevelShellWidgetClass,
			       main_widgets.shell, args, 4);
	main_layout =
	    XtVaCreateManagedWidget("doublelayout", layoutWidgetClass,
				    main_widgets.shell,
				    XtVaTypedArg, XtNlayout, XtRString,
#include "layouts/double.l"
				    (int)sizeof(String), NULL);
    }

    text_layout =
	XtVaCreateManagedWidget("textlayout", layoutWidgetClass,
				main_widgets.second_shell ?
				main_widgets.second_shell : main_layout,
				XtVaTypedArg, XtNlayout, XtRString,
#include "layouts/text.l"
				(int)sizeof(String), NULL);

    main_widgets.top_layout =
	XtVaCreateManagedWidget("toplayout", layoutWidgetClass, main_layout,
				XtVaTypedArg, XtNlayout, XtRString,
#include "layouts/top.l"
				(int)sizeof(String), NULL);

    XtSetArg(args[0], XtNcenter, False);
    main_widgets.message_line =
	XtCreateManagedWidget("message", messageWidgetClass,
			      main_layout, args, 1);
    knapp_layout =
	XtVaCreateManagedWidget("knapplayout", layoutWidgetClass, main_layout,
				XtVaTypedArg, XtNlayout, XtRString,
#include "layouts/knapp.l"
				(int)sizeof(String), NULL);

    main_widgets.top_manager =
	XtCreateManagedWidget("topmanager", managerWidgetClass,
			      main_widgets.top_layout, NULL, 0);

    sash = XtCreateManagedWidget("sash1", sashWidgetClass,
				 main_layout, NULL, 0);
    XtAddCallback(sash, XtNcallback, sash_callback, NULL);
    sash = XtCreateManagedWidget("sash2", sashWidgetClass,
				 main_layout, NULL, 0);
    XtAddCallback(sash, XtNcallback, sash_callback, NULL);
    if (one_win) {
	sash = XtCreateManagedWidget("sash3", sashWidgetClass,
				     main_layout, NULL, 0);
	XtAddCallback(sash, XtNcallback, sash_callback, NULL);
    }

    XtSetArg(args[0], XtNcoverHoriz, True);
    XtSetArg(args[1], XtNcoverVert, True);
    XtSetArg(args[2], XtNcontainHoriz, False);
    XtSetArg(args[3], XtNcontainVert, False);
    XtSetArg(args[4], XtNpixmapWidth, HOT_PIXMAP_SIZE);
    XtSetArg(args[5], XtNpixmapHeight, HOT_PIXMAP_SIZE);
    XtSetArg(args[6], XtNdepthOne, False);
    main_widgets.arttree =
	XtCreateWidget("arttree", artTreeWidgetClass,
		       main_widgets.top_manager, args, 7);
    args[2].value = (XtArgVal)True;
    args[3].value = (XtArgVal)True;
    XtSetArg(args[7], XtNusePixmaps, False);
    main_widgets.group_list =
	XtCreateManagedWidget("grouplist", scrListWidgetClass,
			      main_widgets.top_manager, args, 8);
    XtSetArg(args[7], XtNusePixmaps, True);
    XtSetArg(args[8], XtNatLeastOne, True);
    XtSetArg(args[9], XtNatMostOne, True);
    main_widgets.thread_list =
	XtCreateWidget("threadlist", scrListWidgetClass,
		       main_widgets.top_manager, args, 10);

    main_widgets.top_vbar =
	XtCreateManagedWidget("topvbar", scrBarWidgetClass,
			      main_widgets.top_layout, NULL, 0);
    XtSetArg(args[0], XtNvertical, False);
    XtSetArg(args[1], XtNstepSize, ARTTREE_STEPSIZE);
    main_widgets.top_hbar =
	XtCreateManagedWidget("tophbar", scrBarWidgetClass,
			      main_widgets.top_layout, args, 1);

    main_widgets.text =
	XtCreateManagedWidget("text", artTextWidgetClass,
			      text_layout, NULL, 0);
    main_widgets.text_scrbar =
	XtCreateManagedWidget("textscrbar", scrBarWidgetClass,
			      text_layout, NULL, 0);

    XtSetArg(args[0], XtNresizable, False);
    for (i = 0 ; i < 12 ; i++) {
	int	has_menu = False;

	if (i == 3) {
	    XtSetArg(args[1], XtNmenuName, NULL);
	    has_menu = True;
	} else if (i == 4) {
	    XtSetArg(args[1], XtNmenuName, "postshell");
	    has_menu = True;
	}

	main_widgets.knapp[i] =
	    XtCreateManagedWidget(knapp_name[i], has_menu ?
				  menuKnappWidgetClass : knappWidgetClass,
				  knapp_layout, args, has_menu ? 2 : 1);
    }

    XtRealizeWidget(main_widgets.shell);
    add_WM_DELETE_WINDOW_callback(main_widgets.shell,
				  delete_window_callback, NULL);
    XtInstallAllAccelerators(main_widgets.shell, main_widgets.shell);

    if (main_widgets.second_shell) {
	XtRealizeWidget(main_widgets.second_shell);
	add_WM_DELETE_WINDOW_callback(main_widgets.second_shell,
				      delete_window_callback, NULL);
	XtInstallAllAccelerators(main_widgets.second_shell,
				 main_widgets.shell);
    }

    create_misc_menu1(main_widgets.shell);
    create_misc_menu2(main_widgets.shell);
    create_post_menu(main_widgets.shell);

    XtAddCallback(main_widgets.knapp[0], XtNcallback,
		  knapp0_callback, NULL);
    XtAddCallback(main_widgets.knapp[1], XtNcallback,
		  knapp1_callback, NULL);
    XtAddCallback(main_widgets.knapp[2], XtNcallback,
		  knapp2_callback, NULL);
    XtAddCallback(main_widgets.knapp[5], XtNcallback,
		  knapp5_callback, NULL);
    XtAddCallback(main_widgets.knapp[6], XtNcallback,
		  knapp6_callback, NULL);
    XtAddCallback(main_widgets.knapp[7], XtNcallback,
		  knapp7_callback, NULL);
    XtAddCallback(main_widgets.knapp[8], XtNcallback,
		  knapp8_callback, NULL);
    XtAddCallback(main_widgets.knapp[9], XtNcallback,
		  knapp9_callback, NULL);
    XtAddCallback(main_widgets.knapp[10], XtNcallback,
		  knapp10_callback, NULL);
    XtAddCallback(main_widgets.knapp[11], XtNcallback,
		  knapp11_callback, NULL);
    XtAddCallback(main_widgets.top_vbar, XtNscrollCallback,
		  top_vbar_callback, NULL);
    XtAddCallback(main_widgets.top_hbar, XtNscrollCallback,
		  top_hbar_callback, NULL);
    XtAddCallback(main_widgets.text_scrbar, XtNscrollCallback,
		  text_scrbar_callback, NULL);
    XtAddCallback(main_widgets.top_manager, XtNresizeCallback,
		  top_manager_callback, NULL);
    XtAddCallback(main_widgets.text, XtNresizeCallback,
		  text_resize_callback, NULL);
    XtAddCallback(main_widgets.text, XtNurlCallback,
		  text_url_callback, NULL);
    XtAddCallback(main_widgets.arttree, XtNselectCallback,
		  arttree_sel_callback, NULL);
    XtAddCallback(main_widgets.arttree, XtNscrollCallback,
		  arttree_scroll_callback, NULL);
    XtAddCallback(main_widgets.arttree, XtNouterCallback,
		  arttree_tag_callback, NULL);
    XtAddCallback(main_widgets.thread_list, XtNselectCallback,
		  thread_list_sel_callback, NULL);
    XtAddCallback(main_widgets.thread_list, XtNcallback,
		  thread_list_callback, NULL);
    XtAddCallback(main_widgets.group_list, XtNcallback,
		  group_list_callback, NULL);
    XtAddCallback(main_widgets.group_list, XtNselectCallback,
		  group_list_sel_callback, NULL);
    XtAddCallback(main_widgets.group_list, XtNdndCallback,
		  group_list_dnd_callback, NULL);

#if HAVE_XMU && (XtSpecificationRelease > 4)
    XtAddEventHandler(main_widgets.shell, (EventMask)0, True,
		      _XEditResCheckMessages, NULL);
    if (main_widgets.second_shell)
	XtAddEventHandler(main_widgets.second_shell, (EventMask)0,
			  True, _XEditResCheckMessages, NULL);

#endif
}

void purge_hot(short hot)
{
    Pixmap	pixmap = global.hot_pixmaps[hot];
    SUBJECT	*subj;
    ARTICLE	*art;

    if (global.mode != NewsModeGroup &&
	global.mode != NewsModeThread)
	return;

    ScrListPurgePixmap(main_widgets.thread_list, pixmap);

    for (art = get_articles(main_thr) ; art ; art = art->next)
	if (art->hot == hot)
	    art->hot = 0;

    for (subj = get_subjects(main_thr) ; subj ; subj = subj->next)
	if (subj->hot == hot)
	    update_subj_hot_value(subj);

    if (global.mode == NewsModeThread && global.curr_subj) {
	for (art = global.curr_subj->thread ; art ;
	     art = next_in_thread_preorder(art))
	    if (ArtTreeNodeGetPixmap(main_widgets.arttree,
				     (ART_TREE_NODE *)art) == pixmap)
		ArtTreeNodeSetPixmap(main_widgets.arttree,
				     (ART_TREE_NODE *)art, None);

	update_subj_hot_value(global.curr_subj);
    }
}
