#include <stdio.h>
#include "config.h"

#ifndef VMS
#include <sys/types.h>
#endif

#ifndef NO_STDLIB_H
# include <stdlib.h>
#endif

#ifndef NO_UNISTD_H
# include <unistd.h>
#endif

#include "window.h"
#include "search.h"
#include "display.h"
#include "most.h"
#include "sysdep.h"
#include "file.h"
#include "line.h"

#ifdef USE_SLANG
#include "slang.h"
#endif

char *Most_Global_Msg = "Press `Q' to quit, `H' for help, and SPACE to scroll.";

Window *Most_Win;
Window *Most_Top_Win;
int Most_Top_Line;		       /* row number of top window */
int Most_Curs_Row;
int Most_Curs_Col;
int Most_Column = 1;
int Most_Restore_Width_To = 0;
char Most_Mini_Buf[256];

static int Beep_Mini = 0;
static int Minibuffer_Selected;
unsigned char *Most_Curs_Pos;

#ifndef USE_SLANG
static int read_string(char *str)
{
    char ch;
    int i;

    i = strlen(str);
    do 
      {
	 ch = most_getkey();
	 if (ch == 2)		       /* ^B erases line */
	   {
	      while(i)
		{
		   ch = str[--i];
		   if ((ch < 32) || ((unsigned char) ch >= 127)) 
		     most_send_string_to_term("\b \b"); /* erase the ^ */
		   most_send_string_to_term("\b \b");
		}
	   }
	 else if ((ch == '\b') || (ch == 127))
            {
                if (i > 0)
                  {
		     ch = str[--i];
		     if ((ch < 32) || ((unsigned char) ch >= 127)) 
		       most_send_string_to_term("\b \b"); /* erase the ^ */
		     most_send_string_to_term("\b \b");
                  }
                else most_beep();
            }
          else if ((ch < 32) && (ch != 10) && (ch != 13))
            {
		if ((ch == '\007') || (ch == '\025')) /* ^G or ^U aborts */
		   {
		      most_beep();
		      return(-1);    /* ^G quits */
		   }
		if (ch == '\033')  /* Allow KP ENTER as a terminator */
		   {
		      ch = most_getkey();
		      if (ch == 'O')
			 {
			    ch = most_getkey();
			    if (ch == 'M')
			       {
				  str[i] = '\0';
				  return(i);
			       }
			    else
			       {
				  most_send_string_to_term("^[O");
				  most_tt_putchar(ch);
				  str[i++] = 27;
				  str[i++] = 'O';
				  str[i++] = ch;
			       }
			 }
		      else
			 {
			    most_send_string_to_term("^[");
			    most_tt_putchar(ch);
			    str[i++] = 27;
			    str[i++] = ch;
			 }
		   }
		else
		   {
		      str[i++] = ch;
		      most_tt_putchar('^');
		      most_tt_putchar(ch + '@');
		   }
            }
          else if (ch == '`')   /* quoted insert */
            {
                ch = most_getkey();
                str[i++] = ch;
                if ((ch < ' ') || (ch == 127)) 
                  {
                      if (ch == 127) ch = '?'; else ch = ch + '@';
                      most_tt_putchar('^');
                  }
                most_tt_putchar(ch);
            }
	  else if ((ch != 10) && (ch != 13))
            {
                str[i++] = ch;
                most_tt_putchar(ch);
            }
          
          fflush(stdout);
      }
    while ((ch != 10) && (ch != 13));
      
    str[i] = '\0';
    return(i);
}
#endif

void most_message(char *what, int how)
{
    strcpy(Most_Mini_Buf,what);
    if (how) Beep_Mini = 1; else Beep_Mini = 0;
}

static char *Last_Message;
void most_select_minibuffer()
{
   most_goto_rc(Most_Screen_Height,1);
   most_tt_erase_line();
   Last_Message = NULL;
   Minibuffer_Selected = 1;
   fflush(stdout);
}

void most_exit_minibuffer()
{
   Minibuffer_Selected = 0;   
}

/* put out string, expanding control chars */
static void nicely_puts(char *str)
{
   unsigned char ch;
   int len = 1;
    while ((ch = (unsigned char) *str++) != 0)
      {
	 if (len >= Most_Screen_Width) break;
          if ((ch < ' ') || ((ch >= 127) && (ch < 160)))
            {
	       if (ch & 0x80)
		 {
		    most_tt_putchar ('~');
		    ch &= 0x7F;
		    len++;
		 }
	       if (len >= Most_Screen_Width) break;
	       most_tt_putchar('^');
	       len++;
	       if (ch != 127) ch = ch + '@'; else ch = '?';
            }
	 if (len >= Most_Screen_Width) break;
	 most_tt_putchar((char) ch);
	 len++;
      }
}

static void put_message_1 (char *what)
{
   most_select_minibuffer();
   if (*what) nicely_puts(what);
   most_exit_minibuffer();
   Last_Message = what;
}

void most_put_message ()
{
   if (Beep_Mini) most_beep();
   put_message_1 ((char *) Most_Mini_Buf);
#ifdef USE_SLANG
   if (Beep_Mini)
     {
	/* wait half a second */
	while (SLang_input_pending (5)) 
	  {
	     (void) SLang_getkey ();
	     SLang_Error = SLKeyBoard_Quit = 0;
	  }
     }
#endif
   Beep_Mini = 0;
   *Most_Mini_Buf = 0;
}

/* puts 'what in the minibuffer to be edited. */
/* returns number of chars read */

#ifdef USE_SLANG
SLang_RLine_Info_Type *Most_RLI;

static void most_goto_column (int col)
{
   /* This most version does not use buffered stdout but readline library
    * does.  Flush it now.
    */
   fflush (stdout);
   most_goto_rc(Most_Screen_Height, col + 1);
}

static SLang_RLine_Info_Type  *init_readline (void)
{
   unsigned char *buf = NULL;
   SLang_RLine_Info_Type *rli;
   
   if ((NULL == (rli = (SLang_RLine_Info_Type *) SLMALLOC (sizeof(SLang_RLine_Info_Type))))
       || (NULL == (buf = (unsigned char *) SLMALLOC (256))))
     {
	fprintf(stderr, "malloc error.\n");
	exit(-1);
     }
   
   MEMSET ((char *) rli, 0, sizeof (SLang_RLine_Info_Type));
   rli->buf = buf;
   rli->buf_len = 255;
   rli->tab = 8;
   rli->dhscroll = 20;
   rli->getkey = SLang_getkey;
   rli->tt_goto_column = most_goto_column;
   if (SLang_init_readline (rli) < 0)
     {
	most_exit_error ("Unable to initialize readline library.");
     }
   
   return rli;
}

#endif

int most_read_from_minibuffer(char *prompt, char *what)
{
   int i;
   
#ifdef USE_SLANG

   if (Most_RLI == NULL) 
     {
	Most_RLI = init_readline ();
     }
   
   Most_RLI->edit_width = Most_Screen_Width - 1;
   Most_RLI->prompt = prompt;
   *Most_RLI->buf = 0;
   most_select_minibuffer();
   
   /* do not use default.  The up arrow can always get it back. */
   if ((*what) && (what != Most_Search_Str))
     {
	strcpy ((char *) Most_RLI->buf, what);
	Most_RLI->point = strlen (what);
     }
   
   /* next routine returns -1 upon quit */ 
   i = SLang_read_line (Most_RLI);
   if ((i > 0) && !SLang_Error && !SLKeyBoard_Quit)
     {
	SLang_rline_save_line (Most_RLI);
	strcpy(what, (char *) Most_RLI->buf);
     }
   if (SLKeyBoard_Quit) i = -1;
   SLang_Error = SLKeyBoard_Quit = 0;
   
#else
   char str[256];
   char p[256];
   
   if (*what)
     {
	sprintf(p, "%s(^B kills): ", prompt);
     }
   else	sprintf(p, "%s", prompt);
   
   most_select_minibuffer();
   most_send_string_to_term(p);
   if (*what != '\0')
     nicely_puts(what);
   strcpy(str,what);
   i = read_string(str);
   if (i > 0) strcpy(what,str);
#endif
   
    most_tt_erase_line();
    most_exit_minibuffer();
   
   if (i == -1) most_message ("Aborted!", 1);
    return(i);
}

    
void most_clear_minibuffer()
{
    Most_Mini_Buf[0] = '\0';
    Beep_Mini = 0;
    most_put_message();
}

void most_check_minibuffer ()
{
   if (*Most_Mini_Buf) most_put_message ();
   else
     if (Last_Message != Most_Global_Msg) put_message_1 (Most_Global_Msg);
}


static int get_scroll(int *line)
{
    /* line is the line we want at the top of the window if possible */
    int dtop, dbot,n,top,bot,wsize;

    top = Most_Win->beg_line;
    wsize = Most_Win->bot - Most_Win->top; /* actually 1 less than window size */
    bot = top + wsize;

    if ((*line == 1) && (top == 1))
      {
	 Most_Curs_Col = Most_Curs_Row = 1;
	 if (Most_Beg != Most_C_Pos) 
	   {
	      Most_C_Pos = Most_Beg;
	      Most_C_Line = 1;
	      return 0;
	   }
          most_message("Top of Buffer.",1);
          return(0);
      }
    
    /* handles endof file in a window */
    if ((bot > Most_Num_Lines) && *line > Most_C_Line)
      {
          *line = top;
          most_message("End of Buffer.",1);
          return(0);
      }
    
    if (Most_Num_Lines <= wsize)     /* short file */
      {
          *line = 1;
          return(0);
      }

    dtop = top - 1;
    dbot = Most_Num_Lines - bot;
    n = *line - top;

    if ((n>0) && (dbot < n))
      {
          n = dbot;
          *line = top + n;
          if (!n) most_message("End of buffer.",1);
      }
    else if ((n < 0) && (dtop + n < 0))
      {
          n = -dtop;
          *line = n + top;
          if (!n) 
	   {
	      Most_Curs_Col = Most_Curs_Row = 1;
	      if (Most_Beg != Most_C_Pos) 
		{
		   Most_C_Pos = Most_Beg;
		   Most_C_Line = 1;
		   return 0;
		}
	      most_message("Top of buffer.",1);
	   }
      }
    return(n);
}


void most_update_window(int line)
{
    int n,max_n, save_line, save_col, npos;
    unsigned char *save_pos;

   most_read_to_line(line);
    if (Most_Column != Most_Win->col)
      {
          if (Most_Column < 1) Most_Column = 1;
          if (Most_Column != Most_Win->col)
            {
                save_pos = Most_Curs_Pos; save_line = Most_Curs_Row; save_col = Most_Curs_Col;
                most_redraw_window();
                most_update_status(0);
                Most_Win->curs_pos = Most_Curs_Pos = save_pos;
                Most_Win->curs_line = Most_Curs_Row = save_line;
                Most_Win->curs_col = Most_Curs_Col = save_col;
            }
          return;
      }

    
    n = get_scroll(&line);
    max_n = Most_Win->bot - Most_Win->top;
    if (Most_Term_Cannot_Scroll || (abs(n) > max_n))
      {
          most_goto_line(line);
          most_redraw_window();
          most_update_status(0);
          return;
      }
    if (!n) return;

    most_set_scroll_region(Most_Win->top, Most_Win->bot);
    most_goto_rc(1, 1);
    most_forward_line(n);
    Most_Win->beg_pos = Most_C_Pos;
    Most_Win->beg_line = Most_C_Line;

    if (n>0)
      {
	 npos = 1;
	 most_tt_delete_nlines(n);
	 most_set_scroll_region(1, Most_Screen_Height);
	 most_goto_rc(Most_Win->bot - n + 1,1); 
	 
	 most_forward_line(max_n - n + 1);
      }
    else
      {
          npos = 0;
          Most_Curs_Row = 1; Most_Curs_Col = 1; Most_Curs_Pos = Most_C_Pos;
          n = -n;
          most_reverse_index(n);
	 most_set_scroll_region(1, Most_Screen_Height);
	 most_goto_rc(Most_Win->top, 1);
      }
    n = n - 1;
    most_display_line();
    while(n--)
      {
          most_forward_line(1);
          most_tt_putchar('\n');
          most_display_line();
      }
    if (npos) 
      {
          
          Most_Curs_Row = Most_C_Line - Most_Win->beg_line + 1;
          Most_Curs_Col = 1; Most_Curs_Pos = Most_C_Pos;
      }
    
    Most_C_Pos = Most_Win->beg_pos;
    Most_C_Line = Most_Win->beg_line;
   /* most_set_scroll_region(1, Most_Screen_Height); */
    most_update_status(0);
    fflush(stdout);
}

/* updates current window as well as scroll lock ones */
/* Although current window is update in an absolute fashion, scroll locked
   ones are updated in a relative fashion */
void most_update_windows(int line)
{
    int dline,flg;
    Window *w;
    
    dline = line - Most_C_Line;
   most_update_window(line);
    if (!Most_Win->lock) return;
    flg = 0;
    w = Most_Win;
    while(Most_Win = Most_Win->next, Most_Win != w)
      {
          if (Most_Win->lock)
            {
                flg = 1;
                most_set_window(Most_Win);
                line = Most_C_Line + dline;
                most_update_window(line);
            }
      }
    Most_Win = w;
    if (flg) most_set_window(Most_Win);
}

static void clear_window (void)
{
   int i,n;
   i = Most_Win->top;
   n = Most_Win->bot - Most_Win->top;
   
   most_set_scroll_region(Most_Win->top, Most_Win->bot);
   if ((i == 1) && !Most_Term_Cannot_Scroll)
     {
	most_goto_rc(1, 1);
	/*  most_goto_rc(Most_Win->bot - Most_Win->top + 1,Most_Screen_Width); */
	most_tt_delete_nlines(1 + Most_Win->bot - Most_Win->top);
	/* clr_bos(); */
      }
    else
      {
          most_goto_rc(i - Most_Win->top + 1,1);
          most_tt_erase_line();
          while(n--)
            {
                most_tt_putchar('\n');
                most_tt_erase_line();
            }
      }
    
    most_goto_rc(i - Most_Win->top + 1,1);
   most_set_scroll_region(1, Most_Screen_Height);
    fflush(stdout);
}


void most_redraw_window()
{
    int n,t;
   
   if (Most_Term_Cannot_Scroll) t = 0; else t = Most_Win->top;
    if (t == 1) clear_window();
   
    most_goto_rc(Most_Win->top, 1);
    n = Most_Win->bot - Most_Win->top;
    if ((Most_C_Line + n) > Most_Num_Lines) most_goto_line(Most_Num_Lines - n);
    Most_Win->curs_pos = Most_Curs_Pos = Most_Win->beg_pos = Most_C_Pos;
    Most_Win->beg_line = Most_C_Line;
    Most_Win->col = Most_Column;
    Most_Win->curs_col = Most_Curs_Col = 1;
    Most_Win->curs_line = Most_Curs_Row = 1;
    if (t != 1) 
     {
	most_tt_erase_line();
     }
   
    most_display_line();
    while(n--)
      {
          most_tt_putchar('\n');
          if (t != 1) most_tt_erase_line();
          if (most_forward_line(1)) most_display_line();
      }
    
        
    Most_C_Pos = Most_Win->beg_pos;
    Most_C_Line = Most_Win->beg_line;
}

#define MOST_V_OPT  1
#define MOST_T_OPT  2
#define MOST_B_OPT  4
#define MOST_S_OPT 8
#define MOST_W_OPT 16


/* associates current window with current buffer */
void most_save_win_flags(Window *w)
{
    w->flags = 0;
    if (Most_V_Opt) w->flags |= MOST_V_OPT;
    if (Most_B_Opt) w->flags |= MOST_B_OPT;
    if (Most_T_Opt) w->flags |= MOST_T_OPT;
    if (Most_W_Opt) w->flags |= MOST_W_OPT;
    if (Most_S_Opt) w->flags |= MOST_S_OPT;
    w->n_lines = Most_Num_Lines;
    w->display = Most_Selective_Display;
}

void most_window_buffer()
{
   Most_Win->beg_line = Most_C_Line;
   Most_Win->beg_pos = Most_C_Pos;
   Most_Win->col = Most_Column;
   Most_Win->buf = Most_Buf;
   Most_Selective_Display = 0;
   most_save_win_flags(Most_Win);

   Most_Curs_Row = Most_Win->curs_line = 1;
   Most_Curs_Col = Most_Win->curs_col = 1;
   Most_Curs_Pos = Most_Win->curs_pos = Most_C_Pos;
}


static void restore_win_flags (void)
{
    Most_V_Opt = Most_Win->flags & MOST_V_OPT;
    Most_B_Opt = Most_Win->flags & MOST_B_OPT;
    Most_T_Opt = Most_Win->flags & MOST_T_OPT;
    Most_W_Opt = Most_Win->flags & MOST_W_OPT;
    Most_S_Opt = Most_Win->flags & MOST_S_OPT;
    Most_Num_Lines = Most_Win->n_lines;
    Most_Selective_Display = Most_Win->display;
}


static Window *make_window(int r1,int r2)
{
    int i;
    Window *new;
    new = (Window *) MOSTMALLOC(sizeof(Window));
   memset ((char *) new, 0, sizeof (Window));
    new->status = (char *) MOSTMALLOC(256);
    for (i = 0; i <= Most_Screen_Width; i++) new->status[i] = '\0';
    new->col = Most_Column;
    new->top = r1;
    new->bot = r2;
    new->lock = 0;
   new->buf = NULL;
    most_save_win_flags(new);
    return(new);
}

int Most_Display_Inited = 0;

void most_init_display()
{
   int h = Most_Screen_Height;
   
   if (Most_Display_Inited) return;
   Most_Display_Inited = 1;
   
   most_get_term_dimensions(&Most_Screen_Width, &Most_Screen_Height);
   
   if (Most_Win == NULL)
     {
	Most_Top_Win = Most_Win = make_window(1,Most_Screen_Height - 2);
	Most_Win->prev = Most_Win->next = Most_Win;
     }
   else if (h > Most_Screen_Height) most_resize_display (0);
   else Most_Top_Win->prev->bot = Most_Screen_Height - 2;
   
   most_cls();
   most_set_scroll_region(1, Most_Screen_Height);
   most_goto_rc(Most_Win->top,1);
   fflush(stdout);
   Most_Display_Inited = 1;
}

void most_reset_display()
{
   if (Most_Display_Inited == 0) return;
   most_set_scroll_region(1,Most_Screen_Height);
   most_goto_rc(Most_Screen_Height,1);
   most_tt_erase_line ();
   if (Most_Restore_Width_To)
     {
	most_set_width(Most_Restore_Width_To, 0);
	Most_Restore_Width_To = 0;
     }
    fflush(stdout);
   Most_Display_Inited = 0;
}

static void update_status1(int new_status)
{
    char str[30], ch, *strp;
    static char new[256];
    int i,ii,r,x,line_p = 60, len;
   int max_len;
   Buffer *buf = Most_Win->buf;
   
   max_len = Most_Screen_Width - 25;
    r = Most_Win->bot + 1;
   most_goto_rc(r,1); 

   i = ii = 0;
   new[ii++] = '-';
   if (Most_Win->lock) new[ii++] = '*'; else new[ii++] = '-';
   strp = " MOST: ";
   while(*strp != '\0') new[ii++] = *strp++;
    
   len = strlen (buf->file);
   if (len > max_len)
     {
        new[ii++] = '.'; new[ii++] = '.'; new[ii++] = '.';
	strcpy (&new[ii], buf->file + (len - max_len));
	ii += max_len;
     }
   else 
     {
	strcpy (&new[ii], buf->file);
	ii += len;
     }
   
    while(ii < line_p) new[ii++] = ' ';
   x = (Most_C_Pos - Most_Beg) * 100;
   if (buf->fd == -1) 
     {
	if (Most_Eob == Most_Beg) x = 100; else x = x / (Most_Eob - Most_Beg);
	if (Most_C_Line + (Most_Win->bot - Most_Win->top + 1) >= Most_Num_Lines) x = 100;
     }
   else 
     {
	x = x / (buf->size);
     }

   /* x = (Most_C_Line + Most_Win->bot - Most_Win->top) * 100; x = x / Most_Num_Lines; */

    /* for files with end of file above the bottom row (due to window manipulations) */
    if (x > 100) x = 100;
   
   /* stdin may not be read in yet and I do not know how big it is. */
   if (buf->fd == 0) sprintf(str,"(%d,%d) ",Most_C_Line,Most_Column);
   else
     sprintf(str,"(%d,%d) %d%%",Most_C_Line,Most_Column,x);
   
    i = 0; while(ch = str[i++], ch != '\0') new[ii++] = ch;

    while(ii < Most_Screen_Width) new[ii++] = '-';
    new[Most_Screen_Width] = '\0';
   most_tt_reverse_video();
    if (new_status)
      most_send_string_to_term(new);
    else
      most_smart_puts((char *) new,(char *) Most_Win->status, r, 1);
   most_tt_normal_video();
    strcpy(Most_Win->status,new);
}

void most_update_status(int new_status)
{

    Most_C_Line = Most_Win->beg_line;
    Most_C_Pos = Most_Win->beg_pos;
   /* most_set_scroll_region(1,Most_Screen_Height); */
    update_status1(new_status);
   /* most_set_scroll_region(Most_Win->top,Most_Win->bot); */
    fflush(stdout);
}

/* splits window-- no screen update, does not affect scrolling region */
int most_split_window (void)
{
    Window *new, *old;
    int b2,t2,b1, line;

    b2 = Most_Win->bot;
    b1 = (Most_Win->bot + Most_Win->top) / 2 - 1;
    t2 = b1 + 2;
    if ((b1 == Most_Win->top) || (t2 == b2)) return(0);

    /* line is top line of new window. */
    line = Most_Win->beg_line + t2 - Most_Win->top;
    old = Most_Win;
    Most_Win->bot = b1;
    new = make_window(t2,b2);
    /* add to ring */
    Most_Win->next->prev = new;
    new->next = Most_Win->next;
    new->prev = Most_Win;
    Most_Win->next = new;

    new->beg_line = line;
    new->buf = Most_Buf;
    /* new window status line is at same position as old */
    strcpy(new->status,Most_Win->status);
    return(1);
}

    
void most_two_windows()
{
    int line;
    Window *new, *old;
    if (!most_split_window()) return;

    old = Most_Win;
    new = Most_Win->next;
    line = new->beg_line;
    if (line + new->bot - new->top > Most_Num_Lines)
      {
          most_other_window(1);
          /* since split window left new window undefined... */
          Most_C_Pos = old->beg_pos;
          Most_C_Line = old->beg_line;
          if (Most_Num_Lines <= new->bot - new->top + 1)
            {
                Most_C_Line = new->beg_line = 1;
                Most_C_Pos = new->beg_pos = Most_Buf->beg;
                most_redraw_window();
                most_update_status(0);
            }
          else if (line > Most_Num_Lines)
            {
                most_goto_line(Most_Num_Lines - new->bot + new->top);
                Most_Win->beg_pos = Most_C_Pos;
                Most_Win->beg_line = Most_C_Line;
                most_redraw_window();
                most_update_status(0);
            }    
          else
            {
                most_goto_line(line);
                Most_Win->beg_pos = Most_C_Pos;
                Most_Win->beg_line = Most_C_Line;
                most_update_window(Most_Num_Lines - new->bot + new->top);
            }
          Most_Win->curs_line = 1;
          Most_Win->curs_col = Most_Column;
          Most_Win->curs_pos = Most_C_Pos;
          most_other_window(-1);
      }
    else
      {
          Most_Win = new;
          (void) most_forward_line(line - old->beg_line);
          new->beg_pos = Most_C_Pos;
          new->beg_line = Most_C_Line;
          new->curs_line = 1;
          new->curs_col = Most_Column;
          new->curs_pos = Most_C_Pos;
          most_update_status(0);
          Most_Win = old;
      }    
    most_update_status(1);
}

static void expand_window1(int dn)
{
    int l, save_top, save_line;
    unsigned char *save_pos;

    /* l is line after last line of current window (where status line is) */
    l = Most_Win->beg_line + Most_Win->bot - Most_Win->top + 1;
    save_top = Most_Win->top;
    Most_Win->top = Most_Win->bot + 1;
    Most_Win->bot = Most_Win->bot + dn;
   /* most_set_scroll_region(Most_Win->top, Most_Win->bot); */
    if (l > Most_Num_Lines)
      {
          clear_window();
      }
    else
      {
          /* should add something here for smarter scrolling...
             if ((Most_Win->next->Most_Buf == Most_Buf) && (l >= Most_Win->next->beg_line)
             && (l <= (Most_Win->next->beg_line + Most_Win->next)
             */
                
          save_line = Most_C_Line;
          save_pos = Most_C_Pos;
          most_goto_line(l);
          most_redraw_window();
          Most_Win->beg_line = Most_C_Line = save_line;
          Most_Win->beg_pos = Most_C_Pos = save_pos;
      }
    Most_Win->top = save_top;
   /* most_set_scroll_region(Most_Win->top, Most_Win->bot); */
}

void most_delete_other_windows (void)
{
   Window *w, *tmp;

   if (Most_Win->next == Most_Win) return;
   w = Most_Win;
   Most_Win = Most_Win->next;
   /* delete other windows preserving the ring! */
   while(Most_Win != w)
     {
	most_free_window_buffer(); /* needs a ring */
	tmp = Most_Win->next;
	/* if this is the bottom window, save its status line */
	if (tmp == Most_Top_Win) strcpy(w->status,Most_Win->status);
	tmp->prev = Most_Win->prev;
	Most_Win->prev->next = tmp;
	SLFREE(Most_Win->status);
	SLFREE(Most_Win);
	Most_Win = tmp;
     }
   Most_Win = w;
}

void most_one_window()
{
   int diff;
    
   if (Most_Win->next == Most_Win) return;
   
   most_delete_other_windows ();
   if (Most_Term_Cannot_Scroll) 
     {
	Most_Win->top = 1;
	Most_Win->bot = Most_Screen_Height - 2;
	most_redraw_window();
     }
   else
     {
	/* slide the window to the top and expand it */
	diff = Most_Win->top - 1;
	if (diff)
	  {
	     most_set_scroll_region(1,Most_Screen_Height - 2);
	     most_goto_rc(1,1);
	     most_tt_delete_nlines(diff);
	     Most_Win->top = 1;
	     Most_Win->bot -=  diff;
	     Most_Top_Win = Most_Win;
	  }
	expand_window1(Most_Screen_Height - 2 - Most_Win->bot);
	most_set_scroll_region(1, Most_Screen_Height);
     }
   
    most_update_status(0);
}

          
void most_set_window(Window *w)
{
    Most_Win = w;
    Most_C_Pos = Most_Win->beg_pos;
    Most_Curs_Pos = Most_Win->curs_pos;
    Most_Curs_Row = Most_Win->curs_line;
    Most_Curs_Col = Most_Win->curs_col;
    Most_C_Line = Most_Win->beg_line;
    Most_Column = Most_Win->col;
    Most_Buf = Most_Win->buf;
    most_switch_to_buffer(Most_Buf);
   /* most_set_scroll_region(Most_Win->top,Most_Win->bot); */
    restore_win_flags();
    fflush(stdout);
}

void most_other_window(int n)
{
    if (!n) return;
    Most_Win->beg_pos = Most_C_Pos;
    Most_Win->curs_pos = Most_Curs_Pos;
    Most_Win->curs_line = Most_Curs_Row;
    Most_Win->curs_col = Most_Curs_Col;
    Most_Win->beg_line = Most_C_Line;
    Most_Win->col = Most_Column;
   
   most_save_win_flags(Most_Win);
    if (n < 0)
      while (n++) Most_Win = Most_Win->prev;
    else
      while (n--) Most_Win = Most_Win->next;
    most_set_window(Most_Win);
}



/* kills window by moving lower window up */
static void delete_as_top_window (void)
{
    int t1,t2,b1,b2;
    t1 = Most_Win->top;
    t2 = Most_Win->next->top;
    b1 = Most_Win->bot;
    b2 = Most_Win->next->bot;
    Most_Win->prev->next = Most_Win->next;
    Most_Win->next->prev = Most_Win->prev;

   if (Most_Term_Cannot_Scroll)
     {
	most_other_window(1);
	Most_Win->top = t1;
	most_redraw_window();
     }
   else
     {
	/* scroll contents of window below to top */
	most_set_scroll_region(t1,b2);
	most_goto_rc(1,1);
	most_tt_delete_nlines(t2 - t1);
	most_other_window(1);
	Most_Win->top = t1;
	Most_Win->bot = b2 - t2 + t1;
	expand_window1(b2 - Most_Win->bot);
     }
   
    most_update_status(0);
}

/* free buffer for this window if no other windows are viewing it. */
void most_free_window_buffer (void)
{
   Window *w;
   Buffer *b;
   
    w = Most_Win->next;
   
    while(Most_Win != w)
      {
	 if (!strcmp(Most_Win->buf->file,w->buf->file)) return;
	 w = w->next;
      }
   b = w->buf;
   if (b == NULL) return;
   if (b->fd != -1) close(b->fd);
   if (b->beg != NULL) SLFREE(b->beg);
   Most_Win->buf = NULL;
   SLFREE(b);
}

void most_free_windows (void)
{
   Window *w;
   
   while (Most_Win != NULL)
     {
	most_free_window_buffer ();
	w = Most_Win->next;
	w->prev = Most_Win->prev;
	Most_Win->prev->next = w;
	SLFREE (Most_Win->status);
	SLFREE (Most_Win);
	if (w == Most_Win) w = NULL;
	Most_Win = w;
     }
   
   Most_Buf = NULL;
   Most_Beg = Most_Eob = NULL;
   Most_C_Pos = NULL;
   Most_Num_Lines = Most_C_Line = 0;
}

void most_delete_window()
{
    int new_b, old_b;
    Window *w;
    
    w = Most_Win;
    if (Most_Win->next == Most_Win) return;
    most_free_window_buffer();
    if (Most_Win->next != Most_Top_Win)
      {
          if (Most_Win == Most_Top_Win)
            {
                delete_as_top_window();
                Most_Top_Win = Most_Win;  /* not anymore, this one is */
            }
          else
            delete_as_top_window();
          
          SLFREE(w);
          return;
      }
    old_b = Most_Win->top - 2;
    new_b = Most_Win->bot;
    most_other_window(-1);
   if (Most_Term_Cannot_Scroll)
     {
	Most_Win->bot = new_b;
	most_redraw_window();
     }
   else expand_window1(new_b - old_b);
   
   strcpy(Most_Win->status,w->status); /* share the same line */
    
   Most_Win->next = w->next;
   Most_Win->next->prev = Most_Win;
   SLFREE(w);
   most_update_status(0);
}

void most_redraw_display()
{
   Window *w;
   int n,t;
   most_set_scroll_region(1,Most_Screen_Height);
   most_cls();
   Last_Message = NULL;
   most_save_win_flags(Most_Win);
   w = Most_Win;
   do
     {
	Most_Win = Most_Win->next;
	t = Most_Win->top;
	most_goto_rc(t, 1);
	Most_C_Pos = Most_Win->beg_pos;
	Most_C_Line = Most_Win->beg_line;
	Most_Column = Most_Win->col;
	most_switch_to_buffer(Most_Win->buf);
	
	restore_win_flags();
	n = Most_Win->bot - Most_Win->top;
	most_display_line();
	while(n--)
	  {
	     most_tt_putchar('\n');
	     if (most_forward_line(1)) most_display_line();
	  }
	Most_C_Line = Most_Win->beg_line;
	Most_C_Pos = Most_Win->beg_pos;
	update_status1(1);
     }
   while(Most_Win != w);
   most_set_window(w);
   
   if (Minibuffer_Selected) 
     {
	most_select_minibuffer ();
	most_send_string_to_term ((char *) Most_Mini_Buf);
     }
}

void most_toggle_lock()
{
    Most_Win->lock = !(Most_Win->lock);
    most_update_status(0);
}

