#define Copyright         "Copyright 1995  Ed Casas"

#define Version		  "efix v 0.1a"

/*
    Copyright (C) 1995  Ed Casas

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    Please contact the author if you wish to use efax or efix for
    purposes not covered by the GNU GPL.

    You may contact the author by e-mail at: edc@cce.com, by mail
    at: 2629 West 3rd Ave, Vancouver, BC, Canada, V6K 1M4, or by
    fax at: +1 604 734 5291.

*/

const char *Usage =
  "Usage:\n"
  "  %s [ option ]... [ file ]... \n"
"Options (defaults):\n"
  "  -i  f   input format (fax):\n"
  "     fax    fax (\"Group3\") 1-D coded image\n"
  "     text   text\n"
  "     pbm    raw PBM (portable bit map)\n"
  "  -o  f   output format (fax):\n"
  "     fax    fax (\"Group3\") 1-D coded image\n"
  "     pbm    Portable Bit Map\n"
  "     pgm    Portable Gray Map (decimated by 4)\n"
  "     pcl    HP-PCL (e.g. HP LaserJet)\n"
  "     ps     Postscript (e.g. Apple Laserwriter)\n"
  "  -n pat  printf() pattern for output file name (ofile)\n"
  "  -f fnt  use PBM font file fnt for text (built-in)\n"
  "  -l  n   lines per text page (66)\n"
  "  -v lvl  print messages of type in string lvl (ewi)\n"
  "  -s XxY  scale input by X and Y (Y optional) (1x1)\n"
  "  -r XxY  resolution of output is X by Y (dpi, Y optional) (204x196)\n"
  "  -p WxH  pad/truncate output to width W by height H (215x297mm)\n"
  "  -d R,D  displace output right R, down D (opposite if -ve) (0,0)\n"
  "  -O f    overlay file f (none)\n"
  "\n"
  "Add 'in', 'cm', 'mm', or 'pt' to -p and -d arguments (default in[ches]).\n" 
  ;

#include <ctype.h>		/* ANSI C */
#include <limits.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "efaxlib.h"

#ifndef INT_MAX
#define INT_MAX 32767
#endif

/* Look up a string in a NULL-delimited table where the first
   character of each string is the digit to return if the rest of
   the string matches.  Returns the value of the digit for the
   matching string or -1 if no matches. */

int lookup ( char **tab, char *s )
{
  char **p ;
  for ( p=tab ; *p && strcmp ( *p+1, s ) ; p++ ) ;
  return *p ? ( **p - '0' ) : -1 ;
}


/* Extract pair of values from string.  If it's a `dim'ension,
   two values are required and they are converted to inches, else
   the y value is optional.  Returns 0 or 2 on error. */

int getxy ( char *arg, float *x, float *y, int dim )
{
  int i, n, nc, err=0 ;
  char c ;
  static char *unitstr[] = { "0in", "1cm", "2mm", "3pt", 0 } ;
  static float unitval[] = { 1.0, 2.54, 25.4, 72.0, 1.0 } ;

  if ( ! arg ) {

    err = msg ( "E2 missing argument" ) ;

  } else {

    if ( ! err )
      switch ( n = sscanf ( arg , "%f%c%f%n", x, &c, y, &nc ) ) {
      case 0 : err = msg ( "E2bad X value in (%s)", arg ) ; break ;
      case 2 : err = msg ( "E2bad Y value in (%s)", arg ) ; break ;
      }      
    
    if ( dim ) {
      if ( n != 3 ) {
	err = msg ( "Emissing Y dimension in (%s)", arg ) ;
      } else {
	while ( isspace ( arg [ nc ] ) ) nc++ ;
	if ( arg [ nc ] )
	  if ( ( i = lookup ( unitstr, arg+nc ) ) >= 0 ) {
	    *x /= unitval [ i ] ;
	    *y /= unitval [ i ] ;
	  } else {
	    err = msg ( "E2bad units in (%s)", arg ) ;
	  }
      }
    } else {
      if ( n == 1 ) *y = *x ;
    }

    if ( ! err )
      msg ( "Aconverted (%s) into %f x %f", arg, *x, *y ) ;
  }

  return err ;
}


int main( int argc, char **argv)
{
  int err=0, done=0, i, c ;
  int nr, pels, ovnr, ovpels, no ;	/* run/pixel/repeat counts */
  int page, ilines, olines ;		/* page & line counts */
  int xs, ys, w, h, ixsh, iysh ;	/* integer scale, size & shift */
  short runs [ MAXRUNS ] , ovruns [ MAXRUNS ] ;
  int linesout=0 ;
  
  pbmfont font ;			 /* text font */

  float					 /* defaults: */
    xsc=1.0, ysc=1.0,		         /* scale */
    xsh=0.0, ysh=0.0,			 /* shift */
    xres = 204.145,			 /* o/p res'n: 1728/215mm * 25.4 x */
    yres = 195.58,			 /* 7.7 * 25.4 */
    xsz = 215 / 25.4,			 /* o/p size: 8.5" x A4 */
    ysz = 297 / 25.4 ;

  IFILE ifile, ovfile ;
  OFILE ofile ;
  char *defifnames [ 2 ] = { "-", 0 },  *ovfnames [ 2 ] = { 0, 0 } ;

  char *iformatstr[] = { "0text", "1pbm", "2fax", 0 } ;
  char *oformatstr[] = { "1pbm" , "2fax", "3pcl", "4ps",  "5pgm", 0 } ;

  /* initialize */

  argv0 = argv[0] ;
  verb = "ewi" ;

  newIFILE ( &ifile,  FAX, defifnames ) ;
  newIFILE ( &ovfile, FAX, ovfnames ) ;
  newOFILE ( &ofile,  FAX, 0, xres, yres, xres*xsz, yres*ysz ) ;

  /* process arguments */

  while ( !err && (c=nextopt(argc,argv,"n:i:o:O:v:l:f:r:s:p:d:") ) != -1) {
    switch ( c ) {
    case 'n':
      ofile.fname = optarg ;
      break ;
    case 'i': 
      if ( ( ifile.format = ovfile.format = lookup ( iformatstr, optarg ) ) 
	  < 0 )
	err = msg ( "E2invalid input type (%s)", optarg ) ;
      break ;
    case 'o': 
      if ( ( ofile.format = lookup ( oformatstr, optarg ) ) < 0 )
	err = msg ( "E2invalid output type (%s)", optarg ) ;
      break ;
    case 'O': 
      ovfnames[ 0 ] = optarg ;
      break ;
    case 'v': 
      verb = optarg ;
      break ;
    case 'l':
      if ( sscanf ( optarg , "%d", &(ifile.pglines) ) != 1 ) 
	err = msg ( "E2bad page length (%s)", optarg ) ;
      break ;
    case 'f' :
      if ( ( err = readfont ( optarg , &font ) ) == 0 )
	ifile.font = &font ;
      break ;
    case 's' : err = getxy ( optarg, &xsc , &ysc , 0 ) ; break ;
    case 'r' : err = getxy ( optarg, &xres, &yres, 0 ) ; break ;
    case 'p' : err = getxy ( optarg, &xsz , &ysz , 1 ) ; break ;
    case 'd' : err = getxy ( optarg, &xsh , &ysh , 1 ) ; break ;
    default : fprintf ( stderr, Usage, argv0 ) ; err = 2 ; break ;
    }
  }

  for ( i=0 ; i<argc ; i++ ) msg ( "Aargv[%d]=%s", i, argv[i]) ; 

  if ( optind < argc ) {
    ifile.fname = argv + optind ;
    if ( argv [ argc ] ) err = msg ("E2can't happen(unterminated argv)") ;
  }

  w = xsz * xres + 0.5 ;	      /* output dimensions in pixels */
  h = ysz * yres + 0.5 ;
  
  ixsh = xsh * xres ;		      /* x/y shifts in pixels/lines */
  iysh = ysh * yres ;
  
  xs = xsc * 256 ;		      /* scaling in 8-bit fixed-point format */
  ys = ysc * 256 ;

  if ( xsc < 0 || ysc < 0 || ysz < 0 || xsz < 0 || xres < 0 || yres < 0 )
    err = msg ( "E2negative scaling, size or resolution specified" ) ;
    
  if ( ( w & 7 ) != 0 )		      /* just about everything requires... */
      msg ("Iimage width rounded to %d pixels", 
	   w = ( w + 7 ) & ~7 ) ;
  
  if ( ofile.format == PGM && h & 3 ) /* PGM x4 decimation requires... */
    msg ("IPGM image height rounded up to %d lines", 
	 h = ( h + 3 ) & ~3 ) ;
  
  if ( ofile.format == PCL &&	      /* check for strange PCL resolutions */
      ( xres != yres || ( xres != 300 && xres != 150 && xres != 75 ) ) )
    msg ( "Wstrange PCL resolution (%.0fx%.0f)", xres, yres ) ;

  if ( w > MAXBITS*8 )		      /* make sure output will fit... */
    err = msg( "E2requested output width too large (%d pixels)", w ) ;

  ofile.w = w ; ofile.h = h ; ofile.xres = xres ; ofile.yres = yres ;

  for ( page = 0 ; ! err && ! done ; page++ ) {

    if ( ! nextipage ( &ifile, page ) ) { 
      done=1 ; 
      continue ; 
    }

    if ( *ovfnames )		      /* [re-]open overlay file */
      if ( ! nextipage ( &ovfile , page ) ) { 
	err=2 ; 
	continue ; 
      }

    if ( nextopage ( &ofile, page ) ) { 
      err=2 ; 
      continue ; 
    }
    
    /* y-shift */

    if ( iysh > 0 )
      writeline ( &ofile, ( ( *runs = w ), runs ), 1, linesout = iysh ) ;
    else
      for ( i=0 ; i < -iysh ; i++ ) 
	readline ( &ifile, runs, 0 ) ;
    
    /* copy input to output */
    
    olines = ilines = 0 ; 
    
    while ( linesout < h ) {

      if  ( ( nr = readline ( &ifile, runs, &pels ) ) < 0 )
	break ;

      if ( *ovfnames ) {
	if ( ( ovnr = readline ( &ovfile, ovruns, &ovpels ) ) >= 0 )
	  nr = runor ( runs, nr, ovruns, ovnr, 0, &pels ) ; 
      }

      /* x-scale, x-shift & x-pad input line */
    
      pels  = ( xs == 256 ) ? pels : xscale ( runs, nr, xs ) ;
      pels += ( ixsh == 0 ) ?   0  : xshift ( runs, nr, ixsh ) ;
      nr    = ( pels == w ) ?  nr  : xpad   ( runs, nr, w - pels ) ;

      /* y-scale by deleting/duplicating lines. */

      ilines++ ;
      no = ( ( ilines * ys ) >> 8 ) - olines ;
      olines += ( linesout + no > h ) ? ( no = h - linesout ) : no ;
      
      writeline ( &ofile, runs, nr, no ) ;
      linesout += no ;
    }

    /* y-pad */

    writeline ( &ofile, ( ( *runs = w ), runs ), 1, h - linesout ) ;
    
    if ( ferror ( ifile.f ) ) err = msg ( "ES2input error:" ) ;
  }

  nextopage ( &ofile, EOF ) ;
  
  return err ;
}


