// -*- C++ -*-
/* This file is part of
 * ======================================================
 * 
 *           LyX, the High Level Word Processor
 * 	 
 *	    Copyright (C) 1995 Matthias Ettrich
 *
 *======================================================*/
#ifndef _LYXPARAGRAPH_H
#define _LYXPARAGRAPH_H

#include "definitions.h"
#include "lyxfont.h"
#include "lyxinset.h"
#include "table.h"
#include "textutils.h"

class LyXParagraph  {
private:
	
	struct FontTable  {
		int pos;
		int pos_end;
		LyXFont font;
		FontTable *next;
	};
   
	struct InsetTable {
		int pos;
		Inset *inset;
		InsetTable *next;
	};
   
	FontTable *fonttable;
	InsetTable *insettable;
	int id;

public:
	int GetID(){
		return id;
	}
	void SetID(int id_arg){
		id = id_arg;
	}
	int size;
	/* allocates more memory for the specified paragraph
	 * pos is needed to specify the paragraph correctly. Remember the
	 * closed footnotes */ 
	void Enlarge(int pos, int number);
	
	/* make the allocated memory fit to the needed size
	 * used to make a paragraph smaller */
	void FitSize();
	void read();    
	void readSimpleWholeFile();   
   
	LyXParagraph* Clone();
	char HasSameLayout(LyXParagraph *par);
	void MakeSameLayout(LyXParagraph *par);
	
	char *text;
	char layout;
	char footnoteflag;	       /* no footnote, closed footnote, 
					* open footnote, where footnote
					* means footnote-environment */
   
	char footnotekind;	       /* footnote, margin, fig, tab  */
   
	/* the LyX- DTP-switches */ 
	char line_top;
	char line_bottom;
	
	char fill_top;
	char fill_bottom;
	
	float added_space_top;	       /* in cm   */
	float added_space_bottom;	       /* in cm   */
   
   
	char pagebreak_top;
	char pagebreak_bottom;
   
	char align;
	
	char depth;
	char noindent;
   
	char counter[10];
	char enumdepth;
	char itemdepth;
   
	char* labelstring;
	char* labelwidthstring;
   
	int last;
	LyXParagraph *next;
	LyXParagraph *previous;
 
	/* these function are able to hide closed footnotes */
	LyXParagraph *Next();
	LyXParagraph *Previous();
   
	/* these function are able to hide open and closed footnotes */ 
	LyXParagraph *NextAfterFootnote();  
	LyXParagraph *PreviousBeforeFootnote();  
	LyXParagraph *LastPhysicalPar();    
	LyXParagraph *FirstPhysicalPar();
	
	/* returns the physical paragraph */ 
	LyXParagraph *ParFromPos(int pos);
	/* returns the position in the physical par */
	int PositionInParFromPos(int pos);

	/* for the environments */ 
	LyXParagraph* DepthHook(int depth);
	
	int BeginningOfMainBody();
	char* GetLabelString();
	
	/* the next two functions are for the manual labels */ 
	char* GetLabelWidthString();
	void SetLabelWidthString(char* s);
	
	int GetLayout();
	char GetAlign();
	char GetDepth();
	void SetLayout(char new_layout);
	void SetOnlyLayout(char new_layout);
	
	char GetCounter(int i);
   
	int Last();
   
	LyXParagraph();
	/* this konstruktor inserts the new paragraph in a list */ 
	LyXParagraph(LyXParagraph *par);
	/* the destruktors removes the new paragraph from the list */ 
	~LyXParagraph();
   
	/* this one resets all layout and dtp switches but not the font
	 * of the single characters */ 
	void Clear();
	
	void Erase(int pos);
	
	/* the flag determines wether the layout should be copied */ 
	void BreakParagraph(int pos, int flag);
	void BreakParagraphConservative(int pos);
   
	/* paste this paragraph with the next one
	 * be carefull, this doesent make any check at all */ 
	void PasteParagraph();
   
	/* returns the difference between the characters font
	 * and the layoutfont . This is what is stored in the fonttable */ 
	LyXFont GetFontSettings(int pos);
	
	char GetChar(int pos);
	void SetFont(int pos, LyXFont font);
	void InsertChar(int pos, char c);
	void InsertInset(int pos, Inset *inset);
	Inset* GetInset(int pos);
	/* used to remove the error messages */
	int  AutoDeleteInsets();
	
	Inset* ReturnNextInsetPointer(int &pos);
	
	/* returns -1 if inset not found */
	int GetPositionOfInset(Inset* inset);
	
	/* ok and now some footnote functions */ 
	void OpenFootnotes();
	void OpenFootnote(int pos);
	void CloseFootnotes();
	void CloseFootnote(int pos);
   
	/* important for cut and paste */
	void CopyIntoMinibuffer(int pos);
	void CutIntoMinibuffer(int pos);
	void InsertFromMinibuffer(int pos);
	
	
	LyXParagraph *FirstSelfrowPar();
	
	/* table stuff -- begin*/ 
	LyXTable *table;
	/* table stuff -- end*/
	
	inline bool IsHfill(int pos) {
		return IsHfillChar(GetChar(pos));
	}	
	
	inline bool IsNewline(int pos) {
		if (pos<0)
			return 0;
		
		return IsNewlineChar(GetChar(pos));
	}	
	inline bool IsSeparator(int pos) {
		return IsSeparatorChar(GetChar(pos));
	}	
	inline bool IsLineSeparator(int pos) {
		return IsLineSeparatorChar(GetChar(pos));
	}
	inline bool IsKomma(int pos){
		return IsKommaChar(GetChar(pos));
	}
	inline bool IsWordSeparator(int pos){
		return IsWordSeparatorChar(GetChar(pos));
	}
	
	inline int ClearParagraph(){
		int i=0;
		if (!IsDummy() && !table){
			while (Last()
			       && (IsNewline(0) 
				   || IsLineSeparator(0))){
				Erase(0);
				i++;
			}
		}
		return i;
	}	

	inline bool IsDummy(){
		return (footnoteflag == LYX_NO_FOOTNOTE && previous
			&& previous->footnoteflag != LYX_NO_FOOTNOTE);
	}
};

#endif

