;*---------------------------------------------------------------------*/
;*    Copyright (c) 1996 by Manuel Serrano. All rights reserved.       */
;*                                                                     */
;*                                     ,--^,                           */
;*                               _ ___/ /|/                            */
;*                           ,;'( )__, ) '                             */
;*                          ;;  //   L__.                              */
;*                          '   \    /  '                              */
;*                               ^   ^                                 */
;*                                                                     */
;*                                                                     */
;*    This program is distributed in the hope that it will be useful.  */
;*    Use and copying of this software and preparation of derivative   */
;*    works based upon this software are permitted, so long as the     */
;*    following conditions are met:                                    */
;*           o credit to the authors is acknowledged following         */
;*             current academic behaviour                              */
;*           o no fees or compensation are charged for use, copies,    */
;*             or access to this software                              */
;*           o this copyright notice is included intact.               */
;*      This software is made available AS IS, and no warranty is made */
;*      about the software or its performance.                         */
;*                                                                     */
;*      Bug descriptions, use reports, comments or suggestions are     */
;*      welcome Send them to                                           */
;*        <Manuel.Serrano@inria.fr>                                    */
;*        Manuel Serrano                                               */
;*        INRIA -- Rocquencourt                                        */
;*        Domaine de Voluceau, BP 105                                  */
;*        78153 Le Chesnay Cedex                                       */
;*        France                                                       */
;*---------------------------------------------------------------------*/


;*=====================================================================*/
;*    serrano/prgm/project/bigloo/runtime1.8/Llib/error.scm            */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Fri Jan 20 08:19:23 1995                          */
;*    Last change :  Mon Jan 22 16:54:22 1996 (serrano)                */
;*    -------------------------------------------------------------    */
;*    The error machinery                                              */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    The module                                                       */
;*---------------------------------------------------------------------*/
(module __error

   (use     (__type                    "Llib/type.scm")
	    (__bigloo                  "Llib/bigloo.scm")
	    (__unix                    "Llib/unix.scm")
	    (__reader                  "Read/reader.scm")
	    (__tvector                 "Llib/tvector.scm")
	    (__rgc                     "Rgc/runtime.scm")
	    (__r4_vectors_6_8          "Ieee/vector.scm")
	    (__r4_numbers_6_5_fixnum   "Ieee/fixnum.scm")
	    (__r4_equivalence_6_2      "Ieee/equiv.scm")
	    (__r4_booleans_6_1         "Ieee/boolean.scm")
	    (__r4_pairs_and_lists_6_3  "Ieee/pair-list.scm")
	    (__r4_characters_6_6       "Ieee/char.scm")
	    (__r4_symbols_6_4          "Ieee/symbol.scm")
	    (__r4_strings_6_7          "Ieee/string.scm")
	    (__r4_ports_6_10_1         "Ieee/port.scm")
	    (__r4_control_features_6_9 "Ieee/control.scm")
	    (__r4_output_6_10_3        "Ieee/output.scm")
   	    (__evenv                   "Eval/evenv.scm"))

   (import  (__r4_input_6_10_2         "Ieee/input.scm"))

   (foreign (include "signal.h")
	    
	    (export the_failure "the_failure")
	    (export the_c_failure "the_c_failure")
	    
	    (obj   c-dump-trace-stack (output-port long) "dump_trace_stack")
	    (macro obj c-push-trace   (obj)              "PUSH_TRACE")
	    (macro obj c-pop-trace    ()                 "POP_TRACE")
	    (macro obj c-restore-trace ()                "RESTORE_TRACE")
	    (macro void c-exit        (long)             "exit")
	    (macro int sigfpe                            "SIGFPE")
	    (macro int sigill                            "SIGILL")
	    (macro int sigbus                            "SIGBUS")
	    (macro int sigsegv                           "SIGSEGV"))
   
   (export  (type-error-msg::bstring     ::symbol ::symbol)
	    (inline exit::magic          ::long)
	    (inline error::obj           ::obj ::obj ::obj)
	    (warning                     . args)
	    (warning/location::obj       ::obj ::obj . obj)
	    (error/location::obj         ::obj ::obj ::obj ::obj ::obj)
	    (error/location-file::obj    ::obj ::obj)
	    (notify-error                ::obj ::obj ::obj)
	    (add-error-handler!          ::obj ::obj)
	    (remove-error-handler!)
	    (the_failure::magic          ::obj ::obj ::obj)
	    (the_c_failure::magic        ::string ::string ::obj)
	    *error-notifier*
	    *trace-stack-depth*
	    *debug*
	    *warning*))

;*---------------------------------------------------------------------*/
;*    This is a big hack to prevent the compiler to introduce traces   */
;*    in this module                                                   */
;*---------------------------------------------------------------------*/
(define-macro (no-trace-no-check)
   (set! *compiler-debug* 0)
   (set! *unsafe-type*    #t)
   (set! *unsafe-arity*   #t)
   (set! *unsafe-range*   #t)
   #unspecified)

(no-trace-no-check)

;*---------------------------------------------------------------------*/
;*    *debug* ...                                                      */
;*---------------------------------------------------------------------*/
(define *debug* 0)

;*---------------------------------------------------------------------*/
;*    *warning* ...                                                    */
;*---------------------------------------------------------------------*/
(define *warning* #t)

;*---------------------------------------------------------------------*/
;*    *trace-stack-depth* ...                                          */
;*---------------------------------------------------------------------*/
(define *trace-stack-depth* 10)

;*---------------------------------------------------------------------*/
;*    exit ...                                                         */
;*---------------------------------------------------------------------*/
(define-inline (exit n)
   (c-exit n)
   n)

;*---------------------------------------------------------------------*/
;*    La valeur par defaut du *error-handler*                          */
;*---------------------------------------------------------------------*/
(define *error-handler* '())

;*---------------------------------------------------------------------*/
;*    dump-trace-stack ...                                             */
;*---------------------------------------------------------------------*/
(define-inline (dump-trace-stack)
   (let ((depth (if (integer? *trace-stack-depth*)
		    *trace-stack-depth*
		    10)))
      (c-dump-trace-stack (current-error-port) depth)))
					       
;*---------------------------------------------------------------------*/
;*    add-error-handler! ...                                           */
;*---------------------------------------------------------------------*/
(define (add-error-handler! handler escape)
   (set! *error-handler* (cons (cons handler escape) *error-handler*)))

;*---------------------------------------------------------------------*/
;*    remove-error-handler! ...                                        */
;*---------------------------------------------------------------------*/
(define (remove-error-handler!)
   (if (pair? *error-handler*)
       (set! *error-handler* (cdr *error-handler*)))
   *error-handler*)

;*---------------------------------------------------------------------*/
;*    error ...                                                        */
;*---------------------------------------------------------------------*/
(define-inline (error proc message object)
   (failure proc message object))
   
;*---------------------------------------------------------------------*/
;*    warning ...                                                      */
;*---------------------------------------------------------------------*/
(define (warning . args)
   (if *warning*
       (begin
	  (flush-output-port (current-output-port))
	  (newline (current-error-port))
	  (display "*** WARNING:bigloo:" (current-error-port))
	  (if (not (null? args))
	      (begin
		 (fprint (current-error-port) (car args))
		 (for-each (lambda (a)
			      (display a (current-error-port)))
			   (cdr args))))
	  (newline (current-error-port)))))

;*---------------------------------------------------------------------*/
;*    warning/location ...                                             */
;*---------------------------------------------------------------------*/
(define (warning/location fname location . args)
   (if *warning*
       (cond
	  ((string=? fname "[string]")
	   (apply warning args))
	  ((string=? fname "[stdin]")
	   (apply warning args))
	  (else
	   (warning/location-file fname location args)))))

;*---------------------------------------------------------------------*/
;*    the_failure ...                                                  */
;*---------------------------------------------------------------------*/
(define (the_failure proc message object)
   (reader-reset!)
   (if (not (pair? *error-handler*))
       (default-error-handler proc message object)
       (let ((handler (car (car *error-handler*)))
	     (escape  (cdr (car *error-handler*))))
	  (remove-error-handler!)
	  (if (and (procedure? handler) (=fx (procedure-arity handler) 4))
	      (handler escape proc message object)
	      (incorrect-error-handler handler)))))

;*---------------------------------------------------------------------*/
;*    the_c_failure ...                                                */
;*---------------------------------------------------------------------*/
(define (the_c_failure proc message object)
   (the_failure proc message object))

;*---------------------------------------------------------------------*/
;*    notify-error ...                                                 */
;*---------------------------------------------------------------------*/
(define (notify-error proc mes obj)
   (if (procedure? *error-notifier*)
       (*error-notifier* proc mes obj)
       (begin
	  (flush-output-port (current-output-port))
	  (newline (current-error-port))
	  (let ((old-length (get-write-length)))
	     (set-write-length! 80)
	     (display "*** ERROR:bigloo:" (current-error-port))
	     (fprint (current-error-port) proc ":" #\Newline mes " -- " obj)
	     (if (and (integer? *debug*)
		      (>fx *debug* 0))
		 (dump-trace-stack))
	     (set-write-length! old-length)))))

;*---------------------------------------------------------------------*/
;*    type-error-msg ...                                               */
;*---------------------------------------------------------------------*/
(define (type-error-msg from to)
   (string-append "`"
		  (symbol->string to) "' expected, `"
		  (symbol->string from) "' provided."))

;*---------------------------------------------------------------------*/
;*    *error-notifier* ...                                             */
;*---------------------------------------------------------------------*/
(define *error-notifier* #unspecified)

;*---------------------------------------------------------------------*/
;*    default-error-handler ...                                        */
;*---------------------------------------------------------------------*/
(define (default-error-handler proc mes obj)
   (notify-error proc mes obj)
   -1)

;*---------------------------------------------------------------------*/
;*    incorrect-error-handler ...                                      */
;*---------------------------------------------------------------------*/
(define (incorrect-error-handler handler)
   (default-error-handler "error" "Not an error handler" handler))

;*---------------------------------------------------------------------*/
;*    sigfpe-error-handler ...                                         */
;*---------------------------------------------------------------------*/
(define (sigfpe-error-handler n)
   (error "arithmetic procedure" "`floating point' exception" "reached"))

;*---------------------------------------------------------------------*/
;*    sigill-error-handler ...                                         */
;*---------------------------------------------------------------------*/
(define (sigill-error-handler n)
   (error "bigloo" "`illegal instruction' exception" "reached"))

;*---------------------------------------------------------------------*/
;*    sigbus-error-handler ...                                         */
;*---------------------------------------------------------------------*/
(define (sigbus-error-handler n)
   (error "bigloo" "`bus error' exception" "reached"))

;*---------------------------------------------------------------------*/
;*    sigsegv-error-handler ...                                        */
;*---------------------------------------------------------------------*/
(define (sigsegv-error-handler n)
   (error "bigloo" "`segmentation violation' exception" "reached"))

;*---------------------------------------------------------------------*/
;*    On installe le ratrappage des exceptions                         */
;*---------------------------------------------------------------------*/
(signal sigfpe  sigfpe-error-handler)
(signal sigill  sigill-error-handler)
(signal sigbus  sigbus-error-handler)
(signal sigsegv sigsegv-error-handler)

;*---------------------------------------------------------------------*/
;*    error/location ...                                               */
;*    -------------------------------------------------------------    */
;*    We print error message understable by emacs. We                  */
;*    print them in the following format:                              */
;*      `File "foobar.scm", lines 8, character 20: blah blah'          */
;*---------------------------------------------------------------------*/
(define (error/location proc message object fname location)
   (cond
      ((not (string? fname))
       (error proc message object))
      ((not (fixnum? location))
       (error proc message object))
      ((string=? fname "[string]")
       (error proc message object))
      ((string=? fname "[stdin]")
       (error proc message object))
      (else
       (set! *error-notifier* (error/location-file fname location))
       (error proc message object))))

;*---------------------------------------------------------------------*/
;*    error/location-file ...                                          */
;*---------------------------------------------------------------------*/
(define (error/location-file file-name location)
   ;; we compute the message to print the location
   (let ((port (open-input-file file-name)))
      (if (not (input-port? port))
	  ;; we are enable to re-open the file, we just print a
	  ;; standard error
	  #f
	  ;; we readlines until we reach location
	  (let loop ((line-string (read-line port))
		     (line-num    1))
	     (if (eof-object? line-string)
		 (begin
		    ;; an error we don't know how to print
		    (close-input-port port)
		    #f)
		 (if (>=fx (c-input-port-filepos port) location)
		     (begin
			(close-input-port port)
			(make-location-notifier file-name
						line-num
						location
						line-string
						(+fx
						 (-fx (string-length
						       line-string)
						      (-fx
						       (c-input-port-filepos
							port)
						       location))
						 1)))
		     (let ((old-pos (c-input-port-filepos port)))
			(loop (read-line port)
			      (+fx line-num 1)))))))))

;*---------------------------------------------------------------------*/
;*    make-location-notifier ...                                       */
;*---------------------------------------------------------------------*/
(define (make-location-notifier fname line char string marker)
   (lambda (proc msg obj)
      ;; we first re-install default notifier
      (set! *error-notifier* #f)
      ;; we flush error-port
      (flush-output-port (current-output-port))
      (newline (current-error-port))
      (let ((old-length   (get-write-length))
	    (space-string (if (>fx marker 0)
			      (make-string (-fx marker 1) #\space)
			      "")))
	 ;; we ajust tabulation in space string.
	 (fix-tabulation! marker string space-string)
	 ;; we now print the error message
	 (print-cursor fname line char string space-string)
	 ;; we set the write-length before printing the error message.
	 (set-write-length! 80)
	 ;; we display the error message
	 (fprint (current-error-port) "# *** ERROR:bigloo:" proc)
	 (fprint (current-error-port) "# " msg " -- " obj)
	 (if (and (integer? *debug*)
		  (>fx *debug* 0))
	     (dump-trace-stack))
	 ;; we restore the current-printer
	 (set-write-length! old-length))))

;*---------------------------------------------------------------------*/
;*    warning/location-file ...                                        */
;*---------------------------------------------------------------------*/
(define (warning/location-file file-name location args)
   ;; we compute the message to print the location
   (let ((port (open-input-file file-name)))
      (if (not (input-port? port))
	  ;; we are enable to re-open the file, we just print a
	  ;; standard warning
	  (apply warning args)
	  ;; we readlines until we reach location
	  (let loop ((line-string (read-line port))
		     (line-num    1))
	     (if (eof-object? line-string)
		 (begin
		    ;; an error we don't know how to print
		    (close-input-port port)
		    (apply warning args))
		 (if (>=fx (c-input-port-filepos port) location)
		     (begin
			(close-input-port port)
			(do-warn/location file-name
					  line-num
					  location
					  line-string
					  (+fx
					   (-fx (string-length
						 line-string)
						(-fx
						 (c-input-port-filepos
						  port)
						 location))
					   1)
					  args))
		     (let ((old-pos (c-input-port-filepos port)))
			(loop (read-line port)
			      (+fx line-num 1)))))))))

;*---------------------------------------------------------------------*/
;*    do-warn/location ...                                             */
;*---------------------------------------------------------------------*/
(define (do-warn/location fname line char string marker args)
   (flush-output-port (current-output-port))
   (newline (current-error-port))
   (let ((old-length   (get-write-length))
	 (space-string (if (>fx marker 0)
			   (make-string (-fx marker 1) #\space)
			   "")))
      ;; we ajust tabulation in space string.
      (fix-tabulation! marker string space-string)
      ;; we now print the warning message
      (print-cursor fname line char string space-string)
      ;; we display the warning message
      (display "# *** WARNING:bigloo:" (current-error-port))
      (if (not (null? args))
	  (begin
	     (fprint (current-error-port) (car args))
	     (for-each (lambda (a)
			  (display a (current-error-port)))
		       (cdr args))))
      (newline (current-error-port))))

;*---------------------------------------------------------------------*/
;*    fix-tabulation! ...                                              */
;*---------------------------------------------------------------------*/
(define (fix-tabulation! marker src dst)
   (let loop ((read (-fx marker 1)))
      (cond
	 ((=fx read -1)
	  'done)
	 ((char=? (string-ref src read) #\tab)
	  (string-set! dst read #\tab)
	  (loop (-fx read 1)))
	 (else
	  (loop (-fx read 1))))))

;*---------------------------------------------------------------------*/
;*    print-cursor ...                                                 */
;*---------------------------------------------------------------------*/
(define (print-cursor fname line char string space-string)
   (fprint (current-error-port)
	   "File \"" (nice-fname fname) "\", line " line ", character "
	   char ":"
	   #\Newline
	   "#" string #\Newline
	   "#"
	   space-string
	   "^"))

;*---------------------------------------------------------------------*/
;*    nice-fname ...                                                   */
;*    -------------------------------------------------------------    */
;*    We remove the current path to fname                              */
;*---------------------------------------------------------------------*/
(define (nice-fname fname)
   (let ((pwd (let ((vpwd (getenv "PWD")))
		 (if (string? vpwd)
		     vpwd
		     (pwd)))))
      (if (not (string? pwd))
	  fname
	  (let ((lend (string-length pwd))
		(lenf (string-length fname)))
	     (if (<fx lenf lend)
		 fname
		 (let loop ((r 0))
		    (cond
		       ((=fx r lend)
			(if (and (<fx r lenf)
				 (char=? (string-ref fname r) #\/))
			    (substring fname (+fx r 1) lenf)
			    (substring fname r lenf)))
		       ((char=? (string-ref fname r) (string-ref pwd r))
			(loop (+fx r 1)))
		       (else
			(substring fname r lenf)))))))))

	     
