/* Perform network queries on the contents of a GNATS database.
   Copyright (C) 1994, 1995 Free Software Foundation, Inc.
   Contributed by Brendan Kehoe (brendan@cygnus.com).

This file is part of GNU GNATS.

GNU GNATS is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU GNATS is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU GNATS; see the file COPYING.  If not, write to the Free
Software Foundation, 59 Temple Place - Suite 330, Boston, MA 02111, USA.  */

#include "config.h"
#include "gnats.h"
#include "query.h"

/* The descriptor for talking to the server in non-stdio ways.  */
extern int sockfd;

/* If 1, send information as we make the query.  */
int debug = 0;

/* If 1, ask about every PR in the database.  */
int entirety = 0;

/* If 1, don't allow redirection or viewing of confidential PRs.  */
int restricted = 0;

/* If specified, search for the originator of this PR.  */
char *originator = NULL;

/* If specified, search for the class of this PR.  */
char *class = NULL;

#ifdef GNATS_RELEASE_BASED
/* If specified, search for the class of this PR.  */
char *quarter = NULL;

/* If specified, search for the class of this PR.  */
char *keywords = NULL;
#endif

/* Whether or not to search the text fields, and the regexp to
   search for.  */
char *text_search = NULL;

/* Whether or not to search the multitext fields, and the regexp to
   search for.  */
char *m_text_search = NULL;

/* If set, match PRs with a Date-Required: field before this value.  */
char *before_string = NULL;

/* If set, match PRs with a Date-Required: field after this value.  */
char *after_string = NULL;

/* If 1, print the whole PR out, not just the choice bits.  */
int full_report = 0;

/* If 1, emit info for a SQL database.  */
int sql_format = 0;

/* If 1, don't talk about closed PRs.  */
int skip_closed = 0;

/* How we should print out the format of the PR.  */
unsigned int query_format = FORMAT_REG;

/* What list we're interested in seeing.  */
unsigned int list_format = 0;

/* The name of the GNATS server.  */
char *name;

/* The name this program was run with.  */
char *program_name;

struct option long_options[] =
{
  {"category", 1, NULL, 'c'},
  {"confidential", 1, NULL, 'C'},
  {"debug", 0, NULL, 'D'},
  {"directory", 1, NULL, 'd'},
  {"entire", 0, NULL, 'E'},
  {"full", 0, NULL, 'F'},
  {"help", 0, NULL, 'h'},
  {"host", 1, NULL, 'H'},
  {"multitext", 1, NULL, 'm'},
  {"originator", 1, NULL, 'O'},
  {"class", 1, NULL, 'L'},
#ifdef GNATS_RELEASE_BASED
  {"quarter", 1, NULL, 'Q'},
  {"keywords", 1, NULL, 'K'},
#endif
  {"output", 1, NULL, 'o'},
  {"priority", 1, NULL, 'p'},
  {"print-path", 0, NULL, 'P'},
  {"responsible", 1, NULL, 'r'},
  {"restricted", 0, NULL, 'R'},
  {"severity", 1, NULL, 'e'},
  {"skip-closed", 0, NULL, 'x'},
  {"sql", 0, NULL, 'i'},
  {"state", 1, NULL, 's'},
  {"summary", 0, NULL, 'q'},
  {"submitter", 1, NULL, 'S'},
  {"text", 1, NULL, 't'},
#ifdef GNATS_RELEASE_BASED
  {"required-before", 1, NULL, 'u'},
  {"required-after", 1, NULL, 'U'},
#endif
  {"arrived-before", 1, NULL, 'b'},
  {"arrived-after", 1, NULL, 'a'},
  {"list-categories", 0, NULL, 'j'},
  {"list-responsible", 0, NULL, 'k'},
  {"list-submitters", 0, NULL, 'l'},
  {"version", 0, NULL, 'V'},
  {NULL, 0, NULL, 0}
};

void usage (), version ();

void
ask_about (info, args)
     Index *info;
     char **args;
{
  char *cmd;
  int current = 1024; /* XXX not 1024, the best one... */
  char *buf;
  int i = 0, n;
  int len = 5;

  if (list_format)
    {
      if (list_format == LIST_CATEGORIES)
	{
	  if (debug)
	    fprintf (stderr, "%s: writing `LCAT'\n", program_name);
	  fprintf (serv_write, "LCAT\r\n");
	}
      else if (list_format == LIST_RESPONSIBLE)
	{
	  if (debug)
	    fprintf (stderr, "%s: writing `LRES'\n", program_name);
	  fprintf (serv_write, "LRES\r\n");
	}
      else if (list_format == LIST_SUBMITTERS)
	{
	  if (debug)
	    fprintf (stderr, "%s: writing `LSUB'\n", program_name);
	  fprintf (serv_write, "LSUB\r\n");
	}
      return;
    }

  if (info->category)
    {
      if (debug)
	fprintf (stderr, "%s: writing `CATG %s'\n", program_name,
		 info->category);
      fprintf (serv_write, "CATG %s\r\n", info->category);
      get_reply ();
    }
#if 0 /* not in gnatsd */
  if (info->number)
    {
      fprintf (serv_write, "CATG %s\r\n", info->number);
      get_reply ();
    }
#endif
  if (info->submitter)
    {
      if (debug)
	fprintf (stderr, "%s: writing `SUBM %s'\n", program_name,
		 info->submitter);
      fprintf (serv_write, "SUBM %s\r\n", info->submitter);
      get_reply ();
    }
  if (info->responsible)
    {
      if (debug)
	fprintf (stderr, "%s: writing `RESP %s'\n", program_name,
		 info->responsible);
      fprintf (serv_write, "RESP %s\r\n", info->responsible);
      get_reply ();
    }
  if (info->state)
    {
      if (debug)
	fprintf (stderr, "%s: writing `STAT %s'\n", program_name,
		 info->state);
      fprintf (serv_write, "STAT %s\r\n", info->state);
      get_reply ();
    }
  if (info->confidential)
    {
      if (debug)
	fprintf (stderr, "%s: writing `CONF %s'\n", program_name,
		 info->confidential);
      fprintf (serv_write, "CONF %s\r\n", info->confidential);
      get_reply ();
    }
  if (info->severity)
    {
      if (debug)
	fprintf (stderr, "%s: writing `SVTY %s'\n", program_name,
		 info->severity);
      fprintf (serv_write, "SVTY %s\r\n", info->severity);
      get_reply ();
    }
  if (info->priority)
    {
      if (debug)
	fprintf (stderr, "%s: writing `PRIO %s'\n", program_name,
		 info->priority);
      fprintf (serv_write, "PRIO %s\r\n", info->priority);
      get_reply ();
    }

  if (skip_closed)
    {
      if (debug)
	fprintf (stderr, "%s: writing `NOCL'\n", program_name);
      fprintf (serv_write, "NOCL\r\n");
      get_reply ();
    }

  if (originator)
    {
      if (debug)
	fprintf (stderr, "%s: writing `ORIG %s'\n", program_name, originator);
      fprintf (serv_write, "ORIG %s\r\n", originator);
      get_reply ();
    }

  if (class)
    {
      if (debug)
	fprintf (stderr, "%s: writing `CLSS %s'\n", program_name, class);
      fprintf (serv_write, "CLSS %s\r\n", class);
      get_reply ();
    }

#ifdef GNATS_RELEASE_BASED
  if (quarter)
    {
      if (debug)
	fprintf (stderr, "%s: writing `QRTR %s'\n", program_name, quarter);
      fprintf (serv_write, "QRTR %s\r\n", quarter);
      get_reply ();
    }

  if (keywords)
    {
      if (debug)
	fprintf (stderr, "%s: writing `KYWD %s'\n", program_name, keywords);
      fprintf (serv_write, "KYWD %s\r\n", keywords);
      get_reply ();
    }
#endif

  if (text_search)
    {
      if (debug)
	fprintf (stderr, "%s: writing `TEXT %s'\n", program_name, text_search);
      fprintf (serv_write, "TEXT %s\r\n", text_search);
      get_reply ();
    }

  if (m_text_search)
    {
      if (debug)
	fprintf (stderr, "%s: writing `MTXT %s'\n", program_name,
		 m_text_search);
      fprintf (serv_write, "MTXT %s\r\n", m_text_search);
      get_reply ();
    }

  if (before_string)
    {
      if (debug)
	fprintf (stderr, "%s: writing `BFOR %s'\n", program_name,
		 before_string);
      fprintf (serv_write, "BFOR %s\r\n", before_string);
      get_reply ();
    }

  if (after_string)
    {
      if (debug)
	fprintf (stderr, "%s: writing `AFTR %s'\n", program_name,
		 after_string);
      fprintf (serv_write, "AFTR %s\r\n", after_string);
      get_reply ();
    }

  if (query_format & FORMAT_SUMM)
    cmd = "SUMM";
  else if (query_format & FORMAT_FULL)
    cmd = "FULL";
  else if (query_format & FORMAT_SQL)
    cmd = "SQLF";
  else
    cmd = "QURY";

  buf = (char *) xmalloc (current);
  strcpy (buf, cmd);
  while (args[i])
    {
      sprintf (buf, "%s %s", buf, args[i]);
      len += strlen (args[i++]) + 1;
      if (args[i] == NULL)
	break;
      n = strlen (args[i]);
      if (len + n >= current)
	{
	  current += 1024;
	  buf = (char *) xrealloc (buf, current);
	}
    }

  if (debug)
    fprintf (stderr, "%s: writing `%s'\n", program_name, buf);
  fprintf (serv_write, "%s\r\n", buf);
  xfree (buf);
}


int
main (argc, argv)
     int argc;
     char **argv;
{
  int optc;
  Index *info;
  int formats = 0, lists = 0;

  program_name = basename (argv[0]);
  if (argc == 1)
    usage (1);

  name = gnats_server;

  info = (Index *) xmalloc (sizeof (Index));
  bzero ((Index *) info, sizeof (Index));

#ifdef GNATS_RELEASE_BASED
  while ((optc = getopt_long (argc, argv, "c:C:Dd:e:K:L:m:o:O:p:PQ:s:S:r:t:u:U:VFixhqH:Rjkl",
#else
  while ((optc = getopt_long (argc, argv, "a:b:c:C:Dd:e:L:m:o:O:p:Ps:S:r:t:VFixhqH:Rjkl",
#endif
			      long_options, (int *) 0)) != EOF)
    {
      switch (optc)
	{
	case 'd':
#if 0
	  spec_gnats_root = optarg;
#else
	  fprintf (stderr, "%s: directory option not valid for network query-pr\n",
		   program_name);
	  exit (1);
#endif
	  break;

	case 'D':
	  debug = 1;
	  break;

	case 'E':
	  entirety = 1;
	  break;

	case 'o':
#if 0
	  if (strcmp (optarg, "-") && !restricted)
	    {
	      outfile = fopen (optarg, "w+");
	      if (outfile == (FILE *) NULL)
		{
		  fprintf (stderr, "%s: can not read file %s\n",
			   program_name, optarg);
		  exit (3);
		}
	    }
#endif
	  break;

	case 'r':
	  info->responsible = optarg;
	  break;

	case 'R':
	  restricted = 1;
#if 0
	  outfile = stdout;
#endif
	  info->confidential = "no";
	  break;

	case 'c':
	  info->category = optarg;
	  break;

	case 'C':
	  if (! restricted)
	    info->confidential = optarg;
	  break;

	case 'e':
	  info->severity = optarg;
	  break;

	case 'L':
	  class = optarg;
	  break;

#ifdef GNATS_RELEASE_BASED
	case 'K':
	  keywords = optarg;
	  break;

	case 'Q':
	  quarter = optarg;
	  break;
#endif

	case 'm':
	  m_text_search = optarg;
	  break;

	case 'O':
	  originator = optarg;
	  break;

	case 'p':
	  info->priority = optarg;
	  break;

	case 'P':
	  fprintf (stderr,
		   "%s: -P/--print-path does not work for network queries\n",
		   program_name);
	  exit (1);
	  break;

	case 's':
	  info->state = optarg;
	  break;

	case 'S':
	  info->submitter = optarg;
	  break;

	case 't':
	  text_search = optarg;
	  break;

	case 'V':
	  version ();
	  exit (0);
	  break;

	case 'F':
	  query_format |= FORMAT_FULL;
	  formats++;
	  break;

	case 'i':
	  query_format |= FORMAT_SQL;
	  formats++;
	  break;

	case 'q':
	  query_format |= FORMAT_SUMM;
	  formats++;
	  break;

	case 'x':
	  skip_closed = 1;
	  break;

	case 'j':
	  list_format = LIST_CATEGORIES;
	  lists++;
	  break;

	case 'k':
	  list_format = LIST_RESPONSIBLE;
	  lists++;
	  break;

	case 'l':
	  list_format = LIST_SUBMITTERS;
	  lists++;
	  break;

	case 'u':
	  before_string = optarg;
	  break;

	case 'U':
	  after_string = optarg;
	  break;

	case 'H':
	  name = optarg;
	  break;

	case 'h':
	  usage (0);
	  break;

	default:
	  usage (1);
	}
    }

  if (formats > 1)
    {
      fprintf (stderr, "%s: only one of -i, -F, or -q may be specified\n",
	       program_name);
      exit (3);
    }

  if (lists > 1)
    {
      fprintf (stderr, "%s: only one of -j, -k, or -l may be specified alone\n",
	       program_name);
      exit (4);
    }

  client_init ();

  /* Now make our query, and act on what they said.  */
  ask_about (info, argv + optind);
  get_reply ();

  client_exit ();

  exit (0);
}

void
usage (s)
     int s;
{
#ifdef GNATS_RELEASE_BASED
  fprintf (stderr, "\
Usage: %s [-DFHhiPqVxE] [-C confidential] [-c category] [-d directory]\n\
       [-e severity] [-m mtext] [-O originator] [-o outfile] [-p priority]\n\
       [-L class] [-r responsible] [-S submitter] [-s state] [-t text]\n\
       [--full] [--help] [--sql] [--print-path] [--summary] [--version]\n\
       [--skip-closed] [--category=category] [--confidential=yes|no]\n\
       [--directory=directory] [--output=outfile] [--originator=name]\n\
       [--priority=level] [--responsible=person] [--severity=severity]\n\
       [--state=state] [--submitter=submitter] [--class=class]\n\
       [--quarter=quarter] [--keywords=regexp]\n\
       [--required-before=date] [--required-after=date]\n\
       [--arrived-before=date] [--arrived-after=date]\n\
       [--list-categories] [--list-responsible] [--list-submitters]\n\
       [--text=text] [--multitext=mtext] [--host=server] [--debug] [PR] [PR]...\n",
	   program_name);
#else
  fprintf (stderr, "\
Usage: %s [-DFHhiPqVxE] [-C confidential] [-c category] [-d directory]\n\
       [-e severity] [-m mtext] [-O originator] [-o outfile] [-p priority]\n\
       [-L class] [-r responsible] [-S submitter] [-s state] [-t text]\n\
       [--full] [--help] [--sql] [--print-path] [--summary] [--version]\n\
       [--skip-closed] [--category=category] [--confidential=yes|no]\n\
       [--directory=directory] [--output=outfile] [--originator=name]\n\
       [--priority=level] [--responsible=person] [--severity=severity]\n\
       [--state=state] [--submitter=submitter] [--class=class]\n\
       [--quarter=quarter] [--keywords=regexp]\n\
       [--arrived-before=date] [--arrived-after=date]\n\
       [--list-categories] [--list-responsible] [--list-submitters]\n\
       [--text=text] [--multitext=mtext] [--host=server] [--debug] [PR] [PR]...\n",
	   program_name);
#endif
  exit (s);
}

void
version ()
{
  printf ("net-query-pr %s\n", version_string);
}
