/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

#include "mutt.h"
#include "mutt_curses.h"
#include "keymap.h"
#include "send.h"

#ifdef _PGPPATH
#include "pgp.h"
#endif

#include <ctype.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <dirent.h>

FILE *mutt_open_read (const char *path, pid_t *thepid)
{
  FILE *f;
  int len = strlen (path);

  if (path[len-1] == '|')
  {
    /* read from a pipe */

    char *s = safe_strdup (path);

    s[len-1] = 0;
    endwin ();
    *thepid = mutt_create_filter (s, NULL, &f, NULL);
    free (s);
  }
  else
  {
    f = fopen (path, "r");
    *thepid = -1;
  }
  return (f);
}

static void append_signature (ENVELOPE *env, FILE *f)
{
  char *s;
  FILE *tmpfp;
  int isRemote = 0;
  pid_t thepid;

  if (LocalSig[0] || RemoteSig[0])
  {
    if (env->to && !is_local_site (env->to))
      isRemote++;
    if (env->cc && !is_local_site (env->cc))
      isRemote++;
    if (env->bcc && !is_local_site (env->bcc))
      isRemote++;

    /* if no addresses were given, default to remotesignature */
    if (isRemote || !(env->to || env->cc || env->bcc))
      s = RemoteSig;
    else
      s = LocalSig;
  }
  else
    s = Signature;

  if ((tmpfp = mutt_open_read (s, &thepid)))
  {
    if (option (OPTSIGDASHES))
      fputs ("\n-- \n", f);
    mutt_copy_stream (tmpfp, f);
    fclose (tmpfp);
    if (thepid != -1)
      mutt_wait_filter (thepid);
  }
}

/* compare two e-mail addresses and return 1 if they are equivalent */
static int mutt_addrcmp (ADDRESS *a, ADDRESS *b)
{
  if (!a->mailbox || !b->mailbox)
    return 0;
  if (strcasecmp (a->mailbox, b->mailbox))
    return 0;
  if (!a->host || !b->host)
  {
    if (!a->host && !b->host)
      return 1;
    else
      return 0;
  }
  if (strcasecmp (a->host, b->host))
    return 0;
  return 1;
}

/* search an e-mail address in a list */
static int mutt_addrsrc (ADDRESS *a, ADDRESS *lst)
{
  for (; lst; lst = lst->next)
  {
    if (mutt_addrcmp (a, lst))
      return (1);
  }
  return (0);
}

/* removes addresses from "b" which are contained in "a" */
static ADDRESS *mutt_remove_xrefs (ADDRESS *a, ADDRESS *b)
{
  ADDRESS *top, *p, *prev = NULL;

  top = b;
  while (b)
  {
    p = a;
    while (p)
    {
      if (mutt_addrcmp (p, b))
	break;
      p = p->next;
    }
    if (p)
    {
      if (prev)
      {
	prev->next = b->next;
	b->next = NULL;
	mutt_free_address (&b);
	b = prev;
      }
      else
      {
	top = top->next;
	b->next = NULL;
	mutt_free_address (&b);
	b = top;
      }
    }
    else
    {
      prev = b;
      b = b->next;
    }
  }
  return top;
}

static ADDRESS *remove_user (ADDRESS *a)
{
  ADDRESS *top = 0, *last = 0;

  while (a)
  {
    if (!mutt_addr_is_user (a))
    {
      if (top)
      {
        last->next = a;
        last = last->next;
      }
      else
        last = top = a;
      a = a->next;
      last->next = 0;
    }
    else
    {
      ADDRESS *tmp = a;
      
      a = a->next;
      tmp->next = 0;
      mutt_free_address (&tmp);
    }
  }
  return top;
}

static ADDRESS *find_mailing_lists (ADDRESS *t, ADDRESS *c)
{
  ADDRESS *top = 0, *ptr = t, *tmp;

  while (t)
  {
    if (mutt_is_mail_list (t))
    {
      /*
       * rfc822_cpy_adr() copies the whole list, not just one element, so
       * temporarily cut the list.
       */
      tmp = t->next;
      t->next = 0;
      if (top)
        ptr->next = rfc822_cpy_adr(t);
      else
        ptr = top = rfc822_cpy_adr(t);
      t->next = tmp;
      while (ptr->next) ptr = ptr->next;
    }
    t = t->next;
    if (!t && c)
    {
      /* now look at the CC list */
      t = c;
      c = 0;
    }
  }
  return top;
}

static int edit_envelope (ENVELOPE *en)
{
  char buf[HUGE_STRING];

  buf[0] = 0;
  rfc822_write_address (buf, sizeof (buf), en->to);
  if (ci_get_field ("To: ", buf, sizeof (buf), M_ALIAS) != 0 || !buf[0])
    return (-1);
  mutt_free_address (&en->to);
  mutt_parse_adrlist (&en->to, buf, "@");
  en->to = mutt_expand_aliases (en->to);

  if (option (OPTASKCC))
  {
    buf[0] = 0;
    rfc822_write_address (buf, sizeof (buf), en->cc);
    if (ci_get_field ("Cc: ", buf, sizeof (buf), M_ALIAS) != 0)
      return (-1);
    mutt_free_address (&en->cc);
    mutt_parse_adrlist (&en->cc, buf, "@");
    en->cc = mutt_expand_aliases (en->cc);
  }

  if (option (OPTASKBCC))
  {
    buf[0] = 0;
    rfc822_write_address (buf, sizeof (buf), en->bcc);
    if (ci_get_field ("Bcc: ", buf, sizeof (buf), M_ALIAS) != 0)
      return (-1);
    mutt_free_address (&en->bcc);
    mutt_parse_adrlist (&en->bcc, buf, "@");
    en->bcc = mutt_expand_aliases (en->bcc);
  }

  if (en->subject)
    strfcpy (buf, en->subject, sizeof (buf));
  else
    buf[0] = 0;
  if (ci_get_field ("Subject: ", buf, sizeof (buf), 0) != 0 ||
      (!buf[0] && query_quadoption (OPT_SUBJECT, "No subject, abort?") != 0))
    {
      mutt_message ("No subject, aborting.");
      return (-1);
    }
  safe_free ((void **) &en->subject);
  en->subject = safe_strdup (buf);

  return 0;
}

static void process_user_recips (ENVELOPE *env)
{
  LIST *uh = UserHeader;

  for (; uh; uh = uh->next)
  {
    if (strncasecmp ("to:", uh->data, 3) == 0)
      rfc822_parse_adrlist (&env->to, uh->data + 3, "@");
    else if (strncasecmp ("cc:", uh->data, 3) == 0)
      rfc822_parse_adrlist (&env->cc, uh->data + 3, "@");
    else if (strncasecmp ("bcc:", uh->data, 4) == 0)
      rfc822_parse_adrlist (&env->bcc, uh->data + 4, "@");
  }
}

static void process_user_header (ENVELOPE *env)
{
  LIST *uh = UserHeader;
  LIST *last = env->userhdrs;

  if (last)
    while (last->next)
      last = last->next;

  for (; uh; uh = uh->next)
  {
    if (strncasecmp ("from:", uh->data, 5) == 0)
    {
      /* User has specified a default From: address.  Get rid of the
       * default address.
       */
      mutt_free_address (&env->from);
      rfc822_parse_adrlist (&env->from, uh->data + 5, "@");
    }
    else if (strncasecmp ("reply-to:", uh->data, 9) == 0)
    {
      mutt_free_address (&env->reply_to);
      rfc822_parse_adrlist (&env->reply_to, uh->data + 9, "@");
    }
    else if (strncasecmp ("to:", uh->data, 3) != 0 &&
	     strncasecmp ("cc:", uh->data, 3) != 0 &&
	     strncasecmp ("bcc:", uh->data, 4) != 0)
    {
      if (last)
      {
	last->next = mutt_new_list ();
	last = last->next;
      }
      else
	last = env->userhdrs = mutt_new_list ();
      last->data = safe_strdup (uh->data);
    }
  }
}

LIST *mutt_copy_list (LIST *p)
{
  LIST *t, *r=NULL, *l=NULL;

  for (; p; p = p->next)
  {
    t = (LIST *) safe_malloc (sizeof (LIST));
    t->data = safe_strdup (p->data);
    t->next = NULL;
    if (l)
    {
      r->next = t;
      r = r->next;
    }
    else
      l = r = t;
  }
  return (l);
}

static int include_message (HEADER *cur, FILE *out)
{
  char buffer[STRING];
  MESSAGE *msg;
  STATE s;
  
  memset (&s, 0, sizeof (s));
  s.fpout = out;
  
  if ((msg = mx_open_message (Context, cur->msgno)) == NULL) return (-1);

  s.fpin = msg->fp;

  if (Attribution[0])
  {
    mutt_make_string (buffer, sizeof (buffer), Attribution, cur);
    state_puts (buffer, &s);
    state_putc ('\n', &s);
  }

  mutt_make_string (buffer, sizeof (buffer), Prefix, cur);
  s.prefix = buffer;

  if (option (OPTHEADER))
  {
    fseek (s.fpin, cur->offset, 0);
    mutt_copy_header (s.fpin, cur, s.fpout, CH_DECODE | CH_PREFIX);
  }

#ifdef _PGPPATH
  if (cur->pgp)
  {
    if (cur->pgp == PGPENCRYPT)
    {
      /* make sure we have the user's passphrase before proceeding... */
      pgp_valid_passphrase ();
    }

    unset_option (OPTVERIFYSIG);
  }
#endif /* _PGPPATH */

  mutt_body_handler (cur->content, &s);
  
  if (PostIndentString[0])
  {
    mutt_make_string (buffer, sizeof (buffer), PostIndentString, cur);
    state_puts (buffer, &s);
    state_putc ('\n', &s);
  }

  mx_close_message (&msg);

  return 0;
}

static BODY *make_forward (HEADER *hdr)
{
  char buffer[LONG_STRING];
  char file[_POSIX_PATH_MAX];
  CONTEXT ctx;
  struct stat st;
  BODY *body = mutt_new_body();
  FILE *fpin;
  short old_magic = DefaultMagic;

  mutt_parse_mime_message(hdr);
  body->type = TYPEMESSAGE;
  body->subtype = safe_strdup ("rfc822");
  switch (hdr->content->encoding)
  {
    case ENCBINARY:
    case ENC8BIT:

      body->encoding = hdr->content->encoding;
      break;

    default:

      body->encoding = ENC7BIT;
      break;
  }
  strfcpy (buffer, "Forwarded message from ", sizeof (buffer));
  rfc822_write_address (buffer+23, sizeof (buffer)-23, hdr->env->from);
  body->description = safe_strdup (buffer);
  body->use_disp = 0;

  mutt_mktemp (file);
  DefaultMagic = M_MBOX;

  if (mx_open_mailbox (file, M_APPEND | M_QUIET, &ctx) != NULL)
  {
    MESSAGE *msg;

    if ((msg = mx_open_message (Context, hdr->msgno)) != NULL)
    {
      body->filename = safe_strdup (file);
      body->unlink = 1;

      /* signal that we don't need a "Content-Length:" */
      set_option (OPTFORWARD);
      if (option (OPTFORWDECODE))
	mutt_append_decoded (msg->fp, msg->fp, hdr, hdr->content, &ctx);
      else
	mutt_append_message (msg->fp, hdr,
	  (Context->magic == M_MBOX || Context->magic == M_MMDF), &ctx);
      unset_option (OPTFORWARD);
      mx_close_mailbox (&ctx);
      
      stat (file, &st);
      body->length = (long) st.st_size - 1; /* skip the '\n' separator */
      body->offset = 0;

      if ((fpin = safe_fopen (file, "r")) != NULL) 
      {
	body->parts = mutt_parse_messageRFC822 (fpin, body);
	fclose (fpin);
      }
      mx_close_message (&msg);
    }
  }

  DefaultMagic = old_magic;
  mutt_update_encoding (body);
  return body;
}

static int default_to (ADDRESS **to, ENVELOPE *env)
{
  char buffer[STRING], prompt[STRING];
  int i = 0;

  if (!option (OPTMETOO) && mutt_addr_is_user (env->from))
    /* mail is from the user, assume replying to recipients */
    *to = rfc822_cpy_adr (env->to);
  else if (env->reply_to)
  {
    if (option (OPTIGNORELISTREPLYTO) &&
	mutt_is_mail_list (env->reply_to) &&
	mutt_addrsrc (env->reply_to, env->to))
    {
      /* If the Reply-To: address is a mailing list, assume that it was
       * put there by the mailing list, and use the From: address
       */
      *to = rfc822_cpy_adr (env->from);
    }
    else if (!mutt_addrcmp (env->from, env->reply_to) &&
	quadoption (OPT_REPLYTO) != M_YES)
    {
      /* There are quite a few mailing lists which set the Reply-To:
       * header field to the list address, which makes it quite impossible
       * to send a message to only the sender of the message.  This
       * provides a way to do that.
       */
      buffer[0] = 0;
      mutt_simple_address (buffer, sizeof (buffer), env->reply_to);
      snprintf (prompt, sizeof (prompt), "Reply to %s?", buffer);
      if ((i = query_quadoption (OPT_REPLYTO, prompt)) == M_YES)
	*to = rfc822_cpy_adr (env->reply_to);
      else if (i == M_NO)
	*to = rfc822_cpy_adr (env->from);
      else
	return (-1); /* abort */
    }
    else
      *to = rfc822_cpy_adr (env->reply_to);
  }
  else
    *to = rfc822_cpy_adr (env->from);

  return (0);
}

/* append list 'b' to list 'a' and return the last element in the new list */
static ADDRESS *rfc822_append (ADDRESS **a, ADDRESS *b)
{
  ADDRESS *tmp = *a;

  while (tmp && tmp->next)
    tmp = tmp->next;
  if (!b)
    return tmp;
  if (tmp)
    tmp->next = rfc822_cpy_adr (b);
  else
    tmp = *a = rfc822_cpy_adr (b);
  while (tmp && tmp->next)
    tmp = tmp->next;
  return tmp;
}

static int envelope_defaults (int flags, HEADER *cur, HEADER *msg)
{
  ENVELOPE *curenv = NULL;
  ADDRESS *tmpaddr;
  LIST *tmp;
  char buffer[STRING];
  int i = 0, tag = 0;

  if (Context && flags && flags != SENDPOSTPONED)
  {
    if (!cur)
    {
      tag = 1;
      for (i = 0; i < Context->vcount; i++)
	if (Context->hdrs[Context->v2r[i]]->tagged)
	{
	  cur = Context->hdrs[Context->v2r[i]];
	  curenv = cur->env;
	  break;
	}

      if (!cur)
      {
	/*
	 * This could happen if the user tagged some messages and then did
	 * a limit such that none of the tagged message are visible.
	 */
	mutt_error ("No tagged messages are visible!");
	return (-1);
      }
    }
    else
      curenv = cur->env;
  }

  if (flags & SENDREPLY)
  {
    if (flags & SENDLISTREPLY)
    {
      if ((msg->env->to = find_mailing_lists (curenv->to, curenv->cc)) == 0)
      {
	mutt_error ("Could not find any mailing lists in the message!");
	return (-1);
      }
    }
    else if (!tag || !(flags & SENDGROUPREPLY))
    {
      if (default_to (&msg->env->to, curenv) == -1)
	return (-1); /* abort */
    }

    if (flags & SENDGROUPREPLY)
    {
      if (tag)
      {
	ADDRESS *toRecips = NULL;

	/* add the recipients/originators from each message to the list */

	tmpaddr = NULL;
	msg->env->to = msg->env->cc = NULL;
	for (i = 0; i < Context->vcount; i++)
	{
	  if (Context->hdrs[Context->v2r[i]]->tagged)
	  {
	    /* get the originators */

	    if (default_to (toRecips ? &toRecips->next : &msg->env->to,
			    Context->hdrs[Context->v2r[i]]->env) == -1)
	    {
	      mutt_free_address (&msg->env->to);
	      mutt_free_address (&msg->env->cc);
	      return (-1);
	    }
	    if (!toRecips)
	      toRecips = msg->env->to;

	    while (toRecips && toRecips->next)
	      toRecips = toRecips->next;

	    /* process TO and CC fields into the new CC list */

	    tmpaddr = rfc822_append (tmpaddr ? &tmpaddr : &msg->env->cc,
				     Context->hdrs[Context->v2r[i]]->env->to);
	    tmpaddr = rfc822_append (tmpaddr ? &tmpaddr : &msg->env->cc,
				     Context->hdrs[Context->v2r[i]]->env->cc);
	  }
	}
      }
      else
      {
	if (option (OPTMETOO) || !mutt_addr_is_user (curenv->from))
	  msg->env->cc = rfc822_cpy_adr (curenv->to);

	if ((tmpaddr = msg->env->cc) != NULL)
	{
	  while (tmpaddr->next)
	    tmpaddr = tmpaddr->next;
	  tmpaddr->next = rfc822_cpy_adr (curenv->cc);
	}
	else
	  msg->env->cc = rfc822_cpy_adr (curenv->to);
	rfc822_append (&msg->env->cc, curenv->cc);
      }
    }

    if (! option (OPTMETOO))
    {
      msg->env->to = remove_user (msg->env->to);
      msg->env->cc = remove_user (msg->env->cc);
    }

    if (curenv->real_subj)
    {
      msg->env->subject = safe_malloc (strlen (curenv->real_subj) + 5);
      sprintf (msg->env->subject, "Re: %s", curenv->real_subj);
    }
    else
      msg->env->subject = safe_strdup ("Re: your mail");

    /* add the In-Reply-To field */
    strfcpy (buffer, "In-Reply-To: ", sizeof (buffer));
    mutt_make_string (buffer + 13, sizeof (buffer) - 13, InReplyTo, cur);
    tmp = msg->env->userhdrs;
    while (tmp && tmp->next)
      tmp = tmp->next;
    if (tmp)
    {
      tmp->next = mutt_new_list ();
      tmp = tmp->next;
    }
    else
      tmp = msg->env->userhdrs = mutt_new_list ();
    tmp->data = safe_strdup (buffer);

    msg->env->references = mutt_copy_list (curenv->references);

    if (curenv->message_id)
    {
      LIST *t;

      t = mutt_new_list ();
      t->data = safe_strdup (curenv->message_id);
      t->next = msg->env->references;
      msg->env->references = t;
    }
  }
  else if (flags & SENDFORWARD)
  {
    /* set the default subject for the message. */
    mutt_make_string (buffer, sizeof (buffer), ForwFmt, cur);
    msg->env->subject = safe_strdup (buffer);
  }

  /* the CC field can get cluttered, especially with lists */
  msg->env->to = mutt_remove_duplicates (msg->env->to);
  msg->env->cc = mutt_remove_duplicates (msg->env->cc);
  msg->env->cc = mutt_remove_xrefs (msg->env->to, msg->env->cc);

  return (0);
}

static void mutt_include_forward (HEADER *h, FILE *f)
{
  char buffer[LONG_STRING] = { 0 };
  MESSAGE *msg;

  if ((msg = mx_open_message (Context, h->msgno)) == NULL)
    return;

  rfc822_write_address (buffer, sizeof(buffer), h->env->from);
  fprintf (f, "-----Forwarded message from %s-----\n\n", buffer);
  if (option (OPTFORWDECODE))
  {
    mutt_make_string (buffer, sizeof (buffer), DecodeFmt, h);
    fprintf (f, "%s\n", buffer);
  }
  
  fseek (msg->fp, h->offset, 0);

  mutt_copy_header (msg->fp, h, f,
	CH_XMIT | (option (OPTFORWDECODE) ? (CH_WEED | CH_DECODE) : 0));
  /* mutt_copy_header doesn't add the terminating newline with CH_XMIT */
  fputc ('\n', f);

  /* if this message has a single TEXT/PLAIN part, try to decode it */
  if (option (OPTFORWDECODE) || (h->content->type == TYPETEXT &&
      strcasecmp (h->content->subtype, "plain") == 0))
  {
    STATE s;
    
    memset (&s, 0, sizeof (s));
    s.fpin = msg->fp;
    s.fpout = f;
#ifdef _PGPPATH
    unset_option (OPTVERIFYSIG);
    mutt_parse_mime_message (h);
#endif
    mutt_body_handler (h->content, &s);
  }
  else
  {
    /* just copy the message as-is */
    mutt_copy_bytes (msg->fp, f, h->content->length);
  }
  fputs ("\n-----End of forwarded message-----\n", f);
  mx_close_message (&msg);
}

/* args:
 *	msg		outgoing message
 *	cur		current message, NULL if tag-reply
 *	tempfp		stream for main body of the outgoing message
 */
static int generate_body (int flags, HEADER *msg, HEADER *cur, FILE *tempfp)
{
  int i;

  if (flags & (SENDREPLY | SENDGROUPREPLY))
  {
    if ((i = query_quadoption (OPT_INCLUDE, "Include message in reply?")) == -1)
      return (-1);

    if (i == M_YES)
    {
      if (!cur)
      {
	for (i = 0; i < Context->vcount; i++)
	  if (Context->hdrs[Context->v2r[i]]->tagged)
	  {
	    mutt_parse_mime_message (Context->hdrs[Context->v2r[i]]);
	    if (include_message (Context->hdrs[Context->v2r[i]], tempfp) == -1)
	    {
	      mutt_error ("Could not include all requested messages!");
	      return (-1);
	    }
	    fputc ('\n', tempfp);
	  }
      }
      else
      {
	mutt_parse_mime_message (cur);
	include_message (cur, tempfp);
      }
    }
  }
  else if (flags & SENDFORWARD)
  {
    if (option (OPTMIMEFWD))
    {
      BODY *last = msg->content;
      BODY *tmp;

      while (last && last->next)
	last = last->next;

      if (cur)
      {
	tmp = make_forward (cur);
	if (last)
	  last->next = tmp;
	else
	  msg->content = tmp;
      }
      else
      {
	for (i = 0; i < Context->vcount; i++)
	{
	  if (Context->hdrs[Context->v2r[i]]->tagged)
	  {
	    tmp = make_forward (Context->hdrs[Context->v2r[i]]);
	    if (last)
	    {
	      last->next = tmp;
	      last = tmp;
	    }
	    else
	      last = msg->content = tmp;
	  }
	}
      }
    }
    else
    {
      if (cur)
	mutt_include_forward (cur, tempfp);
      else
	for (i=0; i < Context->vcount; i++)
	  if (Context->hdrs[Context->v2r[i]]->tagged)
	    mutt_include_forward (Context->hdrs[Context->v2r[i]], tempfp);
    }
  }

  return (0);
}

void mutt_choose_fcc (char *s, size_t slen, ENVELOPE *env)
{
  /* check to see if the user has a save-hook defined for a recipient */
  *s = 0;
  if (mutt_address_hook (env, s, slen, M_FCCHOOK) != 0)
  {
    ADDRESS *adr;
    /* no save-hook, so default from the message recipient list */

    if ((option (OPTSAVENAME) || option (OPTFORCENAME)) &&
	(env->to || env->cc || env->bcc))
    {
      char buf[_POSIX_PATH_MAX];

      if (env->to)
        adr = env->to;
      else if (env->cc)
        adr = env->cc;
      else
        adr = env->bcc;

      mutt_safe_path (buf, sizeof (buf), adr);

      snprintf (s, slen, "%s/%s", Maildir, buf);
      if (!option (OPTFORCENAME))
      {
	/* check to see if the file already exists */
	if (access (s, W_OK) == -1)
	  strfcpy (s, Outbox, slen);
      }
    }
    else
      strfcpy (s, Outbox, slen);
  }
}

/* check for the existence of postponed mail */
static int postponed_mail (void)
{
  int magic;
  struct stat st;

  if (stat (Postponed, &st) == -1)
    return 0;

  magic = mx_get_magic (Postponed);
  if (magic == M_MMDF || magic == M_MBOX)
    return (st.st_size != 0);
  else if (magic == M_MAILDIR || magic == M_MH)
  {
    DIR *dirp;
    struct dirent *de;

    if (magic == M_MAILDIR)
    {
      char newpath[_POSIX_PATH_MAX];

      snprintf (newpath, sizeof (newpath), "%s/new", Postponed);
      dirp = opendir (newpath);
    }
    else
      dirp = opendir (Postponed);
    if (!dirp)
    {
      dprint (1, (debugfile, "postponed_mail(): unable to open postponed mailbox.\n"));
      return 0;
    }
    while ((de = readdir (dirp)) != NULL)
    {
      if ((magic == M_MAILDIR && *de->d_name != '.') ||
	  mh_valid_message (de->d_name))
      {
	closedir (dirp);
	return 1;
      }
    }
    closedir (dirp);
  }
  return 0;
}

/* args:
 *	flags		flags to specify which send mode (compose/reply/etc.)
 *	msg		structure that contains the ENVELOPE and BODY
 *			(attachments) for the outgoing message.
 *	tempfile	file to use as temp storage (created by -i or -H)
 *	cur		pointer to the current message (optional)
 */
void ci_send_message (int flags, HEADER *msg, char *tempfile, HEADER *cur)
{
  char buffer[LONG_STRING];
  char fcc[_POSIX_PATH_MAX] = ""; /* where to copy this message */
  FILE *tempfp = NULL;
  BODY *pbody;
  int i;

  if (msg)
  {
    msg->env->to = mutt_expand_aliases (msg->env->to);
    msg->env->cc = mutt_expand_aliases (msg->env->cc);
    msg->env->bcc = mutt_expand_aliases (msg->env->bcc);
  }
  else
  {
    msg = mutt_new_header ();

    if (flags == SENDPOSTPONED)
    {
      if ((flags = mutt_get_postponed (msg, &cur)) < 0)
	goto cleanup;
    }
    else if (!flags && quadoption (OPT_RECALL) != M_NO && postponed_mail ())
    {
      /* If the user is composing a new message, check to see if there
       * are any postponed messages first.
       */
      if ((i = query_quadoption (OPT_RECALL, "Recall postponed message?")) == -1)
	goto cleanup;

      if (i == M_YES)
      {
	if ((flags = mutt_get_postponed (msg, &cur)) < 0)
	  flags = 0;
      }
    }

    if (flags & SENDPOSTPONED)
    {
      if ((tempfp = safe_fopen (msg->content->filename, "a+")) == NULL)
      {
	mutt_perror (msg->content->filename);
	goto cleanup;
      }
    }

    if (!msg->env)
      msg->env = mutt_new_envelope ();
  }

  if (! (flags & SENDPOSTPONED))
  {
    pbody = mutt_new_body ();
    pbody->next = msg->content; /* don't kill command-line attachments */
    msg->content = pbody;
    
    msg->content->type = TYPETEXT;
    msg->content->subtype = safe_strdup ("plain");
    msg->content->unlink = 1;
    msg->content->use_disp = 0;
    
    if (!tempfile)
    {
      mutt_mktemp (buffer);
      tempfp = safe_fopen (buffer, "w+");
      msg->content->filename = safe_strdup (buffer);
    }
    else
    {
      tempfp = safe_fopen (tempfile, "a+");
      msg->content->filename = safe_strdup (tempfile);
    }

    if (!tempfp)
    {
      dprint(1,(debugfile, "newsend_message: can't create tempfile %s (errno=%d)\n", tempfile, errno));
      mutt_perror (tempfile);
      goto cleanup;
    }
  }

  if (flags & SENDBATCH) 
  {
    mutt_copy_stream (stdin, tempfp);

    if (option (OPTHDRS))
    {
      process_user_recips (msg->env);
      process_user_header (msg->env);
    }
  }
  else if (!(flags & SENDPOSTPONED))
  {
    if (envelope_defaults (flags, cur, msg) == -1)
      goto cleanup;

    if (option (OPTHDRS))
      process_user_recips (msg->env);

    if (! (flags & SENDMAILX) &&
	! (option (OPTAUTOEDIT) && option (OPTEDITHDRS)) &&
	! ((flags & SENDREPLY) != 0 && option (OPTFASTREPLY)))
    {
      if (edit_envelope (msg->env) == -1)
	goto cleanup;
    }

    /* change settings based upon recipients */
    {
      ADDRESS *addr;

      /* create a single list of recipients */
      addr = msg->env->to;
      while (addr && addr->next)
	addr = addr->next;
      if (addr)
	addr->next = msg->env->cc;

      mutt_send_hook (msg->env->to ? msg->env->to : msg->env->cc);

      if (addr)
	addr->next = NULL; /* break up the lists */
    }

    if (option (OPTHDRS))
      process_user_header (msg->env);

#ifdef _PGPPATH
    if (! (flags & SENDMAILX))
    {
      if (option (OPTPGPAUTOSIGN))
	msg->pgp |= PGPSIGN;
      if (option (OPTPGPAUTOENCRYPT))
	msg->pgp |= PGPENCRYPT;
      if (option (OPTPGPREPLYENCRYPT) && cur && cur->pgp == PGPENCRYPT)
	msg->pgp |= PGPENCRYPT;
      if (option (OPTPGPREPLYSIGN) && cur && cur->pgp == PGPSIGN)
	msg->pgp |= PGPSIGN;
    }
#endif /* _PGPPATH */

    /* included replies/forwarded messages */
    if (generate_body (flags, msg, cur, tempfp) == -1)
      goto cleanup;

    if (! (flags & SENDMAILX) && strcmp (Editor, "builtin") != 0)
      append_signature (msg->env, tempfp);
  }

  fclose (tempfp);
  tempfp = NULL;

  if (option (OPTUSEFROM) && !msg->env->from)
  {
    if (option (OPTREVNAME))
    {
      ADDRESS *tmp = NULL;
      
      if (cur)
      {
	for (tmp = cur->env->to; tmp; tmp = tmp->next)
	{
	  if (mutt_addr_is_user (tmp))
	    break;
	}

	if (!tmp)
	{
	  for (tmp = cur->env->cc; tmp; tmp = tmp->next)
	  {
	    if (mutt_addr_is_user (tmp))
	      break;
	  }
	}

	if (tmp)
	{
	  ADDRESS *ptr;

	  /* rfc822_cpy_adr() copies the whole list, not just one element */
	  ptr = tmp->next;
	  tmp->next = NULL;
	  msg->env->from = rfc822_cpy_adr (tmp);
	  tmp->next = ptr;

	  if (!msg->env->from->personal)
	    msg->env->from->personal = safe_strdup (Realname);
	}
      }
    }

    if (!msg->env->from)
      msg->env->from = mutt_default_from ();
  }

  if (flags & SENDMAILX)
  {
    if (mutt_builtin_editor (msg->content->filename, &msg->env, cur) == -1)
      goto cleanup;
  }
  else if (! (flags & SENDBATCH))
  {
    struct stat st;
    time_t mtime;

    stat (msg->content->filename, &st);
    mtime = st.st_mtime;

    mutt_update_encoding (msg->content);

    /* If the this isn't a text message, look for a mailcap edit command */
    if (mutt_needs_mailcap (msg->content))
      mutt_edit_attachment (msg->content, 0);
    else if (strcmp ("builtin", Editor) == 0)
      mutt_builtin_editor (msg->content->filename, &msg->env, cur);
    else if (option (OPTEDITHDRS))
      mutt_edit_headers (Editor, msg->content->filename, &msg->env, fcc, sizeof (fcc));
    else
      mutt_edit_file (Editor, msg->content->filename);

    if (! (flags & (SENDPOSTPONED | SENDFORWARD)))
    {
      if (stat (msg->content->filename, &st) == 0)
      {
	/* if the file was not modified, bail out now */
	if (mtime == st.st_mtime &&
	    query_quadoption (OPT_ABORT, "Abort unmodified message?") == M_YES)
	{
	  mutt_message ("Aborted unmodified message.");
	  goto cleanup;
	}
      }
      else
	mutt_perror (msg->content->filename);
    }
  }

  if (!fcc[0])
  {
    /* set the default FCC */
    mutt_choose_fcc (fcc, sizeof (fcc), msg->env);
    mutt_pretty_mailbox (fcc);
  }

  mutt_update_encoding (msg->content);

  if (! (flags & (SENDMAILX | SENDBATCH)))
  {
main_loop:

    if ((i = mutt_send_menu (msg, fcc, sizeof (fcc), cur)) == -1)
    {
      /* abort */
      mutt_message ("Mail not sent.");
      goto cleanup;
    }
    else if (i == 1)
    {
      /* postpone the message until later. */
      if (msg->content->next)
	msg->content = mutt_make_multipart (msg->content);
      mutt_write_fcc (Postponed, msg, (cur && (flags & SENDREPLY)) ? cur->env->message_id : NULL, 1);
      mutt_message ("Message postponed.");
      goto cleanup;
    }
  }

  if (!msg->env->to && !msg->env->cc && !msg->env->bcc)
  {
    if (! (flags & SENDBATCH))
    {
      mutt_error ("No recipients are specified!");
      goto main_loop;
    }
    else
    {
      puts ("No recipients were specified.");
      goto cleanup;
    }
  }

  if (!msg->env->subject && ! (flags & SENDBATCH) &&
      query_quadoption (OPT_SUBJECT, "No subject, abort sending?") != 0)
    goto main_loop;

  if (msg->content->next)
    msg->content = mutt_make_multipart (msg->content);

#ifdef _PGPPATH
  if (msg->pgp)
  {
    if (pgp_protect (msg) != 0)
    {
      if (msg->content->parts)
      {
	/* remove the toplevel multipart structure */
	pbody = msg->content;
	msg->content = msg->content->parts;
	pbody->parts = NULL;
	mutt_free_body (&pbody);
      }
      goto main_loop;
    }
  }
#endif /* _PGPPATH */

  mutt_expand_path (fcc, sizeof (fcc));

  if (!option (OPTNOCURSES) && ! (flags & SENDMAILX))
    mutt_message ("Sending message...");

  if (msg->env->bcc && ! (msg->env->to || msg->env->cc))
  {
    /* some MTA's will put an Apparently-To: header field showing the Bcc:
     * recipients if there is no To: or Cc: field, so attempt to suppress
     * it by using an empty To: field.
     */
    msg->env->to = mutt_new_address ();
    msg->env->to->mailbox = safe_strdup ("undisclosed-recipients");
    msg->env->to->next = mutt_new_address ();
  }

  mutt_send_message (msg, fcc);

  if (!option (OPTNOCURSES) && ! (flags & SENDMAILX))
    mutt_message ("Mail sent.");

  /* now free up the memory used to generate this message. */
  mutt_free_header (&msg);

  if (flags & SENDREPLY)
  {
    if (cur)
      mutt_set_flag (Context, cur, M_REPLIED, 1);
    else if (!(flags & SENDPOSTPONED) && Context && Context->tagged)
    {
      for (i = 0; i < Context->vcount; i++)
	if (Context->hdrs[Context->v2r[i]]->tagged)
	  mutt_set_flag (Context, Context->hdrs[Context->v2r[i]], M_REPLIED, 1);
    }
  }

  return; /* all done */

cleanup:

  if (tempfp)
    fclose (tempfp);
  mutt_free_header (&msg);
}
