/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

#include "mutt.h"
#include "mutt_curses.h"
#include "mutt_regex.h"
#include "keymap_defs.h" /* needed for ci_search_command */

#ifdef _PGPPATH
#include "pgp.h"
#endif

#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <unistd.h>
#include <sys/stat.h>

/*** Note: there must be NO overlap between the entries in the following
 *** two structures, or else the user will not be able to select one.
 ***/

/* operators which select based upon message status */
static struct mapping_t Flags[] = {
  { "A", M_ALL },
  { "D", M_DELETED },
  { "F", M_FLAG },
  { "H", M_THREAD },
  { "N", M_NEW },
  { "O", M_OLD },
  { "R", M_READ },
  { "Q", M_REPLIED },
  { "T", M_TAG },
  { "U", M_UNREAD },
  { NULL, 0 },
};

/* operators which require an argument */
static struct mapping_t Patterns[] = {
  { "b", M_BODY },
  { "c", M_CC },
  { "d", M_DATE },
  { "e", M_SENDER },
  { "f", M_FROM },
  { "h", M_HEADER },
  { "i", M_ID },
  { "m", M_MESSAGE },
  { "r", M_DATE_RECEIVED },
  { "s", M_SUBJECT },
  { "t", M_TO },
  { NULL, 0 }
};

/* macros to select a single bit from a multi-byte vector */
#define setBit(B,V)   V[(B)/8] |= (1 << ((B) % 8))
#define isSet(B, V) (V[(B)/8] & (1 << ((B) % 8))) != 0

short VecLen = 0;			/* how many bytes in a search vector */
char LastSearch[STRING] = { 0 };	/* last pattern searched for */
unsigned char *SearchVector = NULL;	/* last search vector */

int mutt_getvaluebychar (char ch, struct mapping_t *table)
{
  int i;

  for (i = 0; table[i].name; i++)
  {
    if (ch == table[i].name[0])
      return table[i].value;
  }

  return (-1);
}

static unsigned char *allocVec (void)
{
  unsigned char *p;

  VecLen = (Context->msgcount + 7) / 8;
  p = safe_calloc (1, VecLen);
  return p;
}

static int matchesCrit (HEADER *h, int crit, int not)
{
  switch (crit)
  {
    case M_ALL:
      return (!not);
    case M_FLAG:
      return (not ? !h->flagged : h->flagged);
    case M_TAG:
      return (not ? !h->tagged : h->tagged);
    case M_NEW:
      return (not ? h->old || h->read : !(h->old || h->read));
    case M_UNREAD:
      return (not ? h->read : !h->read);
    case M_REPLIED:
      return (not ? !h->replied : h->replied);
    case M_OLD:
      return (not ? (!h->old || h->read) : (h->old && !h->read));
    case M_READ:
      return (not ? !h->read : h->read);
    case M_DELETED:
      return (not ? !h->deleted : h->deleted);
  }
  return 0;
}

static void doThread_r (HEADER *hdr, unsigned char *p)
{
  while (hdr)
  {
    setBit (hdr->msgno, p);
    doThread_r (hdr->child, p);
    hdr = hdr->next;
  }
}

static unsigned char *doThread (HEADER *hdr)
{
  unsigned char *vec = allocVec();

  while (hdr->parent)
    hdr = hdr->parent;
  setBit (hdr->msgno, vec);
  doThread_r (hdr->child, vec);
  return vec;
}

static unsigned char *doKeyword (int crit, int not, HEADER *hdr)
{
  unsigned char *p;
  int i;

  if (crit == M_THREAD)
  {
    p = doThread (hdr);
    if (not)
      for (i = 0; i < VecLen; i++)
	p[i] = ~p[i];
  }
  else
  {
    p = allocVec ();
    for (i = 0; i < Context->msgcount; i++)
      if (matchesCrit (Context->hdrs[i], crit, not))
	setBit (i, p);
  }
  return p;
}

static int getDate (char *s, struct tm *t)
{
  char buf[SHORT_STRING];
  char *p = buf;
  time_t now = time (NULL);
  struct tm *tm = localtime (&now);

  while (*s && *s != '/')
    *p++ = *s++;
  *p = 0;
  t->tm_mday = atoi (buf);
  if (t->tm_mday < 1 || t->tm_mday > 31)
  {
    mutt_error ("Invalid day of month: %s", buf);
    return(-1);
  }
  if (!*s)
  {
    /* fill in today's month and year */
    t->tm_mon = tm->tm_mon;
    t->tm_year = tm->tm_year;
    return 0;
  }
  s++;
  p = buf;
  while (*s && *s != '/')
    *p++ = *s++;
  *p = 0;
  t->tm_mon = atoi (buf) - 1;
  if (t->tm_mon < 0 || t->tm_mon > 11)
  {
    mutt_error ("Invalid month: %s", buf);
    return (-1);
  }
  if (!*s)
  {
    t->tm_year = tm->tm_year;
    return 0;
  }
  s++;
  t->tm_year = atoi (s);
  if (t->tm_year > 1900)
    t->tm_year -= 1900;
  return 0;
}

/* return the local timezone offset from UTC in seconds */
static time_t mutt_local_tz (void)
{
  time_t now = time (NULL);
  struct tm *local = localtime (&now);
  struct tm *utc = gmtime (&now);

  return ((local->tm_yday - utc->tm_yday) * 24 * 3600 + (local->tm_hour - utc->tm_hour) * 3600 + (local->tm_min - utc->tm_min) * 60);
}

/* args:
 *	s	date string to parse
 *	recv	if nonzero, match the recieved time.
 *		else use date-sent
 *	not	if nonzero looking for messages outside the specified
 *		date range
 *	
 * special cases:
 *
 * DD/MM/YY	all messages on DD/MM/YY
 * -DD/MM/YY	all messages on/before DD/MM/YY
 * DD/MM/YY-	all messages on/after DD/MM/YY
 */

static unsigned char *doDate (char *s, int recv, int not)
{
  int i;
  struct tm min;
  struct tm max;
  char *p;
  unsigned char *vec;
  time_t now, mint, maxt, tz;

  memset (&min, 0, sizeof (min));
  memset (&max, 0, sizeof (max));

  /* Arbitrary year in the future.  Don't set this too high
   * or mutt_mktime() returns something larger than will
   * fit in a time_t on some systems
   */
  max.tm_year = 130;
  max.tm_mon = 11;
  max.tm_mday = 31;
  max.tm_hour = 23;
  max.tm_min = 59;
  max.tm_sec = 59;

  if ((p = strchr (s, '-')))
    *p++ = 0;
  if (*s)
  {
    /* mininum date specified */
    if (getDate (s, &min) != 0)
      return NULL;
  }
  if (p)
  {
    if (*p)
    {
      /* max date */
      if (getDate (p, &max) != 0)
	return NULL;
    }
  }
  else
  {
    /* search for messages on a specific day */
    max.tm_year = min.tm_year;
    max.tm_mon = min.tm_mon;
    max.tm_mday = min.tm_mday;
  }

  mint = mutt_mktime (&min);
  maxt = mutt_mktime (&max);

  /* convert to UTC */
  tz = mutt_local_tz ();
  mint -= tz;
  maxt -= tz;

  vec = allocVec ();
  for (i = 0; i < Context->msgcount; i++)
  {
    now = recv ? Context->hdrs[i]->received : Context->hdrs[i]->date_sent;

    if (now >= mint && (now <= maxt))
    {
      if (!not)
	setBit (i, vec);
    }
    else if (not)
      setBit (i, vec);
  }
  return vec;
}

/* if no uppercase letters are given, do a case-insensitive search */
int mutt_which_case (const char *s)
{
  while (*s)
  {
    if (isalpha (*s) && isupper (*s))
      return 0; /* case-sensitive */
    s++;
  }
  return REG_ICASE; /* case-insensitive */
}

static int msg_search (regex_t *rx, char *buf, size_t blen, int is_hdr, int msgno)
{
  char tempfile[_POSIX_PATH_MAX];
  MESSAGE *msg = NULL;
  STATE s;
  struct stat st;
  FILE *fp = NULL;
  long lng;
  int match = 0;

  if ((msg = mx_open_message (Context, msgno)) != NULL)
  {
    if (option (OPTTHOROUGHSRC))
    {
      /* decode the header / body */
      memset (&s, 0, sizeof (s));
      s.fpin = msg->fp;
      mutt_mktemp (tempfile);
      if ((s.fpout = safe_fopen (tempfile, "w+")) == NULL)
      {
	mutt_perror (tempfile);
	return (0);
      }

#ifdef _PGPPATH
      unset_option (OPTVERIFYSIG);
#endif

      if (is_hdr)
	mutt_copy_header (msg->fp, Context->hdrs[msgno], s.fpout, CH_FROM | CH_DECODE);
      else
      {
	mutt_parse_mime_message (Context->hdrs[msgno]);
#ifdef _PGPPATH
	if (Context->hdrs[msgno]->pgp & PGPENCRYPT && !pgp_valid_passphrase())
	{
	  mx_close_message (&msg);
	  fclose (fp);
	  unlink (tempfile);
	  return (0);
	}
#endif
	fseek (msg->fp, Context->hdrs[msgno]->offset, 0);
	mutt_body_handler (Context->hdrs[msgno]->content, &s);
      }

      fp = s.fpout;
      fflush (fp);
      fseek (fp, 0, 0);
      fstat (fileno (fp), &st);
      lng = (long) st.st_size;
    }
    else
    {
      /* raw header / body */
      fp = msg->fp;
      if (is_hdr)
      {
	fseek (fp, Context->hdrs[msgno]->offset, 0);
	lng = Context->hdrs[msgno]->content->offset - Context->hdrs[msgno]->offset;
      }
      else
      {
	fseek (msg->fp, Context->hdrs[msgno]->content->offset, 0);
	lng = Context->hdrs[msgno]->content->length;
      }
    }

    /* search the file "fp" */
    while (lng > 0)
    {
      if (fgets (buf, blen - 1, fp) == NULL)
	break; /* don't loop forever */
      if (regexec (rx, buf, 0, NULL, 0) == 0)
      {
	match = 1;
	break;
      }
      lng -= strlen (buf);
    }
    
    mx_close_message (&msg);

    if (option (OPTTHOROUGHSRC))
    {
      fclose (fp);
      unlink (tempfile);
    }
  }

  return match;
}

/* select messages in the range: [min]-[max] */
static unsigned char *doMessage (char *s, int not)
{
  int i, a = 0, b = Context->msgcount - 1;
  char buf[LONG_STRING];
  char *p, *q;
  unsigned char *tmpvec;

  q = s; /* save the current location */

  p = buf;
  while (isdigit (*s))
    *p++ = *s++;
  *p = 0;
  if (buf[0])
    a = atoi (buf) - 1;

  if (*s)
  {
    s++;
    SKIPWS (s);
    p = buf;
    while (isdigit (*s))
      *p++ = *s++;
    *p = 0;
    if (buf[0])
      b = atoi (buf) - 1;
  }

  if (a < 0 || b > Context->msgcount - 1)
  {
    mutt_error ("Invalid range at: %s", q);
    return NULL;
  }

  tmpvec = allocVec ();
  for (i = 0; i < Context->msgcount; i++)
  {
    if (i >= a && i <= b)
    {
      if (!not)
	setBit (i, tmpvec);
    }
    else if (not)
      setBit (i, tmpvec);
  }

  return tmpvec;
}

static unsigned char *doString (int field, char *s, int not)
{
  unsigned char *vec = allocVec ();
  char buf[LONG_STRING];
  int i, err;
  regex_t exp;
  int match;

  if (field == M_DATE || field == M_DATE_RECEIVED)
    return doDate (s, (field == M_DATE_RECEIVED), not);
  else if (field == M_MESSAGE)
    return doMessage (s, not);

  if ((err = REGCOMP (exp, s, mutt_which_case (s))))
  {
    regerror (err, &exp, buf, sizeof (buf));
    mutt_error ("%s: %s", s, buf);
    safe_free ((void **)&vec);
    regfree (&exp);
    return NULL;
  }

  for (i = 0; i < Context->msgcount; i++)
  {
    if (field == M_BODY)
      match = msg_search (&exp, buf, sizeof (buf), 0, i);
    else if (field == M_HEADER)
      match = msg_search (&exp, buf, sizeof (buf), 1, i);
    else
    {
      buf[0] = 0;
      switch (field)
      {
	case M_SENDER:
	  rfc822_write_address (buf, sizeof (buf), Context->hdrs[i]->env->sender);
	  break;
	case M_FROM:
	  rfc822_write_address (buf, sizeof (buf), Context->hdrs[i]->env->from);
	  break;
	case M_TO:
	  rfc822_write_address (buf, sizeof (buf), Context->hdrs[i]->env->to);
	  break;
	case M_CC:
	  rfc822_write_address (buf, sizeof (buf), Context->hdrs[i]->env->cc);
	  break;
	case M_SUBJECT:
	  if (Context->hdrs[i]->env->subject)
	    strfcpy (buf, Context->hdrs[i]->env->subject, sizeof (buf));
	  break;
	case M_ID:
	  if (Context->hdrs[i]->env->message_id)
	    strfcpy (buf, Context->hdrs[i]->env->message_id, sizeof (buf));
	  break;
      }
      match = (REGEXEC (exp, buf) == 0);
    }

    if (match)
    {
      if (!not)
	setBit (i, vec);
    }
    else if (not)
      setBit (i, vec);
  }
  regfree (&exp);
  return vec;
}

static void doOR (unsigned char *a, unsigned char *b)
{
  int i;

  for (i=0; i < VecLen; i++)
    a[i] |= b[i];
}

static void doAND (unsigned char *a, unsigned char *b)
{
  int i;

  for (i=0; i < VecLen; i++)
    a[i] &= b[i];
}

/* Returns a vector with bits set to indicate which messages matched the given
 * pattern or NULL if there was an error in the pattern.
 *
 * The ``hdr'' argument is a pointer to the current message (required for
 * handling of the ~H (thread) operator)
 *
 * This routine gets called recursively to handle parentheses in the expression
 */

unsigned char *mutt_match_pattern (char *s, HEADER *hdr)
{
  int i;
  int not = 0; /* Do logical NOT */
  int or = 1;  /* Do logical OR.  Initialized to 1 so the first pattern we
		  match gets OR'd with the zero-initialized ``curvec'' */
  char buf[STRING];
  unsigned char *tmpvec = NULL;
  unsigned char *curvec = allocVec ();
  char *p;

  SKIPWS (s);
  while (*s)
  {
    if (*s == '!')
    {
      not = !not;
      s++;
    }
    else if (*s == '~' && (i = mutt_getvaluebychar (*(s + 1), Flags)) != -1)
    {
      s += 2;
      tmpvec = doKeyword (i, not, hdr);
    }
    else if (*s == '~')
    {
      s++;
      if ((i = mutt_getvaluebychar (*s, Patterns)) == -1)
      {
	mutt_error ("Unknown operator: %c", *s);
	safe_free ((void **) &curvec);
	return NULL;
      }

      /* get the string to match */
      s++;
      SKIPWS (s);

      p = buf;
      if (*s == '\'' || *s == '\"') /* handle quotes (for spaces) */
      {
	char q = *s++;

	while (*s && *s != q)
	{
	  if (*s == '\\')
	    s++; /* quote the next character */
	  *p++ = *s++;
	}
	s++; /* move past the end quote char */
      }
      else
        while (*s && strchr (" !|~(", *s) == NULL)
	{
	  if (*s == '\\')
	    s++; /* quote the next character */
	  *p++ = *s++;
	}
      *p = 0;

      if ((tmpvec = doString (i, buf, not)) == NULL)
      {
	safe_free ((void **) &curvec);
	break;
      }
    }
    else if (*s == '(')
    {
      int count = 1; /* parenthesis level */

      s++;
      SKIPWS (s);

      p = buf;
      while (*s && count)
      {
	if (*s == '(')
	  count++;
	else if (*s == ')')
	  count--;
	*p++ = *s++;
      }

      if (count)
      {
	mutt_error ("Mismatched parentheses.");
	safe_free ((void **) &curvec);
	return NULL;
      }
      *--p = 0;

      if ((tmpvec = mutt_match_pattern (buf, hdr)) == NULL)
      {
	safe_free ((void **) &curvec);
	break;
      }

      if (not)
      {
	for (i=0; i < VecLen; i++)
	  tmpvec[i] = ~tmpvec[i];
      }
    }
    else if (*s == '|')
    {
      s++;
      or = 1;
      not = 0;
    }
    else
    {
      mutt_error ("Error in pattern at: %s", s);
      safe_free ((void **) &curvec);
      break;
    }

    if (tmpvec)
    {
      if (or)
	doOR (curvec, tmpvec);
      else
	doAND (curvec, tmpvec);
      safe_free ((void **) &tmpvec);
      or = not = 0;
    }

    SKIPWS (s);
  }

  return curvec;
}

/* convert a simple search into a real request */
static void check_simple (char *s, size_t len)
{
  char tmp[STRING];
  int i, j;

  if (!strpbrk (s, "!()~|")) /* yup, so spoof a real request */
  {
    /* convert old tokens into the new format */
    if (strcasecmp ("all", s) == 0)
      strfcpy (s, "~A", len);
    else if (strcasecmp ("del", s) == 0)
      strfcpy (s, "~D", len);
    else if (strcasecmp ("flag", s) == 0)
      strfcpy (s, "~F", len);
    else if (strcasecmp ("thread", s) == 0)
      strfcpy (s, "~H", len);
    else if (strcasecmp ("new", s) == 0)
      strfcpy (s, "~N", len);
    else if (strcasecmp ("old", s) == 0)
      strfcpy (s, "~O", len);
    else if (strcasecmp ("repl", s) == 0)
      strfcpy (s, "~Q", len);
    else if (strcasecmp ("read", s) == 0)
      strfcpy (s, "~R", len);
    else if (strcasecmp ("tag", s) == 0)
      strfcpy (s, "~T", len);
    else if (strcasecmp ("unread", s) == 0)
      strfcpy (s, "~U", len);
    else
    {
      if (strchr (s, '\'')) /* check for quotes that need quoting */
      {
	for (i = 0, j = 0; s[i] && j < sizeof (tmp) - 1; i++)
	{
	  if (s[i] == '\'')
	    tmp[j++] = '\\';
	  tmp[j++] = s[i];
	}
	tmp[j] = 0;
      }
      else
	strfcpy (tmp, s, sizeof (tmp));

      mutt_expand_fmt (s, len, SimpleSearch, tmp);
    }
  }
}

int mutt_pattern_func (int op, char *prompt, HEADER *hdr)
{
  char buf[STRING] = "";
  unsigned char *vec;
  int i;

  if (ci_get_field (prompt, buf, sizeof (buf), 0) != 0 || !buf[0])
    return (-1);

  mutt_message ("Compiling search pattern...");

  check_simple (buf, sizeof (buf));
  if ((vec = mutt_match_pattern (buf, hdr)) == NULL)
    return (-1);

  mutt_message ("Executing command on matching messages...");

  if (op == M_LIMIT)
  {
    for (i = 0; i < Context->msgcount; i++)
      Context->hdrs[i]->virtual = -1;
    Context->vcount = 0;
  }

  for (i = 0; i < Context->msgcount; i++)
    if (isSet (i, vec))
    {
      switch (op)
      {
	case M_DELETE:
	  mutt_set_flag (Context, Context->hdrs[i], M_DELETE, 1);
	  break;

	case M_UNDELETE:
	  mutt_set_flag (Context, Context->hdrs[i], M_DELETE, 0);
	  break;

	case M_TAG:
	  mutt_set_flag (Context, Context->hdrs[i], M_TAG, 1);
	  break;

	case M_UNTAG:
	  mutt_set_flag (Context, Context->hdrs[i], M_TAG, 0);
	  break;

	case M_LIMIT:
	  Context->hdrs[i]->virtual = Context->vcount;
	  Context->v2r[Context->vcount] = i;
	  Context->vcount++;
	  break;
      }
    }

  mutt_clear_error ();

  if (op == M_LIMIT && !Context->vcount)
  {
    mutt_error ("No messages matched criteria.");
    /* restore full display */
    Context->vcount = 0;
    for (i = 0; i < Context->msgcount; i++)
    {
      Context->hdrs[i]->virtual = i;
      Context->v2r[i] = i;
    }

    Context->vcount = Context->msgcount;
  }

  safe_free ((void **) &vec);
  return 0;
}

int mutt_search_command (int cur, int op)
{
  int i, j;
  char buf[STRING];
  int incr;
  
  strfcpy (buf, LastSearch, sizeof (buf));

  if (op != OP_SEARCH_NEXT)
  {
    if (ci_get_field ((op == OP_SEARCH) ? "Search for: " : "Reverse search for: ",
		      buf, sizeof (buf), M_CLEAR) != 0 || !buf[0])
      return (-1);

    if (op == OP_SEARCH)
      unset_option (OPTSEARCHREVERSE);
    else
      set_option (OPTSEARCHREVERSE);

    if (strcmp (buf, LastSearch) != 0)
    {
      strfcpy (LastSearch, buf, sizeof (LastSearch));
      set_option (OPTSEARCHINVALID);
    }
  }
  else if (!SearchVector)
  {
    mutt_error ("No search pattern.");
    return (-1);
  }

  /* see if the search pattern changed.  if so, recompute the pattern vector.
   * otherwise use the previously computed pattern to speed things up.
   */
  if (!SearchVector || option (OPTSEARCHINVALID))
  {
    safe_free ((void **) &SearchVector);
    check_simple (buf, sizeof (buf));
    mutt_message ("Compiling search pattern...");

    if (!(SearchVector = mutt_match_pattern (buf, Context->hdrs[Context->v2r[cur]])))
      return (-1);

    mutt_clear_error ();
    unset_option (OPTSEARCHINVALID);
  }

  incr = (option (OPTSEARCHREVERSE)) ? -1 : 1;

  for (i = cur + incr, j = 0 ; j != Context->vcount; j++)
  {
    if (i > Context->vcount - 1)
    {
      i = 0;
      mutt_message ("Search wrapped to top.");
    }
    else if (i < 0)
    {
      i = Context->vcount - 1;
      mutt_message ("Search wrapped to bottom.");
    }

    if (isSet (Context->v2r[i], SearchVector))
      return i;

    i += incr;
  }

  mutt_error ("Not found.");
  return (-1);
}

/* search for a regexp in the body of a message starting with "msgno" */
int mutt_body_search (int msgno, const char *pat)
{
  int i;
  char buf[LONG_STRING];
  regex_t re;

  if ((i = REGCOMP (re, pat, mutt_which_case (pat))) != 0)
  {
    regerror (i, &re, buf, sizeof (buf));
    regfree (&re);
    mutt_error ("%s", buf);
    return (-1);
  }

  mutt_message ("Searching...");

  buf[sizeof (buf) - 1] = 0;

  for (i = msgno + 1; i < Context->vcount; i++)
  {
    if (msg_search (&re, buf, sizeof (buf), 0, Context->v2r[i]))
      return (i);
  }

  regfree (&re);

  mutt_error ("Not found.");

  return (-1);
}
