/* cfengine for GNU
 
        Copyright (C) 1995
        Free Software Foundation, Inc.
 
   This file is part of GNU cfengine - written and maintained 
   by Mark Burgess, Dept of Computing and Engineering, Oslo College,
   Dept. of Theoretical physics, University of Oslo
 
   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 2, or (at your option) any
   later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA

*/
 

/*********************************************************************/
/*                                                                   */
/*  TOOLKITS: "object" library                                       */
/*                                                                   */
/*********************************************************************/

#include "cf.defs.h"
#include "cf.extern.h"

/*******************************************************************/
/* Macro substitution based on HASH-table                          */
/*******************************************************************/

Hash(name)

char *name;

{ int i, slot = 0;

 for (i = 0; name[i] != '\0'; i++)
   {
   slot = (macroalphabet * slot + name[i]) % hashtablesize;
   }

if (DEBUG || D1)
   {
   printf("Hash(%s)=%d\n",name,slot);
   }

return slot;
}

/*******************************************************************/

AddMacroValue(name,value)                           /* Like putenv */

char *name,*value;

{ char *sp, buffer[bufsize];
  int slot;

if (DEBUG || D1)
   {
   printf("AddMacroValue(%s=%s)\n",name,value);
   }

if (name == NULL || value == NULL)
   {
   yyerror("Bad macro");
   }

if (strlen(name) > maxvarsize)
   {
   yyerror("macro name too long");
   return;
   }

buffer[0] = NULL;

sprintf(buffer,"%s=%s",name,value);

if ((sp = malloc(strlen(buffer)+1)) == NULL)
   {
   perror("malloc");
   FatalError("aborting");
   }

strcpy(sp,buffer);

slot = Hash(name);

if (HASH[slot] != 0)
   {
   if (CompareMacro(name,HASH[slot]) == 0)
      {
      sprintf(VBUFF,"Redefinition of macro %s",name);
      Warning(VBUFF);
      free(HASH[slot]);
      HASH[slot] = sp;
      return;
      }

   while ((HASH[++slot] != 0))
      {
      if (slot == hashtablesize-1)
         {
         slot = 0;
         }
      if (slot == Hash(name))
         {
         FatalError("AddMacroValue - internal error #1");
         }
      }
   }

HASH[slot] = sp;

if (DEBUG || D1)
   {
   printf("Added Macro at hash address %d: %s\n",slot,sp);
   }
}


/*******************************************************************/

char *GetMacroValue(name)

char *name;

{ char *sp;
  int slot;

slot = Hash(name);

if (CompareMacro(name,HASH[slot]) != 0)
   {
   while (true)
      {
      slot++;

      if (CompareMacro(name,HASH[slot]) == 0)
         {
         for(sp = HASH[slot]; *sp != '='; sp++)
            {
            }

         return(sp+1);
         }

      if (slot == hashtablesize-1)
         {
         slot = 0;
         }

      if (slot == Hash(name))
         {
         return(getenv(name));  /* Look in environment if not found */
         }
      }
   }
else
   {
   for(sp = HASH[slot]; *sp != '='; sp++)
      {
      }

   return(sp+1);
   }   
}

/*******************************************************************/

RecordMacroId(name)

char *name;

{
if (DEBUG || D1)
   {
   printf("RecordMacroId(%s)\n",name);
   }

strcpy(CURRENTITEM,name);
}

/*******************************************************************/

CompareMacro(name,macro)

char *name,*macro;

{ char buffer[bufsize];

if (macro == NULL || name == NULL)
   {
   return 1;
   }

sscanf(macro,"%[^=]",buffer);

if (DEBUG || D1)
   {
   printf("CompareMacro(%s,%s)=%s\n",name,macro,buffer);
   }

return(strcmp(buffer,name));
}

/***************************************************************/
/* Modestring toolkit                                          */
/***************************************************************/

  enum modestate
     {
     who,
     which
     };

ParseModeString(modestring,plusmask,minusmask)

char *modestring;
mode_t *plusmask, *minusmask;

{ char modebuffer[maxvarsize], *sp; 
 int affected = 0, value = 0, gotaction;
  char action = '=';
  enum modestate state = who;

if (DEBUG || D1)
   {
   printf("ParseModeString(%s)\n",modestring);
   }

gotaction = false;
*plusmask = *minusmask = 0;

for (sp = modestring; true ; *sp++)
   {
   switch (*sp)
      {
      case 'a': CheckModeState(who,state,*sp);
                affected |= 07777;
                break;

      case 'u': CheckModeState(who,state,*sp);
                affected |= 00700;
                break;

      case 'g': CheckModeState(who,state,*sp);
                affected |= 00070;
                break;

      case 'o': CheckModeState(who,state,*sp);
                affected |= 00007;
                break;

      case '+':
      case '-':
      case '=': if (gotaction)
                   {
                   yyerror("Too many +-= in mode string");
                   }
                action = *sp;
                state = which;
                gotaction = true;
                break;

      case 'r': CheckModeState(which,state,*sp);
                value |= 0444 & affected;
                break;

      case 'w': CheckModeState(which,state,*sp);
                value |= 0222 & affected;
                break;

      case 'x': CheckModeState(which,state,*sp);
                value |= 0111 & affected;
                break;

      case 's': CheckModeState(which,state,*sp);
                value |= 06000 & affected;
                break;

      case 't': CheckModeState(which,state,*sp);
                value |= 01000;
                break;

      case '0':
      case '1':
      case '2':
      case '3':
      case '4':
      case '5':
      case '6':
      case '7': state = which;
                sscanf(sp,"%o",&value);
                while (isdigit(*sp) && (*sp != NULL))
		   {
                   sp++;
                   }
                sp--;
                break;

      case ',':
                SetMask(action,value,plusmask,minusmask);
                action = '=';
                affected = 0;
                value = 0;
                gotaction = false;
                state = who;
                break;

      case '\0':
                if (state == who || value == 0)
                   {
                   Warning("mode string is incomplete");
                   }

                SetMask(action,value,plusmask,minusmask);
                if (DEBUG || D1)
                   {
                   printf("[PLUS=%o][MINUS=%o]\n",*plusmask,*minusmask);
                   }

                return;

      default:  yyerror ("Invalid mode string");
                break;
      }
   }
}

/*********************************************************/

CheckModeState(stateA,stateB,ch)

enum modestate stateA;
int stateB;
char ch;

{
if ((int)stateA != stateB)
   {
   sprintf(VBUFF,"Mode string constant (%c) used out of context",ch);
   yyerror(VBUFF);
   }
return;
}

/*********************************************************/

SetMask(action,value,p,m)

char action;
int value;
mode_t *p,*m;

{
if (DEBUG || D1)
   {
   printf("SetMask(%c%o)\n",action,value);
   }

switch(action)
   {
   case '+':
             *p |= value;
             *m |= 0;
             return;
   case '-':
             *p |= 0;
             *m |= value;
             return;
   case '=':
             *p |= value;
             *m |= (~value) & 07777;
             return;
   default:
             sprintf(VBUFF,"Mode directive %c is unknown",action);
             yyerror(VBUFF);
             return;
   }
}


/*********************************************************************/
/* TOOLKIT : Various                                                 */
/*********************************************************************/

CheckBuffSpace(str,len)

{
if ((strlen(str)+len) > (bufsize - buffer_margin))
   {
   FatalError("Buffer space exhausted. Increase macro bufsize.");
   }
}

/*********************************************************************/

IsExcluded(exception)

char *exception;

{ char *sp = exception;
  char cbuff[maxvarsize];
  int returnval = false;

if (DEBUG || D1)
   {
   printf("IsExcluded(%s)",exception);
   }

while(*sp != NULL)
   {
   sscanf(sp,"%[^.]",cbuff);

   while ((*sp != NULL) && (*sp !='.'))
      {
      sp++;
      }

   if (*sp == '.')
      {
      sp++;
      }

   if (! IsDefinedClass(cbuff) && ! IsItemIn(VADDCLASSES,cbuff))
      {
      if (DEBUG || D2)
         {
         printf("%s is excluded!\n",cbuff);
         }

      return true;
      }  
   }

return false;
}

/*********************************************************************/
/* TOOLIT : Locks and Signals                                        */
/*********************************************************************/

Locked()
 
{ FILE *lock;

if (IGNORELOCK)
   {
   return false;
   }

if (getuid() != 0)    /* Only establish a lock if we are root      */
   {                  /* Other users can only check parsing anyway */
   return(false);
   }
 
if ((lock = fopen(LOCKFILE,"r")) != NULL)
   {
   printf("cfengine: already running? Lock %s already exists!\n",LOCKFILE);
   printf("          Remove lock and try again.\n\n");
   fclose(lock);
   return(true);
   }
 
if ((lock = fopen(LOCKFILE,"w")) != NULL)
   {
   fprintf(lock,"%d",getpid());
   fclose(lock);
   return(false);
   }
 
printf("cfengine: Can't open %s.",LOCKFILE);
return(true);
}

/********************************************************************/
 
HandleSignal(signum)
 
int signum;
 
{
printf("cfengine: Received signal %s\n",SIGNALS[signum]);

if (signum == 15 || signum == 9)
   {
   Unlock();
   exit(0);
   }
}
 
/************************************************************************/
 
Unlock()
 
{
if (IGNORELOCK)
   {
   return;
   }

if (getuid() != 0)
   {
   if (DEBUG || D2)
      {
      printf(">> Not root, no unlock needed\n");
      }
   return;
   }

if (unlink(LOCKFILE) == -1)
   {
   printf("cfengine: Failed to remove %s\n",LOCKFILE);
   }
}
 
/*********************************************************************/
/* TOOLKIT : Varstring expansion                                     */
/*********************************************************************/

IsVarString(str)

char *str;

{ char *sp;
  char left = 'x', right = 'x';
  int dollar = false;
  int bracks = 0;

for (sp = str; *sp != NULL ; sp++)       /* check for varitems */
   {
   switch (*sp)
      {
      case '$': dollar = true;
                break;
      case '(':
      case '{': left = *sp;
                bracks++;
                break;
      case ')':
      case '}': right = *sp;
                bracks--;
                break;
      }
   }

if (left == 'x' && right == 'x')
   {
   return false;
   }

if (bracks != 0 && dollar)
   {
   yyerror("Incomplete variable syntax");
   return false;
   }

if (bracks != 0)
   {
   return false;
   }

if (left == '(' && right == ')' && dollar)
   {
   return true;
   }

if (left == '{' && right == '}' && dollar)
   {
   return true;
   }

yyerror("Bracket mismatch in variable substitution");
return false;
}

/*********************************************************************/

ExpandVarstring(string,buffer,bserver) 

char *string, *buffer, *bserver;

{ char *sp,*env;
  char varstring = false;
  char currentitem[bufsize], scanstr[6];

if (DEBUG || D1)
   {
   printf(">>Expanding varstring %s\n",string);
   }

buffer[0] = NULL;

for (sp = string; /* No exit */ ; sp++)       /* check for varitems */
   {
   currentitem[0] = NULL;

   sscanf(sp,"%[^<$]",currentitem);

   strcat(buffer,currentitem);
   sp += strlen(currentitem);

   if (*sp == '$')
      {
      switch (*(sp+1))
         {
         case '(': varstring = ')';
                   break;
         case '{': varstring = '}';
                   break;
         default: yyerror("Illegal use of $ symbol or bad variable");
         }
      sp++;
      }

   currentitem[0] = NULL;

   if (*sp == NULL)
      {
      break;
      }
   else
      {
      sprintf(scanstr,"%%[^%c]",varstring);   /* select the correct terminator */
      sscanf(++sp,scanstr,currentitem);               /* reduce item */

      switch (ScanVariable(currentitem))
         {
         case fac:
         case sit:
                   if (VFACULTY[0] == NULL)
                      {
                      yyerror("faculty/site undefined variable");
                      }
                   strcat(buffer,VFACULTY);
                   break;


         case host:
                    strcat(buffer,VSYSNAME.nodename);
                    break;

         case binserv:
                    if (strlen(bserver) == 0)
                       {
                       yyerror("Inappropriate use of variable binserver");
                       FatalError("Bad variable");
                       }
                    strcat(buffer,bserver);
                    break;

         case sysad:
                   if (VSYSADM[0] == NULL)
                      {
                      yyerror("sysadm undefined variable");
                      }
                   strcat(buffer,VSYSADM);
                   break;

         case dom:
                   if (VDOMAIN[0] == NULL)
                      {
                      yyerror("domain undefined variable");
                      }
                   strcat(buffer,VDOMAIN);
                   break;

         case nfstp:
                   strcat(buffer,VNFSTYPE);
                   break;

         case timez:
                   if (VTIMEZONE[0] == NULL)
                      {
                      yyerror("timezone undefined variable");
                      }
                   strcat(buffer,VTIMEZONE);
                   break;

         default:  
                   if ((env = GetMacroValue(currentitem)) != NULL) /* check for environment variable */
                      {
                      if (strlen(env) + strlen(buffer) > bufsize - buffer_margin)
                         {
                         printf("cfengine: Internal software failure expanding varstring\n");
                         printf("          Constant `bufsize' was too small to accomodate\n");
                         printf("          environment variable %s\n",currentitem);
                         FatalError("Aborted");
                         }
                      strcat(buffer,env);
                      break;
                      }

                   printf("Bad variable $(%s)\n",currentitem);
                   FatalError("No such variable or out of context");
         }

      sp += strlen(currentitem);
      currentitem[0] = NULL;
      }
   }

if (DEBUG || D2)
   {
   printf(">>Varstring expanded to %s\n",buffer);
   }

return varstring;
}

/*********************************************************************/

enum vnames ScanVariable(name)

char *name;

{ int i;

for (i = 0; VVNAMES[i] != NULL; i++)
   {
   if (strcmp(VVNAMES[i],name) == 0)
      {
      return (enum vnames) i;
      }
   }
return (enum vnames) i;
}

/*********************************************************************/
/* TOOLKIT : Item list                                               */
/*********************************************************************/

IsItemIn(list,item)

struct Item *list;
char *item;

{ struct Item *ptr; 

for (ptr = list; ptr != NULL; ptr=ptr->next)
   {
   if (strcmp(ptr->name,item) == 0)
      {
      return(true);
      }
   }
return(false);
}

/*********************************************************************/

struct Item *LocateNextItemContaining(list,string) 

struct Item *list;
char *string;

{ struct Item *ip;

for (ip = list; ip != NULL; ip=ip->next)
   {
   if (strstr(ip->name,string))
      {
      return ip;
      }
   }

return NULL;
}

/*********************************************************************/

struct Item *LocateNextItemMatching(list,string) 

struct Item *list;
char *string;

{ struct Item *ip, *next, *last;

for (ip = list; (ip != NULL); ip=ip->next)
   {
   if (strcmp(ip->name,string) == 0)
      {
      return ip;
      }
   last = ip;
   }

return NULL;
}

/*********************************************************************/

struct Item *LocateNextItemStarting(list,string) 

struct Item *list;
char *string;

{ struct Item *ip;

for (ip = list; (ip != NULL); ip=ip->next)
   {
   if (strncmp(ip->name,string,strlen(string)) == 0)
      {
      return ip;
      }
   }

return NULL;
}

/*********************************************************************/

DeleteItemStarting(list,string) /* deletes first item to match the string */

struct Item **list;
char *string;

{ struct Item *ip, *next, *last;

for (ip = *list; ip != NULL; ip=ip->next)
   {
   if (strncmp(ip->name,string,strlen(string)) == 0)
      {
      if (ip == *list)
         {
         free((*list)->name);
         if (ip->classes != NULL) 
	    {
            free(ip->classes);
	    }
         *list = ip->next;
         free((char *)ip);
         if (DEBUG || D2) printf (">> Deleted item %s\n",string);
         NUMBEROFEDITS++;
         return true;
         }
      else
         {
         last->next = ip->next; 
         free(ip->name);
         if (ip->classes != NULL) 
	    {
            free(ip->classes);
	    }
         free((char *)ip);
         if (DEBUG || D2) printf (">> Deleted item %s\n",string);
         NUMBEROFEDITS++;
         return true;
         }

      }
   last = ip;
   }

return false;
}

/*********************************************************************/

DeleteItemContaining(list,string) /* deletes first item to match the string */

struct Item **list;
char *string;

{ struct Item *ip, *next, *last;

for (ip = *list; ip != NULL; ip=ip->next)
   {
   if (strstr(ip->name,string))
      {
      if (ip == *list)
         {
         free((*list)->name);
         *list = ip->next;
         if (ip->classes != NULL) 
	    {
            free(ip->classes);
	    }
         free((char *)ip);
         if (DEBUG || D2) printf (">> Deleted item %s\n",string);
         NUMBEROFEDITS++;
         return true;
         }
      else
         {
         last->next = ip->next; 
         free(ip->name);
         if (ip->classes != NULL) 
	    {
            free(ip->classes);
	    }
         free((char *)ip);
         if (DEBUG || D2) printf (">> Deleted item %s\n",string);
         NUMBEROFEDITS++;
         return true;
         }

      }
   last = ip;
   }

return false;
}

/*********************************************************************/

PrependItem (liststart,itemstring,classes)

struct Item **liststart;
char *itemstring,classes;

{ struct Item *ip;
  char *sp,*spe;

if (DEBUG || D2)
   {
   printf(">> Prepending %s to list\n",itemstring);
   }

if ((ip = (struct Item *)malloc(sizeof(struct Item))) == NULL)
   {
   perror("Can't allocate memory in PrependItem()");
   FatalError("");
   }

if ((sp = malloc(strlen(itemstring)+2)) == NULL)
   {
   perror("Can't allocate memory in PrependItem()");
   FatalError("");
   }

if ((classes!= NULL) && (spe = malloc(strlen(classes)+2)) == NULL)
   {
   perror("Can't allocate memory in PrependItem()");
   FatalError("");
   }

strcpy(sp,itemstring);
ip->name = sp;
ip->next = *liststart;
*liststart = ip;

if (classes != NULL)
   {
   strcpy(spe,classes);
   ip->classes = spe;
   }
else
   {
   ip->classes = NULL;
   }

NUMBEROFEDITS++;
}

/*********************************************************************/

AppendItem (liststart,itemstring,classes)

struct Item **liststart;
char *itemstring,*classes;

{ struct Item *ip, *lp;
  char *sp,*spe;

if (DEBUG || D2)
   {
   printf(">> Appending %s to list\n",itemstring);
   }

if ((ip = (struct Item *)malloc(sizeof(struct Item))) == NULL)
   {
   perror("Can't allocate memory in AppendItem()");
   FatalError("");
   }

if ((sp = malloc(strlen(itemstring)+2)) == NULL)
   {
   perror("Can't allocate memory in AppendItem()");
   FatalError("");
   }

if (*liststart == NULL)
   {
   *liststart = ip;
   }
else
   {
   for (lp = *liststart; lp->next != NULL; lp=lp->next)
      {
      }

   lp->next = ip;
   }


if ((classes!= NULL) && (spe = malloc(strlen(classes)+2)) == NULL)
   {
   perror("Can't allocate memory in PrependItem()");
   FatalError("");
   }


strcpy(sp,itemstring);
ip->name = sp;
ip->next = NULL;

if (classes != NULL)
   {
   strcpy(spe,classes);
   ip->classes = spe;
   }
else
   {
   ip->classes = NULL;
   }

NUMBEROFEDITS++;
}

/*********************************************************************/

DeleteItemList(item)                    /* delete starting from item */
 
struct Item *item;

{ struct Item *ip;

if (item != NULL)
   {
   DeleteItemList(item->next);
   item->next = NULL;

   if (item->name != NULL)
      {
      free (item->name);
      if (item->classes != NULL)
         {
         free (item->classes);
         }
      }

   free((char *)item);
   }
}

/*********************************************************************/

DeleteItem(liststart,item)
 
struct Item **liststart,*item;

{ struct Item *ip, *sp;

if (item != NULL)
   {
   if (item->name != NULL)
      {
      free (item->name);
      }

   if (item->classes != NULL)
      {
      free (item->classes);
      }

   sp = item->next;

   free((char *)item);

   if (item == *liststart)
      {
      *liststart = NULL;
      }
   else
      {
      for (ip = *liststart; ip->next != item; ip=ip->next)
         {
         }

      ip->next = sp;
      }

   NUMBEROFEDITS++;
   }
}

/*********************************************************************/

LoadItemList(liststart,file)

struct Item **liststart;
char *file;

{ FILE *fp;
  struct stat statbuf;

if (stat(file,&statbuf) == -1)
   {
   printf("cfengine: Couldn't stat %s\n",file);
   return false;
   }

if (statbuf.st_size > EDITFILESIZE)
   {
   printf("cfengine: File %s is bigger than the limit <editfilesize>\n",file);
   return(false);
   }

if (! S_ISREG(statbuf.st_mode))
   {
   printf("cfengine: %s is not a plain file\n",file);
   return false;
   }

if ((fp = fopen(file,"r")) == NULL)
   {
   printf("cfengine: Couldn't read file %s for editing\n",file);
   return false;
   }

while(!feof(fp))
   {
   fgets (VBUFF,bufsize,fp);

   if(VBUFF[strlen(VBUFF)-1] == '\n')
      {
      VBUFF[strlen(VBUFF)-1] = NULL;        /* chop */
      }

   if (!feof(fp)) 
      {
      AppendItem(liststart,VBUFF,NULL);
      }

   VBUFF[0] = NULL;
   }

fclose(fp);
return (true);
}

/*********************************************************************/

SaveItemList(liststart,file)

struct Item *liststart;
char *file;

{ struct Item *ip;
  struct stat statbuf;
  FILE *fp;

if (stat(file,&statbuf) == -1)
   {
   printf("cfengine: Couldn't stat %s, which needed editing!\n",file);
   printf("          Check definition in program - is file NFS mounted?\n\n");
   return;
   }

strcpy(VBUFF,file);
strcat(VBUFF,".cfenginesaved");

if (rename(file,VBUFF) == -1)
   {
   printf("cfengine: Error occurred while renaming %s\n",file);
   perror("rename ");
   return;
   }

if ((fp = fopen(file,"w")) == NULL)
   {
   printf("cfengine: Couldn't write file %s after editing\n",file);
   rename(VBUFF,file);
   return NULL;
   }

for (ip = liststart; ip != NULL; ip=ip->next)
   {
   fprintf(fp,"%s\n",ip->name);
   }

printf("cfengine: Edited file %s and moved original to %s\n",file,VBUFF);

fclose(fp);
chmod(file,statbuf.st_mode);                    /* Restore file permissions etc */
chown(file,statbuf.st_uid,statbuf.st_gid);
}

/*********************************************************************/

DebugListItemList(liststart)

struct Item *liststart;

{ struct Item *ptr;

for (ptr = liststart; ptr != NULL; ptr=ptr->next)
   {
   printf("CFDEBUG: [%s]\n",ptr->name);
   }
}

/*********************************************************************/

CommentItemStarting(list,string,comm)

struct Item **list;
char *string,*comm;

{ struct Item *ip, *next, *last;
  char buff[bufsize];

for (ip = *list; ip != NULL; ip=ip->next)
   {
   if (strncmp(ip->name,string,strlen(string)) == 0)
      {
      if (strlen(ip->name)+strlen(comm) > bufsize)
         {
         printf("cfengine: bufsize overflow while commenting line - abort\n");
         return false;
         }

      if (strncmp(ip->name,comm,strlen(comm))== 0)
         {
         continue;
         }

      sprintf(buff,"%s%s",comm,ip->name);
      free(ip->name);

      if ((ip->name = malloc(strlen(buff)+1)) == NULL)
         {
         printf("cfengine: malloc in CommentItemStarting\n ");
         exit(1);
         }

      strcpy(ip->name,buff);
      NUMBEROFEDITS++;

      return true;
      }
   }

return false;
}

/*********************************************************************/

CommentItemContaining(list,string,comm)

struct Item **list;
char *string,*comm;

{ struct Item *ip, *next, *last;
  char buff[bufsize];

for (ip = *list; ip != NULL; ip=ip->next)
   {
   if (strstr(ip->name,string))
      {
      if (strlen(ip->name)+strlen(comm) > bufsize)
         {
         printf("cfengine: bufsize overflow while commenting line - abort\n");
         return false;
         }

      if (strncmp(ip->name,comm,strlen(comm))== 0)
         {
         continue;
         }

      sprintf(buff,"%s%s",comm,ip->name);
      free(ip->name);

      if ((ip->name = malloc(strlen(buff)+1)) == NULL)
         {
         printf("cfengine: malloc in CommentItemContaining\n ");
         exit(1);
         }

      strcpy(ip->name,buff);
      NUMBEROFEDITS++;

      return true;
      }
   }

return false;
}

/*********************************************************************/

CommentItemMatching(list,string,comm)

struct Item **list;
char *string,*comm;

{ struct Item *ip, *next, *last;
  char buff[bufsize];

for (ip = *list; ip != NULL; ip=ip->next)
   {
   if (strcmp(ip->name,string))
      {
      if (strlen(ip->name)+strlen(comm) > bufsize)
         {
         printf("cfengine: bufsize overflow while commenting line - abort\n");
         return false;
         }

      if (strncmp(ip->name,comm,strlen(comm))== 0)
         {
         continue;
         }

      sprintf(buff,"%s%s",comm,ip->name);
      free(ip->name);

      if ((ip->name = malloc(strlen(buff)+1)) == NULL)
         {
         printf("cfengine: malloc in CommentItemContaining\n ");
         exit(1);
         }

      strcpy(ip->name,buff);
      NUMBEROFEDITS++;

      return true;
      }
   }

return false;
}



/*********************************************************************/
/* TOOLKIT : links                                                   */
/*********************************************************************/

LinkChildFiles(from,to)

{ DIR *dirh;
  struct dirent *dirp;
  char pcwdto[bufsize],pcwdfrom[bufsize];
  struct stat statbuf;

if (stat(to,&statbuf) == -1)
   {
   return(false);  /* no error warning, since the higher level routine uses this */
   }

if ((dirh = opendir(to)) == NULL)
   {
   printf("cfengine: LinkChildFiles(): Can't open directory %s\n",to);
   return false;
   }

for (dirp = readdir(dirh); dirp != NULL; dirp = readdir(dirh))
   {
   if (strcmp("lost+found",dirp->d_name) == 0 || strcmp("..",dirp->d_name) == 0 || strcmp(".",dirp->d_name) == 0 )
      {
      continue;
      }

   strcpy(pcwdto,to);                               /* Assemble pathnames */
   AddSlash(pcwdto);
   strcat(pcwdto,dirp->d_name);

   strcpy(pcwdfrom,from);
   AddSlash(pcwdfrom);
   strcat(pcwdfrom,dirp->d_name);

   LinkFiles(pcwdfrom,pcwdto);
   }

return true;
}

/*********************************************************************/

LinkFiles(from,to)           /* should return true if 'to' found */

char *from, *to;

{ struct stat buf,exbuf;
  char linkbuf[bufsize], saved[bufsize], *sp;

if (stat(to,&buf) == -1)
   {
   return(false);  /* no error warning, since the higher level routine uses this */
   }

if (DEBUG || D2)
   {
   printf("Trying to link %s -> %s\n",from,to);
   }

if (lstat(from,&buf) == 0)
   {
   if (! S_ISLNK(buf.st_mode) && ! ENFORCELINKS)
      {
      printf("cfengine: Error linking %s -> %s\n",from,to);
      printf("          Cannot make link: %s exists and is not a link! (uid %d)\n",from,buf.st_uid);
      return(true);
      }

   if (S_ISREG(buf.st_mode) && ENFORCELINKS)
      {
      printf("cfengine: moving %s to %s.cfengine.saved\n",from,from);

      if (DONTDO)
         {
         return true;
         }

      saved[0] = NULL;
      strcpy(saved,from);
      strcat(saved,".cfenginesaved");

      if (rename(from,saved) == -1)
         {
         perror("rename");
         return(true);
         }
      }
   }

for (sp = linkbuf; sp < linkbuf+bufsize; sp++)      /* clear buff, readlink(2) */
   {
   *sp = NULL;
   }

if (readlink(from,linkbuf,bufsize) == -1)
   {
   if (! MakeDirectoriesFor(from))                  /* link doesn't exist */
      {
      printf("cfengine: Couldn't build directory tree up to %s!\n",from);
      printf("          One element was a plain file, not a directory!\n");
      return(true);
      }
   }
else
   {
   if (DEBUG || D2) printf("Comparing old links %s to %s\n",linkbuf,to);

   if (strcmp(linkbuf,to) != 0)
      {
      if (ENFORCELINKS)
         {
         printf("cfengine: remove link %s\n",from);

         if (!DONTDO)
            {
            if (unlink(from) == -1)
               {
               perror("unlink");
               return true;
               }

            DoLink(from,to);
            return true;
            }
         }
      else
         {
         printf("cfengine: Warning old link %s points somewhere else. Doing nothing!\n",from);
         printf("          (Link points to %s not %s)\n\n",linkbuf,to);
         return(true);
         }
      }
   else
      {
      if (VERBOSE || DEBUG || D2)
         {
         printf("cfengine: Link (%s->%s) exists and is okay.\n",from,to);
         }

      KillOldLink(from);             /* Check whether link points somewhere */

      return(true);
      }
   }

DoLink(from,to);
return(true);
}

/*********************************************************************/

DoLink (from,to)

char *from, *to;

{
if (DONTDO)
   {
   if (VERBOSE || DEBUG || D2) 
      {
      printf("cfengine: Link files %s -> %s\n\n",from,to);
      }
   }
else
   {
   printf("cfengine: Linking files %s -> %s\n\n",from,to);

   if (symlink(to,from) == -1)
      {
      perror("symlink failed");
      }
   }
}

/*********************************************************************/

KillOldLink (name)

char *name;

{ char linkbuf[bufsize];
  char linkpath[bufsize],*sp;
  struct stat statbuf;
  short i;

for (i = 0; i < bufsize; i++)
   {
   linkbuf[i] = linkpath[i] = NULL;
   }

if (readlink(name,linkbuf,bufsize) == -1)
   {
   printf("cfengine: KillOldLink() contradiction! Can't read existing link!\n");
   return;
   }

if (linkbuf[0] != '/')
   {
   strcpy(linkpath,name);    /* Get path to link */

   for (sp = linkpath+strlen(linkpath); (*sp != '/') && (sp >= linkpath); sp-- )
     {
     *sp = NULL;
     }
   }

strcat(linkpath,linkbuf);

if (stat(linkpath,&statbuf) == -1)               /* link points nowhere */
   {
   if (VERBOSE || KILLOLDLINKS || DEBUG || D2)
      {
      printf("cfengine: %s is a link which points to %s\n",name,linkpath);
      printf("          but that file doesn't exist!\n");
      }

   if (KILLOLDLINKS)
      {
      printf("cfengine: removing dead link %s\n",name);

      if (! DONTDO)
         {
         unlink(name);  /* May not work on a client-mounted system ! */
         }
      }
   }
}

/*********************************************************************/
/* TOOLKIT : files/directories                                       */
/*********************************************************************/

AddSlash(str)

char *str;

{
if (str[strlen(str)-1] != '/')
   {
   strcat(str,"/");
   }
}

/*********************************************************************/

IsHomeDir(name)

char *name;

{ char *sp;
  struct Item *ip;

for (sp = name+strlen(name); *(sp-1) != '/' && sp >= name; sp--)
   {
   }

for (ip = VHOMEPATLIST; ip != NULL; ip=ip->next)
   {
   if (WildMatch(ip->name,sp))
      {
      return(true);
      }
   }

return(false);
}


/*********************************************************************/

MakeDirectoriesFor(file)  /* Make all directories which underpin file */

char *file;

{ char *sp,*spc;
  char currentpath[bufsize];
  char pathbuf[bufsize];
  struct stat statbuf;

if (DEBUG || D2)
   {
   printf("Want to make directories for %s\n",file);
   }

strcpy(pathbuf,file);                                        /* local copy */

for (sp = pathbuf+strlen(file); *sp != '/'; sp--)        /* skip link name */
   {
   *sp = NULL;
   }

if (lstat(file,&statbuf) != -1)
   {
   if (! S_ISDIR(statbuf.st_mode))
      {
      printf("cfengine: Warning. The object %s is not a directory.\n",file);
      printf("          Cannot make a new directory without deleting it!\n\n");
      return(false);
      }
   }

for (sp = file, spc = currentpath; *sp != NULL; sp++)
   {
   if (*sp != '/' && *sp != '\0')
      {
      *spc = *sp;
      spc++;
      }
   else
      {
      *spc = NULL;

      if (strlen(currentpath) == 0)
         {
         }
      else if (stat(currentpath,&statbuf) == -1)
         {
         if (VERBOSE || DEBUG || D2) 
            {
            printf("cfengine: Making directory %s, mode %o\n",currentpath,DEFAULTMODE);
            }

         if (! DONTDO)
            {
            if (mkdir(currentpath,DEFAULTMODE) == -1)
               {
               printf("cfengine: Unable to make path %s\n",file);
               perror("          ");
               return(false);
               }
            }
         }
      else
         {
         if (! S_ISDIR(statbuf.st_mode))
            {
            printf("MakeDirectoriesFor() failed. %s is not a directory!\n",currentpath);
            return(false);
            }
         }

      *spc = '/';
      spc++;
      }
   }

if (VERBOSE || DEBUG || D2)
   {
   printf("Directory %s exists. Okay\n",file);
   }

return(true);
}

/*********************************************************************/

RecursiveCheck(name,plus,minus,action,uidlist,gidlist,recurse,rlevel)

char *name;
mode_t plus,minus;
struct UidList *uidlist;
struct GidList *gidlist;
enum fileactions action;
int recurse;
int rlevel;

{ DIR *dirh;
  struct dirent *dirp;
  char pcwd[bufsize];
  struct stat statbuf;

if (recurse == -1)
   {
   return;
   }

if (DEBUG || D2)
   {
   printf("cfengine: entering %s\n",name);
   }

if ((dirh = opendir(name)) == NULL)
   {
   printf("cfengine: RecursiveCheck(): Can't open directory %s\n",name);
   perror("opendir");
   return;
   }

for (dirp = readdir(dirh); dirp != NULL; dirp = readdir(dirh))
   {
   if (strcmp("lost+found",dirp->d_name) == 0 || strcmp("..",dirp->d_name) == 0 || strcmp(".",dirp->d_name) == 0 )
      {
      continue;
      }

   if (strcmp(".cfengine.rm",dirp->d_name) == 0)
      {
      continue;
      }

   if (IgnoreFile(name,dirp->d_name))
      {
      continue;
      }

   strcpy(pcwd,name);                                   /* Assemble pathname */
   AddSlash(pcwd);
   strcat(pcwd,dirp->d_name);

   if (strlen(pcwd) > bufsize - buffer_margin)
      {
      printf("cfengine: buffer overflow constructing string in RecursiveCheck\n");
      printf(" culprit: %^s\n",pcwd);
      return;
      }


   if (TRAVLINKS)
      {
      if (stat(pcwd,&statbuf) == -1)
         {
         printf("cfengine: was working in %s when this happened:\n",name);
         printf("          RecursiveCheck(): Can't stat %s\n",dirp->d_name);
         continue;
         }
      }
   else
      {
      if (lstat(pcwd,&statbuf) == -1)
         {
         printf("cfengine: was working in %s when this happened:\n",name);
         printf("          RecursiveCheck(): Can't stat %s\n",dirp->d_name);
         continue;
         }
      }

   if (S_ISLNK(statbuf.st_mode))            /* should we ignore links? */
      {
      CheckExistingFile(pcwd,plus,minus,action,uidlist,gidlist,&statbuf);
      continue;
      }

   if (S_ISDIR(statbuf.st_mode))
      {
      if (IsMountedFileSystem(&statbuf,pcwd,rlevel) || (recurse-1 == 0))
         {
         continue;
         }
      else
         {
         if (strcmp(dirp->d_name,".") != 0)  /* avoid infinite regression */
            {
            CheckExistingFile(pcwd,plus,minus,action,uidlist,gidlist,&statbuf);
            RecursiveCheck(pcwd,plus,minus,action,uidlist,gidlist,recurse-1,rlevel+1);
            }
         else
            {
            CheckExistingFile(pcwd,plus,minus,action,uidlist,gidlist,&statbuf);
            }
         }
      }
   else
      {
      CheckExistingFile(pcwd,plus,minus,action,uidlist,gidlist,&statbuf);
      }
   }

closedir(dirh);
}


/*********************************************************************/

CheckExistingFile(file,plus,minus,action,uidlist,gidlist,stat)

char *file;
mode_t plus,minus;
struct UidList *uidlist;
struct GidList *gidlist;
enum fileactions action;
struct stat *stat;

{ mode_t newperm = stat->st_mode;
  int amroot = true;

if (DEBUG || D2)
   {
   printf("Checking file/dir %s\n",file);
   }

if (getuid() != 0)                            
   {
   amroot = false;
   }

 /* directories must have x set if r set, regardless  */


newperm = (stat->st_mode & 07777) ;
newperm |= plus;
newperm &= ~minus;


if (S_ISREG(stat->st_mode) && (action == fixdirs || action == warndirs)) 
   {
   return;
   }

if (S_ISDIR(stat->st_mode))  
   {
   if (action == fixplain || action == warnplain)
      {
      return;
      }

   if (stat->st_mode & S_IRUSR)
      {
      newperm  |= S_IXUSR;
      }

   if (stat->st_mode & S_IRGRP)
      {
      newperm |= S_IXGRP;
      }

   if (stat->st_mode & S_IROTH)
      {
      newperm |= S_IXOTH;
      }
   }

if (stat->st_uid == 0 && (stat->st_mode & S_ISUID))
   {
   if (newperm & S_ISUID)
      {
      if (! IsItemIn(VSETUIDLIST,file))
         {
         if (amroot) printf("cfengine: NEW SETUID root PROGRAM %s\n",file);
         PrependItem(&VSETUIDLIST,file);
         }
      }
   else
      {
      switch (action)
         {
         case fixall:
         case fixdirs:
         case fixplain: printf("cfengine: removing setuid (root) flag from %s...\n\n",file);
                        break;
         case warnall:
         case warndirs:
         case warnplain: printf("cfengine: WARNING setuid (root) flag on %s...\n\n",file);
                         break;

         default:       break;
         }
      }
   }

if (stat->st_uid == 0 && (stat->st_mode & S_ISGID))
   {
   if (newperm & S_ISGID)
      {
      if (! IsItemIn(VSETUIDLIST,file))
         {
         if (S_ISDIR(stat->st_mode))
            {
            /* setgid directory */
            }
         else
            {
            if (amroot) printf("cfengine: NEW SETGID root PROGRAM %s\n",file);
            PrependItem(&VSETUIDLIST,file);
            }
         }
      }
   else
      {
      switch (action)
         {
         case fixall:
         case fixdirs:
         case fixplain: printf("cfengine: removing setgid (root) flag from %s...\n\n",file);
                        break;
         case warnall:
         case warndirs:
         case warnplain: printf("cfengine: WARNING setgid (root) flag on %s...\n\n",file);
                         break;

         default:        break;
         }
      }
   }

CheckOwner(file,action,uidlist,gidlist,stat);

if (S_ISLNK(stat->st_mode))             /* No point in checking permission on a link */
   {
   KillOldLink(file);
   return;
   }

if ((newperm & 07777) == (stat->st_mode & 07777) && (action != touch))    /* file okay */
   {
   return;
   }

switch (action)
   {
   case linkchildren:

   case warnplain:
                if (S_ISREG(stat->st_mode))
                   {
                   printf("cfengine: %s has permission %o\n",file,stat->st_mode & 07777);
                   printf("          [should be %o]\n",newperm & 07777);
                   }
                break;
   case warndirs:
                if (S_ISDIR(stat->st_mode))
                   {
                   printf("cfengine: %s has permission %o\n",file,stat->st_mode & 07777);
                   printf("          [should be %o]\n",newperm & 07777);
                   }
                break;
   case warnall:   
                printf("cfengine: %s has permission %o\n",file,stat->st_mode & 07777);
                printf("          [should be %o]\n",newperm & 07777);
                break;

   case fixplain:

                if (S_ISREG(stat->st_mode))
                   {
                   if (! DONTDO)
                      {
                      if (chmod (file,newperm & 07777) == -1)
			 {
                         perror("chmod");
                         }
                      }
                   if (VERBOSE || DEBUG || D2)
                      {
                      printf("cfengine: %s had permission %o\n",file,stat->st_mode & 07777);
                      printf("          [changed to %o]\n",newperm & 07777);
                      }
                   }
                break;

   case fixdirs:
                if (S_ISDIR(stat->st_mode))
                   {
                   if (! DONTDO)
                      {
                      chmod (file,newperm & 07777);
                      }
                   if (VERBOSE || DEBUG || D2)
                      {
                      printf("cfengine: %s had permission %o\n",file,stat->st_mode & 07777);
                      printf("          [changed to %o]\n",newperm & 07777);
                      }
                   }
                break;

   case fixall: if (! DONTDO)
                   {
                   if (chmod (file,newperm & 07777) == -1)
		      {
                      perror("chmod");
                      break;
                      }
                   }
                if (VERBOSE || DEBUG || D2)
                   {
                   printf("cfengine: %s had permission %o\n",file,stat->st_mode & 07777);
                   printf("          [changed to %o]\n",newperm & 07777);
                   }
                break;

   case touch:  if (! DONTDO)
                   {
                   if (chmod (file,newperm & 07777) == -1)
		      {
                      perror("chmod");
                      break;
		      }
                   utime (file,NULL);
                   }
                break;

   default:     FatalError("cfengine: internal error CheckExistingFile(): illegal file action\n");
   }
}

/*********************************************************************/

CheckOwner(file,action,uidlist,gidlist,statbuf)

char *file;
enum fileactions action;
struct UidList *uidlist;
struct GidList *gidlist;
struct stat *statbuf;

{ struct passwd *pw;
  struct group *gp;
  struct UidList *ulp;
  struct GidList *glp;
  short uidmatch = false, gidmatch = false;
  int uid; 
  int gid;

for (ulp = uidlist; ulp != NULL; ulp=ulp->next)
   {
   if (ulp->uid == -1 || statbuf->st_uid == ulp->uid)   /* -1 matches anything */
      {
      uid = ulp->uid;
      uidmatch = true;
      break;
      }
   }

for (glp = gidlist; glp != NULL; glp=glp->next)
   {
   if (glp->gid == -1 || statbuf->st_gid == glp->gid)  /* -1 matches anything */
      {
      gid = glp->gid;
      gidmatch = true;
      break;
      }
   }


if (uidmatch && gidmatch)
   {
   return;
   }
else
   {
   if (! uidmatch)
      {
      uid = uidlist->uid;    /* default is first item in list */
      }

   if (! gidmatch)
      {
      gid = gidlist->gid;
      }

   if (S_ISLNK(statbuf->st_mode) && (action == fixdirs || action == fixplain))
      {
      if (DEBUG || D2)
         {
         printf("File %s incorrect type (link), skipping...\n",file);
         }
      return;
      }

   if ((S_ISREG(statbuf->st_mode) && action == fixdirs) || (S_ISDIR(statbuf->st_mode) && action == fixplain))
      {
      if (DEBUG || D2)
         {
         printf("File %s incorrect type, skipping...\n",file);
         }
      return;
      }

   switch (action)
      {
      case fixplain:
      case fixdirs:
      case fixall: 
                  if (VERBOSE || DEBUG || D2)
                     {
                     if (uid == -1 && gid == -1)
                        {
                        printf("cfengine: touching %s\n",file);
                        }
                     else
                        {
                        if (uid != -1)
                           {
                           printf("cfengine: changing owner of %s to uid %d\n",file,uid);
                           }

                        if (gid != -1)
                           {
                           printf("cfengine: changing owner of %s to gid %d\n",file,gid);
                           }
                        }
                     }

                  if (! DONTDO)
                     {
                     if (chown(file,uid,gid) == -1)
                        {
                        printf("cfengine: Cannot set ownership on file %s!\n",file);
                        perror("chown");
                        }
                     }
                  break;

      case linkchildren:
      case touch:
      case warnall: 
      case warndirs:
      case warnplain:
                  if ((pw = getpwuid(statbuf->st_uid)) == NULL)
                     {
                     printf ("cfengine: File %s is not owned by anybody in the passwd database\n",file);
                     printf ("          (uid = %d,gid = %d)\n",statbuf->st_uid,statbuf->st_gid);
                     break;
                     }

                  if ((gp = getgrgid(statbuf->st_gid)) == NULL)
                     {
                     printf ("cfengine: File %s is not owned by any group in /etc/group\n",file);
                     break;
                     }

                  printf ("cfengine: File %s is owned by [%s], group [%s]\n",file,pw->pw_name,gp->gr_name);
                  break;
      }
   }
}

/*********************************************************************/

RecursiveTidy(name,level)

char *name;
int level;

{ struct stat statbuf;
  DIR *dirh;
  struct dirent *dirp;
  char pcwd[bufsize];
  time_t ticks;

if ((dirh = opendir(name)) == NULL)
   {
   printf("cfengine: RecursiveTidy(): Can't open directory %s\n",name);
   return;
   }

if (level == 2)
   {
   strcpy(VLOGFILE,name);
   strcat(VLOGFILE,"/.cfengine.rm");

   if ((VLOGFP = fopen(VLOGFILE,"w")) == NULL)         /* log deleted files for each user */
      {
      printf("cfengine: Couldn't open a file %s\n",VLOGFILE);
      VLOGFP = stderr;
      }
   else
      {
      ticks = time((time_t *)NULL);
      fprintf(VLOGFP,"This file is generated by cfengine %s\n",VERSION);
      fprintf(VLOGFP,"It contains a log of the files which have been tidied.\n");
      fprintf(VLOGFP,"The time of writing is %s\n",ctime(&ticks));
      fprintf(VLOGFP,"If you have any questions about this, send them to %s.\n",VSYSADM);
      fprintf(VLOGFP,"-(Start transcript)---------------\n");
      }
   }

for (dirp = readdir(dirh); dirp != NULL; dirp = readdir(dirh))
   {
   if (strcmp(".",dirp->d_name) == 0 || strcmp("..",dirp->d_name) == 0)
      {
      continue;
      }

   strcpy(pcwd,name);                                 /* Assemble pathname */
   AddSlash(pcwd);
   strcat(pcwd,dirp->d_name);

   if (strlen(pcwd) > bufsize - buffer_margin)
      {
      printf("cfengine: buffer overflow constructing string in RecursiveTidy\n");
      printf(" culprit: %^s\n",pcwd);
      return;
      }

   if (TRAVLINKS)
      {
      if (stat(pcwd,&statbuf) == -1)
         {
         if (DEBUG || D2 || VERBOSE)
            {
            printf("cfengine: RecursiveTidy(): Can't stat %s\n",pcwd);
            }
         continue;
         }
      }
   else
      {
      if (lstat(pcwd,&statbuf) == -1)
         {
         if (DEBUG || D2 || VERBOSE)
            {
            printf("cfengine: RecursiveTidy(): Can't stat %s\n",pcwd);
            if (readlink(pcwd,VBUFF,bufsize) != -1)
               {
               printf("          File is link to -> %s\n",VBUFF);
               }
            }
         continue;
         }
      }


   if (S_ISDIR(statbuf.st_mode))
      {
      if (IsMountedFileSystem(&statbuf,pcwd))
         {
         continue;
         }
      else
         {
         RecursiveTidy(pcwd,level+1);
         }
      }
   else
      {
      TidyExistingFile(pcwd,dirp->d_name,&statbuf);
      }
   }

if (level == 2)
   {
   fclose(VLOGFP);
   chmod(VLOGFILE,DEFAULTMODE);
   }

closedir(dirh);
}


/*********************************************************************/

TidyExistingFile(path,name,statbuf)

char *path;
char *name;
struct stat *statbuf;


{ struct Tidy *tp;
  time_t nowticks, fileticks;


for (tp = VTIDY; tp != NULL; tp=tp->next)
   {
   if (tp->wild == NULL)
      {
      continue;
      }

   if (WildMatch(tp->wild,name) && CheckHomeSubDir(path,tp->path))
      {
      nowticks = time((time_t *)NULL);             /* cmp time in days */
      fileticks = statbuf->st_atime;

      if (nowticks-fileticks < 0)                  /* shouldn't happen */
         {
         printf("cfengine: ALERT! access time for %s is in the future. Check system clock!\n",path);
         return;
         }

      if (DEBUG || D2)
         {
         printf("[%s] is %d days overdue\n",name, (nowticks-fileticks)/TICKSPERDAY-(tp->age));
         }

      if (tp->age*TICKSPERDAY < (nowticks-fileticks))
         {
         fprintf(VLOGFP,"cf: rm %s\n",path);

         if (! DONTDO)
            {
            if (unlink(path) == -1)
	       {
               perror("unlink");
	       }

            if (VERBOSE || DEBUG || D2) 
               {
               printf("cfengine: Deleting %s\n",path);
               }  
            }
         else
            {
            printf("cfengine: want to remove %s\n",path);
            }
         }
      }
   }
}


/*********************************************************************/

CheckHomeSubDir(testpath,tidypath)

char *testpath, *tidypath;

{ char *sp, *subdirstart, *sp1, *sp2;
  char buffer[bufsize];

if (strncmp(tidypath,"home/",5) == 0)
   {
   strcpy(buffer,testpath);

   for (sp = buffer + strlen(buffer)-1; *sp != '/'; sp--)
      {
      *sp = NULL;
      }

   *sp = NULL;

   for (sp = buffer + strlen(buffer)-1; *sp != '/'; sp--)
      {
      *sp = NULL;
      }

   *sp = NULL;

   if (! IsHomeDir(buffer))
      {
      return false;
      }

   if (DEBUG || D2)
      {
      printf("CheckHomeSubDir(%s,%s)\n",testpath,tidypath);
      }

   strcpy(buffer,testpath);

   subdirstart = tidypath + 4;              /* Ptr to start of subdir */

   for (sp = buffer + strlen(buffer)-1; *sp != '/'; sp--)
      {
      *sp = NULL;
      }

   *sp = NULL;

   sp1 = sp-1;
   sp2 = tidypath + strlen(tidypath) - 1; 

   while (*sp2 != *subdirstart)
      {
      if (*sp2 != *sp1)
         {
         return false;
         }

      sp1--;
      sp2--;
      }

   if (DEBUG || D2)
      {
      printf("CheckHomeSubDir(true)\n");
      }

   return(true);
   }

return true;
}

/*********************************************************************/

RecursiveTidySpecialArea(name,wildcard,age,maxrecurse)

char *name,*wildcard;
short age;
int maxrecurse;

{ struct stat statbuf;
  DIR *dirh;
  struct dirent *dirp;
  char pcwd[bufsize];

if (maxrecurse == -1)
   {
   if (DEBUG || D2)
      {
      printf(">>MAXRECURSE ran out, quitting at %s\n",name);
      }
   return;
   }

if (DEBUG || D2)
   {
   printf("Opening dir %s for wildcards %s >%d\n",name,wildcard,age);
   }

if ((dirh = opendir(name)) == NULL)
   {
   printf("cfengine: RecursiveTidySpecialArea(): Can't open directory [%s]\n",name);
   return;
   }

for (dirp = readdir(dirh); dirp != NULL; dirp = readdir(dirh))
   {
   if (strcmp(".",dirp->d_name) == 0 || strcmp("..",dirp->d_name) == 0)
      {
      continue;
      }

   strcpy(pcwd,name);                                   /* Assemble pathname */
   AddSlash(pcwd);
   strcat(pcwd,dirp->d_name);

   if (strlen(pcwd) > bufsize - buffer_margin)
      {
      printf("cfengine: buffer overflow constructing string in RecursiveTidySpecialArea\n");
      printf(" culprit: %^s\n",pcwd);
      return;
      }


   if (TRAVLINKS)
      {
      if (stat(pcwd,&statbuf) == -1)
         {
         if (DEBUG || D2 || VERBOSE)
            {
            printf("cfengine: RecursiveTidy(): Can't stat %s\n",pcwd);
            }
         continue;
         }
      }
   else
      {
      if (lstat(pcwd,&statbuf) == -1)
         {
         if (DEBUG || D2 || VERBOSE)
            {
            printf("cfengine: RecursiveTidy(): Can't stat %s\n",pcwd);
            if (readlink(pcwd,VBUFF,bufsize) != -1)
               {
               printf("          File is link to -> %s\n",VBUFF);
               }
            }
         continue;
         }
      }

   if (S_ISDIR(statbuf.st_mode))
      {
      if (IsMountedFileSystem(&statbuf,pcwd))
         {
         continue;
         }
      else
         {
         if (*(dirp->d_name) != '.')            /* Don't descend into q.dir here */
            {
            RecursiveTidySpecialArea(pcwd,wildcard,age,maxrecurse-1);
            }
         else
            {
            continue;
            }
         }
      }
   else
      {
      TidyParticularFile(pcwd,dirp->d_name,wildcard,age,&statbuf);
      }
   }

closedir(dirh);
}

/*********************************************************************/

TidyParticularFile(path,name,wildcard,age,statbuf)

char *path, *wildcard, *name;
short age;
struct stat *statbuf;

{ time_t nowticks,fileticks;

if (*wildcard == '*' && *name == '.')   /* Don't match dotted files here */
   {
   return;
   }

if (! WildMatch(wildcard,name))
   {
   return;
   }

if (DEBUG || D2)
   {
   printf("Matched %s to %s\n",path,wildcard);
   }

nowticks = time((time_t *)NULL);             /* cmp time in days */
fileticks = statbuf->st_atime;

if (age*TICKSPERDAY < (nowticks-fileticks))
   {
   if (VERBOSE || DEBUG || D2) 
      {
      printf("cfengine: Deleting %s\n",path);
      }

   if (! DONTDO)
      {
      unlink(path);
      }
   }
}

/*********************************************************************/

LinkChildren(path,rootstat,uid,gid)


/* --------------------------------------------------------------------
   Here we try to break up the path into a part which will match the
   last element of a mounted filesytem mountpoint and the remainder
   after that. We parse the path backwards to get a math e.g.

   /fys/lib/emacs ->  lib /emacs
                      fys /lib/emacs
                          /fys/lib/emacs

   we try to match lib and fys to the binserver list e.g. /mn/anyon/fys
   and hope for the best. If it doesn't match, tough! 
   --------------------------------------------------------------------- */

char *path;
struct stat *rootstat;
uid_t uid;
gid_t gid;

{ char *sp;
  char lastlink[bufsize],server[bufsize],from[bufsize],to[bufsize],relpath[bufsize];
  char odir[bufsize];
  DIR *dirh;
  struct dirent *dirp;
  struct stat statbuf;
  int matched = false;

if (! S_ISDIR(rootstat->st_mode))
   {
   printf("cfengine: File %s is not a directory: it has no children to link!\n",path);
   return;
   }

if (VERBOSE || DEBUG || D2)
   {
   printf("Linking the children of %s\n",path);
   }
 
for (sp = path+strlen(path); sp != path-1; sp--)
   {
   if (*(sp-1) == '/')
      {
      relpath[0] = NULL;
      sscanf(sp,"%[^/]%s", lastlink,relpath);

      if (MatchAFileSystem(server,lastlink))
         {
         strcpy(odir,server);
         strcat(odir,relpath);

         if ((dirh = opendir(odir)) == NULL)
            {
            printf("cfengine: LinkChildren: Can't open directory %s\n",path);
            return;
            }

         for (dirp = readdir(dirh); dirp != NULL; dirp = readdir(dirh))
            {
            if (strcmp(".",dirp->d_name) == 0 || strcmp("..",dirp->d_name) == 0)
               {
               continue;
               } 

            if (strcmp("lost+found",dirp->d_name) == 0)
               {
               continue;
               } 

            strcpy(from,path);

            if (from[strlen(from)-2] != '/')
               {
               strcat(from,"/");
               }

            strcat(from,dirp->d_name);

            strcpy(to,odir);

            if (from[strlen(to)-2] != '/')
               {
               strcat(to,"/");
               }

            strcat(to,dirp->d_name);

            if (DEBUG || D2) printf("LinkChild from = %s to = %s\n",from,to);

            if (stat(to,&statbuf) == -1)
               {
               continue;
               }
            else
               {
               matched = LinkFiles(from,to);

               if (matched && !DONTDO)
		 {
                 chown(from,uid,gid);
                 }
               }
            }

         if (matched) return;
         }
      }
   }

printf("cfengine: Couldn't link the children of %s to anything because no\n",path);
printf("          file system was found to mirror it in the defined binservers list.\n");
}

/*********************************************************************/

MatchAFileSystem(server, lastlink)

char *server, *lastlink;

{ struct Item *mp,*ip;
  char *sp;
  char host[maxvarsize];

for (mp = VMOUNTED; mp != NULL; mp=mp->next)
   {
   sscanf (mp->name,"%[^:]",host);

   if (! IsItemIn(VBINSERVERS,host))
      {
      continue;
      }

   if (strcmp(host,VSYSNAME.nodename) == 0)
      {
      continue;                      /* Can't link machine to itself! */
      }

   for (sp = mp->name+strlen(mp->name); *(sp-1) != '/'; sp--)
      {
      }

   if (IsHomeDir(sp))
      {
      continue;
      }

   if (strcmp(sp,lastlink) == 0)
      {
      strcpy(server,mp->name+strlen(host)+1);
      return(true);
      }
   }

return(false);
}

/*********************************************************************/

IsMountedFileSystem (childstat,dir,rlevel)

char *dir;
struct stat *childstat;
int rlevel;

{ struct stat parentstat;

strcpy(VBUFF,dir);

if (VBUFF[strlen(VBUFF)-1] == '/')
   {
   strcat(VBUFF,"..");
   }
else
   {
   strcat(VBUFF,"/..");
   }

if (stat(VBUFF,&parentstat) == -1)
   {
   if (DEBUG || D2)
      {
      printf("File %s couldn't stat its parent directory! Assuming permission\n",dir);
      printf("is denied because the file system is mounted from another host.\n");
      }
   return(true);
   }

if (rlevel == 0)  /* If this is the root of a search, don't stop before we started ! */
   {
   return false;
   }

if (childstat->st_dev != parentstat.st_dev)
   {
   if (DEBUG || D2)
      {
      printf ("[%s is on a different file system, not descending]\n",dir);
      }
   return (true);
   }


return(false);
}

/*********************************************************************/

IgnoreFile (pathto,name)

char *pathto, *name;

{ struct Item *ip;

strcpy(VBUFF,pathto);

if (VBUFF[strlen(VBUFF)-1] != '/')
   {
   strcat(VBUFF,"/");
   }

strcat(VBUFF,name);

for (ip = VIGNORE; ip != NULL; ip=ip->next)
   {
   if (*(ip->name) == '/')
      {
      if (strcmp(VBUFF,ip->name) == 0)
         {
         return true;
         }
      }
   else
      {
      if (WildMatch(ip->name,name))
         {
         return true;
         }
      }
   }

return false;
}

/*********************************************************************/
/* TOOLKIT : actions                                                 */
/*********************************************************************/

enum actions ActionStringToCode (str)

char *str;

{ char *sp;
  int i;

ACTION = none;

for (sp = str; *sp != NULL; sp++)
   {
   *sp = ToLower(*sp);
   }

for (i = 1; ACTIONID[i] != NULL; i++)
   {
   if (strcmp(ACTIONID[i],str) == 0)
      {
      ACTION = (enum actions) i;
      break;
      }
   }

if (ACTION == none)
  {
  yyerror("Indexed macro specified no action");
  FatalError("Could not compile action");
  }

return (enum actions) i;
}


/*********************************************************************/
/* TOOLKIT : classes                                                 */
/*********************************************************************/

enum classes ClassStringToCode (str)

char *str;

{ char *sp;
  int i;

if (DEBUG || D2)
   {
   printf(">> ClassStringToCode(%s)\n",str);
   }

if (strcmp("any",str) == 0)
   {
   return(VSYSTEMHARDCLASS); /* always true */
   }

CLASS = empty;

for (sp = str; *sp != NULL; sp++)
   {
   *sp = ToLower(*sp);
   }

for (i = 1; CLASSTEXT[i] != NULL; i++)
   {
   if (strcmp(CLASSTEXT[i],str) == 0)
      {
      CLASS = (enum classes) i;
      break;
      }
   }

return (enum classes) i;
}

/*********************************************************************/

IsHardClass(sp)  /* true if string matches a hardwired class e.g. hpux */

char *sp;

{ int i;

if (DEBUG || D2 || D1)
   {
   printf("IsHardClass(%s)",sp);
   }

for (i = 2; CLASSTEXT[i] != NULL; i++)
   {
   if (strcmp(CLASSTEXT[i],sp) == 0)
      {
      CLASS = (enum classes) i;

      if (DEBUG || D2 || D1)
         {
         printf("CLASS = %s\n",CLASSTEXT[i]);
         }

      return(true);
      }
   }

for (i = 0; i < 7; i++)
   {
   if (strcmp(DAYTEXT[i],sp)==0)
      {
      if (DEBUG || D1) printf ("-true\n");
      return(true);
      }
   }

if (DEBUG || D1) printf ("-false\n");
return(false);
}

/*********************************************************************/

AddDayClass(str)

char *str;

{ int i;

for (i = 0; i < 7; i++)
   {
   if (strncmp(DAYTEXT[i],str,3)==0)
      {
      AddClassToHeap(DAYTEXT[i]);
      }
   }
}

/*******************************************************************/

AddClassToHeap(class)

char *class;

{ struct Item *ptr;
  char *sp;

if (IsItemIn(VHEAP,class))
   {
   return;
   }

if (DEBUG || D2 || D1) printf(">>Pushing class %s on heap\n",class);

AppendItem(&VHEAP,class,NULL);
}

/*********************************************************************/

IsDefinedClass(class)  /* class.class.class etc. Logical AND of members */

char *class;

{ struct Item *ptr;
  char *sp = class;
  char cbuff[maxvarsize];
  int count = 1;          /* no. of members in class */

if (DEBUG || D1)
   {
   printf("IsDefinedClass(%s)={",class);
   }

while (*sp != '\0')
   {
   if (*sp++ == '.')
      {
      count++;
      }
   }

sp = class;

while(*sp != NULL)
   {
   sscanf(sp,"%[^.]",cbuff);

   while ((*sp != NULL) && (*sp !='.'))
      {
      sp++;
      }

   if (*sp == '.')
      {
      sp++;
      }

   for (ptr = VHEAP; ptr != NULL; ptr=ptr->next)
      {
      if (strcmp(cbuff,ptr->name) == NULL)
         {
         if (DEBUG || D1)
            {
            printf("[%s]",cbuff);
            }
         count--;
         }
      }

   for (ptr = VNEGHEAP; ptr != NULL; ptr=ptr->next)
      {
      if (strcmp(cbuff,ptr->name) == NULL)
         {
         if (DEBUG || D1)
            {
            printf("[%s]",cbuff);
            }
         return(false);
         }
      }
   }

if (DEBUG || D1)
   {
   printf("}\n");
   }

if (count == 0)
   {
   return(true);
   }
else
   {
   return(false);
   }
}

/*********************************************************************/

IsInstallable(class)  /* class.class.class etc. Logical AND of members */

char *class;

{ struct Item *ptr;
  char *sp = class;
  char cbuff[maxvarsize];
  int count = 1;          /* no. of members in class */

if (DEBUG || D1)
   {
   printf("IsDefinedClass(%s)={",class);
   }

while (*sp != '\0')
   {
   if (*sp++ == '.')
      {
      count++;
      }
   }

sp = class;

while(*sp != NULL)
   {
   sscanf(sp,"%[^.]",cbuff);

   while ((*sp != NULL) && (*sp !='.'))
      {
      sp++;
      }

   if (*sp == '.')
      {
      sp++;
      }

   for (ptr = VHEAP; ptr != NULL; ptr=ptr->next)
      {
      if (strcmp(cbuff,ptr->name) == NULL)
         {
         if (DEBUG || D1)
            {
            printf("[%s]",cbuff);
            }
         count--;
         }
      }

   for (ptr = VALLADDCLASSES; ptr != NULL; ptr=ptr->next)
      {
      if (strcmp(cbuff,ptr->name) == NULL)
         {
         if (DEBUG || D1)
            {
            printf("[%s]",cbuff);
            }
         count--;
         }
      }


   for (ptr = VNEGHEAP; ptr != NULL; ptr=ptr->next)
      {
      if (strcmp(cbuff,ptr->name) == NULL)
         {
         if (DEBUG || D1)
            {
            printf("[%s]",cbuff);
            }
         return(false);
         }
      }
   }

if (DEBUG || D1)
   {
   printf("}\n");
   }

if (count == 0)
   {
   return(true);
   }
else
   {
   return(false);
   }
}


/*********************************************************************/

AddCompoundClass(class)

char *class;

{ char *sp = class;
  char cbuff[maxvarsize];

if (DEBUG || D1)
   {
   printf("AddCompoundClass(%s)",class);
   }

while(*sp != NULL)
   {
   sscanf(sp,"%[^.]",cbuff);

   while ((*sp != NULL) && (*sp !='.'))
      {
      sp++;
      }

   if (*sp == '.')
      {
      sp++;
      }

   if (IsHardClass(cbuff))
      {
      FatalError("cfengine: You cannot use -D to define a reserved class!");
      }

   AddClassToHeap(cbuff);
   }
}

/*********************************************************************/

NegateCompoundClass(class,heap)

char *class;
struct Item **heap;

{ char *sp = class;
  char cbuff[maxvarsize];

if (DEBUG || D1)
   {
   printf("AddCompoundClass(%s)",class);
   }

while(*sp != NULL)
   {
   sscanf(sp,"%[^.]",cbuff);

   while ((*sp != NULL) && (*sp !='.'))
      {
      sp++;
      }

   if (*sp == '.')
      {
      sp++;
      }

   if (IsHardClass(cbuff))
      { char err[bufsize];
      yyerror("Illegal exception exception");
      sprintf (err,"Cannot negate the reserved class [%s]\n",cbuff);
      FatalError(err);
      }

   AppendItem(heap,cbuff,NULL);
   }
}

/*********************************************************************/
/* TOOLKIT : Error                                                   */
/*********************************************************************/

FatalError(s)

char *s;

{
fprintf (stderr,"cfengine:%s:%s\n",VCURRENTFILE,s);
Unlock();
exit (1);
}

/*********************************************************************/

Warning(s)

{
if (WARNINGS)
   { 
   fprintf (stderr,"cfengine:%s:%d: Warning: %s\n",VCURRENTFILE,LINENUMBER,s);
   }
}


/*********************************************************************/
/* TOOLKIT : String                                                  */
/*********************************************************************/

char ToLower (ch)

char ch;

{
if (isdigit(ch) || ispunct(ch))
   {
   return(ch);
   }

if (islower(ch))
   {
   return(ch);
   }
else
   {
   return(ch - 'A' + 'a');
   }
}


/*************************************************************************/
/* WILDCARD TOOLKIT : Level 0                                            */
/*************************************************************************/

#define nomatch         0
#define match           1
#define maxlen          20
#define startofstrings  10
#define middleofstrings 11
#define endofstrings    12

#define Wild(c)  (c == '*' || c == '?') ? true : false

#include <string.h>

/*********************************************************************/
 
ExpandWildCardsAndDo(wildpath,buffer,function,argptr)
 
char *wildpath, *buffer;
int (*function)();
void *argptr;

 /* This function recursively expands a path containing wildcards */
 /* and executes the function pointed to by function for each     */
 /* matching file or directory                                    */

 
{ char *rest, extract[bufsize], construct[bufsize],varstring[bufsize], *work;
  struct stat statbuf;
  DIR *dirh;
  struct dirent *dp;
  int count, isdir = false;

varstring[0] = NULL;

ExpandVarstring(wildpath,varstring,NULL);
work = varstring;

if (DEBUG || D2)
   {
   printf("ExpandWildCardsAndDo(%s=%s)\n",wildpath,work);
   }
 
extract[0] = NULL;

if (*work == '/')
   {
   work++;
   isdir = true;
   }

sscanf(work,"%[^/]",extract);
rest = work + strlen(extract);
 
if (strlen(extract) == 0)
   {
   if (isdir)
      {
      strcat(buffer,"/");
      }
   (*function)(buffer,argptr);
   return;
   }
 
if (! IsWildCard(extract))
   {
   strcat(buffer,"/");
   strcat(buffer,extract);
   CheckBuffSpace(buffer,strlen(extract));
   ExpandWildCardsAndDo(rest,buffer,function,argptr);
   return;
   }
else
   { 
   strcat(buffer,"/");
   if ((dirh=opendir(buffer)) == NULL)
      {
      printf("Can't open dir: %s\n",buffer);
      perror("opendir");
      return;
      }

   count = 0;
   strcpy(construct,buffer);                 /* save relative path */
 
   for (dp = readdir(dirh); dp != 0; dp = readdir(dirh))
      {
      if ((strcmp(dp->d_name,".") == 0) || (strcmp(dp->d_name,"..")) == 0)
         {
         continue;
         }

      count++;
      strcpy(buffer,construct);
      strcat(buffer,dp->d_name);

      if (stat(buffer,&statbuf) == -1)
         {
         printf("cfengine: Can't stat %s\n\n",buffer);
         perror("stat");
         continue;
         }
 
      if (S_ISDIR(statbuf.st_mode) && WildMatch(extract,dp->d_name))
         {
         ExpandWildCardsAndDo(rest,buffer,function,argptr);
         }
 
      }
 
   if (count == 0)
      {
      if (VERBOSE || DEBUG || D2)
         {
         printf("cfengine: No directories matching %s in %s\n",extract,buffer);
         }
      return;
      }
   closedir(dirh);
   }
}
 


/*************************************************************************/

IsWildCard (str)

char *str;

{
return (strchr(str,'?') || strchr(str,'*'));
} 

/*************************************************************************/


   WildMatch (wildptr,cmpptr)

   char *wildptr, *cmpptr;

   { char buffer[maxlen];
     char *AfterSubString();
     int i, status = startofstrings;
     char lastwild = '\0';

   if (strstr(wildptr,"*") == NULL && strstr(wildptr,"?") == NULL)
      {
      return (! strcmp(wildptr,cmpptr));
      }

  while (true)
      {
      while (*wildptr == '?')                                /* 1 */
         {
         wildptr++;
         cmpptr++;
         if ((*cmpptr == '\0') && (*wildptr != '\0'))        /* 2 */
            {
            return(nomatch);
            }
         lastwild = '?';
         status = middleofstrings;
         }

      if (*wildptr == '\0' && *cmpptr == '\0')                /* 3 */
         {
         return(match);
         }
      else if (*wildptr == '\0')                              /* 4 */
         {
         return(nomatch);
         }

      if (*wildptr == '*')                                    /* 5 */
         {
         while (*wildptr == '*')                              /* 6 */
            {
            wildptr++;
            }
         if (*wildptr == '\0')                                /* 7 */
            {
            if (*cmpptr == '\0')                              /* 8 */
               {
               return(nomatch);
               }
            else
               {
               return(match);
               }
            }

         cmpptr++;                                            /* 9 */
         status = middleofstrings;
         lastwild = '*';
         }

      for (i = 0; !(Wild(*wildptr) || *wildptr == '\0'); i++) /* 10 */
         {
         buffer[i] = *wildptr++;
         if (*wildptr == '\0')                                /* 11 */
            {
            status = endofstrings;
            }
         }

      buffer[i] = '\0';

      if ((cmpptr = AfterSubString(cmpptr,buffer,status,lastwild)) == NULL)
         {
         return(nomatch);                                      /* 12 */
         }

      status = middleofstrings;
      }
   }

/******************************************************************/
/* Wildcard Toolkit : Level 1                                     */
/******************************************************************/

  char *AfterSubString(big,small,status,lastwild)

   /* If the last wildcard was a ? then this just tries to      */
   /* match the substrings from the present position, otherwise */
   /* looks for next occurrance of small within big and returns */
   /* a pointer to the next character in big after small or     */
   /* NULL if there is no string found to match                 */
   /* If end of strings is signalled, make sure that the string */
   /* is tied to the end of big. This makes sure that there is  */
   /* correct alignment with end of string marker.              */

   char *big,*small;
   int status;
   char lastwild;

   { char *bigptr;

   if (strlen(small) > strlen(big))                       /* 13 */
      {
      return(NULL);
      }

   if (lastwild == '?')                                   /* 14 */
      {
      if (strncmp(big,small,strlen(small)) == 0)
         {
         return(big+strlen(small));
         }
      else
         {
         return(NULL);
         }
      }

   if (status == endofstrings)                             /* 15 */
      {
      big = big + strlen(big) - strlen(small);
      }

   for (bigptr = big; *bigptr != '\0'; ++bigptr)           /* 16 */
      {
      if (strncmp(bigptr,small,strlen(small)) == 0)
         {
         return(bigptr+strlen(small));
         }

      if (status == startofstrings)                        /* 17 */
         {
         return(NULL);
         }
      }

   return(NULL);                                           /* 18 */
   }

/******************************************************************

Comments :

1. ? Matches a single character. Skip over with pointers.
2. Check that the compare string hasn't run out too soon.
3. If the wild-match is complete return MATCH!
4. other wise check that the wild card string hasn't run out
   too soon.

5. * Matches any string.
6. Skip over any *'s in the wildcard string and leave pointer
   on the first character after.
7. If the last character in the wildstring is *, match any
   remaining characters...
8. ...except no string at all. e.g. xyz* doesn't match xyz
9. Advancing this pointer by one, prevents * from matching
   the null string even when the pointers haven't reached
   the end of the string (so that the match is not complete).
   This works because it screws up the pattern match in
   the routine `AfterSubString()' by removing the first
   character from a string which might otherwise match.
   e.g. it stops *.c from matching .c because the strings
   which get passed to AfterSubString are `.c' (from *.c)
   and `c' (from .c).
10.Isolate the next string sandwhiched between wild-cards
   or string delimiters. Copy to buffer.
11.Make a note if we hit the end of wildstring in the process.
   i.e. is this the end of the strings we're matching?
12.The call to this routine checks whether a match is
   allowed between the isolated string and the comparison
   string. If a match is allowed it updates the pointers
   so that they point to the next characters after the
   match. Otherwise it returns NULL.

AfterSubString()

13. You can't find a string larger than big inside big!
    (This could occur because of points 15. and 9.)
14. If the last wildcard was just to match a single
    character then the string match must be anchored to
    the current pointer locations.
15. If this is the end of the strings to be matched, then
    make sure the match also correctly identifies which
    pattern matches the END of the string. e.g.
    to avoid confusion over *.abc in .abc.abc.abc
16. If the last wildcard was a * or none, then
    use a forward floating comparision which can
    skip over anyjunk characters looking for the next
    occurrance of the string.
17. If this is the first two characters in the string
    and there is no match, give up. This anchors the
    floating match to the start to avoid
    a*b matching xaxb by skipping over the first x!
18. If there is no match yet, there's no chance!

Test Data
----------

Wild String        Valid Match           No Match
------------       -----------          ----------

abcd                abcd

*mmm*               abcmmmabc            abcmmm
                                         mmm

*j                  jjjj                 jjjjx

*.c                  a.c                  .c
                     .c.c
*y                   ayyyyyyy             ya
                                           y

?a                    aa                   aaa
                      ba                   ab

???a                  aaaa                 ab
                      xyza                 xyzaa
???                   xyz                  ab
                                           abcd
*                     anything
*.*                   anything.anything
a*                    abc                   a
                                            bad
?mmm?                 ammmb                 abcmmmd
                                            mmm
a*b                   axyzb                 xyz
                      abbb                  axab

*******************************************************************/


/* EOF */
