/* libxenstat: statistics-collection library for Xen
 * Copyright (C) International Business Machines Corp., 2005
 * Authors: Josh Triplett <josht@us.ibm.com>
 *          Judy Fischbach <jfisch@us.ibm.com>
 *          David Hendricks <dhendrix@us.ibm.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 */

#include <limits.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <dirent.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <xs.h>
#include "xenstat.h"

#include "xenctrl.h"

/*
 * Types
 */
#define SHORT_ASC_LEN 5                 /* length of 65535 */
#define VERSION_SIZE (2 * SHORT_ASC_LEN + 1 + sizeof(xen_extraversion_t) + 1)

struct xenstat_handle {
	int xc_handle;
	struct xs_handle *xshandle; /* xenstore handle */
	int page_size;
	FILE *procnetdev;
	DIR *sysfsvbd;
	char xen_version[VERSION_SIZE]; /* xen version running on this node */
};

struct xenstat_node {
	xenstat_handle *handle;
	unsigned int flags;
	unsigned long long cpu_hz;
	unsigned int num_cpus;
	unsigned long long tot_mem;
	unsigned long long free_mem;
	unsigned int num_domains;
	xenstat_domain *domains;	/* Array of length num_domains */
};

struct xenstat_domain {
	unsigned int id;
	char *name;
	unsigned int state;
	unsigned long long cpu_ns;
	unsigned int num_vcpus;		/* No. vcpus configured for domain */
	xenstat_vcpu *vcpus;		/* Array of length num_vcpus */
	unsigned long long cur_mem;	/* Current memory reservation */
	unsigned long long max_mem;	/* Total memory allowed */
	unsigned int ssid;
	unsigned int num_networks;
	xenstat_network *networks;	/* Array of length num_networks */
	unsigned int num_vbds;
	xenstat_vbd *vbds;
};

struct xenstat_vcpu {
	unsigned int online;
	unsigned long long ns;
};

struct xenstat_network {
	unsigned int id;
	/* Received */
	unsigned long long rbytes;
	unsigned long long rpackets;
	unsigned long long rerrs;
	unsigned long long rdrop;
	/* Transmitted */
	unsigned long long tbytes;
	unsigned long long tpackets;
	unsigned long long terrs;
	unsigned long long tdrop;
};

struct xenstat_vbd {
       unsigned int dev;
       unsigned long long oo_reqs;
       unsigned long long rd_reqs;
       unsigned long long wr_reqs;
};
#define SYSFS_VBD_PATH "/sys/devices/xen-backend/"


/*
 * Data-collection types
 */
/* Called to collect the information for the node and all the domains on
 * it. When called, the domain information has already been collected. 
 * Return status is 0 if fatal error occurs, 1 for success. Collectors
 * may prune a domain from the list if it has been deleted between the
 * time the list was setup and the time the colector is called */
typedef int (*xenstat_collect_func)(xenstat_node * node);
/* Called to free the information collected by the collect function.  The free
 * function will only be called on a xenstat_node if that node includes
 * information collected by the corresponding collector. */
typedef void (*xenstat_free_func)(xenstat_node * node);
/* Called to free any information stored in the handle.  Note the lack of a
 * matching init function; the collect functions should initialize on first
 * use.  Also, the uninit function must handle the case that the collector has
 * never been initialized. */
typedef void (*xenstat_uninit_func)(xenstat_handle * handle);
typedef struct xenstat_collector {
	unsigned int flag;
	xenstat_collect_func collect;
	xenstat_free_func free;
	xenstat_uninit_func uninit;
} xenstat_collector;

static int  xenstat_collect_vcpus(xenstat_node * node);
static int  xenstat_collect_networks(xenstat_node * node);
static int  xenstat_collect_xen_version(xenstat_node * node);
static int  xenstat_collect_vbds(xenstat_node * node);
static void xenstat_free_vcpus(xenstat_node * node);
static void xenstat_free_networks(xenstat_node * node);
static void xenstat_free_xen_version(xenstat_node * node);
static void xenstat_free_vbds(xenstat_node * node);
static void xenstat_uninit_vcpus(xenstat_handle * handle);
static void xenstat_uninit_networks(xenstat_handle * handle);
static void xenstat_uninit_xen_version(xenstat_handle * handle);
static void xenstat_uninit_vbds(xenstat_handle * handle);
static char *xenstat_get_domain_name(xenstat_handle * handle, unsigned int domain_id);
static void xenstat_prune_domain(xenstat_node *node, unsigned int entry);

static xenstat_collector collectors[] = {
	{ XENSTAT_VCPU, xenstat_collect_vcpus,
	  xenstat_free_vcpus, xenstat_uninit_vcpus },
	{ XENSTAT_NETWORK, xenstat_collect_networks,
	  xenstat_free_networks, xenstat_uninit_networks },
	{ XENSTAT_XEN_VERSION, xenstat_collect_xen_version,
	  xenstat_free_xen_version, xenstat_uninit_xen_version },
	{ XENSTAT_VBD, xenstat_collect_vbds,
	  xenstat_free_vbds, xenstat_uninit_vbds }
};

#define NUM_COLLECTORS (sizeof(collectors)/sizeof(xenstat_collector))

/*
 * libxenstat API
 */
xenstat_handle *xenstat_init(void)
{
	xenstat_handle *handle;

	handle = (xenstat_handle *) calloc(1, sizeof(xenstat_handle));
	if (handle == NULL)
		return NULL;

#if defined(PAGESIZE)
	handle->page_size = PAGESIZE;
#elif defined(PAGE_SIZE)
	handle->page_size = PAGE_SIZE;
#else
	handle->page_size = sysconf(_SC_PAGE_SIZE);
	if (handle->page_size < 0) {
		perror("Failed to retrieve page size.");
		free(handle);
		return NULL;
	}
#endif

	handle->xc_handle = xc_interface_open();
	if (handle->xc_handle == -1) {
		perror("xc_interface_open");
		free(handle);
		return NULL;
	}

	handle->xshandle = xs_daemon_open_readonly(); /* open handle to xenstore*/
	if (handle->xshandle == NULL) {
		perror("unable to open xenstore\n");
		xc_interface_close(handle->xc_handle);
		free(handle);
		return NULL;
	}

	return handle;
}

void xenstat_uninit(xenstat_handle * handle)
{
	unsigned int i;
	if (handle) {
		for (i = 0; i < NUM_COLLECTORS; i++)
			collectors[i].uninit(handle);
		xc_interface_close(handle->xc_handle);
		xs_daemon_close(handle->xshandle);
		free(handle);
	}
}

xenstat_node *xenstat_get_node(xenstat_handle * handle, unsigned int flags)
{
#define DOMAIN_CHUNK_SIZE 256
	xenstat_node *node;
	xc_physinfo_t physinfo;
	xc_domaininfo_t domaininfo[DOMAIN_CHUNK_SIZE];
	unsigned int new_domains;
	unsigned int i;

	/* Create the node */
	node = (xenstat_node *) calloc(1, sizeof(xenstat_node));
	if (node == NULL)
		return NULL;

	/* Store the handle in the node for later access */
	node->handle = handle;

	/* Get information about the physical system */
	if (xc_physinfo(handle->xc_handle, &physinfo) < 0) {
		free(node);
		return NULL;
	}

	node->cpu_hz = ((unsigned long long)physinfo.cpu_khz) * 1000ULL;
	node->num_cpus =
	    (physinfo.threads_per_core * physinfo.cores_per_socket *
	     physinfo.sockets_per_node * physinfo.nr_nodes);
	node->tot_mem = ((unsigned long long)physinfo.total_pages)
	    * handle->page_size;
	node->free_mem = ((unsigned long long)physinfo.free_pages)
	    * handle->page_size;

	/* malloc(0) is not portable, so allocate a single domain.  This will
	 * be resized below. */
	node->domains = malloc(sizeof(xenstat_domain));
	if (node->domains == NULL) {
		free(node);
		return NULL;
	}

	node->num_domains = 0;
	do {
		xenstat_domain *domain, *tmp;

		new_domains = xc_domain_getinfolist(handle->xc_handle,
						    node->num_domains, 
						    DOMAIN_CHUNK_SIZE, 
						    domaininfo);

		tmp = realloc(node->domains,
			      (node->num_domains + new_domains)
			      * sizeof(xenstat_domain));
		if (tmp == NULL) {
			free(node->domains);
			free(node);
			return NULL;
		}
		node->domains = tmp;

		domain = node->domains + node->num_domains;

		/* zero out newly allocated memory in case error occurs below */
		memset(domain, 0, new_domains * sizeof(xenstat_domain));

		for (i = 0; i < new_domains; i++) {
			/* Fill in domain using domaininfo[i] */
			domain->id = domaininfo[i].domain;
			domain->name = xenstat_get_domain_name(handle, 
							       domain->id);
			if (domain->name == NULL) {
				if (errno == ENOMEM) {
					/* fatal error */
					xenstat_free_node(node);
					return NULL;
				}
				else {
					/* failed to get name -- this means the
					   domain is being destroyed so simply
					   ignore this entry */
					continue;
				}
			}
			domain->state = domaininfo[i].flags;
			domain->cpu_ns = domaininfo[i].cpu_time;
			domain->num_vcpus = (domaininfo[i].max_vcpu_id+1);
			domain->vcpus = NULL;
			domain->cur_mem =
			    ((unsigned long long)domaininfo[i].tot_pages)
			    * handle->page_size;
			domain->max_mem =
			    domaininfo[i].max_pages == UINT_MAX
			    ? (unsigned long long)-1
			    : (unsigned long long)(domaininfo[i].max_pages
						   * handle->page_size);
			domain->ssid = domaininfo[i].ssidref;
			domain->num_networks = 0;
			domain->networks = NULL;
			domain->num_vbds = 0;
			domain->vbds = NULL;

			domain++;
			node->num_domains++;
		}
	} while (new_domains == DOMAIN_CHUNK_SIZE);

	/* Run all the extra data collectors requested */
	node->flags = 0;
	for (i = 0; i < NUM_COLLECTORS; i++) {
		if ((flags & collectors[i].flag) == collectors[i].flag) {
			node->flags |= collectors[i].flag;
			if(collectors[i].collect(node) == 0) {
				xenstat_free_node(node);
				return NULL;
			}
		}
	}

	return node;
}

void xenstat_free_node(xenstat_node * node)
{
	int i;

	if (node) {
		if (node->domains) {
			for (i = 0; i < node->num_domains; i++)
				free(node->domains[i].name);

			for (i = 0; i < NUM_COLLECTORS; i++)
				if((node->flags & collectors[i].flag)
				   == collectors[i].flag)
					collectors[i].free(node);
			free(node->domains);
		}
		free(node);
	}
}

xenstat_domain *xenstat_node_domain(xenstat_node * node, unsigned int domid)
{
	unsigned int i;

	/* FIXME: binary search */
	/* Find the appropriate domain entry in the node struct. */
	for (i = 0; i < node->num_domains; i++) {
		if (node->domains[i].id == domid)
			return &(node->domains[i]);
	}
	return NULL;
}

xenstat_domain *xenstat_node_domain_by_index(xenstat_node * node,
					     unsigned int index)
{
	if (0 <= index && index < node->num_domains)
		return &(node->domains[index]);
	return NULL;
}

const char *xenstat_node_xen_version(xenstat_node * node)
{
	return node->handle->xen_version;
}

unsigned long long xenstat_node_tot_mem(xenstat_node * node)
{
	return node->tot_mem;
}

unsigned long long xenstat_node_free_mem(xenstat_node * node)
{
	return node->free_mem;
}

unsigned int xenstat_node_num_domains(xenstat_node * node)
{
	return node->num_domains;
}

unsigned int xenstat_node_num_cpus(xenstat_node * node)
{
	return node->num_cpus;
}

/* Get information about the CPU speed */
unsigned long long xenstat_node_cpu_hz(xenstat_node * node)
{
	return node->cpu_hz;
}

/* Get the domain ID for this domain */
unsigned xenstat_domain_id(xenstat_domain * domain)
{
	return domain->id;
}

/* Get the domain name for the domain */
char *xenstat_domain_name(xenstat_domain * domain)
{
	return domain->name;
}

/* Get information about how much CPU time has been used */
unsigned long long xenstat_domain_cpu_ns(xenstat_domain * domain)
{
	return domain->cpu_ns;
}

/* Find the number of VCPUs for a domain */
unsigned int xenstat_domain_num_vcpus(xenstat_domain * domain)
{
	return domain->num_vcpus;
}

xenstat_vcpu *xenstat_domain_vcpu(xenstat_domain * domain, unsigned int vcpu)
{
	if (0 <= vcpu && vcpu < domain->num_vcpus)
		return &(domain->vcpus[vcpu]);
	return NULL;
}

/* Find the current memory reservation for this domain */
unsigned long long xenstat_domain_cur_mem(xenstat_domain * domain)
{
	return domain->cur_mem;
}

/* Find the maximum memory reservation for this domain */
unsigned long long xenstat_domain_max_mem(xenstat_domain * domain)
{
	return domain->max_mem;
}

/* Find the domain's SSID */
unsigned int xenstat_domain_ssid(xenstat_domain * domain)
{
	return domain->ssid;
}

/* Get domain states */
unsigned int xenstat_domain_dying(xenstat_domain * domain)
{
	return (domain->state & DOMFLAGS_DYING) == DOMFLAGS_DYING;
}

unsigned int xenstat_domain_crashed(xenstat_domain * domain)
{
	return ((domain->state & DOMFLAGS_SHUTDOWN) == DOMFLAGS_SHUTDOWN)
	    && (((domain->state >> DOMFLAGS_SHUTDOWNSHIFT)
		 & DOMFLAGS_SHUTDOWNMASK) == SHUTDOWN_crash);
}

unsigned int xenstat_domain_shutdown(xenstat_domain * domain)
{
	return ((domain->state & DOMFLAGS_SHUTDOWN) == DOMFLAGS_SHUTDOWN)
	    && (((domain->state >> DOMFLAGS_SHUTDOWNSHIFT)
		 & DOMFLAGS_SHUTDOWNMASK) != SHUTDOWN_crash);
}

unsigned int xenstat_domain_paused(xenstat_domain * domain)
{
	return (domain->state & DOMFLAGS_PAUSED) == DOMFLAGS_PAUSED;
}

unsigned int xenstat_domain_blocked(xenstat_domain * domain)
{
	return (domain->state & DOMFLAGS_BLOCKED) == DOMFLAGS_BLOCKED;
}

unsigned int xenstat_domain_running(xenstat_domain * domain)
{
	return (domain->state & DOMFLAGS_RUNNING) == DOMFLAGS_RUNNING;
}

/* Get the number of networks for a given domain */
unsigned int xenstat_domain_num_networks(xenstat_domain * domain)
{
	return domain->num_networks;
}

/* Get the network handle to obtain network stats */
xenstat_network *xenstat_domain_network(xenstat_domain * domain,
					unsigned int network)
{
	if (domain->networks && 0 <= network && network < domain->num_networks)
		return &(domain->networks[network]);
	return NULL;
}

/* Get the number of VBDs for a given domain */
unsigned int xenstat_domain_num_vbds(xenstat_domain * domain)
{
	return domain->num_vbds;
}

/* Get the VBD handle to obtain VBD stats */
xenstat_vbd *xenstat_domain_vbd(xenstat_domain * domain,
				unsigned int vbd)
{
	if (domain->vbds && 0 <= vbd && vbd < domain->num_vbds)
		return &(domain->vbds[vbd]);
	return NULL;
}

/*
 * VCPU functions
 */
/* Collect information about VCPUs */
static int xenstat_collect_vcpus(xenstat_node * node)
{
	unsigned int i, vcpu, inc_index;

	/* Fill in VCPU information */
	for (i = 0; i < node->num_domains; i+=inc_index) {
		inc_index = 1; /* default is to increment to next domain */

		node->domains[i].vcpus = malloc(node->domains[i].num_vcpus
						* sizeof(xenstat_vcpu));
		if (node->domains[i].vcpus == NULL)
			return 0;
	
		for (vcpu = 0; vcpu < node->domains[i].num_vcpus; vcpu++) {
			/* FIXME: need to be using a more efficient mechanism*/
			xc_vcpuinfo_t info;

			if (xc_vcpu_getinfo(node->handle->xc_handle,
					    node->domains[i].id, vcpu, &info) != 0) {
				if (errno == ENOMEM) {
					/* fatal error */ 
					return 0;
				}
				else {
					/* domain is in transition - remove
					   from list */
					xenstat_prune_domain(node, i);

					/* remember not to increment index! */
					inc_index = 0;
					break;
				}
			}
			else {
				node->domains[i].vcpus[vcpu].online = info.online;
				node->domains[i].vcpus[vcpu].ns = info.cpu_time;
			}
		}
	}
	return 1;
}

/* Free VCPU information */
static void xenstat_free_vcpus(xenstat_node * node)
{
	unsigned int i;
	for (i = 0; i < node->num_domains; i++)
		free(node->domains[i].vcpus);
}

/* Free VCPU information in handle - nothing to do */
static void xenstat_uninit_vcpus(xenstat_handle * handle)
{
}

/* Get VCPU online status */
unsigned int xenstat_vcpu_online(xenstat_vcpu * vcpu)
{
	return vcpu->online;
}

/* Get VCPU usage */
unsigned long long xenstat_vcpu_ns(xenstat_vcpu * vcpu)
{
	return vcpu->ns;
}

/*
 * Network functions
 */

/* Expected format of /proc/net/dev */
static const char PROCNETDEV_HEADER[] =
    "Inter-|   Receive                                                |"
    "  Transmit\n"
    " face |bytes    packets errs drop fifo frame compressed multicast|"
    "bytes    packets errs drop fifo colls carrier compressed\n";

/* Collect information about networks */
static int xenstat_collect_networks(xenstat_node * node)
{
	/* Open and validate /proc/net/dev if we haven't already */
	if (node->handle->procnetdev == NULL) {
		char header[sizeof(PROCNETDEV_HEADER)];
		node->handle->procnetdev = fopen("/proc/net/dev", "r");
		if (node->handle->procnetdev == NULL) {
			perror("Error opening /proc/net/dev");
			return 0;
		}

		/* Validate the format of /proc/net/dev */
		if (fread(header, sizeof(PROCNETDEV_HEADER) - 1, 1,
			  node->handle->procnetdev) != 1) {
			perror("Error reading /proc/net/dev header");
			return 0;
		}
		header[sizeof(PROCNETDEV_HEADER) - 1] = '\0';
		if (strcmp(header, PROCNETDEV_HEADER) != 0) {
			fprintf(stderr,
				"Unexpected /proc/net/dev format\n");
			return 0;
		}
	}

	/* Fill in networks */
	/* FIXME: optimize this */
	fseek(node->handle->procnetdev, sizeof(PROCNETDEV_HEADER) - 1,
	      SEEK_SET);
	while (1) {
		xenstat_domain *domain;
		xenstat_network net;
		unsigned int domid;
		int ret = fscanf(node->handle->procnetdev,
				 "vif%u.%u:%llu%llu%llu%llu%*u%*u%*u%*u"
				 "%llu%llu%llu%llu%*u%*u%*u%*u\n",
				 &domid, &net.id,
				 &net.tbytes, &net.tpackets, &net.terrs,
				 &net.tdrop,
				 &net.rbytes, &net.rpackets, &net.rerrs,
				 &net.rdrop);
		if (ret == EOF)
			break;
		if (ret != 10) {
			unsigned int c;
			do {
				c = fgetc(node->handle->procnetdev);
			} while (c != '\n' && c != EOF);
			if (c == EOF)
				break;
			continue;
		}

		/* FIXME: this does a search for the domid */
		domain = xenstat_node_domain(node, domid);
		if (domain == NULL) {
			fprintf(stderr,
				"Found interface vif%u.%u but domain %u"
				" does not exist.\n", domid, net.id,
				domid);
			continue;
		}
		if (domain->networks == NULL) {
			domain->num_networks = 1;
			domain->networks = malloc(sizeof(xenstat_network));
		} else {
			struct xenstat_network *tmp;
			domain->num_networks++;
			tmp = realloc(domain->networks,
				      domain->num_networks *
				      sizeof(xenstat_network));
			if (tmp == NULL)
				free(domain->networks);
			domain->networks = tmp;
		}
		if (domain->networks == NULL)
			return 0;
		domain->networks[domain->num_networks - 1] = net;
	}

	return 1;
}

/* Free network information */
static void xenstat_free_networks(xenstat_node * node)
{
	unsigned int i;
	for (i = 0; i < node->num_domains; i++)
		free(node->domains[i].networks);
}

/* Free network information in handle */
static void xenstat_uninit_networks(xenstat_handle * handle)
{
	if(handle->procnetdev)
		fclose(handle->procnetdev);
}

/* Get the network ID */
unsigned int xenstat_network_id(xenstat_network * network)
{
	return network->id;
}

/* Get the number of receive bytes */
unsigned long long xenstat_network_rbytes(xenstat_network * network)
{
	return network->rbytes;
}

/* Get the number of receive packets */
unsigned long long xenstat_network_rpackets(xenstat_network * network)
{
	return network->rpackets;
}

/* Get the number of receive errors */
unsigned long long xenstat_network_rerrs(xenstat_network * network)
{
	return network->rerrs;
}

/* Get the number of receive drops */
unsigned long long xenstat_network_rdrop(xenstat_network * network)
{
	return network->rdrop;
}

/* Get the number of transmit bytes */
unsigned long long xenstat_network_tbytes(xenstat_network * network)
{
	return network->tbytes;
}

/* Get the number of transmit packets */
unsigned long long xenstat_network_tpackets(xenstat_network * network)
{
	return network->tpackets;
}

/* Get the number of transmit errors */
unsigned long long xenstat_network_terrs(xenstat_network * network)
{
	return network->terrs;
}

/* Get the number of transmit dropped packets */
unsigned long long xenstat_network_tdrop(xenstat_network * network)
{
	return network->tdrop;
}

/*
 * Xen version functions
 */

/* Collect Xen version information */
static int xenstat_collect_xen_version(xenstat_node * node)
{
	long vnum = 0;
	xen_extraversion_t version;

	/* Collect Xen version information if not already collected */
	if (node->handle->xen_version[0] == '\0') {
		/* Get the Xen version number and extraversion string */
		vnum = xc_version(node->handle->xc_handle,
			XENVER_version, NULL);

		if (vnum < 0)
			return 0;

		if (xc_version(node->handle->xc_handle, XENVER_extraversion,
			&version) < 0)
			return 0;
		/* Format the version information as a string and store it */
		snprintf(node->handle->xen_version, VERSION_SIZE, "%ld.%ld%s",
			 ((vnum >> 16) & 0xFFFF), vnum & 0xFFFF, version);
	}

	return 1;
}

/* Free Xen version information in node - nothing to do */
static void xenstat_free_xen_version(xenstat_node * node)
{
}

/* Free Xen version information in handle - nothing to do */
static void xenstat_uninit_xen_version(xenstat_handle * handle)
{
}

/*
 * VBD functions
 */

static int read_attributes_vbd(const char *vbd_directory, const char *what, char *ret, int cap)
{
	static char file_name[80];
	int fd, num_read;

	sprintf(file_name, "%s/%s/%s", SYSFS_VBD_PATH, vbd_directory, what);
	fd = open(file_name, O_RDONLY, 0);
	if (fd==-1) return -1;
	num_read = read(fd, ret, cap - 1);
	close(fd);
	if (num_read<=0) return -1;
	ret[num_read] = '\0';
	return num_read;
}

/* Collect information about VBDs */
static int xenstat_collect_vbds(xenstat_node * node)
{
	struct dirent *dp;

	if (node->handle->sysfsvbd == NULL) {
		node->handle->sysfsvbd = opendir(SYSFS_VBD_PATH);
		if (node->handle->sysfsvbd == NULL) {
			perror("Error opening " SYSFS_VBD_PATH);
			return 0;
		}
	}

	rewinddir(node->handle->sysfsvbd);

	for(dp = readdir(node->handle->sysfsvbd); dp != NULL ;
	    dp = readdir(node->handle->sysfsvbd)) {
		xenstat_domain *domain;
		xenstat_vbd vbd;
		unsigned int domid;
		int ret;
		char buf[256];


		ret = sscanf(dp->d_name, "vbd-%u-%u", &domid, &vbd.dev);
		if (ret != 2) {
			continue;
		}
		printf("%s is VBD.\n",dp->d_name);

		domain = xenstat_node_domain(node, domid);
		if (domain == NULL) {
			fprintf(stderr,
				"Found interface vbd-%u-%u but domain %u"
				" does not exist.\n",
				domid, vbd.dev, domid);
			continue;
		}

		if((read_attributes_vbd(dp->d_name, "statistics/oo_req", buf, 256)<=0)
		   || ((ret = sscanf(buf, "%llu", &vbd.oo_reqs)) != 1))
		{
			continue;
		}

		if((read_attributes_vbd(dp->d_name, "statistics/rd_req", buf, 256)<=0)
		   || ((ret = sscanf(buf, "%llu", &vbd.rd_reqs)) != 1))
		{
			continue;
		}

		if((read_attributes_vbd(dp->d_name, "statistics/wr_req", buf, 256)<=0)
		   || ((ret = sscanf(buf, "%llu", &vbd.wr_reqs)) != 1))
		{
			continue;
		}


		if (domain->vbds == NULL) {
			domain->num_vbds = 1;
			domain->vbds = malloc(sizeof(xenstat_vbd));
		} else {
			domain->num_vbds++;
			domain->vbds = realloc(domain->vbds,
					       domain->num_vbds *
					       sizeof(xenstat_vbd));
		}
		if (domain->vbds == NULL)
			return 0;
		domain->vbds[domain->num_vbds - 1] = vbd;
	}

	return 1;	
}

/* Free VBD information */
static void xenstat_free_vbds(xenstat_node * node)
{
	unsigned int i;
	for (i = 0; i < node->num_domains; i++)
		free(node->domains[i].vbds);
}

/* Free VBD information in handle */
static void xenstat_uninit_vbds(xenstat_handle * handle)
{
	if (handle->sysfsvbd)
		closedir(handle->sysfsvbd);
}

/* Get the major number of VBD device */
unsigned int xenstat_vbd_dev(xenstat_vbd * vbd)
{
	return vbd->dev;
}

/* Get the number of OO(Out of) requests */
unsigned long long xenstat_vbd_oo_reqs(xenstat_vbd * vbd)
{
	return vbd->oo_reqs;
}

/* Get the number of READ requests */
unsigned long long xenstat_vbd_rd_reqs(xenstat_vbd * vbd)
{
	return vbd->rd_reqs;
}

/* Get the number of WRITE requests */
unsigned long long xenstat_vbd_wr_reqs(xenstat_vbd * vbd)
{
	return vbd->wr_reqs;
}

static char *xenstat_get_domain_name(xenstat_handle *handle, unsigned int domain_id)
{
	char path[80];

	snprintf(path, sizeof(path),"/local/domain/%i/name", domain_id);
	
	return xs_read(handle->xshandle, XBT_NULL, path, NULL);
}	

/* Remove specified entry from list of domains */
static void xenstat_prune_domain(xenstat_node *node, unsigned int entry)
{
	/* nothing to do if array is empty or entry is beyond end */
	if (node->num_domains == 0 || entry >= node->num_domains)
		return;

	/* decrement count of domains */
	node->num_domains--;

	/* shift entries following specified entry up by one */
	if (entry < node->num_domains) {
		xenstat_domain *domain = &node->domains[entry];
		memmove(domain,domain+1,node->num_domains-entry);
	}

	/* zero out original last entry from node -- not
	   strictly necessary but safer! */
	memset(&node->domains[node->num_domains], 0, sizeof(xenstat_domain)); 
}

