#include "fm.h"
#include "parsetagx.h"
#include "myctype.h"
#include <signal.h>
#include <setjmp.h>
#include <errno.h>
#include <math.h>

static JMP_BUF AbortLoading;

static MySignalHandler
KeyAbort(SIGNAL_ARG)
{
    LONGJMP(AbortLoading, 1);
}

struct frameset *
newFrameSet(struct parsed_tag *tag)
{
    struct frameset *f;
    int i;
    char *cols = NULL, *rows = NULL, *p, *q;
    char *length[100];

    f = New(struct frameset);
    f->attr = F_FRAMESET;
    f->name = NULL;
    f->currentURL = NULL;
    parsedtag_get_value(tag, ATTR_COLS, &cols);
    parsedtag_get_value(tag, ATTR_ROWS, &rows);
    i = 0;
    if (cols) {
	length[i] = p = cols;
	while (*p != '\0')
	    if (*p++ == ',') {
		length[++i] = p;
		if (i >= sizeof(length) / sizeof(length[0]) - 2)
		    break;
	    }
	length[++i] = p + 1;
    }
    if (i > 1) {
	f->col = i;
	f->width = New_N(char *, i);
	p = length[i];
	do {
	    i--;
	    q = p - 2;
	    p = length[i];
	    switch (*q) {
		Str tmp;
	    case '%':
		f->width[i] = allocStr(p, q - p + 1);
		break;
	    case '*':
		f->width[i] = "*";
		break;
	    default:
		tmp = Sprintf("%d", atoi(p));
		f->width[i] = tmp->ptr;
		break;
	    }
	} while (i);
    }
    else {
	f->col = 1;
	f->width = New_N(char *, 1);
	f->width[0] = "*";
    }
    i = 0;
    if (rows) {
	length[i] = p = rows;
	while (*p != '\0')
	    if (*p++ == ',') {
		length[++i] = p;
		if (i >= sizeof(length) / sizeof(length[0]) - 2)
		    break;
	    }
	length[++i] = p + 1;
    }
    if (i > 1) {
	f->row = i;
	f->height = New_N(char *, i);
	p = length[i];
	do {
	    i--;
	    q = p - 2;
	    p = length[i];
	    f->height[i] = allocStr(p, q - p + 1);
	} while (i);
    }
    else {
	f->row = 1;
	f->height = New_N(char *, 1);
	f->height[0] = "*";
    }
    i = f->row * f->col;
    f->i = 0;
    f->frame = New_N(union frameset_element, i);
    do {
	i--;
	f->frame[i].element = NULL;
    } while (i);
    return f;
}

struct frame_body *
newFrame(struct parsed_tag *tag, Buffer *buf)
{
    struct frame_body *body;
    char *p;

    body = New(struct frame_body);
    memset(body, 0, sizeof(*body));
    body->attr = F_UNLOADED;
    body->flags = 0;
    body->baseURL = baseURL(buf);
    if (tag) {
	if (parsedtag_get_value(tag, ATTR_SRC, &p))
	    body->url = url_quote_conv_for_buf(p, buf);
	if (parsedtag_get_value(tag, ATTR_NAME, &p) && *p != '_')
	    body->name = url_quote_conv_for_buf(p, buf);
    }
#ifdef MANY_CHARSET
    body->charset = NULL;
#endif
    return body;
}

static void
unloadFrame(struct frame_body *b)
{
    if (b->source && b->flags & FB_TODELETE)
	pushText(fileToDelete, b->source);
    b->attr = F_UNLOADED;
}

void
deleteFrame(struct frame_body *b)
{
    if (b == NULL)
	return;
    unloadFrame(b);
    bzero((void *)b, sizeof(*b));
}

void
addFrameSetElement(struct frameset *f, union frameset_element element)
{
    int i;

    if (f == NULL)
	return;
    i = f->i;
    if (i >= f->col * f->row)
	return;
    f->frame[i] = element;
    f->i++;
}

void
deleteFrameSet(struct frameset *f)
{
    int i;

    if (f == NULL)
	return;
    for (i = 0; i < f->col * f->row; i++) {
	deleteFrameSetElement(f->frame[i]);
    }
    f->name = NULL;
    f->currentURL = NULL;
    return;
}

void
deleteFrameSetElement(union frameset_element e)
{
    if (e.element == NULL)
	return;
    switch (e.element->attr) {
    case F_UNLOADED:
	break;
    case F_BODY:
	deleteFrame(e.body);
	break;
    case F_FRAMESET:
	deleteFrameSet(e.set);
	break;
    default:
	break;
    }
    return;
}

static struct frame_body *
copyFrame(struct frame_body *ob)
{
    struct frame_body *rb;

    rb = New(struct frame_body);
    bcopy((const void *) ob, (void *) rb, sizeof(struct frame_body));
    rb->flags &= ~FB_TODELETE;
    return rb;
}

struct frameset *
copyFrameSet(struct frameset *of)
{
    struct frameset *rf;
    int n;

    rf = New(struct frameset);
    n = of->col * of->row;
    bcopy((const void *)of, (void *)rf, sizeof(struct frameset));
    rf->width = New_N(char *, rf->col);
    bcopy((const void *)of->width,
	  (void *) rf->width, sizeof(char *) * rf->col);
    rf->height = New_N(char *, rf->row);
    bcopy((const void *)of->height,
	  (void *) rf->height, sizeof(char *) * rf->row);
    rf->frame = New_N(union frameset_element, n);
    while (n)
	if (of->frame[--n].element)
	    switch (of->frame[n].element->attr) {
	    case F_UNLOADED:
	    case F_BODY:
		rf->frame[n].body = copyFrame(of->frame[n].body);
		break;
	    case F_FRAMESET:
		rf->frame[n].set = copyFrameSet(of->frame[n].set);
		break;
	    default:
		rf->frame[n].element = NULL;
		break;
	    }
    return rf;
}

void
flushFrameSet(struct frameset *fs)
{
    int n = fs->i;

    while (n)
	if (fs->frame[--n].element)
	    switch (fs->frame[n].element->attr) {
	    case F_UNLOADED:
	    case F_BODY:
		fs->frame[n].body->nameList = NULL;
		break;
	    case F_FRAMESET:
		flushFrameSet(fs->frame[n].set);
		break;
	    default:
		/* nothing to do */
		break;
	    }
}

void
pushFrameTree(struct frameset_queue **fqpp, struct frameset *fs)
{
    struct frameset_queue *rfq, *cfq = *fqpp;

    if (!fs)
	return;

    rfq = New(struct frameset_queue);

    rfq->back = cfq;
    if (cfq) {
	rfq->next = cfq->next;
	if (cfq->next)
	    cfq->next->back = rfq;
	cfq->next = rfq;
    }
    else
	rfq->next = cfq;
    rfq->frameset = fs;
    *fqpp = rfq;
    return;
}

struct frameset *
popFrameTree(struct frameset_queue **fqpp)
{
    struct frameset_queue *rfq = NULL, *cfq = *fqpp;
    struct frameset *rfs = NULL;

    if (!cfq)
	return rfs;

    rfs = cfq->frameset;
    if (cfq->next) {
	(rfq = cfq->next)->back = cfq->back;
    }
    if (cfq->back) {
	(rfq = cfq->back)->next = cfq->next;
    }
    *fqpp = rfq;
    memset(cfq, 0, sizeof(struct frameset_queue));
    return rfs;
}

void
resetFrameElement(union frameset_element *f_element,
		  Buffer *buf, char *referer, FormList *request)
{
    char *f_name;
    struct frame_body *f_body;

    f_name = f_element->element->name;
    if (buf->frameset) {
	/* frame cascade */
	deleteFrameSetElement(*f_element);
	f_element->set = buf->frameset;
	f_element->set->currentURL = New(ParsedURL);
	copyParsedURL(f_element->set->currentURL, &buf->currentURL);
	buf->frameset = popFrameTree(&(buf->frameQ));
	f_element->set->name = f_name;
    }
    else {
	f_body = newFrame(NULL, buf);
	f_body->attr = F_BODY;
	f_body->name = f_name;
	f_body->url = parsedURL2Str(&buf->currentURL)->ptr;
	if (buf->real_scheme == SCM_LOCAL) {
	    f_body->source = buf->sourcefile;
	}
	else {
	    Str tmp = tmpfname(TMPF_FRAME, ".r");
	    rename(buf->sourcefile, tmp->ptr);
	    f_body->source = tmp->ptr;
	    f_body->flags |= FB_TODELETE;
	    buf->sourcefile = NULL;
	}
	f_body->type = buf->type;
	f_body->referer = referer;
	f_body->request = request;
#ifdef MANY_CHARSET
	f_body->charset = buf->document_encoding;
#endif
	deleteFrameSetElement(*f_element);
	f_element->body = f_body;
    }
}

struct frame_source_arg {
    struct frameset *fset, *root;
    int i;
    int nest;
    Buffer *buf;
    Phase0Env *p0env;
};

static struct frameset *
frame_source_downloaded(Buffer *buf, struct frameset *fset, int i)
{
    struct frame_body *b;
    struct frameset *fset1;

    b = fset->frame[i].body;

    if ((buf->bufferprop & BP_ASYNC_MASK) == BP_ASYNC_ABORT) {
	struct frame_element *fe;

	fset->frame[i].element = fe = New(struct frame_element);
	fe->attr = F_ERROR;
	fe->dummy = 0;
	fe->name = b->name;
	return NULL;
    }

    b->url = parsedURL2Str(&buf->currentURL)->ptr;
    b->source = buf->sourcefile;
    b->type = buf->type;
#ifdef USE_SSL
    b->ssl_certificate = buf->ssl_certificate;
#endif

    if (buf->real_scheme != SCM_LOCAL) {
	Str tmp;

	tmp = tmpfname(TMPF_FRAME, ".d");
	rename(buf->sourcefile, tmp->ptr);
	b->source = tmp->ptr;
	b->flags |= FB_TODELETE;
	buf->sourcefile = NULL;
    }

    b->attr = F_BODY;
#ifdef MANY_CHARSET
    b->charset = buf->document_encoding;
#endif

    if ((fset1 = buf->frameset)) {
	fset->frame[i].set = fset1;
	fset1->name = b->name;
	fset1->currentURL = New(ParsedURL);
	copyParsedURL(fset1->currentURL, &buf->currentURL);
	buf->frameset = popFrameTree(&(buf->frameQ));
	deleteFrame(b);
    }

    return fset1;
}

static int
frame_nunloaded_sources(struct frameset *f, int nest)
{
    int r, c, i;
    volatile int nunloaded = 0, m;

    if (f == NULL || nest >= 7)
	return -1;

    for (r = 0; r < f->row; r++)
	for (c = 0; c < f->col; c++) {
	    union frameset_element frame;

	    i = c + r * f->col;
	    frame = f->frame[i];

	    if (frame.element) {
		switch (frame.element->attr) {
		case F_UNLOADED:
		case F_LOADING:
		    ++nunloaded;
		    break;
		case F_FRAMESET:
		    if ((m = frame_nunloaded_sources(frame.set, nest + 1)) > 0)
			nunloaded += m;
		default:
		    break;
		}
	    }
	}

    return nunloaded;
}

static int frame_download_all_sources(struct frameset *f, struct frameset *root, Buffer *current,
				      int nest, int force_reload, Phase0Env *p0env);
static void renderFrame_core(struct frameset *f, Buffer *Cbuf, Phase0Env *p0env);

static void
renderFrame_post(struct frameset *fset, Buffer *buf, Phase0Env *p0env)
{
    pid_t pid;

    flushFrameSet(fset);

    if ((pid = forkWithChannel(parsedURL2Str(&buf->currentURL)->ptr, p0env, &buf)) == -1) {
	fprintf(stderr, "renderFrame(): %s\n", strerror(errno));
	fflush(stderr);
    }
    else if (!pid) {
	renderFrame_core(fset, buf, p0env);
	flush_buffer_and_exit(buf);
    }
}

static void
frame_source_receiver(Buffer *buf, int nlines)
{
    if (nlines < 0) {
	struct frame_source_arg *p;
	struct frameset *fset;

	p = buf->async_buf->p2env->p1env->p0env->receiver_arg;

	if ((fset = frame_source_downloaded(buf, p->fset, p->i)))
	    frame_download_all_sources(fset, p->root, p->buf, p->nest + 1, FALSE, p->p0env);

	if (!frame_nunloaded_sources(p->root, 0))
	    renderFrame_post(p->root, p->buf, p->p0env);
    }
}

static void
frame_download_source(struct frameset *fset, int i, ParsedURL *currentURL,
		      ParsedURL *baseURL, int flag, int nest,
		      struct frameset *root, Buffer *curbuf, Phase0Env *p0env_org)
{
    struct frame_body *b;
    struct frame_source_arg *p;
    Buffer *buf;
    ParsedURL url;
    Phase0Env p0env;

    if (!(b = fset->frame[i].body) || !b->url || !*b->url)
	return;

    p0env = *p0env_org;
    p0env.flag = flag | (RG_DUMP_FRAME | RG_STRSRC);
    p0env.receiver = frame_source_receiver;
    p0env.receiver_arg = p = New(struct frame_source_arg);
    p->fset = fset;
    p->root = root;
    p->i = i;
    p->nest = nest;
    p->buf = curbuf;
    p->p0env = New(Phase0Env);
    *p->p0env = *p0env_org;

    if (b->baseURL)
	baseURL = b->baseURL;

    parseURL2(b->url, &url, currentURL);

    switch (url.scheme) {
    case SCM_LOCAL:
	b->flags = 0;
    default:
	buf = loadGeneralFile(b->url,
			      baseURL ? baseURL : currentURL,
			      b->referer,
			      &p0env,
			      b->request);
	break;
    }

    if (buf == NULL || buf == NO_BUFFER) {
	b->source = NULL;
	b->flags = (buf == NO_BUFFER) ? FB_NO_BUFFER : 0;
	b->attr = F_ERROR;
    }
    else if (buf->async_buf)
	b->attr = F_LOADING;
    else {
	frame_source_downloaded(buf, fset, i);
	discardBuffer(buf);
    }
}

static int
frame_download_all_sources(struct frameset *f, struct frameset *root, Buffer *current,
			   int nest, int force_reload, Phase0Env *p0env)
{
    int n, i;
    ParsedURL *currentURL;
    int flag;
    volatile int nunloaded = 0, m;

    if (f == NULL || nest >= 7)
	return -1;

    if (!nest)
	f->name = "_top";

    currentURL = f->currentURL ? f->currentURL : &current->currentURL;
    flag = p0env->flag;;

    if (force_reload)
	flag |= RG_NOCACHE;

    for (i = 0, n = f->row * f->col ; i < n ; ++i) {
	union frameset_element frame;

	frame = f->frame[i];

	if (frame.element) {
	    if (frame.element->attr == F_BODY && force_reload)
		unloadFrame(frame.body);

	    if (frame.element->attr == F_UNLOADED) {
		if (!frame.body->name && f->name)
		    frame.body->name = Sprintf("%s_%d", f->name, i)->ptr;

		frame_download_source(f, i, currentURL, current->baseURL, flag, nest, root, current, p0env);
		frame = f->frame[i];
	    }

	    switch (frame.element->attr) {
	    case F_UNLOADED:
	    case F_LOADING:
		++nunloaded;
		break;
	    case F_FRAMESET:
		if ((m = frame_download_all_sources(frame.set, root, current, nest + 1, force_reload, p0env)) > 0)
		    nunloaded += m;
	    default:
		break;
	    }
	}
    }

    return nunloaded;
}

static int
createFrameFile(struct frameset *f, FILE *f1, Buffer *current, int nest)
{
    int r, c, t_stack;
    URLFile f2;
#ifdef MANY_CHARSET
    const char *code;
#endif
#ifdef JP_CHARSET
    char code, ic, charset[2];
#endif				/* JP_CHARSET */
    char *d_target, *p_target, *s_target, *t_target;
    ParsedURL *currentURL, base;
    MySignalHandler(* volatile prevtrap)(SIGNAL_ARG) = NULL;
    volatile int ret = 0;

    if (f == NULL)
	return -1;

    if (nest >= 7) {
	fputs("Too many frameset tasked.\n", f1);
	return -1;
    }

    if (!nest) {
	if (frame_nunloaded_sources(f, nest))
	    return -1;

	if (SETJMP(AbortLoading) != 0) {
	    ret = 1;
	    goto sig_end;
	}

	prevtrap = signal(SIGINT, KeyAbort);

	if (fmInitialized)
	    term_cbreak();

	fprintf(f1,
		"<html><head>"
#ifdef MANY_CHARSET
		"<meta http-equiv=\"Content-Type\" content=\"text/html; charset=x-moe-internal\">\n"
#endif
		"<title>%s</title></head><body>\n"
		"<table hborder width=\"100%%\">\n",
		html_quote(current->buffername));
    }
    else
	fputs("<table hborder>\n", f1);

    currentURL = f->currentURL ? f->currentURL : &current->currentURL;
#ifdef JP_CHARSET
    charset[1] = '\0';
#endif

    for (r = 0; r < f->row; r++) {
	fputs("<tr valign=top>\n", f1);

	for (c = 0; c < f->col; c++) {
	    union frameset_element frame;
	    int i = c + r * f->col;
	    char *p = "";
	    Str tok = Strnew();
	    int status;

	    frame = f->frame[i];

	    if (frame.element == NULL) {
		fputs("<td>\n</td>\n", f1);
		continue;
	    }

	    fputs("<td", f1);

	    if (frame.element->name)
		fprintf(f1, " id=\"_%s\"", html_quote(frame.element->name));

	    if (!r)
		fprintf(f1, " width=\"%s\"", f->width[c]);

	    fputs(">\n", f1);

	    switch (frame.element->attr) {
	    default:
		fprintf(f1, "Frameset \"%s\" frame %d: type unrecognized",
			html_quote(f->name), i + 1);
		break;
	    case F_BODY:
		init_stream(&f2, SCM_TMPFILE, NULL);

		if (frame.body->flags & FB_NO_BUFFER)
		    fprintf(f1, "Open %s with other method", frame.body->url);
		else if (frame.body->source == NULL) {
		    if (frame.body->url)
			fprintf(f1, "Source of %s was lost", frame.body->url);
		    else
			fprintf(f1, "This frame (%s) contains no src attribute",
				frame.body->name ? frame.body->name : "(no name)");
		}
		else {
		    Phase0Env p0env;

		    fflush(f1);
		    p0env = main_p0env;
		    p0env.flag &= ~(RG_DUMP_MASK | RG_HALFDUMP_MASK | RG_PROC_MASK | RG_DO_DOWNLOAD);
		    examineFile(frame.body->source, &f2, &p0env);

		    if (!f2.stream)
			fprintf(f1, "Can't open %s (%s)", frame.body->url, strerror(errno));
		}

		if (!f2.stream) {
		    frame.body->attr = F_ERROR;
		    frame.body->source = NULL;
		    break;
		}

		parseURL2(frame.body->url, &base, currentURL);
		p_target = f->name;
		s_target = frame.body->name;
		t_target = "_blank";
		d_target = TargetSelf ? s_target : t_target;
#ifdef MANY_CHARSET
		code = frame.body->charset;
#endif
#ifdef JP_CHARSET
		code = '\0';
#endif				/* JP_CHARSET */
		t_stack = 0;

		if (frame.body->type &&
		    !strcasecmp(frame.body->type, "text/plain")) {
		    Str tmp;

		    fprintf(f1, "<pre>\n");

		    while ((tmp = StrmyUFgets(&f2))->length) {
#ifdef MANY_CHARSET
			tmp = conv_Str2mbStr(tmp, NULL, "@", code);
#endif
#ifdef JP_CHARSET
			if ((ic = checkShiftCode(tmp, code)) != '\0')
			    tmp = conv_str(tmp, (code = ic), InnerCode);
#endif                /* JP_CHARSET */

			cleanup_line(tmp, HTML_MODE);
			fprintf(f1, "%s", html_quote(tmp->ptr));
		    }

		    fprintf(f1, "</pre>\n");
		    UFclose(&f2);
		    break;
		}

		do {
		    status = R_ST_NORMAL;

		    do {
			if (*p == '\0') {
 			    Str tmp = StrmyUFgets(&f2);

			    if (tmp->length == 0 && status != R_ST_NORMAL)
				tmp = correct_irrtag(status);

			    if (tmp->length == 0)
				break;

#ifdef MANY_CHARSET
			    tmp = conv_Str2mbStr(tmp, &code, "");
#endif

#ifdef JP_CHARSET
			    if ((ic = checkShiftCode(tmp, code)) != '\0')
				tmp = conv_str(tmp, (code = ic), InnerCode);
#endif				/* JP_CHARSET */

			    cleanup_line(tmp, HTML_MODE);
			    p = tmp->ptr;
			}

			if (status == R_ST_NORMAL || ST_IS_COMMENT(status))
			    read_token(tok, &p, p + strlen(p), &status, RT_PRE | RT_EOL);
			else
			    read_token(tok, &p, p + strlen(p), &status, RT_PRE | RT_APPEND | RT_EOL);
		    } while (status != R_ST_NORMAL);

		    if (tok->length == 0)
			continue;

		    if (tok->ptr[0] == '<') {
			char *q = tok->ptr;
			int j, a_target = 0;
			struct parsed_tag *tag;
			ParsedURL url;

			if (!(tag = parse_tag(&q, FALSE)))
			    goto token_end;

			switch (tag->tagid) {
			case HTML_TITLE:
			    fputs("<!-- title:", f1);
			    goto token_end;
			case HTML_N_TITLE:
			    fputs("-->", f1);
			    goto token_end;
			case HTML_BASE:
			    if (parsedtag_get_value(tag, ATTR_HREF, &q)) {
				q = url_quote_conv(q, code);
				parseURL(q, &base, NULL);
			    }

			    if (parsedtag_get_value(tag, ATTR_TARGET, &q)) {
				if (!strcasecmp(q, "_self"))
				    d_target = s_target;
				else if (!strcasecmp(q, "_parent"))
				    d_target = p_target;
				else
				    d_target = url_quote_conv(q, code);
			    }
			    /* fall thru, "BASE" is prohibit tag */
			case HTML_HEAD:
			case HTML_N_HEAD:
			case HTML_BODY:
			case HTML_N_BODY:
			case HTML_META:
			case HTML_DOCTYPE:
			    /* prohibit_tags */
			    Strshrinkfirst(tok, 1);
			    Strshrink(tok, 1);
			    fprintf(f1, "<!-- %s -->", tok->ptr);
			    goto token_end;
			case HTML_TABLE:
			    t_stack++;
			    break;
			case HTML_N_TABLE:
			    t_stack--;

			    if (t_stack < 0) {
				t_stack = 0;
				Strshrinkfirst(tok, 1);
				Strshrink(tok, 1);
				fprintf(f1,
					"<!-- table stack underflow: %s -->",
					tok->ptr);
				goto token_end;
			    }

			    break;
			case HTML_THEAD:
			case HTML_N_THEAD:
			case HTML_TBODY:
			case HTML_N_TBODY:
			case HTML_TFOOT:
			case HTML_N_TFOOT:
			case HTML_TD:
			case HTML_N_TD:
			case HTML_TR:
			case HTML_N_TR:
			case HTML_TH:
			case HTML_N_TH:
			    /* table_tags MUST be in table stack */
			    if (!t_stack) {
				Strshrinkfirst(tok, 1);
				Strshrink(tok, 1);
				fprintf(f1, "<!-- %s -->", tok->ptr);
				goto token_end;

			    }

			    break;
			default:
			    break;
			}

			for (j = 0; j < TagMAP[tag->tagid].max_attribute; j++)
			    switch (tag->attrid[j]) {
			    case ATTR_SRC:
			    case ATTR_HREF:
			    case ATTR_ACTION:
				if (!tag->value[j]) {
				    if (tag->tagid == HTML_FORM)
					parseURL2(frame.body->url, &url, currentURL);
				    else
					break;
				}
				else {
				    tag->value[j] = url_quote_conv(tag->value[j], code);
				    parseURL2(tag->value[j], &url, &base);
				}
				if (url.scheme == SCM_MAILTO ||
				    url.scheme == SCM_UNKNOWN ||
				    url.scheme == SCM_MISSING)
				    break;
				a_target |= 1;
				tag->value[j] = parsedURL2Str(&url)->ptr;
				tag->need_reconstruct = TRUE;
				parsedtag_set_value(tag,
						    ATTR_REFERER,
						    parsedURL2Str(&base)->ptr);
#ifdef MANY_CHARSET
				if (code)
				    parsedtag_set_value(tag,
							ATTR_CHARSET, (char *)code);
#endif
#ifdef JP_CHARSET
				if (code != '\0') {
				    charset[0] = code;
				    parsedtag_set_value(tag,
							ATTR_CHARSET, charset);
				}
#endif
				break;
			    case ATTR_TARGET:
				if (!tag->value[j])
				    break;
				a_target |= 2;
				if (!strcasecmp(tag->value[j], "_self")) {
				    parsedtag_set_value(tag,
							ATTR_TARGET, s_target);
				}
				else if (!strcasecmp(tag->value[j], "_parent")) {
				    parsedtag_set_value(tag,
							ATTR_TARGET, p_target);
				}
				break;
			    case ATTR_NAME:
			    case ATTR_ID:
				if (!tag->value[j])
				    break;
				parsedtag_set_value(tag,
						    ATTR_FRAMENAME, s_target);
				break;
			    }

			if (a_target == 1) {
			    /* there is HREF attribute and no TARGET
			     * attribute */
			    parsedtag_set_value(tag, ATTR_TARGET, d_target);
			}

			if (parsedtag_need_reconstruct(tag)) {
			    Str tagstr = parsedtag2str(tag);

			    Strfputs(tagstr, f1);
			}
			else
			    Strfputs(tok, f1);
		    }
		    else
			Strfputs(tok, f1);
		token_end:
		    Strclear(tok);
		} while (*p != '\0' || !iseos(f2.stream));

		while (t_stack--)
		    fputs("</TABLE>\n", f1);

		UFclose(&f2);
		break;
	    case F_FRAMESET:
		if (!frame.set->name && f->name)
		    frame.set->name = Sprintf("%s_%d", f->name, i)->ptr;

		createFrameFile(frame.set, f1, current, nest + 1);
		break;
	    }

	    fputs("</td>\n", f1);
	}

	fputs("</tr>\n", f1);
    }

    fputs("</table>\n", f1);

    if (!nest)
	fputs("</body></html>\n", f1);
sig_end:
    if (fmInitialized)
	term_raw();

    if (prevtrap)
	signal(SIGINT, prevtrap);

    return ret;
}

static void
renderFrame_core(struct frameset *fset, Buffer *newBuf, Phase0Env *p0env_orig)
{
    Str tmp, tmp1;
    FILE *f;
    URLFile uf;
    Phase0Env p0env;

    tmp = tmpfname(TMPF_SRC, ".html");
    pushFileToDelete(tmp->ptr, p0env_orig->flag);

    if (!(f = fopen(tmp->ptr, "w")) || createFrameFile(fset, f, newBuf, 0) < 0)
	return;

    fclose(f);
    init_stream(&uf, SCM_TMPFILE, NULL);
    p0env = main_p0env;
    p0env.flag &= ~(RG_DUMP_MASK | RG_HALFDUMP_MASK | RG_PROC_MASK | RG_DO_DOWNLOAD);
    examineFile(tmp->ptr, &uf, &p0env);
    p0env = *p0env_orig;
    p0env.SearchHeader = FALSE;
    p0env.renderFrameSet = fset;
#ifdef MANY_CHARSET
    p0env.default_content_charset = "x-moe-internal";
#endif
#ifdef JP_CHARSET
    p0env.DocumentCode = CODE_INNER_EUC;
#endif
    loadHTMLstream(&uf, newBuf, NULL, TRUE, &p0env);
    tmp1 = tmpfname(TMPF_FRAME, ".html");
    newBuf->filename = newBuf->sourcefile = rename(tmp->ptr, tmp1->ptr) ? tmp->ptr : tmp1->ptr;
}

static int
renderFrame_pre(struct frameset *fset, Buffer *Cbuf, int force_reload, Phase0Env *p0env, Buffer *newBuf)
{
    copyParsedURL(&newBuf->currentURL, &Cbuf->currentURL);
    newBuf->real_type = Cbuf->real_type;
    newBuf->type = "text/html";
    newBuf->frameset = fset;
    newBuf->bufferprop |= RG_FRAME;
    newBuf->buffername = Cbuf->buffername;
    newBuf->document_header = Cbuf->document_header;
    newBuf->document_header_table = Cbuf->document_header_table;
#ifdef MANY_CHARSET
    newBuf->document_charset = Cbuf->document_charset;
#endif
#ifdef JP_CHARSET
    newBuf->document_code = Cbuf->document_code;
#endif
    return frame_download_all_sources(fset, fset, newBuf, 0, force_reload, p0env);
}

Buffer *
renderFrame(struct frameset *fset, Buffer *Cbuf, int force_reload, Phase0Env *p0env)
{
    Buffer *newBuf;

    if ((p0env->flag & RG_PROC_MASK) == RG_PROC_FORK) {
	AsyncRWBuf *abuf;

	abuf = New(AsyncRWBuf);
	memset(abuf, 0, sizeof(*abuf));
	newBuf = bindBufferWithAsyncRWBuf(NULL, abuf, parsedURL2Str(&Cbuf->currentURL)->ptr, p0env);

	if (!renderFrame_pre(fset, Cbuf, force_reload, p0env, newBuf))
	    renderFrame_post(fset, newBuf, p0env);
    }
    else {
	newBuf = newBuffer(INIT_BUFFER_WIDTH);
	renderFrame_pre(fset, Cbuf, force_reload, p0env, newBuf);
	flushFrameSet(fset);
	renderFrame_core(fset, newBuf, p0env);
	newBuf->topLine = newBuf->firstLine;
    }

    return newBuf;
}

union frameset_element *
search_frame(struct frameset *fset, char *name)
{
    int i;
    union frameset_element *e = NULL;

    for (i = 0; i < fset->col * fset->row; i++) {
	e = &(fset->frame[i]);
	if (e->element != NULL) {
	    if (e->element->name && !strcmp(e->element->name, name)) {
		return e;
	    }
	    else if (e->element->attr == F_FRAMESET &&
		     (e = search_frame(e->set, name))) {
		return e;
	    }
	}
    }
    return NULL;
}

#define DELTA (0.5)

int
parse_frame_colrow(int ctotal, int cmin, int rulesize, double ppc, char **sv, int n, int *cwv)
{
    long ast, isum, *nv;
    char *cv, *e;
    int i;
    double total, real_total, sum, fixed_sum, *wv, t, dmin, fix, avg;

    nv = NewAtom_N(long, n);
    cv = NewAtom_N(char, n);
    wv = NewAtom_N(double, n);
    i = n;
    ast = 0;
    dmin = cmin * ppc;

    if ((total = (double)(ctotal - rulesize * (n - 1)) * ppc) <= dmin)
	return n;

    for (i = n ; i > 0 ;) {
	--i;

	if ((nv[i] = strtoul(sv[i], &e, 10)) < 0) {
	    nv[i] = 0;
	    e = sv[i];
	}

	switch (cv[i] = *e) {
	case '*':
	    if (!nv[i])
		nv[i] = 1;

	    ast += nv[i];
	    break;
	default:
	    cv[i] = '\0';
	case '\0':
	case '%':
	    break;
	}
    }

    real_total = total;

    for (;;) {
	fixed_sum = sum = 0.0;
	ast = 0;
	i = n;

	do {
	    --i;

	    switch (cv[i]) {
	    case '*':
		continue;
	    case '%':
		t = (total * (double)nv[i]) / 100.0;
		break;
	    default:
		t = (double)nv[i];
		break;
	    }

	    sum += t;

	    if ((wv[i] = (t / real_total) * total) < dmin)
		return i;

	    fixed_sum += wv[i];
	} while (i > 0);

	if (ast) {
	    if ((double)ast * dmin + fixed_sum <= total)
		break;

	    return n;
	}
	else {
	    if (fixed_sum + DELTA < total && total + DELTA < fixed_sum)
		break;

	    real_total = sum;
	}
    }

    if (ast) {
	i = n;

	do {
	    --i;

	    if (cv[i] == '*')
		wv[i] = (total - fixed_sum) / ast * (double)nv[i];
	} while (i > 0);
    }

    i = n;
    isum = 0;

    do {
	--i;
	cwv[i] = (long)(wv[i] / ppc);
	isum += cwv[i];
    } while (i > 0);

    while (isum < ctotal) {
	fix = (double)(ctotal - isum) / total;
	avg = (double)(ctotal - isum) / (double)n;

	i = n;

	do {
	    --i;

	    if (fix * wv[i] >= avg) {
		++(cwv[i]);

		if (++isum >= ctotal)
		    break;
	    }
	} while (i > 0);
    }

    return -1;
}

/* Local Variables:    */
/* c-basic-offset: 4   */
/* tab-width: 8        */
/* End:                */
