//========================================================================
//
// Page.cc
//
// Copyright 1996 Derek B. Noonburg
//
//========================================================================

#ifdef __GNUC__
#pragma implementation
#endif

#include <stddef.h>
#include "Object.h"
#include "Array.h"
#include "Dict.h"
#include "XRef.h"
#include "Link.h"
#include "OutputDev.h"
#ifndef PDF_PARSER_ONLY
#include "Gfx.h"
#include "FormWidget.h"
#endif
#include "Error.h"

#include "Params.h"
#include "Page.h"

// get a page boundary of name "boundary" out of "dict" and merge
// it with the Rectangle defined in "box".
static Rectangle getPageBox (char* boundary, Dict* dict, Rectangle& box) {
  Object obj1;
  Rectangle result = box;
  dict->lookup(boundary, &obj1);
  if (obj1.isArray() && obj1.arrayGetLength() == 4) {
    Object obj2;
    obj1.arrayGet(0, &obj2);
    if (obj2.isNum())
      result.x1 = obj2.getNum();
    obj2.free();
    obj1.arrayGet(1, &obj2);
    if (obj2.isNum())
      result.y1 = obj2.getNum();
    obj2.free();
    obj1.arrayGet(2, &obj2);
    if (obj2.isNum())
      result.x2 = obj2.getNum();
    obj2.free();
    obj1.arrayGet(3, &obj2);
    if (obj2.isNum())
      result.y2 = obj2.getNum();
    obj2.free();
  }
  obj1.free();
  return result;
  }

//------------------------------------------------------------------------
// PageAttrs
//------------------------------------------------------------------------

PageAttrs::PageAttrs(PageAttrs *attrs, Dict *dict) {
  Object obj1, obj2;
  double w, h;

  // get old/default values
  if (attrs) {
    mediaBox = attrs->mediaBox;
    cropBox = attrs->cropBox;
    trimBox = attrs->trimBox;
    bleedBox = attrs->bleedBox;
    artBox = attrs->artBox;
    rotate = attrs->rotate;
    attrs->resources.copy(&resources);
  } else {
    // set default MediaBox to 8.5" x 11" -- this shouldn't be necessary
    // but some (non-compliant) PDF files don't specify a MediaBox
    mediaBox.x2 = 612;
    mediaBox.y2 = 792;
    rotate = 0;
    resources.initNull();
  }

  mediaBox = getPageBox ("MediaBox", dict, mediaBox);
  cropBox = getPageBox ("CropBox", dict, cropBox);
  trimBox = getPageBox ("TrimBox", dict, trimBox);
  bleedBox = getPageBox ("BleedBox", dict, bleedBox);
  artBox = getPageBox ("ArtBox", dict, artBox);

  // if the MediaBox is excessively larger than the CropBox,
  // just use the CropBox
  limitToCropBox = gFalse;
  w = 0.25 * (cropBox.x2 - cropBox.x1);
  h = 0.25 * (cropBox.y2 - cropBox.y1);
  if (cropBox.x2 > cropBox.x1 &&
      ((cropBox.x1 - mediaBox.x1) + (mediaBox.x2 - cropBox.x2) > w ||
       (cropBox.y1 - mediaBox.y1) + (mediaBox.y2 - cropBox.y2) > h)) {
    limitToCropBox = gTrue;
  }

  // rotate
  dict->lookup("Rotate", &obj1);
  if (obj1.isInt())
    rotate = obj1.getInt();
  obj1.free();
  while (rotate < 0)
    rotate += 360;
  while (rotate >= 360)
    rotate -= 360;

  // resource dictionary
  dict->lookup("Resources", &obj1);
  if (obj1.isDict()) {
    resources.free();
    obj1.copy(&resources);
  }
  obj1.free();
}

PageAttrs::~PageAttrs() {
  resources.free();
}

//------------------------------------------------------------------------
// Page
//------------------------------------------------------------------------

Page::Page(int num1, Dict *pageDict, PageAttrs *attrs1) {

  ok = gTrue;
  num = num1;

  // get attributes
  attrs = attrs1;

  // annotations
  pageDict->lookupNF("Annots", &annots);
  if (!(annots.isRef() || annots.isArray() || annots.isNull())) {
    error(-1, "Page annotations object (page %d) is wrong type (%s)",
	  num, annots.getTypeName());
    annots.free();
    goto err2;
  }

  // contents
  pageDict->lookupNF("Contents", &contents);
  if (!(contents.isRef() || contents.isArray() ||
	contents.isNull())) {
    error(-1, "Page contents object (page %d) is wrong type (%s)",
	  num, contents.getTypeName());
    contents.free();
    goto err1;
  }

  return;

 err2:
  annots.initNull();
 err1:
  contents.initNull();
  ok = gFalse;
}

Page::~Page() {
  delete attrs;
  annots.free();
  contents.free();
}

void Page::display(OutputDev *out, double dpi, int rotate,
		   Links *links, Catalog *catalog) {
#ifndef PDF_PARSER_ONLY
  Gfx *gfx;
  Object obj;
  Link *link;
  int i;
  FormWidgets *formWidgets;

  if (printCommands) {
    printf("***** MediaBox = ll:%g,%g ur:%g,%g\n",
	   getX1(), getY1(), getX2(), getY2());
    if (isCropped()) {
      printf("***** CropBox = ll:%g,%g ur:%g,%g\n",
	     getCropX1(), getCropY1(), getCropX2(), getCropY2());
    }
    printf("***** BleedBox = ll:%g,%g ur:%g,%g\n",
	   attrs->getBleedBox().x1, attrs->getBleedBox().y1, attrs->getBleedBox().x2, attrs->getBleedBox().y2);
    printf("***** TrimBox = ll:%g,%g ur:%g,%g\n",
	   attrs->getTrimBox().x1, attrs->getTrimBox().y1, attrs->getTrimBox().x2, attrs->getTrimBox().y2);
    printf("***** ArtBox = ll:%g,%g ur:%g,%g\n",
	   attrs->getArtBox().x1, attrs->getArtBox().y1, attrs->getArtBox().x2, attrs->getArtBox().y2);
    printf("***** LogicalMediaBox = ll:%g,%g ur:%g,%g\n",
	   attrs->getLogicalMediaBox().x1, attrs->getLogicalMediaBox().y1, attrs->getLogicalMediaBox().x2, attrs->getLogicalMediaBox().y2);
    printf("***** LogicalCropBox = ll:%g,%g ur:%g,%g\n",
	   attrs->getLogicalCropBox().x1, attrs->getLogicalCropBox().y1, attrs->getLogicalCropBox().x2, attrs->getLogicalCropBox().y2);
    printf("***** LogicalBleedBox = ll:%g,%g ur:%g,%g\n",
	   attrs->getLogicalBleedBox().x1, attrs->getLogicalBleedBox().y1, attrs->getLogicalBleedBox().x2, attrs->getLogicalBleedBox().y2);
    printf("***** LogicalTrimBox = ll:%g,%g ur:%g,%g\n",
	   attrs->getLogicalTrimBox().x1, attrs->getLogicalTrimBox().y1, attrs->getLogicalTrimBox().x2, attrs->getLogicalTrimBox().y2);
    printf("***** LogicalArtBox = ll:%g,%g ur:%g,%g\n",
	   attrs->getLogicalArtBox().x1, attrs->getLogicalArtBox().y1, attrs->getLogicalArtBox().x2, attrs->getLogicalArtBox().y2);
    printf("***** Rotate = %d\n", attrs->getRotate());
  }

  rotate += getRotate();
  if (rotate >= 360) {
    rotate -= 360;
  } else if (rotate < 0) {
    rotate += 360;
  }
  gfx = new Gfx(out, num, attrs->getResourceDict(),
		dpi, getX1(), getY1(), getX2(), getY2(), isCropped(),
		getCropX1(), getCropY1(), getCropX2(), getCropY2(), rotate);
  contents.fetch(&obj);
  if (!obj.isNull()) {
    gfx->display(&obj);
  }
  obj.free();

  // draw links
  if (links) {
    for (i = 0; i < links->getNumLinks(); ++i) {
      link = links->getLink(i);
      out->drawLink(link, catalog);
    }
    out->dump();
  }

  // draw AcroForm widgets
  //~ need to reset CTM ???
  formWidgets = new FormWidgets(annots.fetch(&obj));
  obj.free();
  if (printCommands && formWidgets->getNumWidgets() > 0) {
    printf("***** AcroForm widgets\n");
  }
  for (i = 0; i < formWidgets->getNumWidgets(); ++i) {
    formWidgets->getWidget(i)->draw(gfx);
  }
  if (formWidgets->getNumWidgets() > 0) {
    out->dump();
  }
  delete formWidgets;

  delete gfx;
#endif
}
