/*
** Copyright (C) 1998,1999,2000,2001 Martin Roesch <roesch@clark.net>
** Copyright (C) 1999,2000,2001 Christopher E. Cramer <cec@ee.duke.edu>
** 
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
** 
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

/* $Id: spp_tcp_stream2.c,v 1.17 2001/08/07 11:46:12 fygrave Exp $ */

/* spp_tcp_stream2
 * 
 * Purpose:
 *
 * Construct tcp streams from observed packets
 *
 * Usage:
 *
 * preprocessor stream: <arg set 1>, <arg set 2>, ...
 *
 * Arguments:  one or more of the following separated by commas
 *
 * timeout <timeout value>
 *
 * ports <port 1> ... <port N>
 *
 * maxbytes <maxbytes>
 *
 * <timeout>  - the max time in seconds for which a stream will be 
 *            kept alive if we haven't seen a packet for it
 * <port x>   - a server port to monitor.  we don't want to monitor 
 *            all tcp streams (do we?)
 * <maxbytes> - maximum bytes in our reconstructed packets
 *
 *     example: 
 *       preprocessor stream: timeout 5, ports 21 23 80 8080, maxbytes 16384
 *
 * Effect:
 *
 * creates a buffer for each observed tcp stream.  upon seeing a CRLF
 * or receiving a maximum number of bytes, generate a packet containing 
 * the reconstructed data
 *
 * Comments:
 *
 * the size of <timeout> will directly impact the amount of memory used.  
 * The longer the timeout the more streams kept in memory, the greater 
 * the memory usage.
 *
 * To Do: 
 *    allow tighter checks on what we monitor - reduce memory usage.
 *
 *
 */

#include "spp_tcp_stream2.h"
#ifndef WIN32
    #include <strings.h>
    #include <sys/time.h>
#else
    #include <time.h>
#endif
#include <sys/types.h>

#undef HAVE_64

#define SEQ_GT(x,y)     (y - x > 0x7fffffff)
#define SEQ_LT(x,y)     (x - y > 0x7fffffff)
#define SEQ_EQ(x,y)     (x == y)
#define SEQ_GE(x,y)     (x - y < 0x7fffffff)
#define SEQ_LE(x,y)     (y - x < 0x7fffffff)


struct pseudoheader       /* pseudo header for TCP checksum calculations */
{
    u_int32_t sip, dip;   /* IP addr */
    u_int8_t  zero;       /* checksum placeholder */
    u_int8_t  protocol;   /* protocol number */
    u_int16_t tcplen;     /* tcp packet length */
};
 

/* external globals from rules.c */
extern char *file_name;
extern int file_line;

TcpStream2Data StreamData2;
FILE *TcpStream2File;
int sesscount2 = 0;
Packet *dummy = NULL;
int pcaplen, hdrsize;

/*
 * Function: SetupTcpStream2()
 *
 * Purpose: Registers the preprocessor keyword and initialization 
 *          function into the preprocessor list.  This is the function that
 *          gets called from InitPreprocessors() in plugbase.c.
 *
 * Arguments: None.
 *
 * Returns: void function
 *
 */
void SetupTcpStream2()
{
    /* link the preprocessor keyword to the init function in 
       the preproc list */
    RegisterPreprocessor("stream2", TcpStream2Init);

#ifdef DEBUG
    printf("Preprocessor: TcpStream2 is setup...\n");
#endif
}


/*
 * Function: TcpStream2Init(u_char *)
 *
 * Purpose: Calls the argument parsing function, performs final setup on data
 *          structs, links the preproc function into the function list.
 *
 * Arguments: args => ptr to argument string
 *
 * Returns: void function
 *
 */
void TcpStream2Init(u_char *args)
{
    int i;
#ifdef DEBUG
    printf("Preprocessor: TcpStream2 Initialized\n");
#endif

    /* parse the argument list from the rules file */
    for(i=0;i<256;i++)
        StreamData2.heads[i] = (TcpStream2Session *)NULL;
    ParseTcpStream2Args(args);

    /* Set the preprocessor function into the function list */
    AddFuncToPreprocList(TcpStream2Packet);
    AddFuncToCleanExitList(TcpStream2Exit,NULL);
    AddFuncToRestartList(TcpStream2Restart,NULL);

    dummy = (Packet *) malloc(sizeof(Packet));
    if(dummy == NULL) return;
    pcaplen = 8 + sizeof(struct timeval);
    dummy->pkth = (struct pcap_pkthdr *)malloc(pcaplen + 4 + 40 + 65536);  

    if(dummy->pkth == NULL)
    {
        free(dummy);
        dummy = NULL;
        return;
    }

    pv.fake_packet_flag = 0;
    dummy->iph = (IPHdr *)((u_char*)dummy->pkth + pcaplen + 4);
    hdrsize = 40 + NULL_HDRLEN;
    dummy->tcph = (TCPHdr *)((u_char *) dummy->iph + 20);
    dummy->pkt = ((u_char*)dummy->pkth + pcaplen);
    dummy->iph->ip_ver = 0x04;
    dummy->iph->ip_hlen = 0x05;
    dummy->iph->ip_proto = IPPROTO_TCP;
    dummy->iph->ip_id = 0;
    dummy->iph->ip_off = htons(0);
    dummy->iph->ip_ttl = 255;
    dummy->iph->ip_csum = 0;
    dummy->iph->ip_tos = 0x10;
    memset(dummy->tcph,0,sizeof(TCPHdr));
    dummy->tcph->th_flags = R_PSH | R_ACK;
    dummy->tcph->th_off = 0x05;
}


/*
 * Function: ParseTcpStream2Args(char *)
 *
 * Purpose: Process the preprocessor arguements from the rules file and 
 *          initialize the preprocessor's data struct.  This function doesn't
 *          have to exist if it makes sense to parse the args in the init 
 *          function.
 *
 * Arguments: args => argument list
 *
 * Returns: void function 
 *
 */
void ParseTcpStream2Args(char *args)
{
    /* your parsing function goes here, check out the other spp files
       for examples */

    char **toks, **secs, **ptoks;
    int num_toks, num_secs, num_ptoks;
    int i,j;


    if(args == NULL)
    {
        FatalError("ERROR %s (%d)=> No arguments to TcpStream2 preprocessor!\n", file_name, file_line);
    }
    StreamData2.maxbytes = 0x01<<12;
    StreamData2.minbytes = 1;
    StreamData2.prunetime = 10;
    StreamData2.trunc = 0;
    /* tokenize the argument list */
    secs = mSplit(args, ",", 33, &num_secs, '\\');

    for(i=0;i<num_secs;i++)
    {
        toks = mSplit(secs[i], " ", 33, &num_toks, '\\');
        if(strcmp(toks[0],"timeout") == 0)
        {
            if(num_toks != 2) FatalError("ERROR %s (%d)=> timeout takes 1 argument!\n",file_name,file_line);
            StreamData2.prunetime = atoi(toks[1]);
            if(StreamData2.prunetime <= 0)
            {
                FatalError("ERROR %s (%d)=> Time between prunings must be >0 seconds!\n",file_name,file_line);
            }
        }
        else if(strcmp(toks[0],"maxbytes") == 0)
        {
            if(num_toks != 2) FatalError("ERROR %s (%d)=> maxbytes takes 1 argument!\n",file_name,file_line);
            StreamData2.maxbytes = atoi(toks[1]);
            if(StreamData2.maxbytes < 0 || StreamData2.maxbytes > (0x1 << 15))
            {
                FatalError("ERROR %s (%d)=> maxbytes must be >= 0 and <= 32K bytes!\n",file_name,file_line);
            }
        }
        else if(strcmp(toks[0],"minbytes") == 0)
        {
            if(num_toks != 2) FatalError("ERROR %s (%d)=> minbytes takes 1 argument!\n",file_name,file_line);
            StreamData2.minbytes = atoi(toks[1]);
            if(StreamData2.maxbytes < 1 || StreamData2.maxbytes > (0x1 << 15))
            {
                FatalError("ERROR %s (%d)=> minbytes must be >= 1 and <= 32K bytes!\n",file_name,file_line);
            }
        }
        else if(strcmp(toks[0],"truncate") == 0)
        {
            StreamData2.trunc = 1;
        }
        else if(strcmp(toks[0],"ports") == 0)
        {
            for(j = 1; j < num_toks; j++)
            {
            	ptoks = mSplit(toks[j], "/", 2, &num_ptoks, '\\');
                StreamData2.ports[j-1] = atoi(ptoks[0]);
                StreamData2.port_collect[j-1] = COLLECT_CLIENT | COLLECT_SERVER;
                if(num_ptoks == 2)
                {
                	if(strcmp(ptoks[1],"c") == 0)
                	{
                		StreamData2.port_collect[j-1] = COLLECT_CLIENT;
                	}
                	else if(strcmp(ptoks[1],"s") == 0)
                	{
                		StreamData2.port_collect[j-1] = COLLECT_SERVER;
                	}
                	else if(strcmp(ptoks[1],"b") == 0)
                	{
                		StreamData2.port_collect[j-1] = COLLECT_SERVER | COLLECT_CLIENT;
                	}
                	else
                	{
                		FatalError("ERROR %s (%d)=> port modifier must be 'c', 's' or 'b'\n",file_name,file_line);
                	}
                	free(ptoks[1]);
                }
                free(ptoks[0]);
#ifdef DEBUG
            	printf("New port to monitor: %i/%s%s\n",StreamData2.ports[j-1],StreamData2.port_collect[j-1]&COLLECT_SERVER ? "s" : "", StreamData2.port_collect[j-1]&COLLECT_CLIENT ? "c" : "");
#endif
            }   
            StreamData2.num_entries = num_toks-1;
        }
        else
        {
            FatalError("ERROR %s (%d)=> unknown argument to preprocessor stream: %s!\n",file_name,file_line,toks[0]);
        }
        for(j=0;j<num_toks;j++)
            free(toks[j]);
    }
    if(StreamData2.maxbytes < StreamData2.minbytes)
    {
        FatalError("ERROR %s=> maxbytes must be >= minbytes\n",file_name);    
    }
    for(i=0;i<num_secs;i++)
        free(secs[i]);
}


/*
 * Function: TcpStream2Packet(Packet *)
 *
 * Purpose:  take a packet, determine if it is tcp.  if so, is it the start of a stream?
 *           yes - create new stream.  no - do we have that stream?  yes - deal w/ packet.
 *           no - return
 *
 *
 * Arguments: p => pointer to the current packet data struct 
 *
 * Returns: void function
 *
 */
void TcpStream2Packet(Packet *p)
{
    int i;
#ifdef DEBUG
    char sip[16];
    char dip[16];
#endif
    u_long tm;
    TcpStream2Session * sptr = NULL;
    u_long lack, lseq;
    int server_packet;
    u_long make_packet_size;
    int bin;
    u_char CollectInfo;

    /* statuses (client-status & server-status) from TCP/IP Illustrated v1, p 241:
       0 - closed
       1 - listen
       2 - SYN_RCVD
       3 - SYN_SENT
       4 - ESTABLISHED  (data xfer mode)
       5 - CLOSE_WAIT
       6 - LAST_ACK
       7 - FIN_WAIT_1
       8 - CLOSING
       9 - FIN_WAIT_2
       10- TIME_WAIT
     */

    if(!PacketIsTCP(p))
    {
#ifdef DEBUG
           printf("It isn't TCP session traffic\n");
#endif
        return;
    }

    /* don't accept packets w/ bad checksums */
    if(p->csum_flags & CSE_IP || p->csum_flags & CSE_TCP)
    {
#ifdef DEBUG
        printf("Discarding packet based on checksum errors!\n");
#endif
        return;
    }

    if(pv.fake_packet_flag)
    {
#ifdef DEBUG
        printf("TcpStream2 Preprocessor shouldn't be handling it's own packets! :-)\n");
#endif
        return;
    }

    for(i = 0; i < StreamData2.num_entries; i++)
        if((StreamData2.ports[i] == p->dp) || (StreamData2.ports[i] == p->sp)) break;
    if(i == StreamData2.num_entries) return;  /* not a monitored port */

    pc.tcp_stream_pkts ++;  /* a packet we use */

    CollectInfo = StreamData2.port_collect[i];
    if(StreamData2.ports[i] == p->sp)
    {
        server_packet = 1;
    }
    else
    {
        server_packet = 0;
    }

    tm = TcpStream2Time();
    if(tm - StreamData2.timestamp >= StreamData2.prunetime)
    {
        TcpStream2PruneSessions();
        StreamData2.timestamp = tm;
    }

    lack = ntohl(p->tcph->th_ack);
    lseq = ntohl(p->tcph->th_seq);
#ifdef DEBUG
    strncpy(sip, inet_ntoa(p->iph->ip_src), 16);
    strncpy(dip, inet_ntoa(p->iph->ip_dst), 16);
    printf("Packet (%x %lX/%lX %i) - %s:%i -> %s:%i (%i)\n",p->tcph->th_flags,lseq,lack,p->tcph->th_win,sip,p->sp,dip,p->dp,p->dsize);
#endif 

    sptr = TcpStream2CheckSession(p,StreamData2.ports[i], &bin);
    if(sptr == (TcpStream2Session *)NULL)
    {
        if(p->tcph->th_flags == R_SYN)
        { /* new stream, SYN packet */
            sptr = (TcpStream2Session *)malloc(sizeof(TcpStream2Session));
            if (!sptr)
            {
                printf("couldn't malloc data for a new TcpStream2Session!\n");
                exit(1);
            }
            sptr->next = StreamData2.heads[bin];
            StreamData2.heads[bin] = sptr;
            pc.tcp_streams ++;  /* new tcp stream */
            sptr->client_status = SYN_SENT;
            sptr->server_status = SYN_RCVD;
            sptr->cip = p->iph->ip_src.s_addr;
            sptr->cp = p->sp;
            sptr->sip = p->iph->ip_dst.s_addr;
            sptr->sp = p->dp;
            sptr->c_first_seq = lseq + 1;
            sptr->c_last_ack = lseq + 1;
            sptr->c_data = sptr->s_data = (TcpStream2PacketData *)NULL;
            sptr->timestamp = tm;
        }
        return; /* either way, we are done here */
    }

    sptr->timestamp = tm;

    if(server_packet)
    {
        sptr->s_win = ntohs(p->tcph->th_win);
    }
    else
    {
        sptr->c_win = ntohs(p->tcph->th_win);
    }

    if(p->tcph->th_flags == R_SYN)     /* but we already have a stream */
        return;                 

    if(p->tcph->th_flags & R_RST)
    {
	/* check RST seq - possible fake RST to confuse NIDS */
	if(!server_packet && (SEQ_LT(lseq,sptr->s_last_ack) || SEQ_GE(lseq, (u_long)sptr->s_win + sptr->s_last_ack)) )
	{
#ifdef DEBUG
	    printf("Ignoring bad RST from server, seq: %lX not in range [%lX and %lX)!\n",lseq,sptr->s_last_ack,sptr->s_last_ack+(u_long)sptr->s_win);
#endif
	    return;
	}

	if(server_packet && (SEQ_LT(lseq, sptr->c_last_ack) || SEQ_GE(lseq, (u_long)sptr->c_win + sptr->c_last_ack)) )
	{
#ifdef DEBUG
	    printf("Ignoring bad RST from client, seq: %lX not in range [%lX and %lX)!\n",lseq,sptr->c_last_ack,sptr->c_last_ack+(u_long)sptr->c_win);
#endif
	    return;
	}

        /* handle RST by transitioning to the CLOSED state */
        sptr->client_status = CLOSED;
        sptr->server_status = CLOSED;
        sptr->c_last_ack = lseq + 1;
        sptr->timestamp = tm;
        return;
    }


    if(p->tcph->th_flags == (R_SYN | R_ACK))
    { /* SYN/ACK */
        if(!server_packet) return;  /* Client should not SYN/ACK */
        if(lack != sptr->c_first_seq) return; /* False SYN/ACK - bad seq*/
        if((sptr->client_status == SYN_SENT) && (sptr->server_status == SYN_RCVD))
        {
            sptr->s_first_seq = lseq+1;
            sptr->s_last_ack = lseq+1; 
            sptr->client_status = ESTABLISHED; /* client happy, should send ACK */
        }
        return;
    }

#ifdef DEBUG
    printf("statuses: %i/%i\n",sptr->server_status, sptr->client_status);
#endif

    /* Store Data if we see data and are ESTABLISHED or in CLOSE_WAIT
       and we are collecting data from this half of the stream
     */

    if((sptr->client_status == ESTABLISHED || 
       sptr->client_status == CLOSE_WAIT) && 
       (CollectInfo & COLLECT_CLIENT) && 
       !server_packet)
    {
	/* copy data and then truncate if desired */
    	TcpStream2StoreData(sptr, p, server_packet);
    	if(StreamData2.trunc)
    	{
    	    p->dsize = 0;
    	}
#ifdef DEBUG
        TcpStream2PrintDataList(sptr, !server_packet);
#endif
    }

    if((sptr->server_status == ESTABLISHED || 
       sptr->server_status == CLOSE_WAIT) && 
       (CollectInfo & COLLECT_SERVER) &&
       server_packet)
    {
	/* copy data and then truncate if desired */
    	TcpStream2StoreData(sptr, p, server_packet);
    	if(StreamData2.trunc)
    	{
    	    p->dsize = 0;
    	}
#ifdef DEBUG
        TcpStream2PrintDataList(sptr, !server_packet);
#endif
    }


    /* Check here to see if we inject new server packet */
    if(!server_packet && sptr->server_status >= ESTABLISHED && lack > sptr->s_last_ack)
    { 
        make_packet_size = TcpStream2CheckAck(sptr, lack, server_packet);
        while(make_packet_size >= StreamData2.minbytes) 
        {
            TcpStream2Packetize(sptr, p, make_packet_size, server_packet);
#ifdef DEBUG
            TcpStream2PrintDataList(sptr, !server_packet);
#endif
       	    make_packet_size = TcpStream2CheckAck(sptr, lack, server_packet);
        }
    }

    /* Check here to see if we inject client new packet */
    if(server_packet && sptr->client_status >= ESTABLISHED && lack > sptr->c_last_ack)
    {  
        make_packet_size = TcpStream2CheckAck(sptr, lack, server_packet);
        while(make_packet_size >= StreamData2.minbytes) 
        {
            TcpStream2Packetize(sptr, p, make_packet_size, server_packet);
#ifdef DEBUG
            TcpStream2PrintDataList(sptr, !server_packet);
#endif
	    make_packet_size = TcpStream2CheckAck(sptr, lack, server_packet);
        }
    }

    /* if something's been acknowledged, update last_acked 
     */
    if(p->tcph->th_flags & R_ACK)
    {
        if(server_packet)
            sptr->c_last_ack = lack;
        if(!server_packet)
            sptr->s_last_ack = lack;
    }

    if((sptr->client_status == ESTABLISHED) && (sptr->server_status == SYN_RCVD))
    {
        if(server_packet) return; /* shouldn't see a server packet here */
        if(lack != sptr->s_first_seq) return; /* False ACK - bad seq */
        if(p->tcph->th_flags == R_ACK) sptr->server_status = ESTABLISHED;
        return;
    }

    if((sptr->server_status == ESTABLISHED) && 
            (sptr->client_status == ESTABLISHED) && 
            (p->tcph->th_flags & R_FIN))
    {
        if(server_packet)
        {
            sptr->server_status = FIN_WAIT_1;  /* server does active close */
            sptr->client_status = CLOSE_WAIT;
        }
        else
        {
            sptr->client_status = FIN_WAIT_1; /* client does active close */
            sptr->server_status = CLOSE_WAIT;
        }
    }

    if(sptr->server_status == CLOSE_WAIT && 
            server_packet && p->tcph->th_flags & R_FIN)
    {
        sptr->server_status = LAST_ACK;
        sptr->client_status = TIME_WAIT;
    }
    if(sptr->client_status == CLOSE_WAIT && 
            !server_packet && p->tcph->th_flags & R_FIN)
    {
        sptr->server_status = TIME_WAIT;
        sptr->client_status = LAST_ACK;
    }

    return;
}

/*
 * Function:  TcpStream2DataSize(TcpStream2Session *, int )
 *
 * Purpose:  To determine how much data is currently in the session on 
 *              one side or the other (server or client)
 *
 * Arguments: sptr - a session structure, contains info on both sides of stream
 *            server - are we finding out how much data for server or client
 *
 * Returns: the amount of data in the designated buffer
 *
 */
 
 u_long TcpStream2DataSize(TcpStream2Session *sptr, int server)
{
    TcpStream2PacketData *pdata;
    u_long size;
    u_long last_seq, first_seq;
    
    
#ifdef DEBUG
    printf("Finding maximum amount of data in buffers\n");
#endif

    if(server)
    {
    	pdata = sptr->s_data;
    }
    else
    {
    	pdata = sptr->c_data;
    }
    
    if(pdata == (TcpStream2PacketData *)NULL)
    {
        return 0;
    }
    
    size = 0;
    first_seq = pdata->seq;
    last_seq = pdata->seq;
    for( ; pdata != NULL; pdata = pdata->next)
    	if(SEQ_GT(pdata->seq + (u_long)pdata->size, last_seq))
    	{
    	    last_seq = pdata->seq + (u_long)pdata->size;
    	}
    	
    size = last_seq - first_seq;
    if(size > StreamData2.maxbytes) size = StreamData2.maxbytes;

    return size;
}

/*
 * Function: TcpStream2CheckAck(TcpStream2Session *, u_long, int)
 *
 * Purpose:  To find how much data has been ACK'd on one side or the other
 *
 * Arguments: sptr - pointer to a stream structure
 *            lack - the last ACK number to come in
 *            server_packet - was the ACK from a client or server
 *
 * Returns: the amount of data that has been ACK'd
 *
 */
 
u_long TcpStream2CheckAck(TcpStream2Session *sptr, u_long lack, int server_packet)
{
    TcpStream2PacketData *pdata;
    u_long first_seq;
    u_long size;
    
#ifdef DEBUG
    printf("Finding length of packet based on ACK\n");
#endif


    if(!server_packet)  /* client ack */
    {
    	if(sptr->s_data == (TcpStream2PacketData *)NULL)
    	{
    	    return 0;
    	}
    	first_seq = sptr->s_data->seq;
    	if( SEQ_GE(first_seq, lack) ) return 0;
    	
    	/* find length based on ACK:
    	 *   0        - no LFCR and (ACK - first_seq) < maxbytes
    	 *   size     - LFCR before ACK, last LFCR
    	 *   maxbytes - no LFCR and (ACK - first_seq) >= maxbytes
    	 */
    	 
    	size = 0;
    	for(pdata=sptr->s_data; pdata != NULL; pdata=pdata->next)
    	{
#ifdef DEBUG
	    printf("   size: %lu, packet's lfcr: %i\n",size,pdata->lfcr);
#endif
    	    if(pdata->lfcr > -1  && SEQ_LT((u_long)pdata->lfcr+pdata->seq, lack) )
    	    {
    	    	if((u_long)pdata->lfcr + 1 + (pdata->seq - first_seq) > StreamData2.maxbytes)
    	    	{
    	    		if(size == 0)
    	    		{
    	    			size = StreamData2.maxbytes;
    	    		}
    	    		return size;
    	    	}
    	    	size = (u_long)pdata->lfcr + 1 + (pdata->seq - first_seq);
    	    }
    	    if(SEQ_GE(pdata->seq, lack)) break;
    	}
#ifdef DEBUG
    	printf("   %lX - %lX = %lX\n",lack,first_seq,lack-first_seq);
#endif
    	if(size == 0 && lack - first_seq >= StreamData2.maxbytes) size = StreamData2.maxbytes;
#ifdef DEBUG
    	printf("   size: %lu\n",size);
#endif
    	return size;
    }
    else                /* server ack */
    {
    	if(sptr->c_data == (TcpStream2PacketData *)NULL)
    	{
    	    return 0;
    	}    	
    	
    	first_seq = sptr->c_data->seq;
    	if( SEQ_GE(first_seq, lack) ) return 0;
    	
    	/* find length based on ACK:
    	 *   0        - no LFCR and (ACK - first_seq) < maxbytes
    	 *   size     - LFCR before ACK, last LFCR
    	 *   maxbytes - no LFCR and (ACK - first_seq) >= maxbytes
    	 */
    	 
    	size = 0;
    	for(pdata=sptr->c_data; pdata != NULL; pdata=pdata->next)
    	{
#ifdef DEBUG
	    printf("   size: %lu, packet's lfcr: %i\n",size,pdata->lfcr);
#endif
    	    if(pdata->lfcr > -1  && SEQ_LT((u_long)pdata->lfcr+pdata->seq, lack) )
    	    {
    	    	if((u_long)pdata->lfcr + 1 + (pdata->seq - first_seq) > StreamData2.maxbytes)
    	    	{
    	    		if(size == 0)
    	    		{
    	    			size = StreamData2.maxbytes;
    	    		}
    	    		return size;
    	    	}
    	    	size = (u_long)pdata->lfcr + 1 + (pdata->seq - first_seq);
    	    }
    	    if(SEQ_GE(pdata->seq, lack) ) break;
    	}
    	
#ifdef DEBUG
    	printf("   %lX - %lX = %lX\n",lack,first_seq,lack-first_seq);
#endif
    	if(size == 0 && lack - first_seq >= StreamData2.maxbytes) size = StreamData2.maxbytes;
#ifdef DEBUG
    	printf("   size: %lu\n",size);
#endif
    	
    	return size;
    }
}    

/*
 * Function: TcpStream2StoreData(TcpStream2Session *, Packet *, int)
 *
 * Purpose:  Store newly arriving data in the stream buffer structure
 *
 * Arguments: sptr - pointer to a stream buffer structure
 *            p - pointer to a packet
 *            server_packet - is the packet client or server?
 *
 * Returns: nothing
 *
 */
 
void TcpStream2StoreData(TcpStream2Session * sptr, Packet * p, int server_packet)
{

    /* do we need a +1 on seq if SYN or FIN? */
    
    TcpStream2PacketData *pdata, *prev, *cur;
    u_long seq;
    int i;
    
#ifdef DEBUG
    printf("pulling data from packet\n");
#endif

    if(p->dsize == 0)
    {
#ifdef DEBUG
    	printf("No data in packet\n");
#endif
    	return;
    }
    
  
    seq = ntohl(p->tcph->th_seq);
    
    /* need to see if we should deal w/ this data */
    
    if(!server_packet && SEQ_LT(seq, sptr->c_last_ack) )
    {
#ifdef DEBUG
	printf("Getting old client data, seq: %lX, c_last_ack: %lX\n",seq, sptr->c_last_ack);
#endif
    	return;
    }
    else if(server_packet && SEQ_LT(seq, sptr->s_last_ack) )
    {
#ifdef DEBUG
	printf("Getting old server data, seq: %lX, s_last_ack: %lX\n",seq, sptr->s_last_ack);
#endif
    	return;
    }
    else if(!server_packet && seq + (u_long)p->dsize - sptr->c_last_ack > sptr->s_win)
    {
#ifdef DEBUG
	printf("Getting too much client data\n");
#endif
    	return;
    }
    else if(server_packet && seq + (u_long)p->dsize - sptr->s_last_ack > sptr->c_win)
    {
#ifdef DEBUG
	printf("Getting too much server data\n");
#endif
    	return;
    }
    
    pdata = malloc(sizeof(TcpStream2PacketData));
    if(!pdata)
    {
    	printf("Not enough memory!\n");
    	exit(1);
    }
    pdata->seq = seq;
    pdata->size = p->dsize;
    pdata->data = (u_char *)malloc(pdata->size);
    if(!pdata->data)
    {
    	printf("Not enough memory!\n");
    	exit(1);
    }
    pdata->lfcr = -1;
    pdata->next = (TcpStream2PacketData *)NULL;
    memcpy(pdata->data, p->data, pdata->size);

    for(i=0; (u_long)(i)<pdata->size; i++)
    {
    	if(pdata->data[i] == 0xa || pdata->data[i] == 0xd) pdata->lfcr = i;
    }

    if(server_packet)
    {
    	prev = (TcpStream2PacketData *)NULL;
    	cur = sptr->s_data;
    	
    	while (cur != (TcpStream2PacketData *)NULL && SEQ_GE(pdata->seq, cur->seq) )
    	{
    	    prev = cur;
    	    cur = cur->next;
    	}
    	if (prev == (TcpStream2PacketData *)NULL)
    	{
#ifdef DEBUG
	    printf("   putting in first location\n");
#endif
    	    pdata->next = cur;
    	    sptr->s_data = pdata;
    	}
    	else 
    	{
#ifdef DEBUG
	    printf("   putting in middle (or end)\n");
#endif
    	    pdata->next = cur;
    	    prev->next = pdata;
    	}
    }
    else
    {
    	prev = (TcpStream2PacketData *)NULL;
    	cur = sptr->c_data;
    	while (cur != (TcpStream2PacketData *)NULL && SEQ_GE(pdata->seq, cur->seq) )
    	{
    	    prev = cur;
    	    cur = cur->next;
    	}
    	if (prev == (TcpStream2PacketData *)NULL)
    	{
#ifdef DEBUG
	    printf("   putting in first location\n");
#endif
    	    pdata->next = cur;
    	    sptr->c_data = pdata;
    	}
    	else 
    	{
#ifdef DEBUG
	    printf("   putting in middle (or end)\n");
#endif
    	    pdata->next = cur;
    	    prev->next = pdata;
    	}
    }
}

/*
 * Function: TcpStream2Time()
 *
 * Purpose:  what can i say, i like having my own timing routines
 *
 * Arguments: none
 *
 * Returns: u_long number of seconds since beginning of time (unix-centric)
 *
 */
u_long TcpStream2Time()
{
    struct timeval t;
    gettimeofday(&t, (struct timezone *)NULL);
    return(t.tv_sec);
}

/*
 * Function: TcpStream2CheckSession(Packet *, int sp)
 *
 * Purpose:  return the TcpStream2Session of current packet (if any) w/ 
 * respect to server port sp
 *
 *
 * Arguments: 
 *
 * Returns: appropriate TcpStream2Session (or null)
 *
 */
TcpStream2Session * TcpStream2CheckSession(Packet *p, int sp, int *bin)
{
    TcpStream2Session *sptr;
    u_char hash;
    u_char *ptr;
    int i;

    for(ptr=(u_char *)&p->iph->ip_src, i=0, hash=0; i<8; i++,ptr++)
        hash += *ptr;  
    *bin = hash;

#ifdef DEBUG
    printf("HASH: %x\n", hash);
#endif

    if(sp == p->sp)
    {  /* this is coming from the server */
        for(sptr=StreamData2.heads[hash]; sptr!=NULL; sptr=sptr->next)
        {
            if(sptr->sip == p->iph->ip_src.s_addr && sptr->cip == p->iph->ip_dst.s_addr && sptr->sp == sp && sptr->cp == p->dp)
            {
                return sptr;
            }
        }
    }
    else
    {  /* data from the client */
        for(sptr=StreamData2.heads[hash]; sptr!=NULL; sptr=sptr->next)
        {
            if(sptr->sip == p->iph->ip_dst.s_addr && sptr->cip == p->iph->ip_src.s_addr && sptr->sp == sp && sptr->cp == p->sp)
            {
                return sptr;
            }
        }
    }
    return(TcpStream2Session *)NULL;
}


/*
 * Function: TcpStream2PruneSessions()
 *
 * Purpose:  kill all inactive sessions
 *
 * Arguments: none
 *
 * Returns: void function
 *
 */
void TcpStream2PruneSessions()
{
    TcpStream2Session * sptr = NULL;
    TcpStream2Session * lastptr, *tmpptr;
    u_long tm;
    int i;
    struct pseudoheader ph;    /* pseudo header declaration */
    grinder_t tmp_grinder;
    int psize;
    
#ifdef DEBUG
    int prunecount = 0;
    int remain=0;
    int orig=0;
#endif
    tmp_grinder = grinder;
    grinder = DecodeNullPkt;
    pv.fake_packet_flag = 1;

#ifdef DEBUG
    for(i=0;i<256;i++)
        for(sptr=StreamData2.heads[i]; sptr!=NULL; sptr=sptr->next)
        {
            orig++;
        }

    printf("Trying to PRUNE!   %i streams\n",orig);
#endif

    tm = TcpStream2Time();

    for(i=0;i<256;i++)
    {
        for(sptr=StreamData2.heads[i], lastptr=NULL; sptr!=NULL;)
        {
            if(tm - sptr->timestamp >= StreamData2.prunetime)
            {
                tmpptr = sptr->next;

                if(lastptr == NULL)
                    StreamData2.heads[i] = sptr->next;
                else
                    lastptr->next = sptr->next;

                if(sptr != NULL)
                {
                	psize = TcpStream2DataSize(sptr,1);
		    while(psize > 0) 
                    {
                        pc.rebuilt_tcp ++;  /* new phoney packet */
                        dummy->pkth->caplen = dummy->pkth->len = hdrsize + psize;
                        dummy->iph->ip_src.s_addr = sptr->sip;
                        dummy->iph->ip_dst.s_addr = sptr->cip;
                        dummy->iph->ip_len = htons( (u_short) (40 + psize));
                        dummy->iph->ip_csum = checksum((u_short *)dummy->iph, 20, (u_short *)NULL, 0);
                        dummy->tcph->th_sport = sptr->sp;
                        dummy->tcph->th_dport = sptr->cp;
                        ph.sip = (u_int32_t)(dummy->iph->ip_src.s_addr);
                        ph.dip = (u_int32_t)(dummy->iph->ip_dst.s_addr);
                        ph.zero = 0;
                        ph.protocol = dummy->iph->ip_proto;
                        ph.tcplen = htons( (u_short) (20 + psize));
				dummy->tcph->th_seq = TcpStream2FillBuffer(sptr, (u_char *)dummy->pkth+hdrsize+pcaplen, psize, 1);
				dummy->tcph->th_ack = sptr->c_last_ack;
				dummy->tcph->th_win = htons( (u_short) (sptr->s_win));
                        dummy->tcph->th_sum = checksum((u_int16_t *)&ph, 12, (u_int16_t *)(dummy->tcph), 20 + psize);
                        ProcessPacket(NULL, dummy->pkth, dummy->pkt);
                        psize = TcpStream2DataSize(sptr,1);
                    }

                    psize = TcpStream2DataSize(sptr,0);
                    while(psize > 0)
                    {
                        pc.rebuilt_tcp ++;  /* new phoney packet */
                        dummy->pkth->caplen = dummy->pkth->len = hdrsize + psize;
                        dummy->iph->ip_src.s_addr = sptr->cip;
                        dummy->iph->ip_dst.s_addr = sptr->sip;
                        dummy->iph->ip_len = htons( (u_short) (40 + psize));
                        dummy->iph->ip_csum = checksum((u_short *)dummy->iph, 20, (u_short *)NULL, 0);
                        dummy->tcph->th_sport = sptr->cp;
                        dummy->tcph->th_dport = sptr->sp;
                        ph.sip = (u_int32_t)(dummy->iph->ip_src.s_addr);
                        ph.dip = (u_int32_t)(dummy->iph->ip_dst.s_addr);
                        ph.zero = 0;
                        ph.protocol = dummy->iph->ip_proto;
                        ph.tcplen = htons( (u_short) (20 + psize));
				dummy->tcph->th_seq = TcpStream2FillBuffer(sptr, (u_char *)dummy->pkth+hdrsize+pcaplen, psize, 0);
				dummy->tcph->th_ack = sptr->s_last_ack;
				dummy->tcph->th_win = htons( (u_short) (sptr->c_win));
                        dummy->tcph->th_sum = checksum((u_int16_t *)&ph, 12, (u_int16_t *)(dummy->tcph), 20 + psize);
                        ProcessPacket(NULL, dummy->pkth, dummy->pkt);
                        psize = TcpStream2DataSize(sptr,0);
                    }

                    free(sptr);

#ifdef DEBUG
                    prunecount++;
#endif
                }

                sptr = tmpptr;
            }
            else
            {
                lastptr = sptr;
                sptr = sptr->next;
            }
        }
    }

#ifdef DEBUG
    for(i=0;i<256;i++)
    {
        for(sptr=StreamData2.heads[i]; sptr!=NULL; sptr=sptr->next)
        {
            remain++;
        }
    }

    printf("Done PRUNING!  %i deleted, %i remain\n",prunecount,remain);
#endif
    grinder = tmp_grinder;
    pv.fake_packet_flag = 0;
    return;
}

void TcpStream2Packetize(TcpStream2Session *sptr, Packet *pb, u_long psize, int server_packet)
{
    struct pseudoheader ph;    /* pseudo header declaration */
    grinder_t tmp_grinder;

#ifdef DEBUG
    printf("Trying to make a new packet!  psize: %lu\n", psize);
#endif

    pc.rebuilt_tcp ++;  /* new phoney packet */

    dummy->pkth->ts.tv_sec = pb->pkth->ts.tv_sec;
    dummy->pkth->ts.tv_usec = pb->pkth->ts.tv_usec;
    dummy->pkth->caplen = dummy->pkth->len = hdrsize + psize;
    dummy->iph->ip_src.s_addr = pb->iph->ip_dst.s_addr;
    dummy->iph->ip_dst.s_addr = pb->iph->ip_src.s_addr;
    dummy->iph->ip_len = htons( (u_short) (40 + psize));
    dummy->iph->ip_csum = checksum((u_short *)dummy->iph, 20, (u_short *)NULL, 0);
    dummy->tcph->th_sport = pb->tcph->th_dport;
    dummy->tcph->th_dport = pb->tcph->th_sport;
    ph.sip = (u_int32_t)(pb->iph->ip_dst.s_addr);
    ph.dip = (u_int32_t)(dummy->iph->ip_dst.s_addr);
    ph.zero = 0;
    ph.protocol = dummy->iph->ip_proto;
    ph.tcplen = htons( (u_short) (20 + psize));
    dummy->tcph->th_seq = TcpStream2FillBuffer(sptr, (u_char *)dummy->pkth+hdrsize+pcaplen, psize, !server_packet);
    if(!server_packet)
    {
	dummy->tcph->th_ack = sptr->s_last_ack;
	dummy->tcph->th_win = htons( (u_short) (sptr->c_win));
    }
    else
    {
	dummy->tcph->th_ack = sptr->c_last_ack;
	dummy->tcph->th_win = htons( (u_short) (sptr->s_win));
    }
    dummy->tcph->th_sum = checksum((u_int16_t *)&ph, 12, (u_int16_t *)(dummy->tcph), 20 + psize);
    tmp_grinder = grinder;
    grinder = DecodeNullPkt;
    pv.fake_packet_flag = 1;
    ProcessPacket(NULL, dummy->pkth, dummy->pkt);
    pv.fake_packet_flag = 0;
    grinder = tmp_grinder;
}



/*
 * Function: TcpStream2FillBuffer(TcpStream2Session *, u_char *, u_long, int)
 *
 * Purpose:  Fill a data buffer with data in the stream buffer structure
 *
 * Arguments: sptr - pointer to a stream buffer structure
 *            buf - the buffer in which to put reconstructed data
 *            psize - how many bytes to fill in
 *            server - grab data from server or client size
 *
 * Returns: the sequence number of the first byte in the buffer
 *
 */
 
u_long TcpStream2FillBuffer(TcpStream2Session *sptr, u_char * buf, u_long psize, int server)
{
    TcpStream2PacketData *pdata, *tmp;
    u_long first_seq;
    int i, pos;
    
#ifdef DEBUG
    printf("In fill buffer on %s side with size: %lu\n",server?"server":"client",psize);
#endif

    /* we really need to 0 out the buffer in case we have gaps */
    
    memset(buf,0,psize);
    if(server)
    {
    	pdata = sptr->s_data;
    }
    else
    {
    	pdata = sptr->c_data;
    }
    
    first_seq = pdata->seq;
    while (pdata != (TcpStream2PacketData *)NULL)
    {
        if(pdata->seq - first_seq > psize)
        {
    	    break;
    	}

        if(pdata->seq + pdata->size - first_seq < psize)  /* whole packet */
        {
            memcpy(buf + pdata->seq - first_seq, pdata->data, pdata->size);

       	    if(server)
    	    	sptr->s_data = pdata->next;
    	    else
    	    	sptr->c_data = pdata->next;
    	    tmp = pdata;
    	    pdata = pdata->next;
    	    free(tmp->data);
    	    free(tmp);
    	}
    	else    /* part of packet */
    	{
    	    pos = psize - pdata->seq + first_seq;
    	    memcpy(buf + pdata->seq - first_seq, pdata->data, pos);
    	    pdata->lfcr = -1;
    	    for(i=pos; ((u_long)i)<pdata->size; i++)
    	    {
    	    	pdata->data[i-pos] = pdata->data[i];
	    	if(pdata->data[i-pos] == 0xa || pdata->data[i-pos] == 0xd) pdata->lfcr = i-pos;
    	    }	
    	    pdata->size = pdata->size - pos;
    	    pdata->seq += pos;
    	    pdata = pdata->next;
    	}
    }
    return first_seq;
}


/*
 * Function: TcpStream2Exit(int, void *)
 *
 * Purpose:  stuff to do when we've been told to exit, including dump all data in 
 *           buffer to new packets
 *
 * Arguments: signal - the signal that kills us
 *            arg - arguments
 *
 * Returns: nothing
 *
 */
 
void TcpStream2Exit(int signal, void *arg)
{
    TcpStream2Session * sptr = NULL;
    TcpStream2Session * lastptr, *tmpptr;
    int i;
    struct pseudoheader ph;    /* pseudo header declaration */
    grinder_t tmp_grinder;
    u_long psize;

    tmp_grinder = grinder;
    grinder = DecodeNullPkt;
    pv.fake_packet_flag = 1;

    for(i=0;i<256;i++)
    {
        for(sptr=StreamData2.heads[i], lastptr=NULL; sptr!=NULL;)
        {
            tmpptr = sptr->next;

            if(lastptr == NULL)
                StreamData2.heads[i] = sptr->next;
            else
                lastptr->next = sptr->next;

            if(sptr != NULL)
            {
                psize = TcpStream2DataSize(sptr, 1);
                if(psize > 0)
                {
                    pc.rebuilt_tcp ++;  /* new phoney packet */
                    dummy->pkth->caplen = dummy->pkth->len = hdrsize + psize;
                    dummy->iph->ip_src.s_addr = sptr->sip;
                    dummy->iph->ip_dst.s_addr = sptr->cip;
                    dummy->iph->ip_len = htons( (u_short) (40 + psize));
                    dummy->iph->ip_csum = checksum((u_short *)dummy->iph, 20, (u_short *)NULL, 0);
                    dummy->tcph->th_sport = htons(sptr->sp);
                    dummy->tcph->th_dport = htons(sptr->cp);
                    ph.sip = (u_int32_t)(dummy->iph->ip_src.s_addr);
                    ph.dip = (u_int32_t)(dummy->iph->ip_dst.s_addr);
                    ph.zero = 0;
                    ph.protocol = dummy->iph->ip_proto;
                    ph.tcplen = htons( (u_short) (20 + psize));
                    dummy->tcph->th_seq = TcpStream2FillBuffer(sptr, (u_char *)dummy->pkth+hdrsize+pcaplen, psize, 1);
			dummy->tcph->th_ack = sptr->c_last_ack;
			dummy->tcph->th_win = htons( (u_short) (sptr->s_win));
                    dummy->tcph->th_sum = checksum((u_int16_t *)&ph, 12, (u_int16_t *)(dummy->tcph), 20 + psize);
                    ProcessPacket(NULL, dummy->pkth, dummy->pkt);
                }

                psize = TcpStream2DataSize(sptr, 0);
                if(psize > 0)
                {
                    pc.rebuilt_tcp ++;  /* new phoney packet */
                    dummy->pkth->caplen = dummy->pkth->len = hdrsize + psize;
                    dummy->iph->ip_src.s_addr = sptr->cip;
                    dummy->iph->ip_dst.s_addr = sptr->sip;
                    dummy->iph->ip_len = htons( (u_short) (40 + psize));
                    dummy->iph->ip_csum = checksum((u_short *)dummy->iph, 20, (u_short *)NULL, 0);
                    dummy->tcph->th_sport = htons(sptr->cp);
                    dummy->tcph->th_dport = htons(sptr->sp);
                    ph.sip = (u_int32_t)(dummy->iph->ip_src.s_addr);
                    ph.dip = (u_int32_t)(dummy->iph->ip_dst.s_addr);
                    ph.zero = 0;
                    ph.protocol = dummy->iph->ip_proto;
                    ph.tcplen = htons( (u_short) (20 + psize));
                    dummy->tcph->th_ack = TcpStream2FillBuffer(sptr, (u_char *)dummy->pkth+hdrsize+pcaplen, psize, 0);
			dummy->tcph->th_ack = sptr->s_last_ack;
			dummy->tcph->th_win = htons( (u_short) (sptr->c_win));
                    dummy->tcph->th_sum = checksum((u_int16_t *)&ph, 12, (u_int16_t *)(dummy->tcph), 20 + psize);
                    ProcessPacket(NULL, dummy->pkth, dummy->pkt);
                }

                free(sptr);

            }

            sptr = tmpptr;
        }
    }
    grinder = tmp_grinder;
    pv.fake_packet_flag = 0;

    /* we're really done w/ dummy, free it */ 
    if( dummy != NULL )
    {
        free(dummy->pkth);
        free(dummy);
    }

    dummy = NULL;

}


/*
 * Function: TcpStream2Restart(int, void *)
 *
 * Purpose:  What do we do when sent a restart? - nothing
 *
 * Arguments: signal - what signal generated this
 *            arg - arguments
 *
 * Returns: nothing
 *
 */
 
void TcpStream2Restart(int signal, void *arg)
{

}


/*
 * Function: TcpStream2PrintDataList(TcpStream2Session *, int)
 *
 * Purpose:  A debug function to print info on data in a buffer
 *
 * Arguments: sptr - a pointer to a stream buffer structure
 *            server - printer server or client data
 *
 * Returns: nothing
 *
 */
 
void TcpStream2PrintDataList(TcpStream2Session *sptr, int server)
{
    TcpStream2PacketData *cur;
    int i;
    
    if(server)
    {
    	cur = sptr->s_data;
    }
    else
    {
    	cur = sptr->c_data;
    }
    
    printf("Data list for: %s\n",server?"server":"client");
    for (i=0 ; cur != (TcpStream2PacketData *)NULL; cur = cur->next, i++)
    {
    	printf("  %i: %lX(%lu)\n",i,cur->seq,cur->size);
    }
    printf("\n");
}

