/*
 * Resource manager daemon
 *
 * Copyright (C) 2001-2002, Olaf Kirch <okir@lst.de>
 */

#include "resmgrd.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/signal.h>
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>
#include <stdio.h>
#include <errno.h>
#include <pwd.h>
#include <grp.h>

#define _PATH_CONSOLE	"/dev/console"

static int	getpeercred(int, struct ucred *);

void
svc_start(struct svc *svc, const char *sockpath)
{
	socklen_t	len;
	int		on = 1;

	memset(svc, 0, sizeof(*svc));

	if (sockpath[0] == '\0')
		fatal("no unix socket path specified");
	if (strlen(sockpath) >= sizeof(svc->sun.sun_path))
		fatal("path too long: %s", sockpath);

	strcpy(svc->sun.sun_path, sockpath);
	svc->sun.sun_family = AF_UNIX;

	svc->fd = socket(PF_UNIX, SOCK_STREAM, 0);
	if (svc->fd < 0)
		fatal("unable to create UNIX socket: %m");

	if (setsockopt(svc->fd, SOL_SOCKET, SO_PASSCRED, &on, sizeof(on)) < 0)
		fatal("setsockopt failed: %m");

	len = SUN_LEN(&svc->sun);
	unlink(svc->sun.sun_path);
	if (bind(svc->fd, (struct sockaddr *) &svc->sun, len) < 0)
		fatal("unable to bind to %s: %m", svc->sun.sun_path);

	if (chmod(svc->sun.sun_path, 0666) < 0)
		fatal("unable to fchmod %s: %m", svc->sun.sun_path);

	if (listen(svc->fd, 10) < 0)
		fatal("unable to listen on %s: %m", svc->sun.sun_path);
}

struct conn *
svc_accept(struct svc *svc)
{
	struct conn	*conn;
	char		*reason;
	struct passwd	*pwd;
	struct group	*grp;

	conn = (struct conn *) calloc(1, sizeof(*conn));
	if (conn == NULL)
		return NULL;
	conn->fd = -1;
	conn->passfd = -1;

	conn->fd = accept(svc->fd, NULL, 0);
	if (conn->fd < 0)
		goto close;

	if (fcntl(conn->fd, F_SETFL, O_NONBLOCK) < 0)
		goto close;

	if (getpeercred(conn->fd, &conn->cred) < 0) {
		log("getpeercred failed: %m");
		goto close;
	}

	reason = "user doesn't exist";
	if ((pwd = getpwuid(conn->cred.uid)) == NULL)
		goto reject;

	reason = "user name too long";
	if (strlen(pwd->pw_name) >= sizeof(conn->user))
		goto reject;
	strcpy(conn->user, pwd->pw_name);
	endpwent();

	reason = "group doesn't exist";
	if ((grp = getgrgid(conn->cred.gid)) == NULL)
		goto reject;

	reason = "group name too long";
	if (strlen(grp->gr_name) >= sizeof(conn->group))
		goto reject;
	strcpy(conn->group, grp->gr_name);
	endgrent();

#if 0
	reason = "not allowed to connect";
	if (!rsm_check(conn, "watch"))
		goto reject;
#endif

	if (opt_debug)
		conn->debug = 1;

	return conn;

reject:
	log("rejected connection from uid %d/gid %d: %s",
	    conn->cred.uid, conn->cred.gid, reason);
close:
	rsm_close(conn);
	return NULL;
}

static int
getpeercred(int fd, struct ucred *cred)
{
	socklen_t	len = sizeof(*cred);

	return getsockopt(fd, SOL_SOCKET, SO_PEERCRED, cred, &len);
}
