#include <stdio.h>
#include <string.h>
#include <sys/types.h>

#include <config.h>
#include <support.h>

#include "option.h"
#include "log.h"
#include "timer.h"
#include "fsm.h"

static struct {
    char *name;
    u_int8_t
	flag_to:1;	/* The state should timeout */
} fsmState[]={
    {"Initial", 0},
    {"Starting", 0},
    {"Closed", 0},
    {"Stopped", 0},
    {"Closing", 1},
    {"Stopping", 1},
    {"Req-Sent", 1},
    {"Ack-Rcvd", 1},
    {"Ack-Sent", 1},
    {"Opened", 0},
    {"Unexpected", 0}
};

static void
FsmSetState(struct fsmreg_s *fsm, char *ev, fsmstate_t st)
{
    if (st == FSM_KEEP) st = fsm->state;
    if (ISLOG(LOG_FSM))
	Logf(LOG_FSM, "%s: %s -[%s]-> %s\n", fsm->name,
	     fsmState[fsm->state].name, ev, fsmState[st].name);
    if (st != FSM_UNEXP) fsm->state = st;
    fsm->timer.st_paused = fsmState[fsm->state].flag_to ? 0: 1;
}

void
FsmInit(struct fsmreg_s *fsm)
{
    fsm->state = FSM_INITIAL;
    fsm->id = 0;
    fsm->timer.rest = fsm->init_r_to;
    fsm->r_count = fsm->init_r_count;
    fsm->opt_r = fsm->opt_p = fsm->nak_rcr = fsm->rej_rcr = 0;
    fsm->opt_x = 1;
    fsm->timer.callback = FsmTO;
    fsm->timer.st_paused = 1;
}

/* FSM action functions */

#define	FsmAction(f,a)	{if (f->a) f->a(f);}

static void
FsmDRC(struct fsmreg_s *fsm)
{
    fsm->r_count --;
    fsm->timer.rest = fsm->init_r_to;
    fsm->timer.st_expired = 0;
}

static void
FsmIRC(struct fsmreg_s *fsm)
{
    fsm->timer.rest = fsm->init_r_to;
    fsm->r_count = fsm->init_r_count;
    TimerAdd(&(fsm->timer));
}

/* FSM event functions */

int
FsmUp(struct fsmreg_s *fsm)
{
    fsmstate_t new=FSM_KEEP;

    switch(fsm->state) {
    case FSM_INITIAL:
	new = FSM_CLOSED;
	break;
    case FSM_STARTING:
	FsmIRC(fsm);	/* FsmAction(fsm, irc) */
	FsmAction(fsm, scr);
	new = FSM_REQSENT;
	break;
    default:
	new = FSM_UNEXP;
	break;
    }
    FsmSetState(fsm, "Up", new);
    return(new == FSM_UNEXP);
}

int
FsmDown(struct fsmreg_s *fsm)
{
    fsmstate_t new=FSM_KEEP;

    switch(fsm->state) {
    case FSM_CLOSED:
    case FSM_CLOSING:
	new = FSM_INITIAL;
	break;
    case FSM_OPENED:
	FsmAction(fsm,tld);
    case FSM_STOPPING:
    case FSM_REQSENT:
    case FSM_ACKRCVD:
    case FSM_ACKSENT:
	new = FSM_CLOSED;
	break;
    case FSM_STOPPED:
	FsmAction(fsm,tls);
	new = FSM_CLOSED;
	break;
    default:
	new = FSM_UNEXP;
	break;
    }
    FsmSetState(fsm, "Down", new);
    return(new == FSM_UNEXP);
}

int
FsmOpen(struct fsmreg_s *fsm)
{
    fsmstate_t new=FSM_KEEP;

    switch(fsm->state) {
    case FSM_INITIAL:
	FsmAction(fsm, tls);
    case FSM_STARTING:
    case FSM_REQSENT:
    case FSM_ACKRCVD:
    case FSM_ACKSENT:
	break;
    case FSM_CLOSED:
	if (fsm->opt_p) {
	    new = FSM_STOPPED;
	    break;
	}
	FsmIRC(fsm);	/* FsmAction(fsm, irc) */
	FsmAction(fsm, scr);
	new = FSM_REQSENT;
	break;
    case FSM_CLOSING:
	new = FSM_STOPPING;
    case FSM_STOPPED:
    case FSM_STOPPING:
    case FSM_OPENED:
	if (fsm->opt_r) {
	    FsmDown(fsm);
	    FsmUp(fsm);
	}
	break;
    default:
	new = FSM_UNEXP;
	break;
    }
    FsmSetState(fsm, "Open", new);
    return(new == FSM_UNEXP);
}

int
FsmClose(struct fsmreg_s *fsm)
{
    fsmstate_t new=FSM_KEEP;

    switch(fsm->state) {
    case FSM_INITIAL:
    case FSM_STARTING:
	new = FSM_INITIAL;
	break;
    case FSM_CLOSED:
    case FSM_STOPPED:
	new = FSM_CLOSED;
	break;
    case FSM_OPENED:
	FsmAction(fsm, tld);
    case FSM_REQSENT:
    case FSM_ACKRCVD:
    case FSM_ACKSENT:
	FsmIRC(fsm);	/* FsmAction(fsm, irc) */
	FsmAction(fsm, str);
    case FSM_CLOSING:
    case FSM_STOPPING:
	new = FSM_CLOSING;
	break;
    default:
	new = FSM_UNEXP;
	break;
    }
    FsmSetState(fsm, "Close", new);
    return(new == FSM_UNEXP);
}

void
FsmTO(struct fsmreg_s *fsm)
{
    fsmstate_t new=FSM_KEEP;

    switch(fsm->state) {
    case FSM_CLOSING:
	if (fsm->r_count > 0) {
	    FsmDRC(fsm);
	    FsmAction(fsm, str);
	} else {
	    FsmAction(fsm, tlf);
	    new = FSM_CLOSED;
	}
	break;
    case FSM_STOPPING:
	if (fsm->r_count > 0) {
	    FsmDRC(fsm);
	    FsmAction(fsm, str);
	} else {
	    FsmAction(fsm, tlf);
	    new = FSM_STOPPED;
	}
	break;
    case FSM_REQSENT:
    case FSM_ACKRCVD:
    case FSM_ACKSENT:
	if (fsm->r_count > 0) {
	    FsmDRC(fsm);
	    FsmAction(fsm, scr);
	    new = (fsm->state == FSM_ACKSENT)
		? FSM_ACKSENT: FSM_REQSENT;
	} else {
/*	    if (fsm->opt_p) FsmAction(fsm, tlf);*/
	    FsmAction(fsm, tlf);
	    new = FSM_STOPPED;
	}
	break;
    default:
	new = FSM_UNEXP;
	break;
    }
    FsmSetState(fsm, "TO", new);
}

int
FsmRCR(struct fsmreg_s *fsm)
{
    fsmstate_t new=FSM_KEEP;

    switch(fsm->state) {
    case FSM_CLOSED:
	FsmAction(fsm, sta);
	break;
    case FSM_STOPPED:
	FsmIRC(fsm);	/* FsmAction(fsm, irc) */
	FsmAction(fsm, scr);
	if (fsm->nak_rcr || fsm->rej_rcr) {
	    FsmAction(fsm, scn);
	    new = FSM_REQSENT;
	} else {
	    FsmAction(fsm, sca);
	    new = FSM_ACKSENT;
	}
	break;
    case FSM_CLOSING:
    case FSM_STOPPING:
	break;
    case FSM_REQSENT:
	if (fsm->nak_rcr || fsm->rej_rcr) {
	    FsmAction(fsm, scn);
	} else {
	    FsmAction(fsm, sca);
	    new = FSM_ACKSENT;
	}
	break;
    case FSM_ACKRCVD:
	if (fsm->nak_rcr || fsm->rej_rcr) {
	    FsmAction(fsm, scn);
	} else {
	    FsmAction(fsm, sca);
	    FsmAction(fsm, tlu);
	    new = FSM_OPENED;
	}
	break;
    case FSM_ACKSENT:
	if (fsm->nak_rcr || fsm->rej_rcr) {
	    FsmAction(fsm, scn);
	    new = FSM_REQSENT;
	} else {
	    FsmAction(fsm, sca);
	}
	break;
    case FSM_OPENED:
	FsmAction(fsm, tld);
	FsmAction(fsm, scr);
	if (fsm->nak_rcr || fsm->rej_rcr) {
	    FsmAction(fsm, scn);
	    new = FSM_REQSENT;
	} else {
	    FsmAction(fsm, sca);
	    new = FSM_ACKSENT;
	}
	break;
    default:
	new = FSM_UNEXP;
	break;
    }
    FsmSetState(fsm,
		fsm->rej_rcr ? "RCR!": fsm->nak_rcr ? "RCR-": "RCR+",
		new);
    return(new == FSM_UNEXP);
}

int
FsmRCA(struct fsmreg_s *fsm)
{
    fsmstate_t new=FSM_KEEP;

    switch(fsm->state) {
    case FSM_CLOSED:
    case FSM_STOPPED:
	FsmAction(fsm, sta);
    case FSM_CLOSING:
    case FSM_STOPPING:
	break;
    case FSM_REQSENT:
	FsmIRC(fsm);	/* FsmAction(fsm, irc) */
	new = FSM_ACKRCVD;
	break;
    case FSM_ACKRCVD:
	if (fsm->opt_x) {
	    FsmAction(fsm, scr);
	    new = FSM_REQSENT;
	}
	break;
    case FSM_ACKSENT:
	FsmIRC(fsm);	/* FsmAction(fsm, irc) */
	FsmAction(fsm, tlu);
	new = FSM_OPENED;
	break;
    case FSM_OPENED:
	if (fsm->opt_x) {
	    FsmAction(fsm, tld);
	    FsmAction(fsm, scr);
	    new = FSM_REQSENT;
	}
	break;
    default:
	new = FSM_UNEXP;
	break;
    }
    FsmSetState(fsm, "RCA", new);
    return(new == FSM_UNEXP);
}

int
FsmRCN(struct fsmreg_s *fsm)
{
    fsmstate_t new=FSM_KEEP;

    switch(fsm->state) {
    case FSM_CLOSED:
    case FSM_STOPPED:
	FsmAction(fsm, sta);
    case FSM_CLOSING:
    case FSM_STOPPING:
	break;
    case FSM_REQSENT:
	FsmIRC(fsm);	/* FsmAction(fsm, irc) */
	FsmAction(fsm, scr);
	break;
    case FSM_ACKRCVD:
	if (fsm->opt_x) {
	    FsmAction(fsm, scr);
	    new = FSM_REQSENT;
	}
	break;
    case FSM_ACKSENT:
	FsmIRC(fsm);	/* FsmAction(fsm, irc) */
	FsmAction(fsm, scr);
	break;
    case FSM_OPENED:
	if (fsm->opt_x) {
	    FsmAction(fsm, tld);
	    FsmAction(fsm, scr);
	    new = FSM_REQSENT;
	}
	break;
    default:
	new = FSM_UNEXP;
	break;
    }
    FsmSetState(fsm, "RCN", new);
    return(new == FSM_UNEXP);
}

int
FsmRTR(struct fsmreg_s *fsm)
{
    fsmstate_t new=FSM_KEEP;

    switch(fsm->state) {
    case FSM_ACKRCVD:
    case FSM_ACKSENT:
	new = FSM_REQSENT;
    case FSM_CLOSED:
    case FSM_STOPPED:
    case FSM_CLOSING:
    case FSM_STOPPING:
    case FSM_REQSENT:
	FsmAction(fsm, sta);
	break;
    case FSM_OPENED:
	FsmAction(fsm, tld);
	fsm->r_count = 0;	/* FsmAction(fsm, zrc) */
	FsmAction(fsm, sta);
	new = FSM_STOPPING;
	break;
    default:
	new = FSM_UNEXP;
	break;
    }
    FsmSetState(fsm, "RTR", new);
    return(new == FSM_UNEXP);
}

int
FsmRTA(struct fsmreg_s *fsm)
{
    fsmstate_t new=FSM_KEEP;

    switch(fsm->state) {
    case FSM_CLOSED:
    case FSM_STOPPED:
    case FSM_REQSENT:
	break;
    case FSM_CLOSING:
	FsmAction(fsm, tlf);
	new = FSM_CLOSED;
	break;
    case FSM_STOPPING:
	FsmAction(fsm, tlf);
	new = FSM_STOPPED;
	break;
    case FSM_ACKRCVD:
	new = FSM_REQSENT;
	break;
    case FSM_ACKSENT:
	new = FSM_ACKSENT;
	break;
    case FSM_OPENED:
	FsmAction(fsm, tld);
	FsmAction(fsm, scr);
	new = FSM_REQSENT;
	break;
    default:
	new = FSM_UNEXP;
	break;
    }
    FsmSetState(fsm, "RTA", new);
    return(new == FSM_UNEXP);
}

int
FsmRUC(struct fsmreg_s *fsm)
{
    fsmstate_t new=FSM_KEEP;

    switch(fsm->state) {
    case FSM_CLOSED:
    case FSM_STOPPED:
    case FSM_CLOSING:
    case FSM_STOPPING:
    case FSM_REQSENT:
    case FSM_ACKRCVD:
    case FSM_ACKSENT:
	FsmAction(fsm, scj);
	break;
    case FSM_OPENED:
	FsmAction(fsm, tld);
	FsmAction(fsm, scj);
	FsmAction(fsm, scr);
	new = FSM_REQSENT;
	break;
    default:
	new = FSM_UNEXP;
	break;
    }
    FsmSetState(fsm, "RUC", new);
    return(new == FSM_UNEXP);
}

int
FsmRXJ(struct fsmreg_s *fsm)
{
    fsmstate_t new=FSM_KEEP;

    switch(fsm->state) {
    case FSM_CLOSED:
	if (fsm->fatal_rxj) FsmAction(fsm, tlf);
	break;
    case FSM_STOPPED:
	if (fsm->fatal_rxj) FsmAction(fsm, tlf);
	break;
    case FSM_CLOSING:
	if (fsm->fatal_rxj) {
	    FsmAction(fsm, tlf);
	    new = FSM_CLOSED;
	}
	break;
    case FSM_ACKRCVD:
	if (!fsm->fatal_rxj) {
	    new = FSM_REQSENT;
	    break;
	}
    case FSM_STOPPING:
    case FSM_REQSENT:
    case FSM_ACKSENT:
	if (fsm->fatal_rxj) {
	    FsmAction(fsm, tlf);
	    new = FSM_STOPPED;
	}
	/* should SCR? */
	break;
    case FSM_OPENED:
	if (fsm->fatal_rxj) {
	    FsmAction(fsm, tld);
	    FsmIRC(fsm);	/* FsmAction(fsm, irc) */
	    FsmAction(fsm, str);
	    new = FSM_STOPPING;
	}
	break;
    default:
	new = FSM_UNEXP;
	break;
    }
    FsmSetState(fsm, fsm->fatal_rxj ? "RXJ-": "RXJ+", new);
    return(new == FSM_UNEXP);
}

int
FsmRXR(struct fsmreg_s *fsm)
{
    fsmstate_t new=FSM_KEEP;

    switch(fsm->state) {
    case FSM_OPENED:
	FsmAction(fsm, ser);
    case FSM_CLOSED:
    case FSM_STOPPED:
    case FSM_CLOSING:
    case FSM_STOPPING:
    case FSM_REQSENT:
    case FSM_ACKRCVD:
    case FSM_ACKSENT:
	break;
    default:
	new = FSM_UNEXP;
	break;
    }
    FsmSetState(fsm, "RXR", new);
    return(new == FSM_UNEXP);
}

