/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: filterfactory.cxx,v $
 *
 *  $Revision: 1.20 $
 *
 *  last change: $Author: rt $ $Date: 2005/09/09 01:42:56 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

//_________________________________________________________________________________________________________________
//	my own includes
//_________________________________________________________________________________________________________________

#ifndef __FRAMEWORK_SERVICES_FILTERFACTORY_HXX_
#include <services/filterfactory.hxx>
#endif

#ifndef __FRAMEWORK_CLASSES_ARGUMENTANALYZER_HXX_
#include <classes/argumentanalyzer.hxx>
#endif

#ifndef __FRAMEWORK_CLASSES_CONVERTER_HXX_
#include <classes/converter.hxx>
#endif

#ifndef __FRAMEWORK_THREADHELP_TRANSACTIONGUARD_HXX_
#include <threadhelp/transactionguard.hxx>
#endif

#ifndef __FRAMEWORK_THREADHELP_READGUARD_HXX_
#include <threadhelp/readguard.hxx>
#endif

#ifndef __FRAMEWORK_THREADHELP_WRITEGUARD_HXX_
#include <threadhelp/writeguard.hxx>
#endif

#ifndef __FRAMEWORK_SERVICES_H_
#include <services.h>
#endif

#ifndef __FRAMEWORK_QUERIES_H_
#include <queries.h>
#endif

//_________________________________________________________________________________________________________________
//	interface includes
//_________________________________________________________________________________________________________________

#ifndef _COM_SUN_STAR_UNO_ANY_HXX_
#include <com/sun/star/uno/Any.hxx>
#endif

//_________________________________________________________________________________________________________________
//	includes of other projects
//_________________________________________________________________________________________________________________

//_________________________________________________________________________________________________________________
//	namespace
//_________________________________________________________________________________________________________________

namespace framework{

using namespace ::com::sun::star::beans					;
using namespace ::com::sun::star::container				;
using namespace ::com::sun::star::lang					;
using namespace ::com::sun::star::registry				;
using namespace ::com::sun::star::uno					;
using namespace ::com::sun::star::util					;
using namespace ::cppu									;
using namespace ::osl									;
using namespace ::rtl									;

//_________________________________________________________________________________________________________________
//	non exported const
//_________________________________________________________________________________________________________________

//_________________________________________________________________________________________________________________
//	non exported definitions
//_________________________________________________________________________________________________________________

//_________________________________________________________________________________________________________________
//	declarations
//_________________________________________________________________________________________________________________

//*****************************************************************************************************************
//	constructor
//*****************************************************************************************************************
FilterFactory::FilterFactory( const Reference< XMultiServiceFactory >& xFactory )
		//	Init baseclasses first
        :   ThreadHelpBase      (                                )
        ,   TransactionBase     (                                )
        ,   OWeakObject         (                                )
		// Init member
        ,   m_xFactory          ( xFactory                       )
        ,   m_aCache            (                                )
        ,   m_aListenerContainer( m_aLock.getShareableOslMutex() )
{
	// Safe impossible cases
	// a) We can't work without valid initialization.
	// b) We need our cache for working.
    LOG_ASSERT2( implcp_FilterFactory( xFactory )         , "FilterFactory::FilterFactory()", "Invalid parameter detected!"                           )
    LOG_ASSERT2( m_aCache.isValidOrRepairable()==sal_False, "FilterFactory::FilterFactory()", "Initializing of cache failed. Factory can't work so!"  )

	// Enable working mode on object!
    m_aTransactionManager.setWorkingMode( E_WORK );
}

//*****************************************************************************************************************
//	destructor
//*****************************************************************************************************************
FilterFactory::~FilterFactory()
{
	// Our cache is cleared automaticly!
	// He use a ref count to do that!
    m_aListenerContainer.clear();
}

//*****************************************************************************************************************
//	XInterface, XTypeProvider, XServiceInfo
//*****************************************************************************************************************
DEFINE_XINTERFACE_8						(	FilterFactory							,
											OWeakObject								,
											DIRECT_INTERFACE(XTypeProvider			),
											DIRECT_INTERFACE(XServiceInfo			),
											DIRECT_INTERFACE(XMultiServiceFactory	),
											DIRECT_INTERFACE(XNameContainer			),
											DIRECT_INTERFACE(XNameReplace			),
											DIRECT_INTERFACE(XNameAccess			),
											DIRECT_INTERFACE(XElementAccess			),
											DIRECT_INTERFACE(XFlushable				)
										)

DEFINE_XTYPEPROVIDER_8					(	FilterFactory							,
											XTypeProvider							,
											XServiceInfo							,
											XMultiServiceFactory					,
											XNameContainer							,
											XNameReplace							,
											XNameAccess								,
											XElementAccess							,
											XFlushable
										)

DEFINE_XSERVICEINFO_ONEINSTANCESERVICE	(	FilterFactory							,
                                            OWeakObject                             ,
											SERVICENAME_FILTERFACTORY				,
											IMPLEMENTATIONNAME_FILTERFACTORY
										)

DEFINE_INIT_SERVICE                     (   FilterFactory,
                                            {
                                            }
                                        )

//*****************************************************************************************************************
//	XMultiServiceFactory
//*****************************************************************************************************************
Reference< XInterface > SAL_CALL FilterFactory::createInstance( const OUString& sTypeName ) throw(	Exception		,
						  														 	  				RuntimeException)
{
    return createInstanceWithArguments( sTypeName, Sequence< Any >() );
}

//*****************************************************************************************************************
//	XMultiServiceFactory
//*****************************************************************************************************************
Reference< XInterface > SAL_CALL FilterFactory::createInstanceWithArguments(	const	OUString&			sTypeName	,
																				const	Sequence< Any >&	lArguments) throw(	Exception		,
																																RuntimeException)
{
	/* UNSAFE AREAY -------------------------------------------------------------------------------------------- */
	LOG_ASSERT2( implcp_createInstanceWithArguments( sTypeName, lArguments ), "FilterFactory::createInstanceWidthArguments()", "Invalid parameter detected!" )

    // Look for rejected calls.
    TransactionGuard aTransaction( m_aTransactionManager, E_HARDEXCEPTIONS );

	/* SAFE AREAY ---------------------------------------------------------------------------------------------- */
    ReadGuard aReadLock( m_aLock );

    // Hold member by references!
    FilterCache                         aCacheRef                   ;
    Reference< XMultiServiceFactory >   xFactoryRef =   m_xFactory  ;

    aReadLock.unlock();
    /* UNSAFE AREAY ---------------------------------------------------------------------------------------- */

    Sequence< PropertyValue > lProperties = Converter::convert_seqAny2seqProp( lArguments );
    ArgumentAnalyzer          aAnalyzer   ( lProperties )                                  ;
    OUString                  sFilterName                                                  ;
    Filter                    aFilter                                                      ;
    Reference< XInterface >   xFilter                                                      ;

    //-------------------------------------------------------------------------------------------------------------
    // I)   If a filter name was given by optional arguments ... use it!
    //      Caller doesn't want the default filter for this type.
    if( aAnalyzer.getArgument( E_FILTERNAME, sFilterName ) == sal_True )
    {
        aFilter = aCacheRef.getFilter( sFilterName );
        if (aFilter.sFilterService.getLength()>0)
            xFilter = Reference< XInterface >( xFactoryRef->createInstance( aFilter.sFilterService ), UNO_QUERY );
    }
    else
    //-------------------------------------------------------------------------------------------------------------
    // II)  Otherwise ... try all registered filters for these type.
    //      first come, first serve
    {
        CheckedStringListIterator   pIterator   ;
        OUString                    sFilterName ;
        while   (
                    ( aCacheRef.searchFilterForType( sTypeName, pIterator, sFilterName )    ==  sal_True    )   &&
                    ( xFilter.is()                                                          ==  sal_False   )
                )
        {
            // Try to create filter.
            aFilter = aCacheRef.getFilter( sFilterName );
            if (aFilter.sFilterService.getLength()>0)
                xFilter = Reference< XInterface >( xFactoryRef->createInstance( aFilter.sFilterService ), UNO_QUERY );
        }
    }

    // Set configuration data on filter.
    // Support given filter the new property mechanism?
    // It's a MUST!
    // If xFilter = NULL => xInit = NULL too! (Don't check xFilter again!)
    Reference< XInitialization > xInit( xFilter, UNO_QUERY );
    if( xInit.is() == sal_True )
    {
        // Yes these filter support the new mechanism.
        // Try to set configuration values on it.
        Sequence< Any > lProperties( 1 );
        lProperties[0] <<= aCacheRef.getFilterProperties( aFilter.sName );
        xInit->initialize( lProperties );
    }

	// Return result of operation.
	return xFilter;
}

//*****************************************************************************************************************
//	XMultiServiceFactory
//*****************************************************************************************************************
Sequence< OUString > SAL_CALL FilterFactory::getAvailableServiceNames() throw( RuntimeException )
{
	LOG_ERROR( "FilterFactory::getAvailableServiceNames()", "Not supported! Please use XNameContainer interface instead of these!" )
	return Sequence< OUString >();
}

//*****************************************************************************************************************
//	XNameContainer
//*****************************************************************************************************************
void SAL_CALL FilterFactory::insertByName(	const	OUString&	sFilterName			,
											const	Any&		aFilterProperties	) throw(	IllegalArgumentException	,
																								ElementExistException		,
																								WrappedTargetException		,
																								RuntimeException			)
{
	/* UNSAFE AREAY -------------------------------------------------------------------------------------------- */
    // Look for rejected calls.
    TransactionGuard aTransaction( m_aTransactionManager, E_HARDEXCEPTIONS );

    // Check arguments.
    if (sFilterName.getLength()<1)
        throw IllegalArgumentException( DECLARE_ASCII("FilterFactory::insertByName()\nInvalid parameter \"sFilterName\" detected! We don't accept NULL pointer or empty values.\n"), static_cast< OWeakObject* >( this ), 0 );

    Sequence< PropertyValue > lProperties;
    if (!(aFilterProperties >>= lProperties))
        throw IllegalArgumentException( DECLARE_ASCII("FilterFactory::insertByName()\nInvalid parameter \"aFilterProperties\" detected! We don't accept NULL pointer or empty values.\n"), static_cast< OWeakObject* >( this ), 1 );

	/* SAFE AREAY ---------------------------------------------------------------------------------------------- */
    WriteGuard aWriteLock( m_aLock );

    // Attention: Some exceptions are thrown by addFilter() automaticly. (last parameter = sal_True)
    // But some of them must be wrapped for the outside code!
    try
    {
        m_aCache.addFilter( sFilterName, lProperties, sal_True );
    }
    catch(const InvalidRegistryException& exInvalid)
    {
        throw WrappedTargetException(DECLARE_ASCII("wrapped by service method FilterFactory::insertByName()"), static_cast< ::cppu::OWeakObject* >(this), css::uno::makeAny(exInvalid));
    }
}

//*****************************************************************************************************************
//	XNameContainer
//*****************************************************************************************************************
void SAL_CALL FilterFactory::removeByName( const OUString& sFilterName ) throw(	NoSuchElementException	,
																				WrappedTargetException	,
																				RuntimeException		)
{
	/* UNSAFE AREAY -------------------------------------------------------------------------------------------- */
	LOG_ASSERT2( implcp_removeByName( sFilterName ), "FilterFactory::removeByName()", "Invalid parameter detected!" )
    // Look for rejected calls.
    TransactionGuard aTransaction( m_aTransactionManager, E_HARDEXCEPTIONS );

	/* SAFE AREAY ---------------------------------------------------------------------------------------------- */
    WriteGuard aWriteLock( m_aLock );

    // Attention: Some exceptions are thrown by addFilter() automaticly. (last parameter = sal_True)
    // But some of them must be wrapped for the outside code!
    try
    {
        m_aCache.removeFilter( sFilterName, sal_True );
    }
    catch(const InvalidRegistryException& exInvalid)
    {
        throw WrappedTargetException(DECLARE_ASCII("wrapped by service method FilterFactory::removeByName()"), static_cast< ::cppu::OWeakObject* >(this), css::uno::makeAny(exInvalid));
    }
}

//*****************************************************************************************************************
//	XNameReplace
//*****************************************************************************************************************
void SAL_CALL FilterFactory::replaceByName(	const	OUString&	sFilterName			,
											const	Any&		aFilterProperties	) throw(	IllegalArgumentException,
																								NoSuchElementException	,
																								WrappedTargetException	,
																								RuntimeException		)
{
	/* UNSAFE AREAY -------------------------------------------------------------------------------------------- */
    // Look for rejected calls.
    TransactionGuard aTransaction( m_aTransactionManager, E_HARDEXCEPTIONS );

    // check parameter.
    if (sFilterName.getLength() < 1)
        throw IllegalArgumentException( DECLARE_ASCII("FilterFactory::replaceByName()\nInvalid parameter detected! We don't accept NULL pointer or empty values.\n"), static_cast< OWeakObject* >( this ), 0 );

    Sequence< PropertyValue > lProperties;
    if (!(aFilterProperties >>= lProperties))
        throw IllegalArgumentException( DECLARE_ASCII("FilterFactory::replaceByName()\nInvalid parameter detected! We don't accept NULL pointer or empty values.\n"), static_cast< OWeakObject* >( this ), 1 );

	/* SAFE AREAY ---------------------------------------------------------------------------------------------- */
    WriteGuard aWriteLock( m_aLock );

    // Attention: Some exceptions are thrown by addFilter() automaticly. (last parameter = sal_True)
    // But some of them must be wrapped for the outside code!
    try
    {
        m_aCache.replaceFilter( sFilterName, lProperties, sal_True );
    }
    catch(const InvalidRegistryException& exInvalid)
    {
        throw WrappedTargetException(DECLARE_ASCII("wrapped by service method FilterFactory::replaceByName()"), static_cast< ::cppu::OWeakObject* >(this), css::uno::makeAny(exInvalid));
    }
}

//*****************************************************************************************************************
//	XNameAccess
//*****************************************************************************************************************
Any SAL_CALL FilterFactory::getByName( const OUString& sName ) throw(	NoSuchElementException	,
				  														WrappedTargetException	,
				  														RuntimeException		)
{
	/* UNSAFE AREAY -------------------------------------------------------------------------------------------- */
	LOG_ASSERT2( implcp_getByName( sName ), "FilterFactory::getByName()", "Invalid parameter detected!" )
    // Look for rejected calls.
    TransactionGuard aTransaction( m_aTransactionManager, E_HARDEXCEPTIONS );

	/* SAFE AREAY ---------------------------------------------------------------------------------------------- */
    // Hold cache as reference!
    ReadGuard aReadLock( m_aLock );
    FilterCache aCacheRef;
    aReadLock.unlock();
    /* UNSAFE AREAY -------------------------------------------------------------------------------------------- */

    Any aResult;
    //---------------------------------------------------------------------------------------------------------
    // I)   Look for special supported query mode first!
    //      User whish to get more information about our internal values.
    if( QueryAnalyzer::isQuery( sName ) == sal_True )
    {
        aResult = aCacheRef.queryFilters( sName );
    }
    else
    //---------------------------------------------------------------------------------------------------------
    // II)  User whish to use normal access mode.
    //      Look for existing filters by using given name.
    {
        /* SAFE AREAY ------------------------------------------------------------------------------------------ */
        // Attention: We need a threadsynchronization between "exist()" and get...Properties()"!
        aReadLock.lock();
        if( aCacheRef.existsFilter( sName ) == sal_False )
        {
            throw NoSuchElementException( DECLARE_ASCII("FilterFactory::getByName()\nSpecified filter not exist!\n"), static_cast< OWeakObject* >( this ) );
        }
        aResult <<= aCacheRef.getFilterProperties( sName );
        aReadLock.unlock();
        /* UNSAFE AREAY ---------------------------------------------------------------------------------------- */
    }
	// Return result of operation.
	return aResult;
}

//*****************************************************************************************************************
//	XNameAccess
//*****************************************************************************************************************
Sequence< OUString > SAL_CALL FilterFactory::getElementNames() throw( RuntimeException )
{
    // Look for rejected calls.
    TransactionGuard aTransaction( m_aTransactionManager, E_HARDEXCEPTIONS );

    // I think we don't need any lock here ... because:
    // - cache live if we live
    // - he is threadsafe by himself
    // - this transaction is non breakable by a dispose() call!
    return m_aCache.getAllFilterNames();
}

//*****************************************************************************************************************
//	XNameAccess
//*****************************************************************************************************************
sal_Bool SAL_CALL FilterFactory::hasByName( const OUString& sName ) throw( RuntimeException )
{
    // Look for rejected calls.
    TransactionGuard aTransaction( m_aTransactionManager, E_HARDEXCEPTIONS );

    // I think we don't need any lock here ... because:
    // - cache live if we live
    // - he is threadsafe by himself
    // - this transaction is non breakable by a dispose() call!
    return m_aCache.existsFilter( sName );
}

//*****************************************************************************************************************
//	XElementAccess
//*****************************************************************************************************************
Type SAL_CALL FilterFactory::getElementType() throw( RuntimeException )
{
    // Look for rejected calls!
    TransactionGuard aTransaction( m_aTransactionManager, E_HARDEXCEPTIONS );

	return( ::getCppuType( (const Sequence< PropertyValue >*)NULL ) );
}

//*****************************************************************************************************************
//	XElementAccess
//*****************************************************************************************************************
sal_Bool SAL_CALL FilterFactory::hasElements() throw( RuntimeException )
{
    // Look for rejected calls.
    TransactionGuard aTransaction( m_aTransactionManager, E_HARDEXCEPTIONS );

    // I think we don't need any lock here ... because:
    // - cache live if we live
    // - he is threadsafe by himself
    // - this transaction is non breakable by a dispose() call!
    return m_aCache.hasFilters();
}

//*****************************************************************************************************************
//	XElementAccess
//*****************************************************************************************************************
void SAL_CALL FilterFactory::flush() throw ( RuntimeException)
{
    // Look for rejected calls.
    TransactionGuard aTransaction( m_aTransactionManager, E_HARDEXCEPTIONS );

    // I think we don't need any lock here ... because:
    // - cache and container lives if we live
    // - they are threadsafe by herself
    // - this transaction is non breakable by a dispose() call!
    m_aCache.validateAndRepairFilter();
    m_aCache.flush(DataContainer::E_FILTER);

    ::cppu::OInterfaceContainerHelper* pContainer = m_aListenerContainer.getContainer( ::getCppuType( ( const css::uno::Reference< css::util::XFlushListener >*) NULL ) );
	if( pContainer != NULL )
	{
        css::lang::EventObject aEvent;
        aEvent.Source = static_cast< ::cppu::OWeakObject* >(this);
        ::cppu::OInterfaceIteratorHelper pIterator( *pContainer );
        while( pIterator.hasMoreElements() == sal_True )
		{
            try
            {
                ((css::util::XFlushListener *)pIterator.next())->flushed( aEvent );
            }
            catch( css::uno::RuntimeException& )
            {
                pIterator.remove();
            }
		}
	}
}

//*****************************************************************************************************************
//	XElementAccess
//*****************************************************************************************************************
void SAL_CALL FilterFactory::addFlushListener( const Reference< XFlushListener >& xListener ) throw ( RuntimeException )
{
    // Look for rejected calls.
    TransactionGuard aTransaction( m_aTransactionManager, E_HARDEXCEPTIONS );
    // We dont need any lock here - helper is threadsafe!
    m_aListenerContainer.addInterface( ::getCppuType( (const Reference< XFlushListener >*)NULL ), xListener );
}

//*****************************************************************************************************************
//	XElementAccess
//*****************************************************************************************************************
void SAL_CALL FilterFactory::removeFlushListener( const Reference< XFlushListener >& xListener ) throw ( RuntimeException )
{
    // Look for rejected calls.
    TransactionGuard aTransaction( m_aTransactionManager, E_SOFTEXCEPTIONS );
    // We dont need any lock here - helper is threadsafe!
    m_aListenerContainer.removeInterface( ::getCppuType( (const Reference< XFlushListener >*)NULL ), xListener );
}

//*****************************************************************************************************************
//  XContainerQuery
//*****************************************************************************************************************
css::uno::Reference< css::container::XEnumeration > SAL_CALL FilterFactory::createSubSetEnumerationByQuery( const ::rtl::OUString& sQuery )
    throw(css::uno::RuntimeException)
{
    return css::uno::Reference< css::container::XEnumeration >();
}

//*****************************************************************************************************************
//  XContainerQuery
//*****************************************************************************************************************
css::uno::Reference< css::container::XEnumeration > SAL_CALL FilterFactory::createSubSetEnumerationByProperties( const css::uno::Sequence< css::beans::NamedValue >& lProperties )
    throw(css::uno::RuntimeException)
{
    return css::uno::Reference< css::container::XEnumeration >();
}

//_________________________________________________________________________________________________________________
//	debug methods
//_________________________________________________________________________________________________________________

/*-----------------------------------------------------------------------------------------------------------------
	The follow methods checks the parameter for other functions. If a parameter or his value is non valid,
	we return "sal_False". (else sal_True) This mechanism is used to throw an ASSERT!

	ATTENTION

		If you miss a test for one of this parameters, contact the autor or add it himself !(?)
		But ... look for right testing! See using of this methods!
-----------------------------------------------------------------------------------------------------------------*/

#ifdef ENABLE_ASSERTIONS

//*****************************************************************************************************************
sal_Bool FilterFactory::implcp_FilterFactory( const Reference< XMultiServiceFactory >& xFactory )
{
	return	(
				( &xFactory		==	NULL		)	||
				( xFactory.is()	==	sal_False	)
			);
}

//*****************************************************************************************************************
sal_Bool FilterFactory::implcp_createInstanceWithArguments	(	const	OUString&			sTypeName	,
																const	Sequence< Any >&	lArguments	)
{
	return	(
				( &sTypeName			==	NULL	)	||
				( sTypeName.getLength()	<	1		)	||
				( &lArguments			==	NULL	)
			);
}

//*****************************************************************************************************************
sal_Bool FilterFactory::implcp_getByName( const OUString& sName )
{
	return	(
				( &sName			==	NULL	)	||
				( sName.getLength()	<	1		)
			);
}

//*****************************************************************************************************************
sal_Bool FilterFactory::implcp_hasByName( const OUString& sName )
{
	return	(
				( &sName			==	NULL	)	||
				( sName.getLength()	<	1		)
			);
}

//*****************************************************************************************************************
sal_Bool FilterFactory::implcp_removeByName( const OUString& sFilterName )
{
	return	(
				( &sFilterName				==	NULL	)	||
				( sFilterName.getLength()	<	1		)
			);
}

#endif	//	#ifdef ENABLE_ASSERTIONS

}		//	namespace framework
