/*
 * Copyright 1997-2007 Sun Microsystems, Inc.  All Rights Reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Sun Microsystems, Inc., 4150 Network Circle, Santa Clara,
 * CA 95054 USA or visit www.sun.com if you need additional information or
 * have any questions.
 */

package javax.crypto.spec;

import java.math.BigInteger;
import java.security.spec.AlgorithmParameterSpec;

/**
 * This class specifies the set of parameters used with the Diffie-Hellman
 * algorithm, as specified in PKCS #3: <i>Diffie-Hellman Key-Agreement
 * Standard</i>.
 *
 * <p>A central authority generates parameters and gives them to the two
 * entities seeking to generate a secret key. The parameters are a prime
 * <code>p</code>, a base <code>g</code>, and optionally the length
 * in bits of the private value, <code>l</code>.
 *
 * <p>It is possible that more than one instance of parameters may be
 * generated by a given central authority, and that there may be more than
 * one central authority. Indeed, each individual may be its own central
 * authority, with different entities having different parameters.
 *
 * <p>Note that this class does not perform any validation on specified
 * parameters. Thus, the specified values are returned directly even
 * if they are null.
 *
 * @author Jan Luehe
 *
 * @see javax.crypto.KeyAgreement
 * @since 1.4
 */
public class DHParameterSpec implements AlgorithmParameterSpec {

    // The prime modulus
    private BigInteger p;

    // The base generator
    private BigInteger g;

    // The size in bits of the random exponent (private value) (optional)
    private int l;

    /**
     * Constructs a parameter set for Diffie-Hellman, using a prime modulus
     * <code>p</code> and a base generator <code>g</code>.
     *
     * @param p the prime modulus
     * @param g the base generator
     */
    public DHParameterSpec(BigInteger p, BigInteger g) {
        this.p = p;
        this.g = g;
        this.l = 0;
    }

    /**
     * Constructs a parameter set for Diffie-Hellman, using a prime modulus
     * <code>p</code>, a base generator <code>g</code>,
     * and the size in bits, <code>l</code>, of the random exponent
     * (private value).
     *
     * @param p the prime modulus
     * @param g the base generator
     * @param l the size in bits of the random exponent (private value)
     */
    public DHParameterSpec(BigInteger p, BigInteger g, int l) {
        this.p = p;
        this.g = g;
        this.l = l;
    }

    /**
     * Returns the prime modulus <code>p</code>.
     *
     * @return the prime modulus <code>p</code>
     */
    public BigInteger getP() {
        return this.p;
    }

    /**
     * Returns the base generator <code>g</code>.
     *
     * @return the base generator <code>g</code>
     */
    public BigInteger getG() {
        return this.g;
    }

    /**
     * Returns the size in bits, <code>l</code>, of the random exponent
     * (private value).
     *
     * @return the size in bits, <code>l</code>, of the random exponent
     * (private value), or 0 if this size has not been set
     */
    public int getL() {
        return this.l;
    }
}
