/* build a new stat data file based on data dictionary entries */

#include <stdio.h>

struct fdesc 				/* description of field */
{
	char	*fd_name;		/* name of field */
	int	fd_fldno;		/* field number in source file */
	struct fdesc	*fd_next;	/* next entry */
};
#define FDNULL ((struct fdesc *) 0)
struct fdesc	*newfd();

typedef char Bool;
#define True	1
#define False	0

#define CNULL ((char *) 0)
#define FNULL ((FILE *) 0)

Bool	argcheck();
char	*buildcmd();
void	data_filter();
int	build_outlist();
Bool	loaddd();
char	*Malloc();

char	*Cmdname;
static char	*def_file = ".datadict";	/* default data dict name */
static struct fdesc	*dd_current;	/* used by firstdd, nextdd */

int
main(argc, argv)
	int	argc;
	char	*argv[];
{
	struct fdesc	*fl_base = FDNULL, *curr_fl;	/* fields in cmd line */
	struct fdesc	*dd_base = FDNULL, *curr_dd;	/* fields from dict */
	char	*ddname = CNULL;
	int	fld_count = 0;
	Bool	exclude = False;
	extern char *	Cmdname;

	Cmdname = argv[0];
	if ( !argcheck( argc, argv, &fl_base, &ddname, &exclude ) )
		exit(1);
	if ( ! loaddd( &dd_base, ddname ) )
		exit(1);
	fld_count = build_outlist( &fl_base, dd_base, ddname, exclude );
	data_filter( fl_base, fld_count );
	exit(0);
}


Bool
argcheck( argc, argv, p_fl_base, p_ddname, p_exclude )
	int	argc;
	char	*argv[];
	struct fdesc	**p_fl_base;
	char	**p_ddname;
	char	*p_exclude;
{
	int	idx;
	char	Usage[256];
	Bool	gotdd = False;	/* data dictionary specified on cmd line */
	Bool	result = True;
	struct fdesc	*curr_fl;
	extern char	*Cmdname;

	sprintf( Usage,
		"Usage: %s -x -f <data-dictionary> <field> [<field>]\n",
		Cmdname);
	if ( argc <= 1 )
	{
		fprintf(stderr, Usage);
		result = False;
	}

	for ( idx = 1; idx < argc && result; idx++)
	{
		if ( *argv[idx] == '-' )
		{
			switch ( argv[idx][1] )
			{
			case 'f':		/* next arg is data dictionary*/
				if ( idx == argc - 1 ) /* -f is last argument */
				{
					fprintf(stderr, Usage );
					result = False;
				}
				else if ( gotdd ) /* already have a -f */
				{
					fprintf(stderr, Usage);
					result = False;
				}
				else
				{
					gotdd = True;
					*p_ddname = argv[ ++idx ];
				}
				break;
			case 'x':		/* exclude named fields */
				*p_exclude = True;
				break;
			default:		/* unkown flag */
				fprintf(stderr, Usage);
				result = False;
				break;
			}
		}
		else	/* not a flag, must be a field name */
		{
			if (*p_fl_base == FDNULL )
				curr_fl = newfd(p_fl_base, argv[idx], 0) ;
			else
				curr_fl = newfd( &(curr_fl->fd_next),
					argv[idx],0);
		}
	}
	return result;
}



Bool
loaddd( p_dd_base, ddname )
	struct fdesc	**p_dd_base;
	char	*ddname;
{
	Bool	result = True;
	struct fdesc	*curr_dd;
	FILE	*ddfp;
	char	namebuf[80];
	int	fieldno;
	extern char	*Cmdname;
	extern char	*def_file;

	if ( ddname == CNULL )
		ddname = def_file;

	if ( (ddfp = fopen( ddname, "r" )) == FNULL )
	{
		fprintf(stderr,
			"%s: unable to open data dictionary file\n",
			Cmdname);
		perror( ddname );
		result = False;
	}
	else
		/* load up the data dictionary list */
		while (fscanf(ddfp, "%s %d\n", namebuf, &fieldno ) == 2)
		{
			if ( *p_dd_base == FDNULL )
				curr_dd = newfd( p_dd_base,
					namebuf, fieldno );
			else
				curr_dd = newfd( &(curr_dd->fd_next),
					namebuf, fieldno);
			if ( curr_dd->fd_fldno < 1 )
			{
				fprintf(stderr,
					"%s: field \"%s\" has an illegal field number (%d) in %s\n",
					Cmdname, curr_dd->fd_name, 
					curr_dd->fd_fldno, ddname );
				result = False;
				break;
			}
		}
	return result;
}

Bool
finddd( p_ddrec, dd_base, searchstr )
	struct fdesc	**p_ddrec;
	struct fdesc	*dd_base;
	char	*searchstr;
{
	struct fdesc	*curr_dd;
	Bool	found = False;

	/* find in data dictionary */
	for ( curr_dd = dd_base;
	      curr_dd != FDNULL && !found;
	      curr_dd = curr_dd->fd_next )
		if ( strcmp( curr_dd->fd_name, searchstr ) == 0 )
		{
			found = True;
			*p_ddrec = curr_dd;
		}
	return found;
}


struct fdesc *
firstdd( dd_base )
	struct fdesc	*dd_base;
{
	extern struct fdesc	*dd_current;

	dd_current = dd_base;
	return dd_current;
}

struct fdesc *
nextdd()
{
	extern struct fdesc	*dd_current;

	if ( dd_current != FDNULL)
		dd_current = dd_current->fd_next;

	return dd_current;
}


struct fdesc *
newfd( p_next, name, num )
	struct fdesc	**p_next;
	char	*name;
	int	num;
{
	*p_next = (struct fdesc *) Malloc( sizeof( struct fdesc ) );

	(*p_next)->fd_name = Malloc( strlen( name ) );

	strcpy( (*p_next)->fd_name, name );
	(*p_next)->fd_fldno = num;
	(*p_next)->fd_next = FDNULL;

	return *p_next;
}


char *
buildcmd( fl_base, fld_count )
	struct fdesc	*fl_base;
	int	fld_count;
{
	char	*cmdbuf;
	char	numbuf[6];
	struct fdesc	*curr_fl;
	int	idx;

	cmdbuf = Malloc( 15 + ( 4 * fld_count ) );
	strcpy( cmdbuf, "awk '{print" );
	for ( idx = 1, curr_fl = fl_base;
	      curr_fl != FDNULL;
	      idx++, curr_fl = curr_fl->fd_next )
	{

		if ( idx == fld_count )
			sprintf(numbuf, " $%d;", curr_fl->fd_fldno );
		else
			sprintf(numbuf, " $%d,", curr_fl->fd_fldno );
		strcat( cmdbuf, numbuf );
	}
	strcat( cmdbuf, "}'" );

	return cmdbuf;
}

void
data_filter( fl_base, fld_count )
	struct fdesc	*fl_base;
	int	fld_count;
{
	char	*systemcmd;
	extern char	*Cmdname;

	systemcmd = buildcmd( fl_base, fld_count );
	if ( system( systemcmd ) == 127 )
		fprintf( stderr, "%s: couldn't execute: %s\n",
				Cmdname, systemcmd);
}


int
build_outlist( p_fl_base, dd_base, ddname, exclude )
	struct fdesc	**p_fl_base;
	struct fdesc	*dd_base;
	char	*ddname;
	Bool	exclude;
{
	struct fdesc	*curr_fl, *curr_dd, *new_base, *new_fl;
	int	fld_count = 0;
	extern char	*Cmdname;
	Bool	found;
	
	if ( exclude )
	{
		/* need to rebuild arglist with unnamed dictionary fields */
		for ( curr_dd = firstdd( dd_base );
		      curr_dd != FDNULL;
		      curr_dd = nextdd() )
		{
			/* if current dd entry not in arglist, put in the
			 * new output list
			 */
			for ( curr_fl = *p_fl_base, found = False;
			      curr_fl != FDNULL & !found;
			      curr_fl = curr_fl->fd_next )
				if ( strcmp( curr_dd->fd_name, curr_fl->fd_name) == 0 )
					found = True;
			if ( !found )
			{
				if ( new_base == FDNULL )
					new_fl = newfd( &new_base,
						curr_dd->fd_name, curr_dd->fd_fldno);
				else
					new_fl = newfd( &(new_fl->fd_next),
						curr_dd->fd_name, curr_dd->fd_fldno);
				fld_count++;
			}
		}
		*p_fl_base = new_base;
	}
	else
	{
		/* for each specified field, find it's field number in the dict */
		for ( curr_fl = *p_fl_base;
		      curr_fl != FDNULL;
		      curr_fl = curr_fl->fd_next )
		{
			if ( !finddd( &curr_dd, dd_base, curr_fl->fd_name ) )
			{
				fprintf(stderr,
					"%s: field \"%s\" not found in %s\n",
					Cmdname, curr_fl->fd_name, ddname );
				exit(1);
			}
			else
				curr_fl->fd_fldno = curr_dd->fd_fldno;
			fld_count++;
		}
	}

#ifdef DEBUG
	for ( curr_fl = *p_fl_base; curr_fl != FDNULL; curr_fl = curr_fl->fd_next )
	{
		fprintf(stderr,
			"arglist entry: %s\t%d\n",
			curr_fl->fd_name, curr_fl->fd_fldno);
	}
#endif /* DEBUG */
	
	return fld_count;
}

char *
Malloc( size )
	register int	size;
{
	register char *vast_tracts ;
	char	*malloc();
	extern char *	Cmdname;

	if ( ! ( vast_tracts = malloc( size ) ) )
	{
		fprintf(stderr, "%s: unable to allocate more space\n", Cmdname);
		exit(1);
	}
	return( vast_tracts );
}
