/*!
  @file           Tools_Arguments.cpp
  @author         Bernd Vorsprach - bernd.vorsprach@sap.com
  @brief          argument parsing - Implementation

\if EMIT_LICENCE

    ========== licence begin  GPL
    Copyright (c) 2003-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


\endif
*/

/*
  -----------------------------------------------------------------------------
  includes
  -----------------------------------------------------------------------------
 */
#include <ctype.h>
#include "SAPDB/ToolsCommon/Tools_Arguments.hpp"

#define MASKCHAR   '\\'
#define ASSIGNCHAR '='
#define STRINGCHAR '"'
#define OPTIONCHAR '-'

/*! @brief   Constructor */
Tools_Arguments :: Tools_Arguments
    ( Tools_DynamicUTF8String::ConstPointer src)
{
  setArguments(src);
} // end Tools_Arguments :: Tools_Arguments

/*! @brief   Constructor */
Tools_Arguments :: Tools_Arguments
    ( )
{
} // end Tools_Arguments :: Tools_Arguments

/*! @brief   Destructor */
Tools_Arguments :: ~Tools_Arguments
    (  )
{
} // end Tools_Arguments :: ~Tools_Arguments

/*! @brief   public function */
void Tools_Arguments :: setArguments
    ( Tools_DynamicUTF8String::ConstPointer src)
{
  Tools_DynamicUTF8String sArguments(src);
  Tools_DynamicUTF8String sValue = "";
  Tools_DynamicUTF8String sName  = "";
  bool                    bIgnoreNext = false;
  bool                    bInString   = false;

  Tools_DynamicUTF8String::BasisElementIndex nIndex = 0;

  this->nCount       = 0;
  this->oValues.clear();
  this->oNames.clear();

  for (nIndex = 0; nIndex < sArguments.Length(); ++nIndex) {
    switch (sArguments[nIndex]) {
      case MASKCHAR:
        if (nIndex+1 < sArguments.Length()) {
          if ( (sArguments[nIndex + 1] == ASSIGNCHAR) ||
               (sArguments[nIndex + 1] == STRINGCHAR)    ) {
             bIgnoreNext = true;
          } else {
            sValue.Append(sArguments[nIndex]);
          } // end if
        } else {
          sValue.Append(sArguments[nIndex]);
        } // end if
        break;
      case ASSIGNCHAR:
        if (!bIgnoreNext && !bInString && sName.Empty()) {
          sName = sValue;
          sValue = "";
        } else {
          sValue.Append(sArguments[nIndex]);
          bIgnoreNext = false;
        } // end if
        break;
      case STRINGCHAR:
        if (!bIgnoreNext) {
          bInString = !bInString;
        } else {
          sValue.Append(sArguments[nIndex]);
          bIgnoreNext = false;
        } // end if
        break;
      default:
        if (bInString || !isspace(sArguments[nIndex])) {
          sValue.Append(sArguments[nIndex]);
        } else {
          if (!sValue.Empty() || !sName.Empty()) {
            // end of value found add name/value to the list
            if (sName.Empty() && sValue[0] == OPTIONCHAR) {
              sName = sValue.SubStr(1);
              sValue = "";
            } // end if
            this->oValues[this->nCount] = sValue;
            this->oNames [this->nCount] = sName;
            ++this->nCount;
            sName  = "";
            sValue = "";
          } // end if
        } // end if
        bIgnoreNext = false;
        break;
    } // end switch
  } // end for

  // end handling
  if (!sValue.Empty() || !sName.Empty()) {
    // end of value found add name/value to the list
    if (sName.Empty() && sValue[0] == OPTIONCHAR) {
      sName = sValue.SubStr(1);
      sValue = "";
    } // end if
    this->oValues[this->nCount] = sValue;
    this->oNames [this->nCount] = sName;
    ++this->nCount;
    sName  = "";
    sValue = "";
  } // end if

} // end Tools_Arguments :: Tools_Arguments

/*! @brief  public member */
Tools_DynamicUTF8String Tools_Arguments :: getValue
    ( const Tools_DynamicUTF8String & sName,
            bool                      bCasesensitiv )
{
  return getValue(this->getNumber(sName, bCasesensitiv));
} // end Tools_Arguments :: getValue

/*! @brief  public member */
Tools_DynamicUTF8String Tools_Arguments :: getValue
    ( SAPDB_Int nValue )
{
  if (nValue > 0 && nValue <= this->nCount) {
    return oValues[nValue - 1];
  } // end if
  
  return "";
} // end Tools_Arguments :: getValue

/*! @brief  public member */
bool Tools_Arguments :: existsValue
    ( const Tools_DynamicUTF8String & sValue,
            bool                      bCasesensitiv )
{
  SAPDB_Int nIndex = 0;
  Tools_DynamicUTF8String sTmp  = sValue;
  Tools_DynamicUTF8String sTmp2;;

  if (!bCasesensitiv) {
    sTmp.ToUpper();
  } // end if

  for (nIndex = 0; nIndex < this->nCount; ++nIndex) {
    sTmp2 = this->oValues[nIndex];
    if (!bCasesensitiv) {
      sTmp2.ToUpper();
    } // end if
    if (sTmp.Compare(sTmp2) == 0) {
      return true;
    } // end if
  } // end for

  return false;
} // end Tools_Arguments :: existsValue

/*! @brief  public member */
Tools_DynamicUTF8String Tools_Arguments :: getOption
    ( SAPDB_Int nOption )
{
  if (nOption > 0 && nOption <= this->nCount) {
    return oNames[nOption - 1];
  } // end if
  
  return "";
} // end Tools_Arguments :: getValue

/*! @brief  public member */
bool Tools_Arguments :: getOption
    ( const Tools_DynamicUTF8String & sName,
            bool                      bCasesensitiv )
{
  Tools_DynamicUTF8String sArg1;
  Tools_DynamicUTF8String sArg2;
  Tools_DynamicUTF8String sArg3;
  return getOption(sName, 0, sArg1, sArg2, sArg3, bCasesensitiv);
} // end Tools_Arguments :: getOption

/*! @brief  public member */
bool Tools_Arguments :: getOption
    ( const Tools_DynamicUTF8String & sName,
            Tools_DynamicUTF8String & sArg1,
            bool                      bCasesensitiv )
{
  Tools_DynamicUTF8String sArg2;
  Tools_DynamicUTF8String sArg3;
  return getOption(sName, 1, sArg1, sArg2, sArg3, bCasesensitiv);
} // end Tools_Arguments :: getOption

/*! @brief  public member */
bool Tools_Arguments :: getOption
    ( const Tools_DynamicUTF8String & sName,
            Tools_DynamicUTF8String & sArg1,
            Tools_DynamicUTF8String & sArg2,
            bool                      bCasesensitiv )
{
  Tools_DynamicUTF8String sArg3;
  return getOption(sName, 2, sArg1, sArg2, sArg3, bCasesensitiv);
} // end Tools_Arguments :: getOption

/*! @brief  public member */
bool Tools_Arguments :: getOption
    ( const Tools_DynamicUTF8String & sName,
            Tools_DynamicUTF8String & sArg1,
            Tools_DynamicUTF8String & sArg2, 
            Tools_DynamicUTF8String & sArg3,
            bool                      bCasesensitiv )
{
  return getOption(sName, 3, sArg1, sArg2, sArg3, bCasesensitiv);
} // end Tools_Arguments :: getOption

/*! @brief  private member */
bool Tools_Arguments :: getOption
    ( const Tools_DynamicUTF8String & sName,
            SAPDB_Int                 nArgs,
            Tools_DynamicUTF8String & sArg1,
            Tools_DynamicUTF8String & sArg2,
            Tools_DynamicUTF8String & sArg3,
            bool                      bCasesensitiv )
{
  SAPDB_Int nOption = (sName[0] == OPTIONCHAR) 
                      ? this->getNumber(sName.SubStr(1), bCasesensitiv) 
                      : this->getNumber(sName, bCasesensitiv);
  Tools_DynamicUTF8String sValueName;

  if (nOption == 0) {
    return false;
  } // end if

  if (nOption + nArgs > this->getCount()) {
    return false;
  } // end if

  switch (nArgs) {
    case 3:
      sArg3      = getValue(nOption+3);
      sValueName = getName (nOption+3);
      if (sArg3.Empty() || !sValueName.Empty()) {
        return false;
      } // end if
    case 2:
      sArg2      = getValue(nOption+2);
      sValueName = getName (nOption+2);
      if (sArg2.Empty() || !sValueName.Empty()) {
        return false;
      } // end if
    case 1:
      sArg1      = getValue(nOption+1);
      sValueName = getName (nOption+1);
      if (sArg1.Empty() || !sValueName.Empty()) {
        return false;
      } // end if
  } // end switch

  return true;
} // end Tools_Arguments :: getOption

/*! @brief  public member */
Tools_DynamicUTF8String Tools_Arguments :: getName
    ( SAPDB_Int nValue )
{
  return oNames[nValue - 1];
} // end Tools_Arguments :: getName

/*! @brief  public member */
SAPDB_Int  Tools_Arguments :: getNumber
    ( const Tools_DynamicUTF8String & sName,
            bool                      bCasesensitiv )
{
  SAPDB_Int nIndex = 0;
  Tools_DynamicUTF8String sTmp = sName;
  Tools_DynamicUTF8String sTmp2;
  if (!bCasesensitiv) sTmp.ToUpper();

  for (nIndex = 0; nIndex < this->nCount; ++nIndex) {
    sTmp2 = oNames[nIndex];
    if (!bCasesensitiv) sTmp2.ToUpper();
    if (sTmp.Compare(sTmp2) == 0) {
      return nIndex + 1;
    } // end if
  } // end for

  return 0;
} // end Tools_Arguments :: getNumber

/*! @brief  public member */
SAPDB_Int Tools_Arguments :: getCount
    ( )
{
  return this->nCount;
} // end Tools_Arguments :: getCount
