/*!========================================================================
  
  @file         RTEIPC_NamedSharedMemory.hpp
  @author       StefanP
  @ingroup      Interprocess Communication
  @brief        Primitives for Named Shared Memory Handling

  @since        2003-09-29  10:50
  @sa           

  =======================================================================

  \if EMIT_LICENCE
    
    ========== licence begin  GPL
    Copyright (c) 2003-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end
  \endif
  ----------------------------------------------------------------------------*/


#ifndef RTEIPC_NAMEDSHAREDMEMORY_HPP
#define RTEIPC_NAMEDSHAREDMEMORY_HPP


/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/
#include    "RunTime/RTE_MessageList.hpp"
#include    "SAPDBCommon/SAPDB_ToString.hpp"

/*===========================================================================*
 *  DEFINES                                                                  *
 *===========================================================================*/

#if defined(AIX) || defined(LINUX)
#define RTE_WELL_KNOWN_RAMDISK_PATH         "/rel/ramdisk/SAPDB/"
#endif

/*===========================================================================*
 *  MACROS                                                                   *
 *===========================================================================*/


/*===========================================================================*
 *  CLASSES, STRUCTURES, TYPES, UNIONS ...                                   *
 *===========================================================================*/



/*!
    @class          RTEIPC_NamedSharedMemory
    @brief    Basic class for Named Shared Memory handling

                    Named shared memory objects can be shared between different processes. 
                    This class offers the basic methods for named shared memory handling. 

                    At construction time one has to determine the open mode of the shared memory object. 
                    One can open a new object as well as an existing object. The constructor does not open 
                    the shared memory object. This is first done by a call to the OPEN method! 

                    The base address for the shared memory object can either be determined by the system or
                    can be assigned to a fixed address.
*/

class   RTEIPC_NamedSharedMemory
{
public:

    enum SHMRet {NoError = 0,
                 Error,
                 ResourceMissing,
                 BadAddress,
                 AlreadyExist
                };

  /*!
     @brief   Constructor

                    The constructor does not open or create the shared memory. This is done by a call 
                    to the open method.
     
     @return        none

  */
    
    RTEIPC_NamedSharedMemory        ();

  /*!
     @brief   Destructor
     @return        none

  */

    ~RTEIPC_NamedSharedMemory      ();


  /*!
     @brief    Open or create a shared memory object 
                     
     @param          pSharedMemoryName [in] - Name of the shared memory object (e.g. SAPDBTech-COMMSEG-<UniqueName>).
                                              There is no limitation to the length of the name.
     @param          pBaseAddress [in]      - Address at which the shared memory should be mapped to.
                                              The address has to be a multiple of the system page size. 
                                              If the shared memory can not assigned to this address the
                                              functions returns RTEIPC_ErrorAddress.
     @param          openMode [in]          - RTE_OpenExisting or RTE_OpenNew
     @param          pSA [in]               - Pointer to security attributes (NT only)
     @param          permission [in]        - The permission bits of the shared memory object (UNIX only)
     @param          shmSize [in]           - Size of the shared memory to be created. The size has to be a
                                              multiple of system page size!
     @param          ppSharedMemory [out]   - Pointer to the address of the shared memory
     @param          messageList [in/out]   - Message list

     @return         ResourceMissing, BadAddress, Error, AlreadyExist or NoError

  */

    SHMRet           Open            (SAPDB_Char const * const       pSharedMemoryName,
                                     void *                         pBaseAddress,
                                     RTE_OpenMode                   openMode,
#if defined (_WIN32)
                                     PSECURITY_ATTRIBUTES const     pSA,   
#else
                                     int const                      permission,
#endif
                                     SAPDB_UInt8                    shmSize,
                                     void **                        ppSharedMemory,                   
                                     SAPDBErr_MessageList &         messageList);
            
 /*!
     @brief    Close a shared memory object
                     
     @param          messageList [in/out]   - Message list
     @return         Error, NoError
 */

    SHMRet          Close              (SAPDBErr_MessageList &         messageList);

#if defined (_WIN32)

  /*!
     @brief   

     @param         

     @return

  */
    SAPDB_Bool      Lock                (SAPDBErr_MessageList &         messageList);
#endif

private:
    // It is not allowed to call the Copy Constructor from outside
	inline   RTEIPC_NamedSharedMemory  (const RTEIPC_NamedSharedMemory & shm) {;};

#if !defined (_WIN32)
    SAPDB_Int UnlinkSHMObject       ();

    SAPDB_Int OpenSHMObject         (SAPDB_Int const                openFlags, 
                                     mode_t const                   openMode = (mode_t)0);
#endif

    SAPDB_UInt8                         m_Size;

    RTE_FileHandle                      m_Handle;
    void *                              m_pMem;
    SAPDB_Char *                        m_pName;
    RTE_OpenMode                        m_OpenMode;
};

inline SAPDB_ToStringClass SAPDB_ToString(RTEIPC_NamedSharedMemory::SHMRet     rc)
{
    switch (rc)
    {
    case RTEIPC_NamedSharedMemory::NoError: 
        return SAPDB_ToStringClass("NOERROR");
    case RTEIPC_NamedSharedMemory::ResourceMissing: 
        return SAPDB_ToStringClass("RESOURCEMISSING");
    case RTEIPC_NamedSharedMemory::BadAddress: 
        return SAPDB_ToStringClass("BADADDRESS");
    case RTEIPC_NamedSharedMemory::AlreadyExist: 
        return SAPDB_ToStringClass("ALREADYEXIST");
    case RTEIPC_NamedSharedMemory::Error: 
        return SAPDB_ToStringClass("ERROR");
    default:                          
        break;
    }

    return SAPDB_ToStringClass((SAPDB_Int4)rc); 
}


#endif  /* RTEIPC_NAMEDSHAREDMEMORY_HPP */
