/**
 *
 * $Id: FontList.c,v 1.25 2001/07/27 18:51:56 dannybackx Exp $
 *
 * Copyright (C) 1995 Free Software Foundation, Inc.
 * Copyright (C) 1995-2001 LessTif Development Team
 *
 * This file is part of the GNU LessTif Library.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 **/

static const char rcsid[] = "$Id: FontList.c,v 1.25 2001/07/27 18:51:56 dannybackx Exp $";

#include <LTconfig.h>

#include <string.h>
#include <locale.h>

#include <XmI/XmI.h>

#include <Xm/XmP.h>

#include <XmI/DebugUtil.h>


/**************************** INTERNAL FUNCTIONS **************************/
static XmFontList
__XmFontListAlloc(int numberOfEntries)
{
    XmFontList newFontList = (XmFontList)XtCalloc(numberOfEntries + 1,
                                                  sizeof(struct _XmFontListRec));

    return newFontList;
}


static int
__XmFontListNumEntries(XmFontList flist)
{
    int cnt;

    for (cnt = 0; flist[cnt].tag != NULL; cnt++)
    {
    }

    return cnt;
}


static void
__XmFontListDealloc(XmFontList list)
{
    int i;

    for (i = 0; list[i].tag != NULL; i++)
    {
	XtFree(list[i].tag);
    }

    XtFree((char *)list);
}


/************************* LOW LEVEL FUNCTIONS **************************/
extern XmFontList
_XmFontListCreateDefault(Display *d)
{
    struct _XmFontListRec defaultEntry;
    XFontStruct *fs = XLoadQueryFont(d, XmDEFAULT_FONT);

    defaultEntry.tag = XmFONTLIST_DEFAULT_TAG;
    defaultEntry.type = XmFONT_IS_FONT;
    defaultEntry.font = (XtPointer)fs;

    return XmFontListAppendEntry(NULL, &defaultEntry);
}


/************************** PUBLIC FUNCTIONS ***********************/
extern XmFontList
XmFontListAppendEntry(XmFontList old,
		      XmFontListEntry entry)
{
    XmFontList newFontList;
    int i;

    if (entry == NULL)
    {
	/* We can't just return "old" here, because of memory allocation
	 * rules: old can be freed by the caller; while the result of
	 * XmFontListAppendEntry should be newly allocated (and hence
	 * should be freed by the caller as well).
         *
     	 *    return XmFontListCopy(old);
	 *
         * amai: well thought, but a more literal interpretation of the
	 *       man page indeed indicates that 'old' is returned literally and
	 *       a quick test on DU/ Motif 1.2.4 confirms that!
	 */
	 return old;
    }

    if (old == NULL)
    {
	newFontList = __XmFontListAlloc(1);
	i = 0;
    }
    else
    {
	newFontList = __XmFontListAlloc(__XmFontListNumEntries(old) + 1);

	for (i = 0; old[i].tag != NULL; i++)
	{
	    newFontList[i].tag  = XtNewString(old[i].tag);
	    newFontList[i].type = old[i].type;
	    newFontList[i].font = old[i].font;
	}

	__XmFontListDealloc(old);
    }

    newFontList[i].tag = XtNewString(entry->tag);
    newFontList[i].type = entry->type;
    newFontList[i].font = entry->font;

    return newFontList;
}


extern XmFontList
XmFontListCreate(XFontStruct *font, XmStringCharSet charset)
{
    /* this function is obsolete 1.2 and above and should not be used.
       Use XmFontListAppendEntry instead! */

    XmFontList r;
    struct _XmFontListRec entry;
    static Boolean warn=False;

#if 0
    if (!warn) {
       warn=True;
       _XmWarning(NULL, "XmFontListCreate() is an obsolete function!\n");
    }
#endif

    entry.tag = charset;	/* XmFontListAppendEntry *copies* this */
				/* so we don't need to */
    entry.type = XmFONT_IS_FONT;
    entry.font = (XtPointer)font;

    r = XmFontListAppendEntry(NULL, &entry);

    return r;
}

/*
 * this function is "obsolete" since 1.2 ; however Xmt uses it.
 */
extern XmFontList
XmFontListAdd(XmFontList old, XFontStruct *font, XmStringCharSet charset)
{
    XmFontList newFontList;
    struct _XmFontListRec newfont;
    static Boolean warn=False;

#if 0
    if (!warn) {
       warn=True;
       _XmWarning(NULL, "XmFontListAdd() is an obsolete function!\n");
    }
#endif

    if (!old)
    	return (XmFontList)NULL;
    if (!font || !charset)
      return old;

    newfont.tag = charset;
    newfont.type = XmFONT_IS_FONT;
    newfont.font = (XtPointer)font;

    newFontList = XmFontListAppendEntry(old, &newfont);
    /* amai: de-allocation of 'old' is done by XmFontListAppendEntry() */

    return newFontList;
}


extern XmFontList
XmFontListCopy(XmFontList fontlist)
{
    XmFontList newFontList;
    int i;

    if (!fontlist)
    {
	return (XmFontList)NULL;
    }

    newFontList = __XmFontListAlloc(__XmFontListNumEntries(fontlist));

    for (i = 0; fontlist[i].tag != NULL; i++)
    {
	newFontList[i].tag  = XtNewString(fontlist[i].tag);
	newFontList[i].type = fontlist[i].type;
	newFontList[i].font = fontlist[i].font;
    }

    return newFontList;
}


extern XmFontListEntry
XmFontListEntryCreate(char *tag,
		      XmFontType type,
		      XtPointer font)
{
    XmFontListEntry entry = (XmFontListEntry)XtMalloc(sizeof(struct _XmFontListRec));

    entry->tag  = XtNewString(tag);
    entry->type = type;
    entry->font = font;

    return entry;
}


extern void
XmFontListEntryFree(XmFontListEntry *entry)
{
    DEBUGOUT(_LtDebug(__FILE__, NULL, "XmFontListEntryFree(0x%x)\n", entry));

    if (entry)
    {
	XtFree((*entry)->tag);
	XtFree((char *)*entry);

	/* should we close the Font? */
    }
}


extern XtPointer
XmFontListEntryGetFont(XmFontListEntry entry,
		       XmFontType *type_return)
{
    if (entry == NULL)
    {
	DEBUGOUT(_LtDebug(__FILE__, NULL, "XmFontListEntryGetFont(NULL)\n"));
	if (type_return)
	    *type_return = XmFONT_IS_FONT;
	return (XtPointer)NULL;
    }

    DEBUGOUT(_LtDebug(__FILE__, NULL, "XmFontListEntryGetFont()\n"));

    if (type_return)
    {
	*type_return = entry->type;
    }

    return (XtPointer)entry->font;
}


extern char *
XmFontListEntryGetTag(XmFontListEntry entry)
{
    return XtNewString(entry->tag);
}


extern XmFontListEntry
XmFontListEntryLoad(Display *display,
		    char *font_name,
		    XmFontType type,
		    char *tag)
{
    XmFontListEntry entry;
    XrmValue fromString, toFont, cvtArg[2];
    Boolean success = False;
    char *p;

    DEBUGOUT(_LtDebug(__FILE__, NULL, "XmFontListEntryLoad(%s, tag %s)\n",
		      font_name, tag));

    /* Strip trailing semi-colon */
    for (p = font_name; *p; p++)
    {
    }

    while (p > font_name && *(p - 1) == ',')
    {
	p--;
	*p = '\0';
    }

    if (*font_name == '"' && *(p-1) == '"') {
	font_name++;
	p--;
	*p = '\0';
    }

    fromString.addr = font_name;
    fromString.size = strlen(font_name) + 1;

    cvtArg[0].addr = (XPointer)&display;
    cvtArg[0].size = sizeof(Display *);

    entry = (XmFontListEntry)XtMalloc(sizeof(struct _XmFontListRec));
    entry->font = NULL; /* for the XtCallConverter() call */

    switch (type)
    {
    case XmFONT_IS_FONT:
	toFont.addr = (XPointer)&(entry->font);
	toFont.size = sizeof(XFontStruct *);

	success = XtCallConverter(display, XtCvtStringToFontStruct, cvtArg,
	                          (Cardinal)1, &fromString, &toFont, NULL);
	break;

    case XmFONT_IS_FONTSET:
	toFont.addr = (XPointer)&(entry->font);
	toFont.size = sizeof(XFontSet);

	/* XtCvtStringToFontSet needs 2 arguments : display and locale */
	cvtArg[1].addr = (XPointer)XtNewString(setlocale(LC_CTYPE, NULL))/*&"C"*/;	/* FIX ME */
	cvtArg[1].size = sizeof(char *);

	success = XtCallConverter(display, XtCvtStringToFontSet, cvtArg,
	                          (Cardinal)2, &fromString, &toFont, NULL);
	XtFree(cvtArg[1].addr);
	break;
    }

    if (!success || !entry->font)
    {
	DEBUGOUT(_LtDebug(__FILE__, NULL, "XmFontListEntryLoad => NULL\n"));
	XtFree((char *)entry);
	return (XmFontListEntry)NULL;
    }
    else
    {
	DEBUGOUT(_LtDebug(__FILE__, NULL, "XmFontListEntryLoad success!\n"));
	entry->tag = XtNewString(tag);
	entry->type = type;
	return entry;
    }
}


extern void
XmFontListFree(XmFontList list)
{
    if (!list || list == (XmFontList)XmUNSPECIFIED)
    {
	return;
    }

    DEBUGOUT(_LtDebug(__FILE__, NULL, "XmFontListFree(0x%X)\n", list));

    __XmFontListDealloc(list);
}


extern void
XmFontListFreeFontContext(XmFontContext context)
{
    DEBUGOUT(_LtDebug(__FILE__, NULL,
		      "XmFontListFreeFontContext(0x%X)\n", context));
    XtFree((char *)context);
}


extern Boolean
XmFontListInitFontContext(XmFontContext *context,
			  XmFontList fontlist)
{
    if (fontlist && context)
    {
	*context = (XmFontContext)XtMalloc(sizeof(struct _XmFontListContextRec));
	(*context)->fontlist = fontlist;
	(*context)->current_entry = -1;

	return True;
    }
    else
    {
	return False;
    }
}


extern XmFontListEntry
XmFontListNextEntry(XmFontContext context)
{
    context->current_entry++;

    if (context->current_entry < __XmFontListNumEntries(context->fontlist))
    {
	return &context->fontlist[context->current_entry];
    }
    else
    {
	return NULL;
    }
}


extern Boolean
XmFontListGetNextFont(XmFontContext context, XmStringCharSet *charset, XFontStruct **font)
{
    context->current_entry++;

    if (context->current_entry < __XmFontListNumEntries(context->fontlist))
    {
	if (context->fontlist[context->current_entry].type == XmFONT_IS_FONT)
	{
	    *font = (XFontStruct *)(context->fontlist
				    [context->current_entry].font);
	    *charset = XtNewString(context->fontlist
				   [context->current_entry].tag);
	}
	else
	{
	    XFontStruct **foo;
	    char **bar;

	    if (XFontsOfFontSet((XFontSet)(context->fontlist
					   [context->current_entry].font),
				&foo,
				&bar) < 1)
	    {
		*font = NULL;
	    }
	    else
	    {
		*font = foo[0];
	    }

	    *charset = XtNewString(context->fontlist
				   [context->current_entry].tag);
	}
	return True;
    }
    else
    {
	return False;
    }
}


extern XmFontList
XmFontListRemoveEntry(XmFontList oldlist,
		      XmFontListEntry entry)
{
    XmFontList newFontList;
    int i, j;
    int oldnr, newnr;

    if (!oldlist)
       return (XmFontList)NULL;
    if (!entry)
       return oldlist;

    /* entry match >= 0 entries in oldlist, right?! */
    oldnr = __XmFontListNumEntries(oldlist);
    newFontList = __XmFontListAlloc(oldnr);
    j = 0;
    for (i = 0; oldlist[i].tag != NULL; i++)
    {
	if (!(!strcmp(entry->tag, oldlist[i].tag) &&
	      entry->type == oldlist[i].type &&
	      entry->font == oldlist[i].font))
	{
	    newFontList[j].tag  = XtNewString(oldlist[i].tag);
	    newFontList[j].type = oldlist[i].type;
	    newFontList[j].font = oldlist[i].font;

	    j++;
	}
    }
    newnr = j;

    /* Motif manpage tells to check here */
    if (oldnr==newnr)
    {
       __XmFontListDealloc(newFontList);
       return oldlist;
     }
    else
    {
       __XmFontListDealloc(oldlist);
       return newFontList;
     }
}
