/***************************************************************************
                          game.c  -  description
                             -------------------
    begin                : Thu Sep 6 2001
    copyright            : (C) 2001 by Michael Speck
    email                : kulkanie@gmx.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "lbreakout.h"
#include "list.h"
#include "event.h"
#include "config.h"
#include "difficulty.h"
#include "shrapnells.h"
#include "levels.h"
#include "player.h"
#include "chart.h"
#include "shine.h"
#include "credit.h"
#include "bricks.h"
#include "shots.h"
#include "frame.h"
#include "paddle.h"
#include "balls.h"
#include "extras.h"
#include "game.h"
#ifdef SOUND
#include "audio.h"
#endif

SDL_Surface *bkgnd = 0; /* current background picture */
SDL_Surface *offscreen = 0; /* buffer with frame, background and bricks */
int screenshot = 0; /* index of current screenshot */
extern Font *font; /* standard font */
Player *player = 0; /* current player */
extern Config config; /* lbreakout config struct */
extern int term_game; /* terminate game */
extern Sdl sdl; /* SDL struct */
extern Brick bricks[MAP_WIDTH][MAP_HEIGHT];
Diff *diff = 0; /* current difficulty information */
extern int keystate[SDLK_LAST];
extern int buttonstate[BUTTON_COUNT];
extern int active[EX_NUMBER];
extern int shot_bricks_hit, shot_mx, shot_my; /* position at which brick (and maybe neighbor) was destroyed */
extern int brick_count;
extern char **levelset_names; /* names of all available levelsets */
char *levelset_name = 0; /* pointer (may not be deleted) to the current levelset's name */
extern Player players[MAX_PLAYERS]; /* player infos */
#ifdef SOUND
extern Sound_Chunk *wav_click;
extern Sound_Chunk *wav_damn, *wav_dammit, *wav_wontgiveup, *wav_excellent, *wav_verygood;
#endif
extern List *exp_bricks;
extern List *heal_bricks;

/* grow stuff put here because of interaction between balls and bricks */
extern int grow;
extern int grow_mask[MAP_WIDTH][MAP_HEIGHT]; /* indicates which tiles are blocked by a ball */
extern Brick_Conv brick_conv_table[BRICK_COUNT];
extern int brick_count;
extern List *balls;
extern int ball_dia;
extern int shadow_size;

/*
====================================================================
Locals
====================================================================
*/

/*
====================================================================
Create all bricks by ids stored to grow_mask.
====================================================================
*/
void game_grow_bricks()
{
    int i, j;
    int px, py, refresh_w, refresh_h;
	Ball *list_ball;

    /* clear occupied grow parts */
    list_reset( balls );
    while ( ( list_ball = list_next( balls ) ) != 0 )  {
        grow_mask[(list_ball->x) / BRICK_WIDTH][(list_ball->y) / BRICK_HEIGHT] = -1;
        grow_mask[(list_ball->x + ball_dia) / BRICK_WIDTH][(list_ball->y) / BRICK_HEIGHT] = -1;
        grow_mask[(list_ball->x) / BRICK_WIDTH][(list_ball->y + ball_dia) / BRICK_HEIGHT] = -1;
        grow_mask[(list_ball->x + ball_dia) / BRICK_WIDTH][(list_ball->y + ball_dia) / BRICK_HEIGHT] = -1;
    }
    /* refresh tile size */
    refresh_w = BRICK_WIDTH;
    refresh_h = BRICK_HEIGHT;
    if ( config.shadow ) {
        refresh_w += shadow_size;
        refresh_h += shadow_size;
    }
    /* add bricks */
    px = py = 0;
    for ( j = 0; j < MAP_HEIGHT; j++ ) {
        for ( i = 0; i < MAP_WIDTH; i++ ) {
            if ( grow_mask[i][j] >= BRICK_GROW_FIRST && grow_mask[i][j] <= BRICK_GROW_LAST ) {
                bricks[i][j].id = brick_conv_table[grow_mask[i][j]].id;
                bricks[i][j].type = brick_conv_table[grow_mask[i][j]].type;
                bricks[i][j].score = brick_conv_table[grow_mask[i][j]].score;
                bricks[i][j].dur = brick_conv_table[grow_mask[i][j]].dur;
                /* keep the extra that is already assigned to this position */
                bricks[i][j].exp_time = -1;
                bricks[i][j].heal_time = -1;
                if ( !active[EX_DARKNESS] ) {
                    if ( config.shadow )
                         brick_draw_complex( i, j, px, py );
                    else
                         brick_draw( offscreen, i, j, 0 );
                }   
                /* add id to grown_brick mask */
                player->grown_bricks[i][j] = brick_conv_table[grow_mask[i][j]].id;
                /* adjust brick count */
                brick_count++;
                /* refresh*/
                if ( !active[EX_DARKNESS] ) {
                    DEST( sdl.screen, px, py, refresh_w, refresh_h );
                    SOURCE( offscreen, px, py );
                    blit_surf();
                    add_refresh_rect( px, py, refresh_w, refresh_h );
                }
            }	
            px += BRICK_WIDTH;
        }            
        py += BRICK_HEIGHT;
        px  = 0;
    }
    grow = 0;
    /* get new targets */
    balls_get_new_targets( -1, -1 );
}

/*
====================================================================
Initate current level from player's level resource pointer.
Recreates offscreen (frame, background, bricks) and resets paddle,
balls, extras. Do not add the bricks already removed:
player->bricks[x][y] == 0.
Return Value: True if successful
====================================================================
*/
enum { WITH_CREDITS = 1 };
int game_init_level( Player *player, int with_credits )
{
    int i, j, k;

    /* load bricks&extras from level data */
    bricks_create( player->level );
    /* remove cleared bricks */
    for ( i = 1; i < MAP_WIDTH - 1; i++ )
        for ( j = 1; j < MAP_HEIGHT - 1; j++ ) {
            if ( player->bricks[i][j] == -99 ) {
                /* initiate duration -- 0 means this brick was removed */
                if ( bricks[i][j].type == MAP_EMPTY )
                    player->bricks[i][j] = 0;
                else
                    player->bricks[i][j] = bricks[i][j].dur;
            }
            else
                if ( player->bricks[i][j] == 0 ) {
                    if ( bricks[i][j].dur > 0 ) brick_count--;
                    bricks[i][j].id = -1;
                    bricks[i][j].dur = -1;
                    bricks[i][j].type = MAP_EMPTY;
                }
                else 
                    if ( bricks[i][j].type != MAP_BRICK_HEAL ) /* generative bricks are kept reset */
                        if ( bricks[i][j].dur > 1 ) { 
                            /*if multiple duration we need to use the player->bricks value */
                            for ( k = 0; k < bricks[i][j].dur - player->bricks[i][j]; k++ )
                                bricks[i][j].id--;
                            bricks[i][j].dur = player->bricks[i][j];
                        }
            /* regnerative bricks with a durability of less than three shall be added to the regen list */
            if ( bricks[i][j].type == MAP_BRICK_HEAL && bricks[i][j].dur < 3 ) {
                bricks[i][j].mx = i;
                bricks[i][j].my = j;
                bricks[i][j].heal_time = BRICK_HEAL_TIME;
                list_add( heal_bricks, &bricks[i][j] );
            }
            if ( player->grown_bricks[i][j] ) {
                if ( bricks[i][j].type != MAP_EMPTY ) 
                    fprintf( stderr, "WARNING: Grown brick is blocked by another brick at %i,%i.\n", i, j );
                else {
                    /* add grown brick */
                    bricks[i][j].id = player->grown_bricks[i][j];
                    bricks[i][j].dur = 1;
                    bricks[i][j].type = MAP_BRICK;
                    brick_count++;
                }
            }
        }

    /* recreate offscreen */
    if ( offscreen ) SDL_FreeSurface( offscreen );
    offscreen = create_surf( sdl.screen->w, sdl.screen->h, SDL_SWSURFACE );
    SDL_SetColorKey( offscreen, 0, 0 );

    /* add&create background */
    if ( bkgnd ) SDL_FreeSurface( bkgnd );
    bkgnd = bkgnd_draw( -1 );
    /* add frame */
    frame_draw();
    /* add bricks */
    bricks_draw();
    /* draw lives */
    frame_draw_lives( player->lives, diff->max_lives );
    /* reset score */
    frame_score_reset( player->score );
    /* draw player's name */
    frame_draw_player_name( player->name );
    /* reset paddle */
    paddle_reset();
    /* event reset */
    event_reset();
    /* shots reset */
    shots_reset();
    /* reset balls */
    balls_set_vel( diff->v_start, diff->v_change, diff->v_max );
    balls_reset();
    /* reset extras */
    extras_reset();
    /* reset shrapnells */
    shrapnells_reset();
    /* reset shine */
    shine_reset();
    /* reset explosions */
    exps_clear();
    exps_set_dark( 0 );
    /* initiate credit */
    if ( with_credits )
        credit_init( player->level->name, player->level->author, levels_get_id( player->level ) );

    /* show offscreen */
    FULL_DEST( sdl.screen ); FULL_SOURCE( offscreen ); blit_surf();
    /* no refresh rect as we want to use dim effect */
    return 1;
}

/*
====================================================================
Display info about player
====================================================================
*/
void player_info()
{
    char info[128];
    /* short info about actual player */
    /* no info if this is the last active player */
    if ( players_count() > 1 ) {
        sprintf( info, "Next Player: %s", player->name );
        confirm( font, info, CONFIRM_ANY_KEY );
    }
    else
        refresh_screen( 0, 0, 0, 0 );
}
/*
====================================================================
Runs Lbreakout until player looses ball or explicitly wishes
to quit, restart. (just request! confirmation not done here)
Assumes that the current offscreen was already displayed on screen
by game_init_level.
====================================================================
*/
enum {
    REQUEST_NONE = 0,
    REQUEST_RESTART,
    REQUEST_QUIT,
    REQUEST_NEXT_LEVEL,
    REQUEST_NEXT_PLAYER,
    REQUEST_PAUSE
};
int breakout_run() {
    SDL_Event event;
    int result = REQUEST_NONE;
    int ms;
    int fps_delay = 0;
	Brick *brick;
    
    /* delay */
    switch ( config.fps ) {
        case 1: fps_delay = 20; break;
        case 2: fps_delay = 10; break;
        case 3: fps_delay = 5; break;
    }
    /* grab input if wanted */
    event_grab_input();
    /* run main loop */
    event_reset();
    reset_timer();
    while( result == REQUEST_NONE ) {
        if ( config.fps ) SDL_Delay( fps_delay );
        if ( event_poll( &event ) ) {
            switch ( event.type ) {
                case SDL_QUIT: term_game = 1; result = REQUEST_QUIT; break;
                case SDL_KEYDOWN:
                    if ( keystate[config.k_fire] && weapon_installed() && !weapon_firing() ) weapon_start_fire();
                    break;
                case SDL_KEYUP:
                    /* stop firing shots */
                    if ( weapon_firing() && !keystate[config.k_fire] ) weapon_stop_fire();
                    /* return balls that may do so */
                    if ( event.key.keysym.sym == config.k_return )
                        balls_return();
                    /* various requests */
                    switch ( event.key.keysym.sym ) {
						case SDLK_q:
                        case SDLK_ESCAPE: result = REQUEST_QUIT; break;
                        case SDLK_r:
                            if (  player->lives <= 1 && player->score < diff->life_cost ) break;
                            result = REQUEST_RESTART;
                            player->lives--;
                            break;
                        case SDLK_f:
                            event_ungrab_input();
                            config.fullscreen = !config.fullscreen;
                            set_video_mode( std_video_mode( config.fullscreen ) );
                            /* redraw offscreen to get background */
                            FULL_DEST( sdl.screen ); FULL_SOURCE( offscreen ); blit_surf();
                            refresh_screen( 0, 0, 0, 0 );
                            event_grab_input();
                            break;
                        case SDLK_s:
#ifdef SOUND
                            config.sound = !config.sound;
                            sound_enable( config.sound );
#endif
                            break;
                        case SDLK_a:
                            config.anim++;
                            if ( config.anim >= 4 ) config.anim = 0;
                            break;
                        case SDLK_TAB:
                            take_screenshot( screenshot++ );
                            break;
                        case SDLK_p:
                            result = REQUEST_PAUSE;
                            break;
                        default: break;
                    }
                    break;
                case SDL_MOUSEBUTTONDOWN:
                    if ( buttonstate[LEFT_BUTTON] && weapon_installed() && !weapon_firing() ) weapon_start_fire();
                    break;
                case SDL_MOUSEBUTTONUP:
                    if ( weapon_firing() && !buttonstate[LEFT_BUTTON] ) weapon_stop_fire();
                    if ( event.button.button == MIDDLE_BUTTON )
                        balls_return();
                    break;
            }
        }
        ms = get_time();
        /* hide */
        frame_info_hide();
        extras_hide();
        paddle_hide();
        balls_hide();
        shots_hide();
        shrapnells_hide();
        wall_hide();
        if ( !active[EX_DARKNESS] ) frame_score_hide();
        shine_hide();
        exps_hide();
        credit_hide();
        /* update */
        paddle_update( ms );
        if ( !balls_update( ms ) ) {
            result = REQUEST_NEXT_PLAYER;
            player->lives--;
        }
        shots_update( ms );
        extras_update( ms );
        wall_update( ms );
        shrapnells_update( ms );
        frame_score_update( ms );
        shine_update( ms );
        exps_update( ms );
        credit_update( ms );
        /* check if bricks were destroyed by shots */
        if ( shot_bricks_hit ) {
            balls_get_new_targets( shot_mx, shot_my );
            if ( shot_bricks_hit == 2 ) /* hit neighbor, too */
                balls_get_new_targets( shot_mx + 1, shot_my );
            shot_bricks_hit = 0;
        }
		/* check if bricks were destroyed by explosion */
		if ( exp_bricks->count > 0 ) {
			list_reset( exp_bricks );
			while ( ( brick = list_next( exp_bricks ) ) != 0 ) {
				brick->exp_time -= ms;
				if ( brick->exp_time <= 0 ) {
					brick->exp_time = -1;
					brick_remove( brick->mx, brick->my, SHR_BY_EXPL, vector_get( 0, 0 ) );
					balls_get_new_targets( brick->mx, brick->my );
					list_delete_current( exp_bricks );
				}
			}
		}
        /* check if bricks regenerate */
        if ( heal_bricks->count > 0 ) {
            list_reset( heal_bricks );
            while ( ( brick = list_next( heal_bricks ) ) != 0 ) {
                /* skip brick if destroyed meanwhile */
                if ( brick->type != MAP_BRICK_HEAL ) {
                    list_delete_current( heal_bricks );
                    continue;
                }
                brick->heal_time -= ms;
                if ( brick->heal_time < 0 ) {
                    brick->dur++;
                    brick->id++;
                    /* redraw */
                    if ( !active[EX_DARKNESS] ) {
                        brick_draw( offscreen, brick->mx, brick->my, 0 );
                        brick_draw( sdl.screen, brick->mx, brick->my, 0 );
                        add_refresh_rect( brick->mx * BRICK_WIDTH, brick->my * BRICK_HEIGHT, BRICK_WIDTH, BRICK_HEIGHT );
                    }
                    if ( brick->dur < 3 ) {
                        /* initate next healing step */
                        brick->heal_time = BRICK_HEAL_TIME;
                    }
                    else {
                        brick->heal_time = -1;
                        list_delete_current( heal_bricks );
                    }
                }
            }
        }
        /* check if this results in growing */
        if ( grow ) game_grow_bricks();
        /* update score */
        if ( player_score_changed( player ) ) frame_score_set( player->score );
        /* update ball speed */
        if ( !active[EX_SLOW] && !active[EX_FAST] ) balls_inc_vel( ms );
        /* check if all bricks where destroyed */
        if ( brick_count == 0 ) result = REQUEST_NEXT_LEVEL;
        /* show -- some things will be missing if darkness is enabled */
        if ( config.ball_level == BALL_BELOW_BONUS )
			balls_show();
       	shots_show();
       	extras_show();
		if ( config.ball_level == BALL_ABOVE_BONUS )
			balls_show();
        paddle_show();
        if ( !active[EX_DARKNESS] ) wall_show();
        shine_show();
        exps_show();
        if ( !active[EX_DARKNESS] ) {
			shrapnells_show();
			frame_score_show();
        	frame_info_show();
		}	
        credit_show();
        /* update anything that was changed */
        refresh_rects();
    }
    /* release input */
    event_ungrab_input();
    return result;
}
/*
====================================================================
Fade all animations until they disappear
====================================================================
*/
void fade_anims()
{
    float alpha = 0;
    int ms;
    frame_remove_life();
    reset_timer();
    while ( alpha < 255 ) {
        paddle_hide();
        balls_hide();
        extras_hide();
        shrapnells_hide();
        shots_hide();
        wall_hide();
        credit_hide();
        ms = get_time();
        alpha += 0.5 * ms;
        if ( alpha > 255 ) alpha = 255;
        shrapnells_update( ms );
        paddle_alphashow( alpha );
        balls_alphashow( alpha );
        extras_alphashow( alpha );
        shots_alphashow( alpha );
        shrapnells_show();
        wall_alphashow( alpha );
        credit_alphashow( alpha );
        refresh_rects();
    }
}

/*
====================================================================
Publics
====================================================================
*/

/*
====================================================================
Load all static resources like frame, bricks, balls, extras...
====================================================================
*/
void game_create()
{
    bricks_load();
    paddle_load();
    ball_load();
    shot_load();
    extras_load();
    frame_create();
    shrapnells_init();
    shine_load();
    /* load names of all valid levelsets */
    levelsets_load_names();
}
/*
====================================================================
Delete anything created by game_create();
====================================================================
*/
void game_delete()
{
    bricks_delete();
    paddle_delete();
    ball_delete();
    shot_delete();
    extras_delete();
    frame_delete();
    shrapnells_delete();
    shine_delete();
    /* free levelset names */
    levelsets_delete_names();
}
/*
====================================================================
Initiates players and the first level.
Return Value: True if successful
====================================================================
*/
int game_init()
{
    int i;
    char path[512];
    char *setname = 0;
    /* load level */
    setname = levelset_names[config.levelset_id];
    if ( levelset_names[config.levelset_id][0] == '~' ) {
        sprintf( path, "%s/%s/lbreakout2-levels", (getenv( "HOME" )?getenv( "HOME" ):"."), CONFIG_DIR_NAME );
        setname++;
    }
    else
        sprintf( path, "%s/levels", SRC_DIR );
    if ( !levels_load( path, setname ) ) return 0;
	/* set global pointer to set name */
	levelset_name = levelset_names[config.levelset_id];
    /* set difficulty */
    diff = diff_get( config.diff );
    /* initiate players from config information */
    players_clear();
    for ( i = 0; i < config.player_count; i++ )
        player_add( config.player_names[i], diff->lives, levels_get_first() );
    player = players_get_first();
    /* initiate level */
    return game_init_level( player, WITH_CREDITS );
}

/*
====================================================================
Free all memory allocated by game_init()
====================================================================
*/
void game_clear()
{
    if ( offscreen ) SDL_FreeSurface( offscreen ); offscreen = 0;
    if ( bkgnd ) SDL_FreeSurface( bkgnd ); bkgnd = 0;
    credit_clear();
}
/*
====================================================================
Run game after first level was initiated. Initiates next levels,
too.
====================================================================
*/
void game_run()
{
    int leave = 0;
    int next_player = 0;
    int result;
    char str[64];
    int i;

    fade_screen( FADE_IN, FADE_DEF_TIME );
    player_info();
    while ( !leave && !term_game ) {
        /* run game until player wants to quit or restart or lost the ball or reached next level */
        result = breakout_run();
        switch ( result ) {
            case REQUEST_PAUSE:
                confirm( font, "Pause", CONFIRM_PAUSE );
                break;
            case REQUEST_QUIT:
                if ( term_game || confirm( font, "Exit Game? y/n", CONFIRM_YES_NO ) ) 
                    leave = 1;
                break;
            case REQUEST_NEXT_LEVEL:
#ifdef SOUND
                if ( config.speech ) {
                    if ( rand() % 2 )
                        sound_play( wav_excellent );
                    else
                        sound_play( wav_verygood );
                }
#endif
                player->level = levels_get_next( player->level );
                player_reset_bricks( player );
                if ( !player->level ) {
                    confirm( font, "Congratulations! No more levels left!", CONFIRM_ANY_KEY );
                    /* player finished all levels so ''deactivate'' him */
                    player->lives = 0;
                    /* if this was the last player show congrats */
                    if ( players_count() == 0 ) {
                        /* congrats */
                        confirm( font, "Game Over!", CONFIRM_ANY_KEY );
                        leave = 1;
                        break;
                    }
                    /*
                    else {
                        player = players_get_next();
                        next_player = 1;
                    }
                    */
                }
                /* switch player in any case after level was finished so the other 
                   players don't have to wait too long */
                if ( players_count() > 1 ) {
                    player = players_get_next();
                    next_player = 1;
                }
                if ( next_player ) {
                    fade_anims();
                    game_init_level( player, WITH_CREDITS );
                    player_info();
                    next_player = 0;
                }
                else {
                    fade_screen( FADE_OUT, FADE_DEF_TIME );
                    game_init_level( player, WITH_CREDITS );
                    fade_screen( FADE_IN, FADE_DEF_TIME );
                }
                break;
            case REQUEST_RESTART:
                /* if there are no lifes remaining this is a continue request */
                if ( player->lives == 0 ) {
                    sprintf( str, "Buy a continue? (%i points) y/n", diff->life_cost );
                    if ( confirm( font , str, CONFIRM_YES_NO ) ) {
                        player->score -= diff->life_cost;
                        player->lives++;
                    }
                    else
                        break;
                }
                else
                    if ( !confirm( font, "Restart Level? y/n", CONFIRM_YES_NO ) ) break;
                player_reset_bricks( player );
            case REQUEST_NEXT_PLAYER:
                /* if we got here the current player lost a life */
#ifdef SOUND
                if ( result == REQUEST_NEXT_PLAYER && config.speech ) {
                    if ( rand() % 2 )
                        sound_play( wav_damn );
                    else
                        sound_play( wav_dammit );
                }
#endif
                /* if he has no lives left but enough score to buy one he may do so
                else he will be kicked out of the game as lives == 0 */
                if ( player->lives == 0 && player->score >= diff->life_cost ) {
                    sprintf( str, "Buy a continue? (%i points) y/n", diff->life_cost );
                    if ( confirm( font , str, CONFIRM_YES_NO ) ) {
                        player->score -= diff->life_cost;
                        player->lives++;
#ifdef SOUND
                        if ( config.speech )
                            sound_play( wav_wontgiveup );
#endif
                    }
                }
                if ( players_count() >= 1 )
                    player = players_get_next();
                else {
                    /* all players died - quit game */
                    leave = 1;
                    confirm( font, "Game Over!", CONFIRM_ANY_KEY );
                    break;
                }
                fade_anims();
                game_init_level( player, WITH_CREDITS );
                player_info();
                break;
        }
    }
    fade_screen( FADE_OUT, FADE_DEF_TIME );
	/* modify score of involved players */
    if ( diff->score_mod != 1.0 )
		for ( i = 0; i < config.player_count; i++ )
        	players[i].score = (int)(diff->score_mod * players[i].score);
    /* add players to chart */
    chart_clear_new_entries();
    for ( i = 0; i < config.player_count; i++ )
        chart_add( chart_set_query( levelset_name), players[i].name, levels_get_id( players[i].level ) + 1, players[i].score );
	/* and save... maybe it crashes and we don't want to loose highscore results, right? */
	chart_save();
}

/*
====================================================================
Test this level until ball gets lost.
====================================================================
*/
void game_test_level( Level *level )
{
    Player test_player;
    /* initate testplayer */
    strcpy( test_player.name, "Player" );
    test_player.score = 0;
    test_player.old_score = 0;
    test_player.lives = 1;
    test_player.level = level;
    player_reset_bricks( &test_player );
    player = &test_player;
    /* difficulty */
    diff = diff_get( config.diff );
    /* init game */
    game_init_level( player, 0 /* no credits */ );
    /* run */
    fade_screen( FADE_IN, FADE_DEF_TIME );
    breakout_run();
    fade_screen( FADE_OUT, FADE_DEF_TIME );
}
