
#include "irc.h"
#include "struct.h"
#include "dcc.h"
#include "ircaux.h"
#include "ctcp.h"
#include "cdcc.h"
#include "input.h"
#include "status.h"
#include "lastlog.h"
#include "screen.h"
#include "vars.h" 
#include "misc.h"
#include "output.h"
#include "module.h"
#include "hook.h"
#include "hash2.h"
#include "napster.h"
#include "md5.h"

#include "bsdglob.h"
#include "modval.h"

#include <sys/time.h>
#include <sys/stat.h>
#if !defined(WINNT) || !defined(__EMX__)
#include <sys/mman.h>
#endif

#define DEFAULT_FILEMASK "*.mp3"

#ifndef MAP_FAILED
#define MAP_FAILED (void *) -1
#endif

Stats statistics = { 0, };


static unsigned char _buffer[32];
static int _bptr = 0;

static int current_sending = 0;



Files *fserv_files = NULL;
GetFile *napster_sendqueue = NULL;


void clear_files(Files **f)
{
Files *last, *f1 = *f;
	while (f1)
	{
		last = f1->next;
		new_free(&f1->filename);
		new_free(&f1->checksum);
		new_free(&f1);
		f1 = last;
	}
	*f = NULL;
}

static char *convertnap_dos(char *str)
{
register char *p;
	for (p = str; *p; p++)
		if (*p == '/')
			*p = '\\';
	return str;
}

static char *convertnap_unix(char *arg)
{
register char *x = arg;
	while (*x)
	{
		if (*x == '\\')
			*x = '/';
		x++;
	}
	return arg;
}


char *mode_str(int mode)
{
	switch(mode)
	{	
		case 0:
			return "Stereo";
		case 1:
			return "Joint-Stereo";
		case 2:
			return "Dual-Channel";
		case 3:
			return "Mono";
	}
	return empty_string;
}

char *print_time(time_t input)
{
	static char	buff[40];
	time_t		seconds,
			minutes;
	seconds = input;
	minutes = seconds / 60;
	seconds = seconds % 60;
	sprintf(buff, "%02u:%02u", (unsigned int)minutes, (unsigned int)seconds);
	return buff;
}                                        

char *make_mp3_string(FILE *fp, Files *f, char *fs, char *dirbuff)
{
	static char	buffer[2*NAP_BUFFER_SIZE+1];
	char		*s,
			*loc,
			*p,
			*fn;

	if (!fs || !*fs)
		return empty_string;
	memset(buffer, 0, sizeof(buffer));

	loc = LOCAL_COPY(f->filename);
	fn = strrchr(loc, '/');
	*fn++ = 0;
	if ((p = strrchr(loc, '/')))
		*p++ = 0;
	/* fn should point to the filename and p to the dir */
	/* 
	 * init the dir keeper 
	 * or cmp the old dir with the new
	 */
	if (dirbuff && (!*dirbuff || strcmp(dirbuff, p)))
	{
		strcpy(dirbuff, p);
		if (fp)
			fprintf(fp, "\nDirectory [ %s ]\n", dirbuff);
		else
			return NULL;
	}
	/* size bitrate [time] filename */
	s = buffer;
	while (*fs)
	{
		if (*fs == '%')
		{
			int prec = 0, fl = 0;
			fs++;
			if (isdigit(*fs))
			{
				prec = strtol(fs, &fs, 0);
				if (*fs == '.')
					fl = strtoul(fs+1, &fs, 0);
			}
			switch(*fs)
			{
				case '%':
					*s++ = *fs;
					break;
				case 'b':
					sprintf(s, "%*u", prec, f->bitrate);
					break;
				case 's':
					if (!prec) prec = 3;
					sprintf(s, "%*.*f%s", prec, fl, _GMKv(f->filesize), _GMKs(f->filesize));
					break;
				case 't':
					strcpy(s, print_time(f->time));
					break;
				case 'T':
					strcpy(s, ltoa(f->time));
					break;
				case 'f':
					strcpy(s, fn);
					break;
				case 'F':
					strcpy(s, f->filename);
					break;
				case 'M':
					strcpy(s, f->checksum);
					break;
				case 'S':
					strcpy(s, mode_str(f->stereo));
					break;
				case 'H':
					sprintf(s, "%*.*f", prec, fl, ((double)f->freq) / ((double)1000.0));
					break;
				case 'h':
					sprintf(s, "%*u", prec, f->freq);
					break;
				default:
					*s++ = *fs;
					break;
			}
		}
		else if (*fs == '\\')
		{
			fs++;
			switch(*fs)
			{
				case 'n':
					strcpy(s, "\n");
					break;
				case 't':
					strcpy(s, "\t");
					break;
				default:
					*s++ = *fs++;
			}
		}
		else
			*s++ = *fs;
		while (*s) s++;
		fs++;
	}
	if (fp && *buffer)
		fprintf(fp, buffer);
	return buffer;
}


int read_glob_dir(char *path, int globflags, glob_t *globpat, int recurse)
{
	char	buffer[NAP_BUFFER_SIZE+1];
	
	sprintf(buffer, "%s/*", path);
	bsd_glob(buffer, globflags, NULL, globpat);
	if (recurse)
	{
		int i = 0;
		int old_glpathc = globpat->gl_pathc;
		for (i = 0; i < old_glpathc; i++)
		{
			char *fn;
			fn = globpat->gl_pathv[i];
			if (fn[strlen(fn)-1] != '/')
				continue;
			sprintf(buffer, "%s*", fn);
			bsd_glob(buffer, globflags|GLOB_APPEND, NULL, globpat);
		}
		while (i < globpat->gl_pathc)
		{
			for (i = old_glpathc, old_glpathc = globpat->gl_pathc; i < old_glpathc; i++)
			{
				char *fn;
				fn = globpat->gl_pathv[i];
				if (fn[strlen(fn)-1] != '/')
					continue;
				sprintf(buffer, "%s*", fn);
				bsd_glob(buffer, globflags|GLOB_APPEND, NULL, globpat);
			}
		}
	}
	return 0;
}

unsigned int print_mp3(char *pattern, char *format, int freq, int number, int bitrate, int md5)
{
unsigned int count = 0;
Files *new;
char dir[NAP_BUFFER_SIZE];
char *fs = NULL;
	*dir = 0;
	for (new = fserv_files; new; new = new->next)
	{
		if (!pattern || (pattern && wild_match(pattern, new->filename)))
		{
			char *p;
			p = base_name(new->filename);
			if ((bitrate != -1) && (new->bitrate != bitrate))
				continue;
			if ((freq != -1) && (new->freq != freq))
				continue;
			if (do_hook(MODULE_LIST, "NAP MATCH %s %s %u %lu", p, new->checksum, new->bitrate, new->time))
			{
				if (!format || !*format)
				{
					if (md5)
						put_it("\"%s\" %s %dk [%s]", p, new->checksum, new->bitrate, print_time(new->time));
					else
						put_it("\"%s\" %s %dk [%s]", p, mode_str(new->stereo), new->bitrate, print_time(new->time));
				}
				else
				{
					if ((fs = make_mp3_string(NULL, new, format, dir)))
						put_it("%s", fs);
					else
						put_it("%s", make_mp3_string(NULL, new, format, dir));
				}
			}
		}
		if ((number > 0) && (count == number))
			break;
		count++;
	}
	return count;
}

BUILT_IN_DLL(print_napster)
{
	int	count = 0;
	int	bitrate = -1;
	int	number = -1;
	int	freq = -1;
	int	md5 = 0;
	char 	*fs_output = NULL;
	char	*tmp_pat = NULL;
	
	if ((get_dllstring_var("napster_format")))
		fs_output = m_strdup(get_dllstring_var("napster_format"));
	if (args && *args)
	{
		char *tmp;
		while ((tmp = next_arg(args, &args)) && *tmp)
		{
			int len;
			len = strlen(tmp);
			if (!my_strnicmp(tmp, "-BITRATE", len))
			{
				if ((tmp = next_arg(args, &args)))
					bitrate = (unsigned int) my_atol(tmp);
			}
			else if (!my_strnicmp(tmp, "-COUNT", len))
			{
				if ((tmp = next_arg(args, &args)))
					number = (unsigned int) my_atol(tmp);
			} 
			else if (!my_strnicmp(tmp, "-FREQ", 3))
			{
				if ((tmp = next_arg(args, &args)))
					freq = (unsigned int)my_atol(tmp);
			} 
			else if (!my_strnicmp(tmp, "-MD5", 3))
			{
				md5 = 1;
			} 
			else if (!my_strnicmp(tmp, "-FORMAT", 3))
			{
				if ((tmp = new_next_arg(args, &args)))
					malloc_strcpy(&fs_output, tmp);
			} 
			else
			{
				count += print_mp3(tmp, fs_output, freq, number, bitrate, md5);
				m_s3cat(&tmp_pat, " ", tmp);
			}
		}
	}
	else
		count += print_mp3(NULL, fs_output, freq, number, bitrate, md5);
	
	if (do_hook(MODULE_LIST, "NAP MATCHEND %d %s", count, tmp_pat ? tmp_pat : "*"))
		nap_say("Found %d files matching \"%s\"", count, tmp_pat ? tmp_pat : "*");
	new_free(&tmp_pat);
	new_free(&fs_output);
}

int _get_input(int file, unsigned char *bp, int size)
{
	if (read(file,  bp, size) != size)
		return -1;
	return 0;
}
                                        
static inline int readsync(int file)
{
	_bptr=0;
	_buffer[0]=_buffer[1];
	_buffer[1]=_buffer[2];
	_buffer[2]=_buffer[3];
	return _get_input(file, &_buffer[3], 1);
}


static inline int _fillbfr(int file, unsigned int size)
{
	_bptr=0;
        return _get_input(file, _buffer, size);
}


static inline unsigned int _getbits(int n)
{
	unsigned int	pos,
			ret_value;

        pos = _bptr >> 3;
	ret_value = _buffer[pos] << 24 |
		    _buffer[pos+1] << 16 |
		    _buffer[pos+2] << 8 |
		    _buffer[pos+3];
        ret_value <<= _bptr & 7;
        ret_value >>= 32 - n;
        _bptr += n;
        return ret_value;
}       


/*
 * header and side info parsing stuff ******************************************
 */
static inline void parse_header(AUDIO_HEADER *header) 
{
        header->IDex=_getbits(1);
        header->ID=_getbits(1);
        header->layer=_getbits(2);
        header->protection_bit=_getbits(1);
        header->bitrate_index=_getbits(4);
        header->sampling_frequency=_getbits(2);
        header->padding_bit=_getbits(1);
        header->private_bit=_getbits(1);
        header->mode=_getbits(2);
        header->mode_extension=_getbits(2);
        if (!header->mode) 
        	header->mode_extension=0;
        header->copyright=_getbits(1);
        header->original=_getbits(1);
        header->emphasis=_getbits(2);

	header->stereo = (header->mode == 3) ? 1 : 2;
	header->true_layer = 4 - header->layer;
}

int gethdr(int file, AUDIO_HEADER *header)
{
	int	retval;

	if ((retval=_fillbfr(file, 4))) 
		return retval;

	while (_getbits(11) != 0x7ff) 
	{
		if ((retval=readsync(file)) !=0) 
			return retval;
	}
	parse_header(header);
	return 0;
}

long get_bitrate(int fdes, time_t *mp3_time, unsigned int *freq_rate, unsigned long *filesize, int *stereo, long *id3)
{
	short t_bitrate[2][3][15] = {{
	{0,32,48,56,64,80,96,112,128,144,160,176,192,224,256},
	{0,8,16,24,32,40,48,56,64,80,96,112,128,144,160},
	{0,8,16,24,32,40,48,56,64,80,96,112,128,144,160}
	},{
	{0,32,64,96,128,160,192,224,256,288,320,352,384,416,448},
	{0,32,48,56,64,80,96,112,128,160,192,224,256,320,384},
	{0,32,40,48,56,64,80,96,112,128,160,192,224,256,320}
	}};

	int t_sampling_frequency[2][2][3] = {
		{ /* MPEG 2.5 samplerates */ 
  			{ 11025, 12000, 8000},
			{ 0,0,0 }
		},{ /* MPEG 2.0/1.0 samplerates */
			{ 22050 , 24000 , 16000},
			{ 44100 , 48000 , 32000}
		}
	};

	AUDIO_HEADER header;
	unsigned long btr = 0;

	struct stat	st;
	unsigned long	framesize = 0,
			totalframes = 0;
	
	if (freq_rate)
		*freq_rate =0;
	gethdr(fdes, &header);
	if (header.ID > 1 || header.layer > 2 || header.bitrate_index > 14) 
		return 0;
	btr = t_bitrate[header.ID][3-header.layer][header.bitrate_index];

	fstat(fdes, &st);

	if (t_sampling_frequency[header.IDex][header.ID][header.sampling_frequency] > 0)
		framesize = (header.ID ? 144000 : 72000) * btr / t_sampling_frequency[header.IDex][header.ID][header.sampling_frequency];
	totalframes = (st.st_size / (framesize + 1)) - 1;
	if (t_sampling_frequency[header.IDex][header.ID][header.sampling_frequency] > 0)
		*mp3_time = (time_t) (totalframes * (header.ID==0?576:1152)/t_sampling_frequency[header.IDex][header.ID][header.sampling_frequency]);
	*filesize = st.st_size;

	if (freq_rate)
		*freq_rate = t_sampling_frequency[header.IDex][header.ID][header.sampling_frequency];
	if (id3)
	{
		char buff[130];
		int rc;
		lseek(fdes, 0, SEEK_SET);
		*id3 = 0;
		rc = read(fdes, buff, 128);
		if (!strncmp(buff, "ID3", 3))
		{
			struct id3v2 {
				char tag[3];
				unsigned char ver[2];
				unsigned char flag;
				unsigned char size[4];
			} id3v2;
			unsigned char bytes[4];
			/* this is id3v2 */
			memcpy(&id3v2, buff, sizeof(id3v2));
			bytes[3] = id3v2.size[3] | ((id3v2.size[2] & 1 ) << 7);
			bytes[2] = ((id3v2.size[2] >> 1) & 63) | ((id3v2.size[1] & 3) << 6);
			bytes[1] = ((id3v2.size[1] >> 2) & 31) | ((id3v2.size[0] & 7) << 5);
			bytes[0] = ((id3v2.size[0] >> 3) & 15);

			*id3 = (bytes[3] | (bytes[2] << 8) | ( bytes[1] << 16) | ( bytes[0] << 24 )) + sizeof(struct id3v2);
		}
		lseek(fdes, st.st_size-128, SEEK_SET);
		rc = read(fdes, buff, 128);
		if ((rc == 128) && !strncmp(buff, "TAG", 3))
			*id3 = *id3 ? (*id3 * -1) : 1;
	}
	*stereo = header.mode;
	return btr;
}

off_t file_size (char *filename)
{
	struct stat statbuf;

	if (!stat(filename, &statbuf))
		return (off_t)(statbuf.st_size);
	else
		return -1;
}

char *calc_md5(int r, unsigned long mapsize)
{
unsigned char digest[16];			
md5_state_t state;
char buffer[BIG_BUFFER_SIZE+1];
struct stat st;
unsigned long size = 300000;
int di = 0;
int rc;

#if !defined(WINNT) && !defined(__EMX__)
	char *m;
#endif

	*buffer = 0;
	md5_init(&state);
	if ((fstat(r, &st)) == -1)
		return m_strdup("");
		
	if (!mapsize)
	{
		if (st.st_size < size)
			size = st.st_size;
	}
	else if (st.st_size < mapsize)
		size = st.st_size;
	else
		size = mapsize;

#if defined(WINNT) || defined(__EMX__)
	while (size)
	{
		unsigned char md5_buff[8 * NAP_BUFFER_SIZE+1];
		rc = (size >= (8 * NAP_BUFFER_SIZE)) ?  8 * NAP_BUFFER_SIZE : size;
		rc = read(r, md5_buff, rc);
		md5_append(&state, (unsigned char *)md5_buff, rc);
		if (size >= 8 * NAP_BUFFER_SIZE)
			size -= rc;
		else
			size = 0;
	}						
	md5_finish(digest, &state);
	{
#else
	if ((m = mmap((void *)0, size, PROT_READ, MAP_PRIVATE, r, 0)) != MAP_FAILED)
	{
		md5_append(&state, (unsigned char *)m, size);
		md5_finish(digest, &state);
		munmap(m, size);
#endif
		memset(buffer, 0, 200);
		for (di = 0, rc = 0; di < 16; ++di, rc += 2)
			snprintf(&buffer[rc], BIG_BUFFER_SIZE, "%02x", digest[di]);
		strcat(buffer, "-");
		strcat(buffer, ltoa(st.st_size));
	}
	return m_strdup(buffer);
}

unsigned int scan_mp3_dir(char *path, int recurse, int reload, int share)
{
	int	mt = 0;
	glob_t	globpat;
	int	i = 0;
	Files	*new;
	int	count = 0;
	int	r;
	char	buffer[2*NAP_BUFFER_SIZE+1];
	
	memset(&globpat, 0, sizeof(glob_t));
	read_glob_dir(path, GLOB_MARK|GLOB_NOSORT, &globpat, recurse);
	for (i = 0; i < globpat.gl_pathc; i++)
	{
		char	*fn;
		long	id3 = 0;
		fn = globpat.gl_pathv[i];
		if (fn[strlen(fn)-1] == '/')
			continue;
		if (!(mt = wild_match(DEFAULT_FILEMASK, fn)))
			continue;
		if (reload && find_in_list((List **)&fserv_files, globpat.gl_pathv[i], 0))
			continue;
		new = (Files *) new_malloc(sizeof(Files));
		new->filename = m_strdup(fn);
		if ((r = open(fn, O_RDONLY)) == -1)
			continue;
		new->bitrate = get_bitrate(r, &new->time, &new->freq, &new->filesize, &new->stereo, &id3);
		if (new->filesize && new->bitrate)
		{
			unsigned long size = 300000;
			switch(id3)
			{
				case 1:
				{
					if (new->filesize < size)
						size = new->filesize - 128;
				}
				case 0:
					lseek(r, 0, SEEK_SET);
					break;
				default:
				{
					lseek(r, (id3 < 0) ? -id3 : id3, SEEK_SET);
					if (id3 > 0)
					{
						if ((new->filesize - id3) < size)
							size = new->filesize - id3;
					}
					else
					{
						/*blah. got both tags. */
						if ((new->filesize + id3 - 128) < size)
							size = new->filesize + id3 - 128;
					}
					break;
				}
			}
			new->checksum = calc_md5(r, size);
			add_to_list((List **)&fserv_files, (List *)new);
			statistics.total_files++;
			statistics.total_filesize += new->filesize;
			count++;
			if (share && (nap_socket != -1))
			{
				sprintf(buffer, "\"%s\" %s %lu %u %u %lu", new->filename, 
					new->checksum, new->filesize, new->bitrate, new->freq, new->time);
				send_ncommand(CMDS_ADDFILE, convertnap_dos(buffer));
				statistics.shared_files++;
				statistics.shared_filesize += new->filesize;
			}
			if (!(count % 25))
			{
				lock_stack_frame();
				io("scan_mp3_dir");
				unlock_stack_frame();
				build_napster_status(NULL);
			}
			close(r);
		}
		else
		{
			new_free(&new->filename);
			new_free(&new);
		}
	}
	bsd_globfree(&globpat);
	return count;
}

void load_shared(char *fname)
{
char buffer[BIG_BUFFER_SIZE+1];
char *expand = NULL;
FILE *fp;
int count = 0;
Files *new;
	if (!fname || !*fname)
		return;
	if (!strchr(fname, '/'))
		sprintf(buffer, "%s/%s", get_string_var(CTOOLZ_DIR_VAR), fname);
	else
		sprintf(buffer, "%s", fname);
	expand = expand_twiddle(buffer);
	if ((fp = fopen(expand, "r")))
	{
		char *fn, *md5, *fs, *br, *fr, *t, *args;
		while (!feof(fp))
		{
			if (!fgets(buffer, BIG_BUFFER_SIZE, fp))
				break;
			args = &buffer[0];
			fn = new_next_arg(args, &args);
			if (fn && *fn && find_in_list((List **)&fserv_files, fn, 0))
				continue;
			if (!(md5 = next_arg(args, &args)) || 
				!(fs = next_arg(args, &args)) ||
				!(br = next_arg(args, &args)) ||
				!(fr = next_arg(args, &args)) ||
				!(t = next_arg(args, &args)))
				continue;
			new = (Files *) new_malloc(sizeof(Files));
			new->filename = m_strdup(fn);
			new->checksum = m_strdup(md5);
			new->time = my_atol(t);
			new->bitrate = my_atol(br);
			new->freq = my_atol(fr);
			new->filesize = my_atol(fs);
			new->stereo = 1;			
			add_to_list((List **)&fserv_files, (List *)new);
			count++;
			statistics.total_files++;
			statistics.total_filesize += new->filesize;
		}
		fclose(fp);
	} else
		nap_say("Error loading %s[%s]", buffer, strerror(errno));
	if (count)
		nap_say("Finished loading %s/%s. Sharing %d files", get_string_var(CTOOLZ_DIR_VAR), fname, count);
	new_free(&expand);
}

void save_shared(char *fname)
{
char buffer[BIG_BUFFER_SIZE+1];
char *expand = NULL;
FILE *fp;
int count = 0;
Files *new;
	if (!fname || !*fname)
		return;
	if (!strchr(fname, '/'))
		sprintf(buffer, "%s/%s", get_string_var(CTOOLZ_DIR_VAR), fname);
	else
		sprintf(buffer, "%s", fname);
	expand = expand_twiddle(buffer);
	if ((fp = fopen(expand, "w")))
	{
		for (new = fserv_files; new; new = new->next)
		{
			fprintf(fp, "\"%s\" %s %lu %u %u %lu\n",
				new->filename, new->checksum, new->filesize, 
				new->bitrate, new->freq, new->time);
			count++;
		}
		fclose(fp);
		nap_say("Finished saving %s [%d]", buffer, count);
		statistics.total_files = 0;
		statistics.total_filesize = 0;
	} else
		nap_say("Error saving %s %s", buffer, strerror(errno));
	new_free(&expand);
}

static int in_load = 0;
BUILT_IN_DLL(load_napserv)
{
	char	*path = NULL;
	int	recurse = 1;
	char	*pch;
	int	count = 0;
	int	reload = 0;
	int	share = 0;
	char	fname[] = "shared.dat";
		
	if (command && !my_stricmp(command, "NRELOAD"))
		reload = 1;
	
	if (in_load)
	{
		nap_say("Already loading files. Please wait");
		return;
	}
	in_load++;
	if (args && *args)
	{
		if (!my_stricmp(args, "-clear"))
		{
			clear_files(&fserv_files);
			in_load--;
			return;
		}
		else if (!my_stricmp(args, "-file"))
		{
			char *fn;
			next_arg(args, &args);
			fn = next_arg(args, &args);
			load_shared(fn ? fn : fname);
			in_load--;
			return;
		}
		else if (!my_stricmp(args, "-save"))
		{
			char *fn;
			next_arg(args, &args);
			fn = next_arg(args, &args);
			save_shared(fn ? fn : fname);
			in_load--;
			return;
		}
		while ((path = next_arg(args, &args)) && *path)
		{
			int len = strlen(path);
			if (!my_strnicmp(path, "-recurse", len))
			{
				recurse ^= 1;
				continue;
			}
			if (!my_strnicmp(path, "-share", len))
			{
				share ^= 1;
				continue;
			}
			count += scan_mp3_dir(path, recurse, reload, share);
		}
	}
	else
	{
		path = get_dllstring_var("napster_dir");

		if (!path || !*path)
		{
			nap_say("No path. /set napster_dir first.");
			in_load = 0;
			return;
		}

		pch = LOCAL_COPY(path);
		while ((path = next_arg(pch, &pch)))
			count += scan_mp3_dir(path, recurse, reload, share);
	}

	build_napster_status(NULL);
	if (!fserv_files || !count)
		nap_say("Could not read dir");
	else if (do_hook(MODULE_LIST, "NAP LOAD %d", count))
		nap_say("Found %d files%s", count, share ? "" : ". To share these type /nshare");
	in_load = 0;
	return;
}

static int in_sharing = 0;

BUILT_IN_DLL(share_napster)
{
unsigned long count = 0;
char buffer[2*NAP_BUFFER_SIZE+1];
Files	*new;
	if (in_sharing)
	{
		nap_say("Already Attempting share");
		return;
	}
	in_sharing++;
	for (new = fserv_files; new && (nap_socket != -1); new = new->next, count++)
	{
		int rc;
		int len;
		if (!new->checksum || !new->filesize || !new->bitrate || !new->freq || !new->filename)
			continue;
		sprintf(buffer, "\"%s\" %s %lu %u %u %lu", new->filename, 
			new->checksum, new->filesize, new->bitrate, new->freq, new->time);
		len = strlen(buffer);
		statistics.shared_files++;
		statistics.shared_filesize += new->filesize;
		if ((rc = send_ncommand(CMDS_ADDFILE, convertnap_dos(buffer))) == -1)
		{
			nclose(NULL, NULL, NULL, NULL, NULL);
			in_sharing = 0;
			return;
		}
		while (rc != len)
		{
			int i;
			if (!(count % 2))
			{
				lock_stack_frame();
				io("share napster");
				unlock_stack_frame();
				build_napster_status(NULL);
			}
			if ((nap_socket > -1) && (i = write(nap_socket, buffer+rc, strlen(buffer+rc))) != -1)
				rc += i;
			else
			{
				nclose(NULL, NULL, NULL, NULL, NULL);
				in_sharing = 0;
				return;
			}
		}
		if (!(count % 20))
		{
			lock_stack_frame();
			io("share napster");
			unlock_stack_frame();
			build_napster_status(NULL);
		}
	}
	build_napster_status(NULL);
	if (do_hook(MODULE_LIST, "NAP SHARE %d", count))
		nap_say("%s", cparse("Sharing $0 files", "%l", count));
	in_sharing = 0;
}

int nap_finished_file(int snum, GetFile *gf)
{
SocketList *s = NULL;
	if (snum > 0)
	{
		if ((s = get_socket(snum)))
		{
			s->is_write = 0;
			s->info = NULL;
		}
		close_socketread(snum);
	}
	if (gf)
	{
		if (gf->write > 0)
			close(gf->write);
		new_free(&gf->nick);
		new_free(&gf->filename);
		new_free(&gf->checksum);
		new_free(&gf->realfile);
		new_free(&gf->ip);
		if (gf->up == NAP_UPLOAD)
			current_sending--;
		new_free(&gf);
	}
	return 0;
}

void sendfile_timeout(int snum)
{
GetFile *f = NULL;
	if ((f = (GetFile *)get_socketinfo(snum)))
	{
		f = find_in_getfile(&napster_sendqueue, 1, f->nick, NULL, f->filename, -1, NAP_UPLOAD);
		if (do_hook(MODULE_LIST, "NAP SENDTIMEOUT %s %s", f->nick, strerror(errno)))
			nap_say("%s", cparse("Send to $0 timed out [$1-]", "%s %s", f->nick, strerror(errno)));
		if (f->socket)
			send_ncommand(CMDS_UPDATE_SEND, NULL);
	}
	nap_finished_file(snum, f);
	build_napster_status(NULL);
	return;
}

int clean_queue(GetFile **gf, int timeout)
{
GetFile *ptr;
int count = 0;
	if (!gf || !*gf || timeout <= 0)
		return 0;
	ptr = *gf;
	while (ptr)
	{
		if (ptr->addtime && (ptr->addtime <= (now - timeout)))
		{
			if (!(ptr = find_in_getfile(gf, 1, ptr->nick, NULL, ptr->filename, -1, NAP_UPLOAD)))
				continue;
			if (ptr->write > 0)
				close(ptr->write);
			if (ptr->socket > 0)
			{
				SocketList *s;
				s = get_socket(ptr->socket);
				s->is_write = 0;
				s->info = NULL;
				close_socketread(ptr->socket);
				send_ncommand(CMDS_UPDATE_SEND, NULL);
			}
			new_free(&ptr->nick);
			new_free(&ptr->filename);
			new_free(&ptr->checksum);
			new_free(&ptr->realfile);
			new_free(&ptr->ip);
			if (ptr->up == NAP_UPLOAD)
				current_sending--;
			new_free(&ptr);
			ptr = *gf;
			count++;
		} 
		else 
			ptr = ptr->next;
	}
	if (count)
		nap_say("Cleaned queue of stale entries");
	return count;
}

NAP_COMM(cmd_accepterror)
{
char *nick, *filename;
	nick = next_arg(args, &args);
	filename = new_next_arg(args, &args);
	if (nick && filename)
	{
		GetFile *gf;
		if (!(gf = find_in_getfile(&napster_sendqueue, 1, nick, NULL, filename, -1, NAP_UPLOAD)))
			return 0;
		nap_say("%s", cparse("Removing $0 from the send queue. Accept error", "%s", nick));
		nap_finished_file(gf->socket, gf);
	}
	return 0;
}

void naplink_handleconnect(int);

NAP_COMM(cmd_firewall_request) /* 501 */
{
char	*nick,
	*ip,
	*filename,
	*md5;
unsigned short port = 0;
	nick = next_arg(args, &args);
	ip = next_arg(args, &args);
	port = my_atol(next_arg(args, &args));
	filename = new_next_arg(args, &args);
	convertnap_unix(filename);
	md5 = next_arg(args, &args);
	if (!port)
		nap_say("Unable to send to a firewalled system");
	else
	{
		GetFile *gf;
		int getfd;
		struct sockaddr_in socka;
#ifndef NO_STRUCT_LINGER
		int len;
		struct linger	lin;
		len = sizeof(lin);
		lin.l_onoff = lin.l_linger = 1;
#endif
		if (!(gf = find_in_getfile(&napster_sendqueue, 1, nick, NULL, filename, -1, -1)))
		{
			nap_say("no such file requested %s %s", nick, filename);
			return 0;
		}
		gf->checksum = m_strdup(md5);
		
		getfd = socket(AF_INET, SOCK_STREAM, 0);
		socka.sin_addr.s_addr = strtoul(ip, NULL, 10);
		socka.sin_family = AF_INET;
		socka.sin_port = htons(port);
		alarm(get_int_var(CONNECT_TIMEOUT_VAR));
		if (connect(getfd, (struct sockaddr *)&socka, sizeof(struct sockaddr)) != 0) 
		{
			nap_say("ERROR connecting [%s]", strerror(errno));
			send_ncommand(CMDR_DATAPORTERROR, gf->nick);
			new_free(&gf->nick);
			new_free(&gf->filename);
			new_free(&gf->ip);
			new_free(&gf->checksum);
			new_free(&gf->realfile);
			new_free(&gf);
			return 0;
		}
		alarm(0);
#ifndef NO_STRUCT_LINGER
		setsockopt(getfd, SOL_SOCKET, SO_LINGER, (char *)&lin, len);
#endif
		gf->socket = getfd;
		gf->next = napster_sendqueue;
		napster_sendqueue = gf;
		add_socketread(getfd, getfd, 0, inet_ntoa(socka.sin_addr), naplink_handleconnect, NULL);
		set_socketinfo(getfd, gf);
		write(getfd, "1", 1);
	}
	return 0;
}

NAP_COMM(cmd_filerequest)
{
char *nick;
Files *new = NULL;
char *filename;
int count = 0;

	nick = next_arg(args, &args);
	filename = new_next_arg(args, &args);
	if (!nick || !filename || !*filename || check_nignore(nick))
		return 0;
	convertnap_unix(filename);
	for (new = fserv_files; new; new = new->next)
	{
		if (!strcmp(filename, new->filename))
			break;
	}
	if (new)
	{
		char buffer[2*NAP_BUFFER_SIZE+1];
		GetFile *gf;
		int dl_limit, dl_count;
		for (gf = napster_sendqueue; gf; gf = gf->next)
		{
			if (!gf->filename)
			{
				nap_say("ERROR in cmd_filerequest. gf->filename is null");
				return 0;
				
			}
			count++;
			if (!strcmp(filename, gf->filename) && !strcmp(nick, gf->nick))
			{
				if (do_hook(MODULE_LIST, "NAP SENDFILE already queued %s %s", gf->nick, gf->filename))
					nap_say("%s", cparse("$0 is already queued for $1-", "%s %s", gf->nick, gf->filename));
				break;
			}
		}
		dl_limit = get_dllint_var("napster_max_send_nick");
		dl_count = count_download(nick);
		if (!get_dllint_var("napster_share") || (count >= get_dllint_var("napster_send_limit")) || (dl_limit && (dl_count >= dl_limit)))
		{
			sprintf(buffer, "%s \"%s\" %d", nick, convertnap_dos(filename), (dl_limit && dl_count >= dl_limit) ? dl_limit : get_dllint_var("napster_send_limit"));
			send_ncommand(CMDS_SENDLIMIT, buffer);
			return 0;
		}
		if (do_hook(MODULE_LIST, "NAP SENDFILE %s %s", nick, filename))
			nap_say("%s", cparse("$0 has requested [$1-]", "%s %s", nick, base_name(filename)));
		sprintf(buffer, "%s \"%s\"", nick, new->filename);
		send_ncommand(CMDS_REQUESTINFO, nick);
		send_ncommand(CMDS_FILEINFO, convertnap_dos(buffer));
		if (!gf)
		{
			gf = new_malloc(sizeof(GetFile));
			gf->nick = m_strdup(nick);
			gf->checksum = m_strdup(new->checksum);
			gf->filename = m_strdup(new->filename);
			if ((gf->write = open(new->filename, O_RDONLY)) < 0)
				nap_say("Unable to open %s for sending [%s]", new->filename, strerror(errno));
			gf->filesize = new->filesize;
			gf->next = napster_sendqueue;
			gf->up = NAP_UPLOAD;
			napster_sendqueue = gf;
			current_sending++;
		}
		gf->addtime = time(NULL);
		clean_queue(&napster_sendqueue, 300);
	} 
	return 0;
}

void napfile_sendfile(int snum)
{
GetFile *gf;
unsigned char buffer[NAP_BUFFER_SIZE+1];
int rc, numread;
	if (!(gf = (GetFile *)get_socketinfo(snum)))
		return;
	gf->addtime = now;
	numread = read(gf->write, buffer, sizeof(buffer)-1);
	switch(numread)
	{
		case -1:
		case 0:
		{
			close(gf->write);
			if ((gf = find_in_getfile(&napster_sendqueue, 1, gf->nick, NULL, gf->filename, -1, NAP_UPLOAD)))
			{
				if ((gf->received + gf->resume) >= gf->filesize)
				{
					double speed;
					char speed1[80];
					statistics.files_served++;
					statistics.filesize_served += gf->received;
					speed = gf->received / 1024.0 / (time(NULL) - gf->starttime);
					if (speed > statistics.max_uploadspeed)
						statistics.max_uploadspeed = speed;
					sprintf(speed1, "%4.2fK/s", speed);
					if (do_hook(MODULE_LIST, "NAP SENDFILE FINISHED %s %s %s", gf->nick, speed1, gf->filename)) 
						nap_say("%s", cparse("Finished Sending $0 [$2-] at $1", "%s %s %s", gf->nick, speed1, base_name(gf->filename)));
				}
				else if (do_hook(MODULE_LIST, "NAP SENDFILE ERROR %s %lu %lu %s", gf->nick, gf->filesize, gf->received + gf->resume, base_name(gf->filename)))
				{
					char rs[60];
					sprintf(rs, "%4.2g%s", _GMKv(gf->received+gf->resume), _GMKs(gf->received+gf->resume));
					nap_say("%s", cparse("Error sending [$2-] to $0 ", "%s %s \"%s\"", gf->nick, rs, base_name(gf->filename)));
				}
			}
			nap_finished_file(snum, gf);
			build_napster_status(NULL);
			send_ncommand(CMDS_UPDATE_SEND, NULL);
			break;
		}
		default:
		{
			rc = send(snum, buffer, numread, 0);	
			if (rc != numread)
			{
				if (rc == -1)
				{
					if (errno == EWOULDBLOCK || errno == ENOBUFS)
						lseek(gf->write, -numread, SEEK_CUR);
					else
					{
						if ((gf = find_in_getfile(&napster_sendqueue, 1, gf->nick, NULL, gf->filename, -1, NAP_UPLOAD)))
						{
							if (do_hook(MODULE_LIST, "NAP SENDFILE ERROR %s %lu %lu \"%s\" %s", gf->nick, gf->filesize, gf->received + gf->resume, base_name(gf->filename), strerror(errno)))
							{
								char rs[60];
								sprintf(rs, "%4.2g%s", _GMKv(gf->received+gf->resume), _GMKs(gf->received+gf->resume));
								nap_say("%s", cparse("Error sending [$2-] to $0 ", "%s %s \"%s\" %s", gf->nick, rs, base_name(gf->filename), strerror(errno)));
							}
						}
						nap_finished_file(snum, gf);
						build_napster_status(NULL);
						send_ncommand(CMDS_UPDATE_SEND, NULL);
					}
					return;
				}
				else 
					lseek(gf->write, -(numread - rc), SEEK_CUR);
			}
			gf->received += rc;
			if (!(gf->received % (10 * (sizeof(buffer)-1))))
				build_napster_status(NULL);
		}
	}
}

void napfirewall_pos(int snum)
{
int rc;
char buff[80];
GetFile *gf;
unsigned long pos;
SocketList *s;
	s = get_socket(snum);
	if (!s || !(gf = (GetFile *)get_socketinfo(snum)))
		return;
	alarm(10);
	if ((rc = read(snum, buff, sizeof(buff)-1)) < 1)
	{
		alarm(0);
		return;
	}
	alarm(0);
	buff[rc] = 0;
	pos = my_atol(buff);
	gf->resume = pos;
	lseek(gf->write, SEEK_SET, pos);
	s->func_read = napfile_sendfile;
	napfile_sendfile(snum);
}

void nap_firewall_start(int snum)
{
GetFile *gf;
unsigned char buffer[NAP_BUFFER_SIZE+1];
SocketList *s;
	s = get_socket(snum);
	if (!s || !(gf = (GetFile *)get_socketinfo(snum)))
		return;
	if ((read(snum, buffer, 4)) < 1)
		return;
	
#if 0
	sprintf(buffer, "%s \"%s\" %lu", gf->nick, gf->filename, gf->filesize);
	write(snum, convertnap_dos(buffer), strlen(buffer));
#endif
	if (*buffer && !strcmp(buffer, "SEND"))
		s->func_read = napfirewall_pos;	
	else
		close_socketread(snum);
}

void napfile_read(int snum)
{
GetFile *gf;
unsigned char buffer[NAP_BUFFER_SIZE+1];
int rc;
SocketList *s;
	s = get_socket(snum);
	if (!(gf = (GetFile *)get_socketinfo(snum)))
	{
		unsigned char buff[2*NAP_BUFFER_SIZE+1];
		unsigned char fbuff[2*NAP_BUFFER_SIZE+1];
		char *nick, *filename, *args;
		
		alarm(10);
		if ((rc = read(snum, buff, sizeof(buff)-1)) < 0)
		{
			alarm(0);
			close_socketread(snum);
			return;
		}
		alarm(0);
		buff[rc] = 0;
		args = &buff[0];
		if (!*args || !strcmp(buff, "FILE NOT FOUND") || !strcmp(buff, "INVALID REQUEST"))
		{
			close_socketread(snum);
			nap_say("Error %s", *args ? args : "unknown read");
			return;
		}

		nick = next_arg(args, &args);
		if ((filename = new_next_arg(args, &args)) && *filename)
		{
			strcpy(fbuff, filename);
			convertnap_unix(fbuff);
		}
		if (!nick || !filename || !*filename || !args || !*args
			|| !(gf = find_in_getfile(&napster_sendqueue, 0, nick, NULL, fbuff, -1, NAP_UPLOAD)) 
			|| (gf->write == -1))
		{
			memset(buff, 0, 80);
			if (!gf)
				sprintf(buff, "0INVALID REQUEST");
			else
			{
				sprintf(buff, "0FILE NOT FOUND");
				if ((gf = find_in_getfile(&napster_sendqueue, 1, nick, NULL, fbuff, -1, NAP_UPLOAD)))
					gf->socket = snum;
			}
			write(snum, buff, strlen(buffer));
			nap_finished_file(snum, gf);
			return;
		}
		gf->resume = strtoul(args, NULL, 0);
		if (gf->resume >= gf->filesize)
		{
			gf = find_in_getfile(&napster_sendqueue, 1, nick, NULL, fbuff, -1, NAP_UPLOAD);
			nap_finished_file(snum, gf);
			return;
		}
		gf->socket = snum;
		lseek(gf->write, SEEK_SET, gf->resume);
		set_socketinfo(snum, gf);
		memset(buff, 0, 80);
		sprintf(buff, "%lu", gf->filesize);
		write(snum, buff, strlen(buff));
		s->func_write = s->func_read;
		s->is_write = s->is_read;
		if (do_hook(MODULE_LIST, "NAP SENDFILE %sING %s %s",gf->resume ? "RESUM" : "SEND", gf->nick, gf->filename))
			nap_say("%s", cparse("$0ing file to $1 [$2-]", "%s %s %s", gf->resume ? "Resum" : "Send", gf->nick, base_name(gf->filename)));
		add_sockettimeout(snum, 0, NULL);
		set_non_blocking(snum);
		build_napster_status(NULL);
		send_ncommand(CMDS_UPDATE_SEND1, NULL);
		return;
	} else if (!gf->starttime)
		gf->starttime = now;
	s->func_read = napfile_sendfile;
	napfile_sendfile(snum);
}

void naplink_handleconnect(int snum)
{
unsigned char buff[2*NAP_BUFFER_SIZE+1];
SocketList *s;
int rc;
	memset(buff, 0, sizeof(buff) - 1);
	switch ((rc = recv(snum, buff, 4, MSG_PEEK)))
	{

		case -1:
			nap_say("naplink_handleconnect %s", strerror(errno));
			close_socketread(snum);
		case 0:
			return;
		default:
			break;
	}

	buff[rc] = 0;
	if (!(s = get_socket(snum)))
	{
		close_socketread(snum);
		return;
	}
	if (rc == 1 && (buff[0] == '1' || buff[0] == '\n'))
	{
		read(snum, buff, 1);
/*		write(snum, "SEND", 4);*/
		s->func_read = nap_firewall_start;
	}
	else if (!strncmp(buff, "GET", 3))
	{
	/* someone has requested a non-firewalled send */
		read(snum, buff, 3);
		set_napster_socket(snum);
		s->func_read = napfile_read;
	}
	else if (!strncmp(buff, "SEND", 4))
	{
	/* we have requested a file from someone who is firewalled */
		read(snum, buff, 4);
		s->func_read = nap_firewall_get;
	}
	else
		close_socketread(snum);
}

void naplink_handlelink(int snum)
{
struct  sockaddr_in     remaddr;
int sra = sizeof(struct sockaddr_in);
int sock = -1;
	if ((sock = accept(snum, (struct sockaddr *) &remaddr, &sra)) > -1)
	{
		add_socketread(sock, snum, 0, inet_ntoa(remaddr.sin_addr), naplink_handleconnect, NULL);
		add_sockettimeout(sock, 180, sendfile_timeout);
#if 0
/*is this really necessary?*/
		for (sf = napster_sendqueue; sf; sf = sf->next)
		{
			unsigned long ip, ip1;
			ip = inet_addr(sf->ip);
			ip1 = htonl(remaddr.sin_addr.s_addr);
			if ((ip == ip1) && !my_atol(sf->port))
				return;
		}
#endif
		write(sock, "\n", 1); 
	}
}

