/* 
 *	HT Editor
 *	textedit.cc
 *
 *	Copyright (C) 1999, 2000, 2001 Stefan Weyergraf (stefan@weyergraf.de)
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 2 as
 *	published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

extern "C" {
#include "evalx.h"
}
#include "htapp.h"
#include "htdialog.h"
#include "htobj.h"
#include "htclipboard.h"
#include "htiobox.h"
#include "textedit.h"

#include <errno.h>
#include <limits.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

// FIXME: put into registry
#define TEXT_VIEWER_TABSIZE			5
#define TEXT_VIEWER_HIGHLIGHT_LINEEND	true

/*
 *	CLASS ht_undo_data
 */
ht_undo_data::ht_undo_data()
{
}

/*
 *	CLASS ht_undo_data_delete_string
 */

ht_undo_data_delete_string::ht_undo_data_delete_string(text_viewer_pos *APos, text_viewer_pos *BPos, void *String, UINT Len)
{
	apos = *APos;
	bpos = *BPos;
	if (Len) {
		string = malloc(Len);
		memcpy(string, String, Len);
	} else {
		string = NULL;
	}
	len = Len;
}

ht_undo_data_delete_string::~ht_undo_data_delete_string()
{
	if (string) free(string);
}

bool ht_undo_data_delete_string::combine(ht_undo_data *ud)
{
	if (ud->object_id()==object_id()) {
		ht_undo_data_delete_string *ud2 = (ht_undo_data_delete_string *)ud;
		if (ud2->apos.line == apos.line) {
			if (ud2->bpos.pofs + ud2->len == bpos.pofs) {
				string = realloc(string, len+ud2->len);
				memmove((void*)((byte*)string+ud2->len), string, len);
				memcpy(string, ud2->string, ud2->len);
				len += ud2->len;
				bpos = ud2->bpos;
				return true;
			}
		}
	}
	return false;
}

UINT ht_undo_data_delete_string::getsize()
{
	return len+sizeof *this;
}

void ht_undo_data_delete_string::gettext(char *text, UINT maxlen)
{
	char *buf = (char *) malloc(len+1);
	bin2str(buf, string, len);
	// FIXME snprintf
	sprintf(text, "deletion of '%s' at %d:%d", buf, bpos.line+1, bpos.pofs+1);
	free(buf);
}

OBJECT_ID ht_undo_data_delete_string::object_id()
{
	return ATOM_HT_UNDO_DATA_DELETE;
}

void ht_undo_data_delete_string::apply(ht_text_editor *te)
{
	te->delete_chars(bpos.line, bpos.pofs, len);
	te->goto_line(bpos.line);
	te->cursor_pput(bpos.pofs);
}

void ht_undo_data_delete_string::unapply(ht_text_editor *te, bool *goto_only)
{
	if (*goto_only && (bpos.line != te->top_line + te->cursory || te->physical_cursorx()!=bpos.pofs)) {
		te->goto_line(bpos.line);
		te->cursor_pput(bpos.pofs);
		return;
	}
	*goto_only = false;
	if (string) {
		te->insert_chars(bpos.line, bpos.pofs, string, len);
	}
	te->goto_line(apos.line);
	te->cursor_pput(apos.pofs);
}

/*
 *	CLASS ht_undo_data_delete_string2
 */

ht_undo_data_delete_string2::ht_undo_data_delete_string2(text_viewer_pos *APos, text_viewer_pos *BPos, void *String, UINT Len)
{
	apos = *APos;
	bpos = *BPos;
	if (Len) {
		string = malloc(Len);
		memcpy(string, String, Len);
	} else {
		string = NULL;
	}
	len = Len;
}

ht_undo_data_delete_string2::~ht_undo_data_delete_string2()
{
	if (string) free(string);
}

bool ht_undo_data_delete_string2::combine(ht_undo_data *ud)
{
	if (ud->object_id()==object_id()) {
		ht_undo_data_delete_string2 *ud2 = (ht_undo_data_delete_string2 *)ud;
		if (ud2->apos.line == apos.line) {
			if (ud2->apos.pofs == apos.pofs) {
				string = realloc(string, len+ud2->len);
				memcpy((void*) ((byte*)string+len) , ud2->string, ud2->len);
				len += ud2->len;
				return true;
			}
		}
	}
	return false;
}

UINT ht_undo_data_delete_string2::getsize()
{
	return len+sizeof *this;
}

void ht_undo_data_delete_string2::gettext(char *text, UINT maxlen)
{
	char *buf = (char *) malloc(len+1);
	bin2str(buf, string, len);
	// FIXME snprintf
	sprintf(text, "deletion of '%s' at %d:%d", buf, apos.line+1, apos.pofs+1);
	free(buf);
}

OBJECT_ID ht_undo_data_delete_string2::object_id()
{
	return ATOM_HT_UNDO_DATA_DELETE2;
}

void ht_undo_data_delete_string2::apply(ht_text_editor *te)
{
	te->delete_chars(apos.line, apos.pofs, len);
	te->goto_line(apos.line);
	te->cursor_pput(apos.pofs);
}

void ht_undo_data_delete_string2::unapply(ht_text_editor *te, bool *goto_only)
{
	if (*goto_only && (apos.line != te->top_line + te->cursory || te->physical_cursorx()!=apos.pofs)) {
		te->goto_line(apos.line);
		te->cursor_pput(apos.pofs);
		return;
	}
	te->goto_line(apos.line);
	te->cursor_pput(apos.pofs);
	*goto_only = false;
	if (string) {
		te->insert_chars(apos.line, apos.pofs, string, len);
	}
}

/*
 *	CLASS ht_undo_data_insert_string
 */

ht_undo_data_insert_string::ht_undo_data_insert_string(text_viewer_pos *APos, text_viewer_pos *BPos, void *String, UINT Len)
{
	apos = *APos;
	bpos = *BPos;
	if (Len) {
		string = malloc(Len);
		memcpy(string, String, Len);
	} else {
		string = NULL;
	}
	len = Len;
}

ht_undo_data_insert_string::~ht_undo_data_insert_string()
{
	if (string) free(string);
}

bool ht_undo_data_insert_string::combine(ht_undo_data *ud)
{
	if (ud->object_id()==object_id()) {
		ht_undo_data_insert_string *ud2 = (ht_undo_data_insert_string *)ud;
		if (ud2->cpos.line == cpos.line) {
			if (ud2->cpos.pofs == cpos.pofs+len) {
				string = realloc(string, len+ud2->len);
				memcpy((void*) ((byte*)string+len) , ud2->string, ud2->len);
				len += ud2->len;
				bpos = ud2->bpos;
				return true;
			}
		}
	}
	return false;
}

UINT ht_undo_data_insert_string::getsize()
{
	return len+sizeof *this;
}

void ht_undo_data_insert_string::gettext(char *text, UINT maxlen)
{
	char *buf = (char *) malloc(len+1);
	bin2str(buf, string, len);
	// FIXME snprintf
	sprintf(text, "insertion of '%s' at %d:%d", buf, apos.line+1, apos.pofs+1);
	free(buf);
}

OBJECT_ID ht_undo_data_insert_string::object_id()
{
	return ATOM_HT_UNDO_DATA_INSERT;
}

void ht_undo_data_insert_string::apply(ht_text_editor *te)
{
	if (string) {
		UINT l = te->get_line_length(apos.line);
		UINT p = apos.pofs;
		if (apos.pofs > l) {
			UINT k = apos.pofs - l;
			void *s = malloc(k + len);
			memset(s, 32, k);
			memmove((char*)s+k, string, len);
			len += k;
			p = l;
			free(string);
			string = s;
		}
		te->insert_chars(apos.line, p, string, len);
		cpos.line = apos.line;
		cpos.pofs = p;
	}
	te->goto_line(bpos.line);
	te->cursor_pput(bpos.pofs);
}

void ht_undo_data_insert_string::unapply(ht_text_editor *te, bool *goto_only)
{
	if (*goto_only && (bpos.line != te->top_line + te->cursory || te->physical_cursorx()!=bpos.pofs)) {
		te->goto_line(bpos.line);
		te->cursor_pput(bpos.pofs);
		return;
	}
	*goto_only = false;
	te->delete_chars(apos.line, apos.pofs, len);
	te->goto_line(apos.line);
	te->cursor_pput(apos.pofs);
}

/*
 *	CLASS ht_undo_data_split_line
 */
ht_undo_data_split_line::ht_undo_data_split_line(text_viewer_pos *APos, text_viewer_pos *BPos)
{
	apos = *APos;
	bpos = *BPos;
}

ht_undo_data_split_line::~ht_undo_data_split_line()
{
}

bool ht_undo_data_split_line::combine(ht_undo_data *ud)
{
	return false;
}

UINT ht_undo_data_split_line::getsize()
{
	return sizeof *this;
}

void ht_undo_data_split_line::gettext(char *text, UINT maxlen)
{
	sprintf(text, "split line at %d:%d", apos.line+1, apos.pofs+1);
}

OBJECT_ID ht_undo_data_split_line::object_id()
{
	return ATOM_HT_UNDO_DATA_SPLIT_LINE;
}

void ht_undo_data_split_line::apply(ht_text_editor *te)
{
	te->split_line(apos.line, apos.pofs);
	te->goto_line(bpos.line);
	te->cursor_pput(bpos.pofs);
}

void ht_undo_data_split_line::unapply(ht_text_editor *te, bool *goto_only)
{
	if (*goto_only && (bpos.line != te->top_line + te->cursory || te->physical_cursorx()!=bpos.pofs)) {
		te->goto_line(bpos.line);
		te->cursor_pput(bpos.pofs);
		return;
	}
	*goto_only = false;
	te->concat_lines(apos.line);
	te->goto_line(apos.line);
	te->cursor_pput(apos.pofs);
}

/*
 *	CLASS ht_undo_data_join_line
 */
ht_undo_data_join_line::ht_undo_data_join_line(text_viewer_pos *APos, text_viewer_pos *BPos)
{
	apos = *APos;
	bpos = *BPos;
}

ht_undo_data_join_line::~ht_undo_data_join_line()
{
}

bool ht_undo_data_join_line::combine(ht_undo_data *ud)
{
	return false;
}

UINT ht_undo_data_join_line::getsize()
{
	return sizeof *this;
}

void ht_undo_data_join_line::gettext(char *text, UINT maxlen)
{
	sprintf(text, "join lines %d and %d", bpos.line+1, bpos.line+2);
}

OBJECT_ID ht_undo_data_join_line::object_id()
{
	return ATOM_HT_UNDO_DATA_JOIN_LINE;
}

void ht_undo_data_join_line::apply(ht_text_editor *te)
{
	te->concat_lines(bpos.line);
	te->goto_line(bpos.line);
	te->cursor_pput(bpos.pofs);
}

void ht_undo_data_join_line::unapply(ht_text_editor *te, bool *goto_only)
{
	if (*goto_only && (bpos.line != te->top_line + te->cursory || te->physical_cursorx()!=bpos.pofs)) {
		te->goto_line(bpos.line);
		te->cursor_pput(bpos.pofs);
		return;     
	}
	*goto_only = false;
	te->split_line(bpos.line, bpos.pofs);
	te->goto_line(apos.line);
	te->cursor_pput(apos.pofs);
}

/*
 *	CLASS ht_text_editor_undo
 */
void ht_text_editor_undo::init(UINT max_undo_size)
{
	ht_clist::init();
	size = 0;
	max_size = max_undo_size;
	clean_state = 0;
	goto_state = true;
	current_position = 0;
}

UINT ht_text_editor_undo::get_current_position()
{
	   return current_position;
}

void ht_text_editor_undo::insert_undo(ht_text_editor *tv, ht_undo_data *undo)
{
	if (undo) {
		if (current_position!=c_entry_count) {
			// remove all pending redo's
			UINT test=c_entry_count;
			for (UINT i=current_position; i<test; i++) {
				ht_undo_data *u = (ht_undo_data*)get(current_position);
				size-=u->getsize();
				del(current_position);
			}
			assert(current_position == c_entry_count);
		}
		undo->apply(tv);
		UINT gsize = undo->getsize();
		while (size + gsize > max_size) {
			if (c_entry_count) {
				size-=((ht_undo_data*)get(0))->getsize();
				del(0);
				if (current_position) current_position--;
				if (clean_state) clean_state--;
			} else {
				delete undo;
				return;
			}
		}
		if (c_entry_count && !is_clean()) {
			ht_undo_data *u = (ht_undo_data*)get(c_entry_count-1);
			UINT zsize = u->getsize();
			if (u->combine(undo)) {
				size-=zsize;
				size+=u->getsize();
				delete undo;
				return;
			}
		}
		current_position++;
		append(undo);
		size+=gsize;
	}
	goto_state = true;
}

void ht_text_editor_undo::mark_clean()
{
	clean_state = current_position;
	goto_state = true;
}

bool ht_text_editor_undo::is_clean()
{
	return clean_state == current_position;
}

void ht_text_editor_undo::redo(ht_text_editor *te)
{
	goto_state = true;
	if (current_position < c_entry_count) {
		ht_undo_data *u = (ht_undo_data*)get(current_position);
		u->apply(te);
		current_position++;
	}
}

void ht_text_editor_undo::undo(ht_text_editor *te)
{
	if (current_position) {
		ht_undo_data *u = (ht_undo_data*)get(current_position-1);
		bool goto_state_test = goto_state;
		u->unapply(te, &goto_state_test);
		if (!goto_state_test) {
			current_position--;
		}
		goto_state = !goto_state_test;
	}
}

/*
 *
 */
int text_viewer_pos_compare(text_viewer_pos *a, text_viewer_pos *b)
{
	if (a->line==b->line) {
		return a->pofs-b->pofs;
	}
	return a->line-b->line;
}

/*
 *	CLASS ht_text_viewer
 */

void ht_text_viewer::init(bounds *b, bool own_t, ht_textfile *t, bool own_l, ht_syntax_lexer *l)
{
	ht_view::init(b, VO_OWNBUFFER | VO_SELECTABLE | VO_RESIZE, "text viewer");
	VIEW_DEBUG_NAME("ht_text_viewer");

	growmode=GM_VDEFORM | GM_HDEFORM;

	own_textfile=false;
	textfile=NULL;
	set_textfile(t, own_t);

	own_lexer=false;
	lexer=NULL;
	set_lexer(l, own_l);

	top_line=0;
	xofs=0;
	cursorx=0;
	cursory=0;
	select_clear();

	selectcursor=false;
}

void ht_text_viewer::done()
{
	if (own_textfile) {
		textfile->done();
		delete textfile;
	}

	if (own_lexer && lexer) {
		lexer->done();
		delete lexer;
	}

	ht_view::done();
}

UINT ht_text_viewer::char_vsize(char c, UINT x)
{
	if (c=='\t') return TEXT_VIEWER_TABSIZE-x%TEXT_VIEWER_TABSIZE;
	return 1;
}

void ht_text_viewer::clipboard_copy_cmd()
{
	if (text_viewer_pos_compare(&sel_start, &sel_end)<0) {
		FILEOFS s, e;
		if (textfile->convert_line2ofs(sel_start.line, sel_start.pofs, &s) &&
		textfile->convert_line2ofs(sel_end.line, sel_end.pofs, &e)) {
			char dsc[1024]; /* FIXME: possible buffer overflow */
			sprintf(dsc, "%s::%s", textfile->get_desc(), desc);
			clipboard_copy(dsc, textfile, s, e-s);
		}
	}
}

UINT ht_text_viewer::cursor_up(UINT n)
{
	if (cursory>n) cursory-=n; else {
		n=scroll_up(n-cursory);
		cursory=0;
	}
	return n;
}

UINT ht_text_viewer::cursor_down(UINT n)
{
	UINT lmh=textfile->linecount()-top_line;
	if (lmh>(UINT)size.h) lmh=size.h;
	if (cursory+n>lmh-1) {
		UINT k = scroll_down(cursory+n-(lmh-1));
		lmh = textfile->linecount()-top_line;
		if (lmh>(UINT)size.h) lmh = size.h;
		n = k+(lmh-1)-cursory;
		cursory = lmh-1;
	} else cursory+=n;
	return n;
}

UINT ht_text_viewer::cursor_left(UINT n)
{
	UINT p;
	if (cursorx+xofs>n) p=cursorx+xofs-n; else p=0;
	cursor_vput(p);
	return 1;
}

UINT ht_text_viewer::cursor_right(UINT n)
{
	cursor_vput(cursorx+xofs+n);
	return 1;
}

void ht_text_viewer::cursor_home()
{
	cursor_vput(0);
}

void ht_text_viewer::cursor_end()
{
	cursor_vput(get_line_vlength(top_line+cursory));
}

void ht_text_viewer::cursor_pput(UINT dx)
{
	UINT vx = 0, px = 0;
//	int v=cursorx+xofs;
	int v = xofs;
	char line[1024];
	char *linep=line;

	UINT linelen;
	if (!textfile->getline(top_line+cursory, 0, line, sizeof line, &linelen, NULL)) return;
	
	while (linelen--) {
		if (px==dx) {
			v=-1;
			break;
		}
		int k=char_vsize(*(linep++), vx);
		vx+=k;
/*		v-=k;
		if (v<0) break;*/
		px++;
	}
/*	while (v>0) {
		if (px==dx) break;
		px++;
		vx++;
		v--;
	}*/
	if (xofs>vx) {
		cursorx=0;
		xofs=vx;
	} else {
		cursorx=vx-xofs;
		if (cursorx>(UINT)size.w-1) {
			xofs=cursorx-(size.w-1);
			cursorx=size.w-1;
		}
	}
}

void ht_text_viewer::cursor_vput(UINT vx)
{
	if ((vx>=xofs) && (vx<xofs+size.w)) {
		cursorx=vx-xofs;
	} else if (vx<xofs) {
		xofs=vx;
		cursorx=0;
	} else if (vx>(UINT)size.w-1) {
		xofs=vx-(size.w-1);
		cursorx=size.w-1;
	} else {
		xofs=0;
		cursorx=vx;
	}
}

void ht_text_viewer::draw()
{
/*	timer_handle h=new_timer();
	start_timer(h);*/
	if (!textfile) return;

	UINT tabsize=TEXT_VIEWER_TABSIZE;
	vcp bgcolor = get_bgcolor();
//	vcp bgcolor=VCP(VC_LIGHT(VC_YELLOW), VC_BLUE);
	clear(bgcolor);
	char line[1024];
	text_viewer_pos pos;
	pos.line=top_line;
	for (int y=0; y<size.h; y++) {
		lexer_state state;

		pos.pofs=0;

//FIXME:debug:		if (!textfile->getline((top_line+y)|0x80000000, line, sizeof line, &state)) break;
		UINT linelen;
		if (!textfile->getline(top_line+y, 0, line, sizeof line, &linelen, &state)) break;
		line[linelen]=0;

		UINT x=0;
		if (lexer) {
			char *linep=(char*)line;
			UINT toklen;
			lexer_token tok;
			bool start_of_line=true;
			bool theend=false;
blabla:
			text_pos p;
			p.line=pos.line;
			p.pofs=pos.pofs;
			while ((tok=lexer->gettoken(linep, &state, &p, &toklen, start_of_line, false))) {
				UINT k, i;
				UINT vtoklen=toklen;
				bool print=true;
				bool is_tab=((toklen==1) && (*linep=='\t'));
				vcp color=lexer->gettoken_color(tok);

				if (is_tab) vtoklen=tabsize-x%tabsize;

				if (x>=xofs) {
					k=x-xofs;
					i=0;
					if (k>(UINT)size.w-1) {
						theend=true;
						break;
					}
				} else if (x+vtoklen>=xofs) {
					k=0;
					i=xofs-x;
				} else {
					print=false;
				}
				if (print) {
					if (is_tab) {
						char tab[17];
						UINT z;
						for (z=0; z<vtoklen; z++) tab[z]=' ';
						tab[z]=0;
						render_str(k, y, color, &pos, vtoklen-i, tab+i, false);
					} else {
						render_str(k, y, color, &pos, vtoklen-i, linep+i, true);
					}
				}
				x+=vtoklen;
				linep+=toklen;
				linelen-=toklen;
				pos.pofs+=toklen;
				start_of_line=false;
				p.line=pos.line;
				p.pofs=pos.pofs;
			}
			if ((linelen) && (!*linep) && (!theend)) {
				while ((linelen) && (!*linep)) {
					UINT k, i;
					bool print=true;
					UINT vtoklen=char_vsize(*linep, x);
					if (x>=xofs) {
						k=x-xofs;
						i=0;
						if (k>(UINT)size.w-1) {
							theend=true;
							break;
						}
					} else if (x+vtoklen>=xofs) {
						k=0;
						i=xofs-x;
					} else {
						print=false;
					}
					if (print) render_str(k, y, bgcolor, &pos, 1, " ", false);
					linep++;
					linelen--;
					pos.pofs++;
					x++;
				}
				if (linelen && !theend) goto blabla;
			}
		} else {
			char *linep=line;
			while (linelen--) {
				UINT vtoklen=char_vsize(*linep, x);
				if (x>=xofs) {
					if (x-xofs>(UINT)size.w-1) break;
					if (*linep=='\t') {
						vcp c=bgcolor;
						char tab[17];
						UINT z;
						for (z=0; z<vtoklen; z++) tab[z]=' ';
						tab[z]=0;
						render_str_color(&c, &pos);
						buf_lprint(x-xofs, y, c, z, tab);
					} else {
						vcp c=bgcolor;
						render_str_color(&c, &pos);
						buf_printchar(x-xofs, y, c, *linep);
					}
				}
				x+=vtoklen;
				linep++;
				pos.pofs++;
			}
		}
		if ((TEXT_VIEWER_HIGHLIGHT_LINEEND) && (pos.line<sel_end.line) &&
		(pos.line>=sel_start.line)) {
			int p = (x>xofs) ? x-xofs : 0;
			fill(p, y, size.w-p, 1, VCP(VC_TRANSPARENT, VC_WHITE), ' ');
		}
		pos.line++;
	}
	if (focused) setcursor(cursorx, cursory);

/*	stop_timer(h);
	int tix=get_timer_tick(h);
	delete_timer(h);
	buf_printf(50, 0, bgcolor, "%dtix", tix);*/
}

char *ht_text_viewer::func(UINT i, bool execute)
{
	switch (i) {
		case 5: {
			if (execute) {
				sendmsg(cmd_text_viewer_goto);
			}
			return "goto";
		}
		default:
			return false;
	}
	return 0;
}

vcp ht_text_viewer::get_bgcolor()
{
	return getcolor(palidx_windialog_body);
}

UINT ht_text_viewer::get_line_length(UINT line)
{
	return textfile->getlinelength(line);
}

UINT ht_text_viewer::get_line_vlength(UINT line)
{
	char l[1024];
	char *linep=l;
	UINT vl=0;

	UINT linelen;
	if (!textfile->getline(line, 0, l, sizeof l, &linelen, NULL)) return 0;
	
	while (linelen--) vl+=char_vsize(*(linep++), vl);
	
	return vl;
}
void ht_text_viewer::get_pindicator_str(char *buf)
{
	sprintf(buf, "%c%d:%d ", modified() ? '*' : ' ', top_line+cursory+1, xofs+cursorx+1);
}

bool ht_text_viewer::get_hscrollbar_pos(int *pstart, int *psize)
{
	return (scrollbar_pos(top_line, size.h, textfile->linecount(), pstart, psize));
}

bool ht_text_viewer::get_vscrollbar_pos(int *pstart, int *psize)
{
	return false;
}

bool ht_text_viewer::goto_line(UINT line)
{
	if (line > textfile->linecount()) {
		return false;
	}
	if ((line >= top_line) && (line - top_line < (UINT)size.h)) {
		cursory = line - top_line;
	} else {
		cursory = 0;
		if (line > top_line) {
			top_line = line;
			cursor_down(cursor_up(size.h));
		} else {
			top_line = line;
			cursor_up(cursor_down(size.h));
		}
	}
	return true;
}

void ht_text_viewer::handlemsg(htmsg *msg)
{
	switch (msg->msg) {
		case msg_keypressed: {
			int k=msg->data1.integer;
			bool sel;
			switch (k) {
/* FIXME: debug */
/*				case 'X':
					textfile->cntl(FCNTL_CACHE_INVD);
					dirtyview();
					clearmsg(msg);
					return;*/
				case K_Alt_S:
					selectcursor=!selectcursor;
					clearmsg(msg);
					return;
				case K_Control_J:
					sendmsg(cmd_text_viewer_goto);
					clearmsg(msg);
					return;
				case K_Up:
				case K_Shift_Up:
					sel=(k==K_Shift_Up) != selectcursor;
					if (sel) select_start();
					cursor_up(1);
					if (sel) select_end();
					dirtyview();
					clearmsg(msg);
					return;
				case K_Down:
				case K_Shift_Down:
					sel=(k==K_Shift_Down) != selectcursor;
					if (sel) select_start();
					cursor_down(1);
					if (sel) select_end();
					dirtyview();
					clearmsg(msg);
					return;
				case K_Left:
				case K_Shift_Left:
					sel=(k==K_Shift_Left) != selectcursor;
					if (sel) select_start();
					cursor_left(1);
					if (sel) select_end();
					dirtyview();
					clearmsg(msg);
					return;
				case K_Right:
				case K_Shift_Right:
					sel=(k==K_Shift_Right) != selectcursor;
					if (sel) select_start();
					cursor_right(1);
					if (sel) select_end();
					dirtyview();
					clearmsg(msg);
					return;
				case K_PageUp:
				case K_Shift_PageUp:
					sel=(k==K_Shift_PageUp) != selectcursor;
					if (sel) select_start();
					scroll_up(size.h-1);
					if (sel) select_end();
					dirtyview();
					clearmsg(msg);
					return;
				case K_PageDown:
				case K_Shift_PageDown:
					sel=(k==K_Shift_Down) != selectcursor;
					if (sel) select_start();
					scroll_down(size.h-1);
					if (sel) select_end();
					dirtyview();
					clearmsg(msg);
					return;
				case K_Home:
				case K_Shift_Home:
					sel=(k==K_Shift_Home) != selectcursor;
					if (sel) select_start();
					cursor_home();
					if (sel) select_end();
					dirtyview();
					clearmsg(msg);
					return;
				case K_End:
				case K_Shift_End:
					sel=(k==K_Shift_End) != selectcursor;
					if (sel) select_start();
					cursor_end();
					if (sel) select_end();
					dirtyview();
					clearmsg(msg);
					return;
				case K_Control_PageUp:
					top_line=0;
					cursorx=0;
					cursory=0;
					dirtyview();
					clearmsg(msg);
					return;
				case K_Control_PageDown:
					top_line=textfile->linecount()-1;
					if (top_line>(UINT)size.h-1) top_line-=size.h-1;
					cursorx=0;
					cursory=0;
					scroll_down(size.h-1);
					dirtyview();
					clearmsg(msg);
					return;
				case K_Alt_C:
				case K_Control_Insert:
					sendmsg(cmd_edit_copy);
					clearmsg(msg);
					return;
			}
			break;
		}
		case cmd_edit_copy: {
			clipboard_copy_cmd();
			dirtyview();
			clearmsg(msg);
			return;
		}
		case cmd_text_viewer_goto: {          
			char line[1024];
			line[0]=0;
			if (inputbox("goto", "line", line, 1024)) {
				scalar_t r;
				if (eval(&r, line, NULL, NULL, NULL)) {
					int_t i;
					scalar_context_int(&r, &i);
					if (!i.value || !goto_line(i.value-1)) {
						errorbox("no line %d!", i.value);
					}
				}
			}
			return;
		}
		case msg_get_scrollinfo:
			switch (msg->data1.integer) {
				case gsi_pindicator: {
					get_pindicator_str((char*)msg->data2.ptr);
					break;
				}
				case gsi_hscrollbar: {
					gsi_scrollbar_t *p=(gsi_scrollbar_t*)msg->data2.ptr;
					get_hscrollbar_pos(&p->pstart, &p->psize);
					break;
				}
				case gsi_vscrollbar: {
					gsi_scrollbar_t *p=(gsi_scrollbar_t*)msg->data2.ptr;
					get_vscrollbar_pos(&p->pstart, &p->psize);
					break;
				}
			}
			clearmsg(msg);
			dirtyview();
			return;
		case msg_funcexec:
			if (func(msg->data1.integer, 1)) {
				clearmsg(msg);
				return;
			}
			break;
		case msg_funcquery: {
			char *s=func(msg->data1.integer, 0);
			if (s) {
				msg->msg=msg_retval;
				msg->data1.str=s;
			}
			break;
		}
	}
	ht_view::handlemsg(msg);
}

void ht_text_viewer::make_pos_physical(text_viewer_pos *p)
{
	UINT l=textfile->getlinelength(p->line);
	if (p->pofs > l) p->pofs=l;
}

bool ht_text_viewer::modified()
{
	bool modified1=true;
	textfile->cntl(FCNTL_MODS_IS_DIRTY, 0, textfile->get_size(), &modified1);
	return modified1;
}

void ht_text_viewer::normalize_selection()
{
	if ((text_viewer_pos_compare(&sel_end, &sel_start)<=0)) {
		sel_start.line=0;
		sel_start.pofs=0;
		sel_end.line=0;
		sel_end.pofs=0;
	}
}

UINT ht_text_viewer::physical_cursorx()
{
	int vx=0, px=0, v=cursorx+xofs;
	char line[1024];
	char *linep=line;

	UINT linelen;
	if (!textfile->getline(top_line+cursory, 0, line, sizeof line, &linelen, NULL)) return 0;
	
	while (linelen--) {
		int k=char_vsize(*(linep++), vx);
		vx+=k;
		v-=k;
		if (v<0) break;
		px++;
	}
	if (v>0) px+=v;
	
	return px;
}

void ht_text_viewer::render_str(int x, int y, vcp color, text_viewer_pos *pos, UINT len, char *str, bool multi)
{
	if ((pos->line == sel_start.line) || (pos->line == sel_end.line)) {
		text_viewer_pos p=*pos;
		vcp c;
		while (len--) {
			c=color;
			render_str_color(&c, &p);
			buf_lprint(x++, y, c, 1, str++);
			if (multi) p.pofs++;
		}
	} else {
		render_str_color(&color, pos);
		buf_lprint(x, y, color, len, str);
	}
}

void ht_text_viewer::render_str_color(vcp *color, text_viewer_pos *pos)
{
	if ((text_viewer_pos_compare(pos, &sel_start)>=0) &&
	(text_viewer_pos_compare(pos, &sel_end)<0)) {
		if (VCP_FOREGROUND(*color)==VC_WHITE) {
			*color=VCP(VC_LIGHT(VC_BLACK), VC_WHITE);
		} else {
			*color=vcp_mix(*color, VCP(VC_TRANSPARENT, VC_WHITE));
		}
	}
}

void ht_text_viewer::resize(int rw, int rh)
{
	ht_view::resize(rw, rh);
	if ((int)cursorx>size.w-1) {
		xofs+=(int)cursorx-(size.w-1);
		cursorx=size.w-1;
	}
	if ((int)cursory>size.h-1) {
		top_line+=(int)cursory-(size.h-1);
		cursory=size.h-1;
	}
}

UINT ht_text_viewer::scroll_up(UINT n)
{
	if (top_line > n) top_line -= n; else {
		int q = top_line;
		top_line = 0;
		cursory = 0;
		return q;
	}
	return n;
}

UINT ht_text_viewer::scroll_down(UINT n)
{
	UINT lc=textfile->linecount();
	if (top_line+n+size.h < lc) top_line+=n; else {
		if (lc-top_line>=(UINT)size.h) {
			int q=top_line;
			top_line=lc-size.h;
			cursory=size.h-1;
			return top_line-q;
		}
		cursory=lc-1;
		return 0;
	}
	return n;
}

UINT ht_text_viewer::scroll_left(UINT n)
{
	if (xofs>n) xofs-=n; else xofs=0;
	return n;
}

UINT ht_text_viewer::scroll_right(UINT n)
{
	xofs+=n;
	return n;
}

void ht_text_viewer::select_add(text_viewer_pos *s, text_viewer_pos *e)
{
	text_viewer_pos start=*s, end=*e;
	make_pos_physical(&start);
	make_pos_physical(&end);
	if (text_viewer_pos_compare(&start, &end)>0) {
		text_viewer_pos temp=start;
		start=end;
		end=temp;
	}
	
	if ((text_viewer_pos_compare(&end, &sel_start)==0)) {
		sel_start=start;
	} else if (text_viewer_pos_compare(&start, &sel_end)==0) {
		sel_end=end;
	} else if (text_viewer_pos_compare(&start, &sel_start)==0) {
		sel_start=end;
	} else if (text_viewer_pos_compare(&end, &sel_end)==0) {
		sel_end=start;
	} else {
		sel_start=start;
		sel_end=end;
	}
	
	if (text_viewer_pos_compare(&sel_start, &sel_end)>0) {
		text_viewer_pos temp=sel_start;
		sel_start=sel_end;
		sel_end=temp;
	}
	normalize_selection();
}

void ht_text_viewer::select_clear()
{
	sel_start.line=0;
	sel_start.pofs=0;
	sel_end.line=0;
	sel_end.pofs=0;
}

void ht_text_viewer::select_set(text_viewer_pos *s, text_viewer_pos *e)
{
	sel_start=*s;
	sel_end=*e;
	normalize_selection();
}

void ht_text_viewer::select_start()
{
	selectmode=true;
	selectstart.line=top_line+cursory;
	selectstart.pofs=physical_cursorx();
}

void ht_text_viewer::select_end()
{
	text_viewer_pos p;
	selectmode=false;
	p.line=top_line+cursory;
	p.pofs=physical_cursorx();
	select_add(&selectstart, &p);
}

void ht_text_viewer::set_lexer(ht_syntax_lexer *l, bool own_l)
{
	if (own_lexer) {
		lexer->done();
		delete lexer;
	}
	lexer=l;
	own_lexer=own_l;
}

void ht_text_viewer::set_textfile(ht_textfile *t, bool own_t)
{
	if (own_textfile) {
		textfile->done();
		delete textfile;
	}
	textfile=t;
	own_textfile=own_t;
}

/*
 *	CLASS ht_text_editor
 */

void ht_text_editor::init(bounds *b, bool own_t, ht_textfile *t, bool own_l, ht_syntax_lexer *l, UINT e)
{
	ht_text_viewer::init(b, own_t, t, own_l, l);
	edit_options=e;
	if (edit_options & TEXTEDITOPT_UNDO) {
		undo_list = new ht_text_editor_undo();
		undo_list->init(1024*1024);
	} else {
		undo_list = NULL;
	}
}

void ht_text_editor::done()
{
	if (undo_list) {
		undo_list->destroy();
		delete undo_list;
	}
	ht_text_viewer::done();
}

void ht_text_editor::clipboard_cut_cmd()
{
	clipboard_copy_cmd();
	clipboard_delete_cmd();
}

void ht_text_editor::clipboard_delete_cmd()
{
	if (text_viewer_pos_compare(&sel_start, &sel_end)<0) {
		FILEOFS s, e;
		if (textfile->convert_line2ofs(sel_start.line, sel_start.pofs, &s) &&
		textfile->convert_line2ofs(sel_end.line, sel_end.pofs, &e)) {
			int k=0;
			int p=sel_start.pofs, l=sel_start.line-top_line;
			if (sel_start.line < sel_end.line) {
				k=textfile->getlinelength(sel_start.line)-sel_start.pofs;
			} else {
				k=sel_end.pofs-sel_start.pofs;
			}
			delete_chars(sel_start.line, sel_start.pofs, k);
			if (sel_start.line < sel_end.line) {
				if (sel_start.line+1 < sel_end.line) {
					delete_lines(sel_start.line+1, sel_end.line-sel_start.line-1);
				}
				delete_chars(sel_end.line, 0, sel_end.pofs);
				concat_lines(sel_start.line);
			}
			select_clear();
			cursory = l;
			cursor_pput(p);
		}
	}
}

void ht_text_editor::clipboard_paste_cmd()
{
	FILEOFS o;
	UINT px=physical_cursorx();
	UINT py=top_line+cursory;
	UINT ll=textfile->getlinelength(py);
// fill gaps with spaces
	while (px>ll) {
		textfile->insert_chars(py, ll, (void*)" ", 1);
		ll++;
		o++;
	}
	textfile->convert_line2ofs(py, px, &o);
	clipboard_paste(textfile, o);
	UINT s=clipboard_getsize();
	text_viewer_pos start, end;
	textfile->convert_ofs2line(o, &start.line, &start.pofs);
	textfile->convert_ofs2line(o+s, &end.line, &end.pofs);
	select_set(&start, &end);
}

void ht_text_editor::concat_lines(UINT a)
{
	UINT b=a+1;
	if (textfile->has_line(a) && textfile->has_line(b)) {
		UINT alen=textfile->getlinelength(a);
		char *aline=(char*)malloc(alen+1);
		UINT alinelen;
		textfile->getline(a, 0, aline, alen+1, &alinelen, NULL);
	
		text_viewer_pos ss, se;

		ss=sel_start;
		se=sel_end;

		insert_chars(b, 0, aline, alinelen);
	
		delete_lines(a, 1);

		if (b>ss.line) {
			if (b==se.line) {
				se.pofs+=alen;
				se.line--;
			} else if (b<se.line) {
				se.line--;
			}
		} else {
			if (b==ss.line) {
				ss.pofs+=alen;
			}
			ss.line--;
			se.line--;
		}

		sel_start=ss;
		sel_end=se;
		normalize_selection();
	}
}

void ht_text_editor::delete_chars(UINT line, UINT ofs, UINT count)
{
	text_viewer_pos pos;
	pos.line=line;
	pos.pofs=ofs;
	if ((sel_end.line==pos.line) &&
	(text_viewer_pos_compare(&pos, &sel_start)>=0) &&
	(text_viewer_pos_compare(&pos, &sel_end)<0)) {
		sel_end.pofs-=count;
	} else if ((sel_start.line==pos.line) &&
	(text_viewer_pos_compare(&pos, &sel_start)<0)) {
		sel_start.pofs-=count;
		if (sel_end.line==pos.line) sel_end.pofs-=count;
	}
	textfile->delete_chars(line, ofs, count);
}

void ht_text_editor::delete_lines(UINT line, UINT count)
{
	if (sel_start.line+1>line) sel_start.line--;
	if (sel_end.line>line) sel_end.line--;
	normalize_selection();
	textfile->delete_lines(line, count);
}

char *ht_text_editor::func(UINT i, bool execute)
{
	switch (i) {
		case 2:
			if (execute) {
				if (textfile->get_filename()) sendmsg(cmd_file_save);
					else app->sendmsg(cmd_file_saveas);
			}
			return "save";
	}
	return ht_text_viewer::func(i, execute);
}

vcp ht_text_editor::get_bgcolor()
{
	return getcolor(palidx_windialog_input_focused);
}

void ht_text_editor::handlemsg(htmsg *msg)
{
	switch (msg->msg) {
		case msg_keypressed: {
			int k=msg->data1.integer;
			switch (k) {
				case K_Return: {
					UINT px=physical_cursorx();
					text_viewer_pos apos, bpos;
					apos.line = top_line+cursory;
					bpos.line = apos.line+1;
					apos.pofs = px;
					bpos.pofs = 0;
					textoperation_apply(new ht_undo_data_split_line(&apos, &bpos));
					
/*					split_line(top_line+cursory, physical_cursorx());
					cursor_home();
					cursor_down(1);*/
					dirtyview();
					clearmsg(msg);
					return;
				}
				case K_Delete: {
					UINT cx=physical_cursorx();
					if (cx<textfile->getlinelength(top_line+cursory)) {
						UINT px=physical_cursorx();
//						delete_chars(top_line+cursory, cx, 1);
//						cursor_pput(px);
						text_viewer_pos apos, bpos;
						char s[1024];
						apos.line = bpos.line = top_line+cursory;
						apos.pofs = px;
						bpos.pofs = px;
//                              textfile->get_char(apos.line, &s, px);
						UINT i;
						textfile->getline(apos.line, 0, s, 1024, &i, NULL);
						textoperation_apply(new ht_undo_data_delete_string2(&apos, &bpos, &s[px], 1));
					} else {
						concat_lines(top_line+cursory);
					}
					dirtyview();
					clearmsg(msg);
					return;
				}
				case K_BackSpace: {
					UINT cx=physical_cursorx();
					if (cx) {
						UINT px=physical_cursorx()-1;
//						delete_chars(top_line+cursory, cx-1, 1);
//						cursor_pput(px);
						text_viewer_pos apos, bpos;
						char s[1024];
						apos.line = bpos.line = top_line+cursory;
						apos.pofs = cx;
						bpos.pofs = px;
//                              textfile->get_char(apos.line, &s, px);
						UINT i;
						textfile->getline(apos.line, 0, s, 1024, &i, NULL);
						textoperation_apply(new ht_undo_data_delete_string(&apos, &bpos, &s[px], 1));
					} else {
						if (top_line+cursory) {
/*							UINT la=top_line+cursory-1;
							cursor_up(1);
							cursor_end();
							concat_lines(la);*/
							UINT px=physical_cursorx();
							text_viewer_pos apos, bpos;
							apos.line = top_line+cursory;
							apos.pofs = px;
							bpos.line = apos.line-1;
							bpos.pofs = get_line_length(bpos.line);
							textoperation_apply(new ht_undo_data_join_line(&apos, &bpos));
						}
					}
					dirtyview();
					clearmsg(msg);
					return;
				}
				case K_Alt_P: {
					sendmsg(cmd_text_editor_protocol);
					clearmsg(msg);
					return;
				}
				case K_Alt_U:
				case K_Alt_Backspace: {
					sendmsg(cmd_text_editor_undo);
					clearmsg(msg);
					return;
				}
				case K_Alt_R: {
					sendmsg(cmd_text_editor_redo);
					clearmsg(msg);
					return;
				}
				case K_Alt_V:
				case K_Shift_Insert:
					sendmsg(cmd_edit_paste);
					clearmsg(msg);
					return;
				case K_Alt_X:
				case K_Shift_Delete:
					sendmsg(cmd_edit_cut);
					clearmsg(msg);
					return;
				case K_Alt_D:
				case K_Control_Delete:
					sendmsg(cmd_edit_delete);
					clearmsg(msg);
					return;
				default: {
					int k=msg->data1.integer;
					if (((k>=' ') && (k<=255)) ||
					((k=='\t') && (edit_options & TEXTEDITOPT_INPUTTABS))) {
						char s=k;
						text_viewer_pos apos, bpos;
						UINT px=physical_cursorx();
						apos.line = bpos.line = top_line+cursory;
						apos.pofs = px;
						bpos.pofs = px+1;
						textoperation_apply(new ht_undo_data_insert_string(&apos, &bpos, &s, 1));
//						insert_chars(top_line+cursory, px, &s, 1);
//						cursor_pput(px+1);
						dirtyview();
						clearmsg(msg);
						return;
					}							
				}
			}
			break;
		}
		case cmd_edit_paste: {
			clipboard_paste_cmd();
			dirtyview();
			clearmsg(msg);
			return;
		}
		case cmd_edit_cut: {
			clipboard_cut_cmd();
			dirtyview();
			clearmsg(msg);
			return;
		}
		case cmd_edit_delete: {
			clipboard_delete_cmd();
			dirtyview();
			clearmsg(msg);
			return;
		}
		case cmd_file_save: {
			clearmsg(msg);
			save();
			return;
		}
		case cmd_text_editor_undo: {
			undo();
			dirtyview();
			clearmsg(msg);
			return;
		}          
		case cmd_text_editor_redo: {
			redo();
			dirtyview();
			clearmsg(msg);
			return;
		}          
		case cmd_text_editor_protocol: {
			show_protocol();
			dirtyview();
			clearmsg(msg);
			return;
		}
	}
	ht_text_viewer::handlemsg(msg);
}

void ht_text_editor::insert_chars(UINT line, UINT ofs, void *chars, UINT len)
{
	text_viewer_pos pos;
	pos.line=line;
	pos.pofs=ofs;
	if ((sel_end.line==pos.line) &&
	(text_viewer_pos_compare(&pos, &sel_start)>=0) &&
	(text_viewer_pos_compare(&pos, &sel_end)<0)) {
		sel_end.pofs+=len;
	} else if ((sel_start.line==pos.line) &&
	(text_viewer_pos_compare(&pos, &sel_start)<0)) {
		sel_start.pofs+=len;
		if (sel_end.line==pos.line) sel_end.pofs+=len;
	}
	textfile->insert_chars(line, ofs, chars, len);
}

void ht_text_editor::insert_lines(UINT line, UINT count)
{
	if (sel_start.line+1>line) sel_start.line++;
	if (sel_end.line>line) sel_end.line++;
	normalize_selection();
	textfile->insert_lines(line, count);
}

void ht_text_editor::save()
{
	if (!textfile->get_filename()) return;
	dirtyview();
	
	char tempfile[PATH_MAX+20];
	sys_dirname(textfile->get_filename(), tempfile);

	strcat(tempfile, "/");

	char *n=tempfile+strlen(tempfile);
	int i=0;
	FILE *exists;
	do {
		sprintf(n, "ht%06d.tmp", i++);
		errno=0;
		exists=fopen(tempfile, "rb");
		if (exists) fclose(exists);
	} while ((errno!=ENOENT) || exists);
	
	ht_file *temp=new ht_file();
	temp->init(tempfile, FAM_CREATE | FAM_WRITE);

	if (temp->get_error()) {
		errorbox("couldn't create tempfile %s", tempfile);
		return;
	}
	ht_streamfile *old = textfile->get_layered();
	char *oldname = strdup(textfile->get_filename());

	textfile->copy_to(temp);

	ht_streamfile *alibi = new ht_streamfile();
	alibi->init();
	
	textfile->set_layered(alibi);

	temp->done();
	delete temp;

	old->done();
	delete old;

	sys_deletefile(oldname);
	rename(tempfile, oldname);

	temp=new ht_file();
	temp->init(oldname, FAM_READ);

	free(oldname);

	if (temp->get_error()) {
		errorbox("arrggh");
		return;
	}

	textfile->set_layered(temp);
	
	alibi->done();
	delete alibi;
	
	if (undo_list) {
		undo_list->mark_clean();
	}
}

void ht_text_editor::show_protocol()
{
	bounds c, b;
	app->getbounds(&c);
	b.w=c.w*5/6;
	UINT bh=b.h=c.h*5/6;
	b.x=(c.w-b.w)/2;
	b.y=(c.h-b.h)/2;
	ht_dialog *dialog;
	NEW_OBJECT(dialog, ht_dialog, &b, "protocol", FS_KILLER | FS_TITLE | FS_MOVE);
	BOUNDS_ASSIGN(b, 1, 0, b.w-4, 1);
	ht_statictext *text;
	NEW_OBJECT(text, ht_statictext, &b, " ", 0);
	dialog->insert(text);
	b.y = 1;
	b.h = bh-5;
	ht_text_listbox *list;
	NEW_OBJECT(list, ht_text_listbox, &b, 1, 0);
	UINT cp = undo_list->get_current_position();
	list->insert_str(0, "--- initial state ---");
	for (UINT i=0; i<undo_list->count(); i++) {
		char buf[1024];
		((ht_undo_data*)undo_list->get(i))->gettext(buf, 1024);
		list->insert_str(i+1, buf);
	}
	list->update();
	list->goto_item(list->getbyid(cp));
	dialog->insert(list);
	int r = dialog->run(0);
	if (r == button_ok) {
		ht_listbox_data d;
		list->databuf_get(&d);
		int a = d.cursor_id;
		int b = cp;
		if (a-b < 0) {
			for (int i=0; i < b-a; i++) {
				undo();
			}
		} else if (a-b > 0) {
			for (int i=0; i < a-b; i++) {
				redo();
			}
		}
	}
	dialog->done();
	delete dialog;
}

void ht_text_editor::split_line(UINT a, UINT pos)
{
	UINT l=textfile->getlinelength(a);
	if (pos>l) pos=l;
	char *aline=(char*)malloc(pos+1);
	UINT alinelen;
	textfile->getline(a, 0, aline, pos+1, &alinelen, NULL);

	text_viewer_pos p, ss, se;
	p.line=a;
	p.pofs=pos;

	ss=sel_start;
	se=sel_end;

	insert_lines(a, 1);
	insert_chars(a, 0, aline, alinelen);

	delete_chars(a+1, 0, pos);
	free(aline);
	
	if (text_viewer_pos_compare(&p, &ss)>0) {
		if (text_viewer_pos_compare(&p, &se)<0) {
			if (se.line==p.line) {
				se.pofs-=pos;
			}
			se.line++;
		}
	} else {
		if (ss.line==p.line) {
			ss.pofs-=pos;
		}
		ss.line++;
		se.line++;
	}
	sel_start=ss;
	sel_end=se;
	normalize_selection();
}

void ht_text_editor::textoperation_apply(ht_undo_data *ud)
{
	if (undo_list) {
		undo_list->insert_undo(this, ud);
	} else {
		ud->apply(this);
		delete ud;
	}
}

void ht_text_editor::redo()
{
	if (undo_list) {
		undo_list->redo(this);
		if (undo_list->is_clean()) {
			textfile->cntl(FCNTL_MODS_CLEAR_DIRTY);
		}
	}
}

void ht_text_editor::undo()
{
	if (undo_list) {
		undo_list->undo(this);
		if (undo_list->is_clean()) {
			textfile->cntl(FCNTL_MODS_CLEAR_DIRTY);
		}
	}
}

