/* $Id: netpolling.h,v 1.7 2000/08/08 01:07:21 bergo Exp $ */

#ifndef NETPOLLING_H
#define NETPOLLING_H

#include "polling.h"

#define MAX_NET_LINE 4096

/**
   network connection (TCP stream) manager. implements I/O
   for the network pollers (derivated classes that poll
   rgpsp remotely). This I/O is already oriented to the
   rpgps protocol (i.e.: lines beginning with '*' are errors).
 */
class NetworkActor {
 public:

  /// opens connection to hostname:port
  int hopen(char *hostname,int port);

  /// an fgets-like input method, returns 0 if ok, <0 on error.
  int hreadln(char *buffer,int max);

  /// closes connection
  int hclose();

  /// reads all available data (garbage?) from socket. Non-blocking.
  void read_dead_output();

  /** requests refresh with space (usually provides a system info
      header followed by non-detailed process list */
  void request_refresh();

  /** requests refresh with  details. when the remote node is
      marked as special, the sent character is Ctrl-L, and the
      process list has LOTS of details. Otherwise sends space. */
  void request_detailed_refresh();

  /** requests refresh but just for the system info header.
      If the remote node is marked as special, sends an 'h'
      character, and retrieves a header without process list.
      If the remote node does not announces itself as special,
      this will send the space character as usual. */
  void request_usage_header();

  /// returns whether the remote node is marked as special.
  int isSpecial();

 protected:

  /// 0 means the connection is ok.
  int ok;
 private:
  int thesocket;
  char privateb[128];
  int thedoor;

  int is_special;

  friend void *thread_network_blocking_stuff(void *p);
};

/// network watch poller, provides a process list from a remote host
class NetworkListPoller : public NetworkActor {
 public:

  /** constructor, connects to host hname on the given tcp port.
      hname may be a hostname or ip address in standard dot notation
      (if gethostbyname() accepts it, we do too) */
  NetworkListPoller(char *hname,int port);

  /// actual polling implementation
  virtual void poll();

  /// clean up, called on destruction.
  void terminate();

  /// returns positive if connection is up, 0 otherwise.
  int isOk();

  /// the process list, as a glib doubly linked list of (ProcessListItem *).
  GList *process_list;

 protected:
  /// clean up for the process list.
  void reset();

 private:
  char hostname[128];
  int tcpport;
};

/// network watch details poller, provides single process info.
class NetworkDetailsPoller : public NetworkActor {
 public:

  /** constructor, opens connection to hname : port.
      hname may be a hostname or ip address in standard dot notation
      (if gethostbyname() accepts it, we do too) */
  NetworkDetailsPoller(char *hname,int port);

  /// frees the buffer
  ~NetworkDetailsPoller();
  
  /// actual polling implementation. fills item with data about whichpid.
  virtual void poll(int whichpid);

  /// poll to an internal buffer
  void poll2buffer();

  /// query data in buffer instead of making new net traffic
  void queryBuffer(int whichpid);
  
  /// clean up, called on destruction.
  void terminate();

  /// returns positive if connection is up, 0 otherwise.
  int isOk();

  /** returns positive if this poller was initialized with hname equal to
      s, 0 otherwise. */
  int isHost(char *s);

  /// contains the details about the last process polled.
  ProcessItem *item;

 private:
  int nb_readln(char *dest,int lim);
  void nb_gotop();

  char *netbuffer;
  char hostname[128];
  int tcpport;

  char *nb_pos;
};

/// network watch poller, provides cpu and memory information.
class NetworkSystemInfoProvider :
 public SystemInfoProvider, NetworkActor {
 public:

  /** constructor, opens connection to hname : port.
      hname may be a hostname or ip address in standard dot notation
      (if gethostbyname() accepts it, we do too) */
  NetworkSystemInfoProvider(char *hname,int port);

  /** actual polling implementation. memory info may be bogus due
      to innacurate rgpsp implementations running on remote site. */
  void update();

  /// clean up called on destruction. closes connection.
  void terminate();

  /// returns the hostname on the other end of the connection.
  char *getHostname();

 private:
  char hostname[128];
  int tcpport;
  int mem_shift[256];
};

#endif
