/**************************************************************************\
 gatos (General ATI TV and Overlay Software)

  Project Coordinated By Insomnia (Steaphan Greene)
  (insomnia@core.binghamton.edu)

  Copyright (C) 1999 Steaphan Greene, yvind Aabling, Octavian Purdila, 
	Vladimir Dergachev, Christian Lupien and Chris Hardy

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.

\**************************************************************************/

#define GATOS_TVOUT_C 1

#include "gatos.h"
#include "tvout.h"
#include "m64regs.h"
#include "r128regs.h"

#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <math.h>

/* MPP_CONFIG values */
#define MPPNORMAL	0x80038CB0L	/* write */
#define MPPREAD		0x84038CB0L	/* prefetch read */
#define MPPNORMALINC	0x80338CB0L	/* write, autoincrement MPP_ADDR */
#define MPPREADINC	0x84338CB0L	/* prefetch read, MPP_ADDR autoincr */
/* MPP_ADDR autoincrement wraps around after 4 adresses (2 LSB's only) */

#define MPPTRIES	100	/* See comments in mpp_wait() /AA */

/* Read a byte with possible bugfix correction from MPP */
#define MPPREADBYTE	(MPP_DATA0 ^ gatos.bugfix_mppread)

/* Private global vars */
static char *ident=NULL, *name=NULL ;
static int id=0, rev=0 ;
static u32 timeouts=0 ;

/* Note:
 * ImpacTV registers are 32 bit, indexed by the 16 bit
 * value in MPP registers 0x0009 (MSB) and 0x0008 (LSB).
 * Value of indexed ImpacTV register is accessed
 * through MPP registers 0x0018 (LSB) to 0x001B (MSB).
 * tvout_read_i2c_cntl8() and tvout_write_i2c_cntl8() assumes that
 * the ImpacTV Register Index points to TV_I2C_CNTL (index 0x0015)
 * and that MPP_ADDR points to low byte of ImpacTV Register Data.
 * All other routines must call tvout_addr(TV_I2C_CNTL) to restore
 * these conditions after accessing other ImpacTV chip registers.
 * AA
 */

/* ------------------------------------------------------------------------ */
/* Initialization routine */

int tvout_init(void) {

  u8 init ; u32 save1, save2, save3 ;

  /* The Mach64 TV Out driver hasn't been ported to the Rage128 yet /AA */
  if (ati.r128) RETURN0 ;

  /* ATI chip with MPP read bug ? */
  if ( ati.gt_c2u1 || ati.gt_c2u2 )
    if ( ati.gp_c2u1 || ati.gq_c2u1 ||
         ati.gp_c2u2 || ati.gq_c2u2 ) {
    if (VERBOSE) {
      printf("TV Out: Chip with MPP read bug (correcting)\n") ;
      fflush(stdout) ; }
    gatos.bugfix_mppread = 0x81 ; }

  save1 = DAC_CNTL ; save2 = MPP_CONFIG ; save3 = MPP_STROBE_SEQ ;
  MPP_STROBE_SEQ = 0x00000383L ; DAC_CNTL &= 0xFFFFBFFFL ;

  /* Read ImpacTV Revision Code and Initialize value */
  MPP_CONFIG = MPPREAD ; mpp_wait("tvout_init()") ;
  MPP_ADDR = 0x0000000AL ; init = MPPREADBYTE ; mpp_wait("tvout_init() 1") ;
  MPP_ADDR = 0x0000000BL ; id   = MPPREADBYTE ; mpp_wait("tvout_init() 2") ;
  MPP_CONFIG = MPPNORMAL ;

  switch (id) {
    case IMPACTV:                    ident = "ImpacTV" ;   rev = 1 ; break ;
    case IMPACTV2:  case IMPACTV2M:  ident = "ImpacTV2" ;  rev = 2 ; break ;
    case IMPACTV2P: case IMPACTV2PM: ident = "ImpacTV2+" ; rev = 3 ; break ;
    default: break ; }
  switch (id) {
    case IMPACTV:    name = "ImpacTV" ; break ;
    case IMPACTV2:   name = "ImpacTV2 (Macrovision disabled)" ; break ;
    case IMPACTV2M:  name = "ImpacTV2 (Macrovision enabled)" ; break ;
    case IMPACTV2P:  name = "ImpacTV2+ (Macrovision disabled)" ; break ;
    case IMPACTV2PM: name = "ImpacTV2+ (Macrovision enabled)" ; break ;
    default: break ; }
  if (init != 0x00) rev = 0 ;

  if (rev) {
    if (VERBOSE) { printf("TV Out: %s\n",name) ; fflush(stdout) ; }
    tvout_addr(TV_I2C_CNTL) ; }
  else { /* Initialization failed, restore registers */
    if (VERBOSE) {
      printf("TV Out: No ImpacTV chip (ID read was %02X)\n", id) ;
      fflush(stdout) ; }
    DAC_CNTL = save1 ; MPP_CONFIG = save2 ; MPP_STROBE_SEQ = save3 ; }

  /* Always return success (even if ImpacTV chip not present :-) */
  RETURN0 ; }

/* ------------------------------------------------------------------------ */
/* Public routines */

/* For MPP/ImpacTV I2C Driver */
int tvout_present(void) { return (rev!=0) ; }

/* Read low byte of ImpacTV TV_I2C_CNTL register */
u8 tvout_read_i2c_cntl8(void) {
  mpp_wait("tvout_read_i2c_cntl8()") ;
  MPP_CONFIG = MPPREAD ; mpp_wait("tvout_read_i2c_cntl8() 1") ;
  return MPPREADBYTE ; }

/* Write low byte of ImpacTV TV_I2C_CNTL register */
void tvout_write_i2c_cntl8(u8 data) {
  mpp_wait("tvout_write_i2c_cntl8()") ;
  MPP_CONFIG = MPPNORMAL ; MPP_DATA0 = data ; }

/* Read ImpacTV register */
u32 tvout_read32(u16 addr) {
  u32 data ;
  tvout_addr(addr) ;
  MPP_CONFIG = MPPREADINC ; mpp_wait("tvout_read32()") ;
  data  = MPPREADBYTE ;       mpp_wait("tvout_read32() 1") ;
  data |= MPPREADBYTE << 8 ;  mpp_wait("tvout_read32() 2") ;
  data |= MPPREADBYTE << 16 ; mpp_wait("tvout_read32() 3") ;
  data |= MPPREADBYTE << 24 ; mpp_wait("tvout_read32() 4") ;
  if (addr != TV_I2C_CNTL) tvout_addr(TV_I2C_CNTL) ; return data ; }

/* Write to ImpacTV register */
void tvout_write32(u16 addr, u32 data) {
  tvout_addr(addr) ;
  mpp_wait("tvout_write32()") ;
  MPP_CONFIG = MPPNORMALINC ;
  MPP_DATA0 = data ;       mpp_wait("tvout_write32() 1") ;
  MPP_DATA0 = data >> 8 ;  mpp_wait("tvout_write32() 2") ;
  MPP_DATA0 = data >> 16 ; mpp_wait("tvout_write32() 3") ;
  MPP_DATA0 = data >> 24 ; mpp_wait("tvout_write32() 4") ;
  if (addr != TV_I2C_CNTL) tvout_addr(TV_I2C_CNTL) ; }

/* ------------------------------------------------------------------------ */
/* Private routines */

/* Set ImpacTV register index and return MPP_ADDR
 * to 0x0018 for ImpacTV register data access. */
static void tvout_addr(u16 addr) {
  mpp_wait("tvout_addr()") ;
  MPP_CONFIG = MPPNORMALINC ; MPP_ADDR = 0x00000008L ;
  MPP_DATA0 = addr ;    mpp_wait("tvout_addr() 1") ;
  MPP_DATA0 = addr>>8 ; mpp_wait("tvout_addr() 2") ;
  MPP_CONFIG = MPPNORMAL ; MPP_ADDR = 0x00000018L ; }

/* Returns 1 if ready, 0 if MPP bus timed out (not ready) */
static int mpp_wait(char *str) {
  u32 tries=MPPTRIES ;
  /* For some reason, it's *necessary* to have *two* while() poll loops
   * to avoid MPP bus timeouts, but doubling MPPTRIES doesn't help ...
   * Maybe the code optimizer is playing tricks on us ?!?
   * Nah, can't be, problem is there with 'gcc -g' as well ...
   * Also, it seems to make no difference if MPPTRIES is 0x100
   * (the official ATI value) or 1, so I've naturally chosen 1 ...  /AA
   */
  while (tries && (MPP_CONFIG3 & 0x40)) tries-- ; if (tries) return 1 ;
//  tries = MPPTRIES ;
//  while (tries && (MPP_CONFIG3 & 0x40)) tries-- ; if (tries) return 1 ;
  if (!(gatos.debug&GATOSQUIET))
    fprintf(stderr,"mpp_wait(): MPP bus timeout in %s\n",str) ;
  timeouts++ ; return 0 ; }

/* ------------------------------------------------------------------------ */
/* Debug and report routines */

#define DUMPREG(NAME)	\
	fprintf(stderr,"%s: %-22s (0x%04X) = 0x%08X\n", \
		ident,#NAME,NAME,tvout_read32(NAME))

void tvout_dumpregs(void) {
  if (!rev) return ;
  DUMPREG(TV_MASTER_CNTL) ;
  DUMPREG(TV_RGB_CNTL) ;
  DUMPREG(TV_CLKOUT_CNTL) ;
  DUMPREG(TV_SYNC_CNTL) ;
  DUMPREG(TV_I2C_CNTL) ;
  DUMPREG(TV_MPP_DATA_CNTL) ;
  DUMPREG(TV_HTOTAL) ;
  DUMPREG(TV_HDISP) ;
  DUMPREG(TV_HSIZE) ;
  DUMPREG(TV_HSTART) ;
  DUMPREG(TV_HCOUNT) ;
  DUMPREG(TV_VTOTAL) ;
  DUMPREG(TV_VDISP) ;
  DUMPREG(TV_VCOUNT) ;
  DUMPREG(TV_FTOTAL) ;
  DUMPREG(TV_FCOUNT) ;
  DUMPREG(TV_FRESTART) ;
  DUMPREG(TV_HRESTART) ;
  DUMPREG(TV_VRESTART) ;
  DUMPREG(TV_SYNC_SIZE) ;
  DUMPREG(TV_TV_PLL_CNTL) ;
  DUMPREG(TV_CRT_PLL_CNTL) ;
  DUMPREG(TV_PLL_CNTL) ;
  DUMPREG(TV_PLL_TEST_CNTL) ;
  DUMPREG(TV_CLOCK_SEL_CNTL) ;
  DUMPREG(TV_FRAME_LOCK_CNTL) ;
  DUMPREG(TV_SYNC_LOCK_CNTL) ;
  DUMPREG(TV_TVO_SYNC_PAT_ACCUM) ;
  DUMPREG(TV_TVO_SYNC_THRESHOLD) ;
  DUMPREG(TV_TVO_SYNC_PAT_EXPECT) ;
  DUMPREG(TV_DELAY_ONE_MAP_A) ;
  DUMPREG(TV_DELAY_ONE_MAP_B) ;
  DUMPREG(TV_DELAY_ZERO_MAP_A) ;
  DUMPREG(TV_DELAY_ZERO_MAP_B) ;
  DUMPREG(TV_TVO_DATA_DELAY_A) ;
  DUMPREG(TV_TVO_DATA_DELAY_B) ;
  DUMPREG(TV_HOST_READ_DATA) ;
  DUMPREG(TV_HOST_WRITE_DATA) ;
  DUMPREG(TV_HOST_RD_WT_CNTL) ;
  DUMPREG(TV_VSCALER_CNTL) ;
  DUMPREG(TV_TIMING_CNTL) ;
  DUMPREG(TV_GAMMA_CNTL) ;
  if (rev == 2) {
    DUMPREG(TV_Y_FALL_CNTL) ;
    DUMPREG(TV_Y_RISE_CNTL) ;
    DUMPREG(TV_Y_SAW_TOOTH_CNTL) ; }
  DUMPREG(TV_MODULATOR_CNTL1) ;
  DUMPREG(TV_MODULATOR_CNTL2) ;
  DUMPREG(TV_PRE_DAC_MUX_CNTL) ;
  DUMPREG(TV_TV_DAC_CNTL) ;
  DUMPREG(TV_CRC_CNTL) ;
  DUMPREG(TV_VIDEO_PORT_SIG) ;
  if (rev == 3) {
    DUMPREG(TV_VBI_20BIT_CNTL) ;
    DUMPREG(TV_VBI_LEVEL_CNTL) ; }
  DUMPREG(TV_UV_ADR) ;
  DUMPREG(TV_FIFO_TEST_CNTL) ;
  /* Report then clear MPP bus timeout stats */
  if (timeouts) fprintf(stderr,"Warning: %d MPP bus timeouts\n",timeouts) ;
  timeouts = 0 ; }

/* ------------------------------------------------------------------------ */
/* Setting up tvout */

#define RefClock 28.63636
//#define RefClock 28.63

static double tvclk;
int tvout_start(void) {
  if(!tvout_present()) RETURN(ENODEV);
  tvout_write32(TV_MASTER_CNTL,0);
  tvout_write32(TV_CLKOUT_CNTL,1);
  tvout_write32(TV_SYNC_CNTL,0x28);
  /* For NTSC, if main cloc kis RefClock MHz */
  tvout_write32(TV_TV_PLL_CNTL,0x00005438);
  tvclk=RefClock*0x54/0x38;
   /* leave macro on defaults (mine are 0xD4)*/
  tvout_write32(TV_PLL_CNTL,tvout_read32(TV_PLL_CNTL)|0x18); 
  tvout_write32(TV_PLL_TEST_CNTL,0);
  tvout_write32(TV_FRAME_LOCK_CNTL,0xf);
  tvout_write32(TV_SYNC_LOCK_CNTL,0x01000000);
  tvout_write32(TV_HOST_RD_WT_CNTL,0);
  if(rev==1)
   tvout_write32(TV_GAMMA_CNTL,0x008D0000);
  else { /* Need to fill this */}
  tvout_write32(TV_MODULATOR_CNTL1,0x003B462C);//??? For NTSC, Phase alt=??
  tvout_write32(TV_MODULATOR_CNTL2,0x00000177); //???For NTSC, Phase alt=??
  if(rev==1) tvout_write32(TV_PRE_DAC_MUX_CNTL,0x0000000c); //composite only
  else { /* Need to fill this */}
  if(rev==1) tvout_write32(TV_TV_DAC_CNTL,0x113); //NTSC, PAL = 0x013 probably
  else { /* Need to fill this */}
  tvout_write32(TV_CRC_CNTL,0);
  tvout_write32(TV_UV_ADR,0x000000BE); // What should this be?
  tvout_write32(TV_FIFO_TEST_CNTL,0); 
  if (VERBOSE) { printf("Ready to setmode!\n") ; fflush(stdout) ; }
  RETURN(tvout_setmode()); }

static u8 olddiv, oldextdiv, tvoutflag=0,ind,oldsel,nsel;

#define OVERSCAN_CLR       (*(MEM_0+0x10))
#define OVERSCAN_WLR       (*(MEM_0+0x11))
#define OVERSCAN_WTB       (*(MEM_0+0x12))
#include <unistd.h>

int tvout_reset(void) {
  if(!tvout_present()) RETURN(ENODEV);
  if(!tvoutflag) RETURN(EPERM);
  OVERSCAN_CLR=0; // should have saved these
  OVERSCAN_WLR=0;
  OVERSCAN_WTB=0;
  TVO_CNTL=0;
  if(rev==1) tvout_write32(TV_PRE_DAC_MUX_CNTL,0x00000008); //composite only
  else { /* Need to fill this */}
  if(rev==1) tvout_write32(TV_TV_DAC_CNTL,0x11a); //NTSC, PAL = 0x013 probably
  else { /* Need to fill this */}
  // should probably disable something: PLL,syncs?
  CLOCK_CNTL= ((nsel|4)<<16)|(((5<<2)|2)<<8)|ind; 
  CLOCK_CNTL= (oldextdiv<<16)|(((11<<2)|2)<<8)|ind;
  CLOCK_CNTL= (olddiv<<16)|(((6<<2)|2)<<8)|ind;
  CLOCK_CNTL= ((oldsel|4)<<16)|(((5<<2)|2)<<8)|ind; 
  usleep(100000);
  CLOCK_CNTL= ((oldsel&~4)<<16)|(((5<<2)|2)<<8)|ind; 
  RETURN0; }

void findNM(double R, u32 *N,u32 *M,u32 htotal, u8 *hadd) {
  int i,n,m,h;
  double diff,rdiff=100.,r;
  for(i=1;i<256;i++) 
   for(h=0;h<8;h++) {
    n=i;
    r=R*(htotal+h)/htotal;
    m=rint(n/r);
    if(m==0) m=1;
    diff=fabs(r-1.*n/m);
    if(diff<rdiff) {rdiff=diff; *N=n; *M=m;*hadd=h;} }
  }

int tvout_setmode(void) {
  double Ref=RefClock,oldclk;
  u32 RefDivider,post_div,ext_div,FBdivider;
  int RGB888mode=1;   /* Otherwise 565 */
  u32 htotal,hdisp,vtotal,vdisp;
  u32 M,N,BytDiv,BytDel,yscal,ythin,hinc,ndiv,nextdiv;
  u8  del[8],hadd;
  double pixclk,tclk;
  // RAGE II approximation
  int uncertainty=2500;
  int ns_per_tap=235;
  int datad[]={14336, 13780, 11520, 11600, 11360, 13390, 11090, 14336};
  int datad_max=0;
  int byte_period,mid_data_eye,no_of_phase,ns_per_phase;
  int i;

  for(i=0;i<8;i++) 
    if(datad[i]>datad_max) datad_max=datad[i]; 
  if(tvoutflag) RETURN(EPERM); 
  htotal=((CRTC_H_TOTAL_DISP&0xffff)+1)*8;
  //assuming extra 7 is 0
  hdisp=(((CRTC_H_TOTAL_DISP>>16)&0xfff)+1)*8;
  vtotal=(CRTC_V_TOTAL_DISP&0xffff)+1;
  vdisp=((CRTC_V_TOTAL_DISP>>16)&0xffff)+1;
  if (VERBOSE) { printf("Setmode: htotal=%i,vtotal=%i  hdisp=%i,vdisp=%i\n",
                  htotal,vtotal,hdisp,vdisp) ; fflush(stdout) ; }

  ind=CLOCK_CNTL0&0x3; 
  CLOCK_CNTL1=(5<<2);  oldsel=CLOCK_CNTL2;
  if((oldsel&0x3)!=3)
     { fprintf(stderr,"TVOUT: current mode is not PLLed!\n"); RETURN(EPERM);}
  CLOCK_CNTL1=(2<<2);  RefDivider=CLOCK_CNTL2;
  CLOCK_CNTL1=(6<<2);  olddiv=CLOCK_CNTL2;
  CLOCK_CNTL1=(11<<2);  oldextdiv=CLOCK_CNTL2;
  CLOCK_CNTL1=((7+ind)<<2);  FBdivider=CLOCK_CNTL2;
  /* I assume these chips have the extended div */
  ext_div=(oldextdiv>>(4+ind))&1;
  post_div=(olddiv>>(2*ind))&3;
  post_div=1<<post_div;
  if(ext_div) {
   if(post_div==1) post_div=3;
   else post_div+=post_div/2; }
  if (VERBOSE) { printf("Setmode: RefDivider=%i, FBdivider=%i, post_div=%i\n",
                  RefDivider,FBdivider,post_div) ; fflush(stdout) ; }
  oldclk=Ref*2*FBdivider/(RefDivider*post_div);
  if (VERBOSE) { printf("Setmode: oldclk=%g\n",
                  oldclk) ; fflush(stdout) ; }
  if((abs(oldclk*1e6/vtotal/htotal-60)>2)||(vtotal>1000)) 
     { fprintf(stderr,"TVOUT: current mode is not good enough!\n");
       RETURN(EPERM);}

  if(oldclk>34) RGB888mode=0; else RGB888mode=1;
  tclk=(60.*1000./1001.)*vtotal*htotal*(RGB888mode?3:2)/1e6;
  BytDiv=floor(180./tclk); /* 180 is my guess CL*/
  if(BytDiv>4) BytDiv=4;
  if(BytDiv<1) BytDiv=1;
  tclk*=BytDiv/2/RefClock;
  N=65;
  M=196;
  if (VERBOSE) { printf("Setmode(in findNM): tclk=%g, N=%i, M=%i, htotal=%i, hadd=%i\n",
                  tclk,N,M,htotal,hadd) ; fflush(stdout) ; }
  findNM(tclk,&N,&M,htotal,&hadd);
  htotal+=hadd;
  pixclk=RefClock*N/M*2/BytDiv/(RGB888mode?3:2);
  if (VERBOSE) { printf("Setmode: M=%i, N=%i, hadd=%i, pixclk=%g, BytDiv=%i, RG888=%i\n",
                  M,N,hadd,pixclk,BytDiv,RGB888mode) ; fflush(stdout) ; }
  no_of_phase=2*BytDiv;
  byte_period=1./(RefClock*N/M*2/BytDiv)*1e12;
  ns_per_phase=byte_period/no_of_phase;
  if(byte_period<12500) mid_data_eye=(byte_period-uncertainty)/3;
  else mid_data_eye=(byte_period-uncertainty)/2;
  if(mid_data_eye > datad_max) {
    BytDel=floor((mid_data_eye-datad_max)/ns_per_phase);
    for(i=0;i<8;i++) {
      del[i]=((mid_data_eye-datad[i])-(BytDel*ns_per_phase))/ns_per_tap;
      if(datad[i]>47) datad[i]=47; }
    if(BytDel!=0) BytDel=no_of_phase-BytDel; }
  else {
    BytDel=floor((datad_max-mid_data_eye)/ns_per_phase)+1;
    for(i=0;i<8;i++) {
      del[i]=(((BytDel*ns_per_phase)+mid_data_eye)-datad[i])/ns_per_tap;
      if(datad[i]>47) datad[i]=47; }
    if(BytDel>(no_of_phase-1)) BytDel=BytDel-no_of_phase; }

  tvout_write32(TV_RGB_CNTL,RGB888mode);
  tvout_write32(TV_HTOTAL,htotal-1);
  tvout_write32(TV_HDISP,hdisp-1);
  tvout_write32(TV_HSIZE,hdisp);
  tvout_write32(TV_HSTART,hdisp+12); /* ????? */
  tvout_write32(TV_VTOTAL,vtotal-1);
  tvout_write32(TV_VDISP,vdisp-1);
  tvout_write32(TV_FTOTAL,1);
  tvout_write32(TV_FRESTART,0);
  tvout_write32(TV_HRESTART,hdisp+12); /* ????? */
  tvout_write32(TV_VRESTART,vdisp+4); /* ????? */
  tvout_write32(TV_SYNC_SIZE,hdisp+8);
  tvout_write32(TV_CRT_PLL_CNTL,(N<<8)|(M)); //N can be 9bits for rev>=
  tvout_write32(TV_CLOCK_SEL_CNTL,(BytDel<<8)|((BytDiv-1)<<2)|0x33);
  tvout_write32(TV_TVO_DATA_DELAY_A,
          (del[3]<<24)|(del[2]<<16)|(del[1]<<8)|(del[0]));
  tvout_write32(TV_TVO_DATA_DELAY_B,
          (del[7]<<24)|(del[6]<<16)|(del[5]<<8)|(del[4]));
  if(rev==1) {
    yscal=vtotal*(1<<0xe)*2/525; //This is for NTSC  525
     //max vtotal for NTSC is 1049
    if(yscal>0xffff) {yscal=0xffff; fprintf(stderr,"Scaling TVOUT to big!");}
    ythin=0; // This is max averaging!!!
    tvout_write32(TV_VSCALER_CNTL,((pixclk<.55*tvclk)?0x07000000:0x03000000)|
                   (ythin<<16)|yscal);}
  else { /* this needs to be done !!!*/}
  if(rev==1) {
    hinc=(htotal+50)*(1<<0xc)/2730; //2730 for NTSC ~= tvclk/(525*30)
                                    // 50 is just to scale picture smaller
     //max (htotal+50) for NTSC is 2729
    if(hinc>0xfff) {hinc=0xfff; fprintf(stderr,"Scaling TVOUT to big!");}
    tvout_write32(TV_TIMING_CNTL,0x00090000|hinc);}
  else { /* this needs to be done !!!*/}
  if(RGB888mode) OVERSCAN_CLR=0x00000100;
  else OVERSCAN_CLR=0x00000800;
  OVERSCAN_WLR=0x00010000;
  OVERSCAN_WTB=0x00020000;
  TVO_CNTL=0x80000008|hadd;
  if(RGB888mode) {
    nextdiv= oldextdiv|(1<<(4+ind)); 
    ndiv=olddiv&~(3<<(2*ind));
  } else {
    nextdiv= oldextdiv&~(1<<(4+ind)); 
    ndiv=olddiv&~(3<<(2*ind));
    ndiv|=(1<<(2*ind)); }
  nsel=(oldsel&0xfc)|2|8;
  if (VERBOSE) { printf("Setmode: olddiv=%x, ndiv=%x  oldextdiv=%x, nextdiv=%x oldsel=%x,  nsel=%x\n", olddiv,ndiv,oldextdiv,nextdiv,oldsel,nsel) ; fflush(stdout) ; }
//  RETURN(EIO);
  CLOCK_CNTL= ((oldsel|4)<<16)|(((5<<2)|2)<<8)|ind;
  CLOCK_CNTL= (nextdiv<<16)|(((11<<2)|2)<<8)|ind;
  CLOCK_CNTL= (ndiv<<16)|(((6<<2)|2)<<8)|ind;
  CLOCK_CNTL= ((nsel|4)<<16)|(((5<<2)|2)<<8)|ind;
  usleep(100000); 
  CLOCK_CNTL= ((nsel&~4)<<16)|(((5<<2)|2)<<8)|ind;
  tvoutflag=1;
  RETURN(0);}

int tvout_monitor(void) {
/* what seems to be neede is 1001 for MON which is 
  bits 12-15 of TV_TV_DAC_CNTL.
  direct access to mpp gives good result:
  m 8 a0
  m 0x18 --> either 33 or 13
  m 8 15  

or setting up: m 8 a0;  MON=1111
  m 0x18
*/ 
  int flag=0;
  if(!tvout_present()) return 0;
  tvout_write32(TV_PRE_DAC_MUX_CNTL,0x00008004);
  tvout_write32(TV_TV_DAC_CNTL,0x0000f113);
  usleep(10000);
  if(tvout_read32(TV_TV_DAC_CNTL)&0x20) flag=1;
  tvout_write32(TV_PRE_DAC_MUX_CNTL,0x00000802);
  usleep(10000);
  if(tvout_read32(TV_TV_DAC_CNTL)&0x20) flag|=2;
  tvout_write32(TV_PRE_DAC_MUX_CNTL,0x00000081);
  usleep(10000);
  if(tvout_read32(TV_TV_DAC_CNTL)&0x20) flag|=4;
  /* enable only composite out for now... */
  tvout_write32(TV_PRE_DAC_MUX_CNTL,0x0000000c);
  tvout_write32(TV_TV_DAC_CNTL,0x00000113);
  return flag;
}

int tvout_getdelay(void) {
  u32 htotal,vtotal,hsync,vsync,hoffvpitch,gencntl,daccntl;
  htotal=CRTC_H_TOTAL_DISP;
  vtotal=CRTC_V_TOTAL_DISP;
  hsync=(*(MEM_0+0x01)); 
  vsync=(*(MEM_0+0x03)); 
  hoffvpitch=CRTC_OFF_PITCH;
  gencntl=CRTC_GEN_CNTL;
  daccntl=DAC_CNTL;

  /* setting up  320x200 mode 24bpp?? */
  CRTC_GEN_CNTL= (gencntl&0xfffff8f0)|0x0000040c;
  /*                                         4=(565) 16bit
                                             5=(888) 24bit
                                             6=(a888) 32bit
                                             2=(322) if DAC_DIRECT=1 
                            DAC_DIREAC(10=ah bit)@DAC_CNTL(31h) */
  DAC_CNTL=(daccntl&0xffff7fff)|0x00008000;
//Modeline "320x200@60d" 12.587   320  352  376  400    200  206  207  225
//+Hsync -Vsync doublescan

  CRTC_H_TOTAL_DISP=0x00270031;
  CRTC_V_TOTAL_DISP=0x018F01C1;
  (*(MEM_0+0x01))=0x0003002B;
  (*(MEM_0+0x03))=0x0022019B;
  CRTC_OFF_PITCH=0x0A000000;

  /*  reseting the mode */
  CRTC_H_TOTAL_DISP=htotal;
  CRTC_V_TOTAL_DISP=vtotal;
  (*(MEM_0+0x01))=hsync;
  (*(MEM_0+0x03))=vsync;
  CRTC_OFF_PITCH=hoffvpitch;
  CRTC_GEN_CNTL=gencntl;
  DAC_CNTL=daccntl;
  return 0;
}

