/* 
 *  Copyright (C) 2001 Ross Burton
 *  Copyright (C) 2001 Ricardo Fernndez Pascual
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/**
 * Saving and loading bookmarks in XBEL
 */

/* Galeon includes */
#include "bookmarks_io.h"
#include "misc_general.h"
#include "misc_gui.h"
#include "misc_string.h"

/* system includes */
#include <string.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-util.h>
#include <gnome-xml/parser.h>
#include <gnome-xml/tree.h>
#include <gnome-xml/xmlmemory.h>

/* This is the only exported function. */
BookmarkTranslator *bookmarks_io_xbel_init (void);


/* private functions */
static BookmarkItem *load_from_file (const gchar *filename, 
				     BookmarkItem **new_default_folder);
static BookmarkItem *load_from_string (const gchar *data, 
				       BookmarkItem **new_default_folder);
static gboolean save_to_file (BookmarkItem *root,
			      BookmarkItem *default_folder,
			      const gchar *file);
static gboolean save_to_string (BookmarkItem *root,
				BookmarkItem *default_folder,
				gchar **data);

/**
 * Info about the translator
 */

static gchar *extensions[] = { "xbel", "xml", NULL };

static BookmarkTranslator xbel = {
	N_("XBEL bookmarks format"),
	extensions,
	load_from_file,
	load_from_string,
	save_to_file,
	save_to_string
};

BookmarkTranslator *
bookmarks_io_xbel_init (void)
{
	if (g_list_find (bookmarks_translators, &xbel) == NULL)
		bookmarks_translators = g_list_prepend (bookmarks_translators,
							&xbel);
	return &xbel;
}


/* saving */
static void xbel_save_root (xmlDocPtr doc, BookmarkItem *b,
			      BookmarkItem *default_folder);
static void xbel_save_galeon_data (xmlNodePtr xmlNode, BookmarkItem *b,
				     BookmarkItem *default_folder);
static void xbel_save_site (xmlNodePtr xmlNode, BookmarkItem *b);
static void xbel_save_folder (xmlNodePtr xmlNode, BookmarkItem *b,
				BookmarkItem *default_folder);
static void xbel_save_alias (xmlNodePtr xmlNode, BookmarkItem *b);
static void xbel_save_node (xmlNodePtr xmlNode, BookmarkItem *b, 
			      BookmarkItem *default_folder);

/* loading */
static BookmarkItem *xbel_read (xmlNodePtr item,
				BookmarkItem **default_bookmarks_root);
static BookmarkItem *xbel_read_node (BookmarkLoaderInfo *li, xmlNodePtr node);
static void xbel_read_populate_bookmark (BookmarkLoaderInfo *li, 
					 xmlNodePtr node,
					 BookmarkItem *item);
static void xbel_read_metadata (BookmarkLoaderInfo *li, xmlNodePtr node,
				BookmarkItem *item);

/* Utility Prototypes */
static xmlNodePtr get_child (xmlNodePtr parent, const gchar *name);
static gchar *get_child_value_string (xmlNodePtr parent, const gchar *name);
static gboolean get_child_value_boolean (xmlNodePtr parent, const gchar *name);
static glong get_child_value_long (xmlNodePtr parent, const gchar *name);

/* safe strlen */
#define sstrlen(s) ((s) ? strlen ((s)) : 0)

#define XBEL_GALEON_OWNER "http://galeon.sourceforge.net/"

/**
 * Write the bookmark tree as a XBEL file.  Takes a gchar* as a
 * filename, and a BookmarkItem as the root of the bookmark tree.
 */
static gboolean
save_to_file (BookmarkItem *root, BookmarkItem *default_folder, 
	      const gchar *filename)
{
        xmlDocPtr doc;

        /* Sanity checks */
        g_return_val_if_fail (!default_folder 
			      || default_folder->type == BM_FOLDER, FALSE);
        g_return_val_if_fail (filename != NULL, FALSE);

        doc = xmlNewDoc ("1.0");
        /* Add the root XBEL node */
        xbel_save_root (doc, root, default_folder);
        /* Try and save this document */
        if (xmlSaveFile (filename, doc) <= 0) {
                xmlFreeDoc (doc);
                return FALSE;
        }
        xmlFreeDoc (doc);
        return TRUE;
}

static gboolean
save_to_string (BookmarkItem *root,
		BookmarkItem *default_folder,
		gchar **data)
{
        xmlDocPtr doc;
	xmlChar *mem;
	gint size;

        /* Sanity checks */
        g_return_val_if_fail (!default_folder 
			      || default_folder->type == BM_FOLDER, FALSE);
        g_return_val_if_fail (data != NULL, FALSE);

        doc = xmlNewDoc ("1.0");
        /* Add the root XBEL node */
        xbel_save_root (doc, root, default_folder);
        /* Try and save this document */
	xmlDocDumpMemory (doc, &mem, &size);
	
	*data = mem;

	xmlFreeDoc (doc);
        return size > 0;
}

/**
 * Treat a BM_FOLDER item specially and use it as the root of the XBEL tree
 */
static void
xbel_save_root (xmlDocPtr doc, BookmarkItem *b, BookmarkItem *default_folder)
{
        GList *list;
        gchar *strippedname;
        xmlNodePtr newNode;

        g_return_if_fail (doc != NULL);
	
	xmlCreateIntSubset 
		(doc, "xbel", 
		 "+//IDN python.org//DTD XML Bookmark "
		 "Exchange Language 1.0//EN//XML", 
		 "http://www.python.org/topics/xml/dtds/xbel-1.0.dtd");

        newNode = xmlNewDocNode (doc, NULL, "xbel", NULL);
        doc->root = newNode;
        xmlSetProp(newNode, "version", "1.0");
        /* In Galeon the root of the tree is a BM_FOLDER. However,
	   XBEL complicates the situation by have a special root element,
	   XBEL. This is identical to FOLDER apart from the name of the
	   tag... Thus this hack.  */
	if (BOOKMARK_ITEM_IS_FOLDER (b))
	{
	        xbel_save_galeon_data (newNode, b, default_folder);
        	strippedname = misc_string_strip_uline_accel (b->name);
	        xmlNewTextChild (newNode, NULL, "title", strippedname);
        	g_free(strippedname);
	}
	if (b->alias) {
		gchar *idstr = g_strdup_printf ("%lu", (gulong) b);
		xmlSetProp (newNode, "id", idstr);
		g_free (idstr);
	}
	if (BOOKMARK_ITEM_IS_FOLDER (b))
	{
        	for (list = b->list; list != NULL; list = list->next)
                	xbel_save_node (newNode, list->data, default_folder);
	}
	else
	{
		xbel_save_node (newNode, b, default_folder);
	}
}

/**
 * Add an item of unknown type to the tree. This delegates the
 * responsibility of adding nodes to the tree to other functions.
 */
static void
xbel_save_node (xmlNodePtr xmlNode, BookmarkItem *b,
		  BookmarkItem *default_folder)
{
        g_return_if_fail (xmlNode != NULL);
	if (b->alias_of) 
	{
		xbel_save_alias (xmlNode, b);
	}
	else
	{
		switch (b->type) 
		{
		case BM_SITE:
			xbel_save_site (xmlNode, b);
			break;
		case BM_AUTOBOOKMARKS:
		case BM_FOLDER:
			xbel_save_folder (xmlNode, b, default_folder);
			break;
		case BM_SEPARATOR:
			xmlNewChild (xmlNode, NULL, "separator", NULL);
			break;
		default:
			g_warning 
				("Detected unknown bookmark item type");
                break;
		}
	}
}


/**
 * Add a bookmark to the tree
 */
static void
xbel_save_site (xmlNodePtr xmlNode, BookmarkItem *b)
{
        xmlNodePtr newNode;
        gchar *strippedname;
        g_return_if_fail(xmlNode != NULL);
        newNode = xmlNewChild (xmlNode, NULL, "bookmark", NULL);
        xmlSetRawProp (newNode, "href", b->url);
        strippedname = misc_string_strip_uline_accel (b->name);
        xmlNewTextChild (newNode, NULL, "title", strippedname);
        g_free (strippedname);
	if (b->alias) {
		gchar *idstr = g_strdup_printf ("%lu", (gulong) b);
		xmlSetProp (newNode, "id", idstr);
		g_free (idstr);
	}
        xbel_save_galeon_data (newNode, b, NULL);
}

/**
 * Add a folder to the tree
 */
static void
xbel_save_folder (xmlNodePtr xmlNode, BookmarkItem *b, 
		    BookmarkItem *default_folder)
{
        GList *list;
        gchar *strippedname;
        xmlNodePtr newNode;
        g_return_if_fail (xmlNode != NULL);

        newNode = xmlNewChild (xmlNode, NULL, "folder", NULL);
        xmlSetProp (newNode, "folded", b->expanded ? "no" : "yes");
        strippedname = misc_string_strip_uline_accel (b->name);
        xmlNewTextChild (newNode, NULL, "title", strippedname);
        g_free (strippedname);
	if (b->alias) {
		gchar *idstr = g_strdup_printf ("%lu", (gulong) b);
		xmlSetProp (newNode, "id", idstr);
		g_free (idstr);
	}
	xbel_save_galeon_data (newNode, b, default_folder);
        /* now iterate over the children of this folder */
	if (!b->alias_of)
		for (list = b->list; list != NULL; list = list->next)
			xbel_save_node (newNode, list->data, default_folder);
}


static void
xbel_save_alias (xmlNodePtr xmlNode, BookmarkItem *b)
{
        xmlNodePtr newNode;
	gchar *refstr = g_strdup_printf
		("%lu", (gulong) bookmarks_find_real_bookmark (b));
        g_return_if_fail (xmlNode != NULL);
        newNode = xmlNewChild (xmlNode, NULL, "alias", NULL);
	xmlSetProp (newNode, "ref", refstr);
	g_free (refstr);
        xbel_save_galeon_data (newNode, b, NULL);
}

/**
 * Export the data which Galeon supports but XBEL doesn't by putting
 * it in metadata elements. This is done by create a metadata element
 * with an owner of "http://galeon.sourceforge.net" and adding child
 * nodes which describe features which XBEL doesn't support.
 */
static void
xbel_save_galeon_data (xmlNodePtr xmlNode, BookmarkItem *b,
			 BookmarkItem *default_folder) {
        xmlNodePtr infoNode;
	GString *s;
        g_return_if_fail(xmlNode != NULL);
	if (!b->alias_of && sstrlen (b->notes) > 0) {
		xmlNewTextChild (xmlNode, NULL, "desc", b->notes);
	}
        if ((!b->alias_of 
	     && ((sstrlen (b->nick) > 0) || (sstrlen(b->pixmap_file) > 0) 
		 || b->create_toolbar || b->create_context_menu
		 || (b == default_folder) || b->toolbar_style
		 || b->time_added || b->time_modified || b->time_visited
		 || b->accel_key || b->accel_mods 
		 || (sstrlen (b->smarturl) > 0)
		 || (b->type == BM_AUTOBOOKMARKS)))
	    || (b->alias_of
		&& (b->create_toolbar || b->create_context_menu 
		    || b->toolbar_style)))
	{
		s = g_string_new ("");
                infoNode = xmlNewChild(xmlNode, NULL, "info", NULL);
                infoNode = xmlNewChild(infoNode, NULL, "metadata", NULL);
                xmlSetProp(infoNode, "owner", XBEL_GALEON_OWNER);
                /* export the nick name */
                if (sstrlen (b->nick) > 0) {
                        xmlNewTextChild (infoNode, NULL, "nick", b->nick);
                }
                /* pixmap path */
		/* TODO: move this into the icon attribute somehow */
                if (sstrlen (b->pixmap_file) > 0) {
                        xmlNewTextChild (infoNode, NULL, "pixmap",
					 b->pixmap_file);
                }
                /* create a toolbar */
                if (b->create_toolbar) {
			xmlNewChild (infoNode, NULL, "create_toolbar", "yes");
                }
                /* create a context menu */
                if (b->create_context_menu) {
			xmlNewChild (infoNode, NULL, "create_context", "yes");
                }
		/* is this the default_folder? */
                if (BOOKMARK_ITEM_IS_FOLDER (b) && (default_folder == b)) {
			xmlNewChild (infoNode, NULL, "default_folder", "yes");
                }
		if (b->toolbar_style != 0) {
			g_string_sprintf (s, "%d", b->toolbar_style);
			xmlNewChild (infoNode, NULL, "toolbar_style", s->str);
		}
		if (b->time_visited != 0) {
			g_string_sprintf (s, "%d", b->time_visited);
			xmlNewChild (infoNode, NULL, "time_visited", s->str);
		}
		if (b->time_modified != 0) {
			g_string_sprintf (s, "%d", b->time_modified);
			xmlNewChild (infoNode, NULL, "time_modified", s->str);
		}
		if (b->time_added != 0) {
			g_string_sprintf (s, "%d", b->time_added);
			xmlNewChild (infoNode, NULL, "time_added", s->str);
		}
		if (b->accel_mods != 0) {
			g_string_sprintf (s, "%d", b->accel_mods);
			xmlNewChild (infoNode, NULL, "accel_mods", s->str);
		}
		if (b->accel_key != 0) {
			g_string_sprintf (s, "%d", b->accel_key);
			xmlNewChild (infoNode, NULL, "accel_key", s->str);
		}
		if (sstrlen (b->smarturl) > 0) {
			xmlNewTextChild (infoNode, NULL, "smarturl",
					 b->smarturl);
		}
		if (b->type == BM_AUTOBOOKMARKS) {
			xmlNewChild (infoNode, NULL, "autobookmarks",
				     "yes");
		}
        }
}

/**
 * Take a node and populate a BookmarkItem with information from the
 * node, such as metadata, etc. 
 */
static void
xbel_read_populate_bookmark (BookmarkLoaderInfo *li,
			     xmlNodePtr node, BookmarkItem *item)
{
        gchar *notes = NULL, *folded_s = NULL;
	gchar *idstr = xmlGetProp (node, "id");
	xmlChar *pixmap = NULL;
	gboolean folded = FALSE;
	xmlNodePtr info_node;

	if (idstr) 
	{
		guint idval = strtoul (idstr, NULL, 10);
		g_hash_table_insert (li->id_bookmark,
				     GINT_TO_POINTER (idval), item);
		xmlFree (idstr);
	}

	/* this is probably wrong, but it is harmless because the setting in
	 * the metadata will take precedence, if present */
	pixmap = xmlGetRawProp (node, "icon");
        if (pixmap) {
		g_free (item->pixmap_file); 
		item->pixmap_file = g_strdup (pixmap); 
	}
	
        notes = get_child_value_string (node, "desc");
        folded_s = xmlGetRawProp (node, "folded");
	if (folded_s
	    && (!g_strcasecmp (folded_s, "yes") 
		|| !g_strcasecmp (folded_s, "true")))
	{
		folded = TRUE;
	}

 
        /* Now extract the Galeon-specific data */
        info_node = get_child (node, "info");
        if (info_node) {
                /* Iterate over the metadata children */
                for (info_node = info_node->childs;
		     info_node != NULL; info_node = info_node->next) {
			if (!strcmp (info_node->name, "metadata")) {
				xbel_read_metadata (li, info_node, item);
                        }
                }
        }
	
        /* Now set the attributes of the bookmark */ 
        if (notes) { g_free (item->notes); item->notes = g_strdup (notes); }
	item->expanded = !folded;
	
	/* the pixmap file can be set here or in read metata */
	if (sstrlen (item->pixmap_file) > 0)
		item->pixmap_data = misc_gui_pixmap_data_new_from_file
			(item->pixmap_file, FALSE);

        /* Now free the memory we don't need any more */
	xmlFree (pixmap);
	xmlFree (folded_s);
	g_free (notes);
}

static void 
xbel_read_metadata (BookmarkLoaderInfo *li, xmlNodePtr node,
		    BookmarkItem *item)
{
	gchar *nick = NULL, *pixmap = NULL, *smarturl = NULL;
	gboolean toolbar = FALSE, context = FALSE, default_f = FALSE,
		autobm = FALSE;
	glong acc_key = 0, acc_mod = 0, tb_style = 0,
		t_add = 0, t_mod = 0, t_vis = 0;
	gchar *owner = xmlGetProp (node, "owner");

	if (!strcmp (owner, XBEL_GALEON_OWNER)) {
		/* This is a Galeon metadata element,
		   extract the data from it */
		nick = get_child_value_string (node, "nick");
		smarturl = get_child_value_string (node, "smarturl");
		pixmap = get_child_value_string	(node, "pixmap");
		toolbar = get_child_value_boolean (node, "create_toolbar");
		context = get_child_value_boolean (node, "create_context");
		default_f = get_child_value_boolean (node, "default_folder");
		autobm = get_child_value_boolean (node, "autobookmarks");
		t_add = get_child_value_long (node, "time_added");
		t_vis = get_child_value_long (node, "time_visited");
		t_mod = get_child_value_long (node, "time_modified");
		tb_style = get_child_value_long	(node, "toolbar_style");
		acc_key = get_child_value_long (node, "accel_key");
		acc_mod = get_child_value_long (node, "accel_mods");
	} else {
		/* FIXME: read the info and store it for saving */
	}

        if (nick) { 
		g_free (item->nick); 
		item->nick = g_strdup (nick); 
	}
	if (smarturl) {
		g_free (item->smarturl); 
		item->smarturl = g_strdup (smarturl); 
	}
        if (pixmap) {
		g_free (item->pixmap_file); 
		item->pixmap_file = g_strdup (pixmap); 
	}

	item->create_toolbar =  toolbar;
	item->create_context_menu = context;
	item->time_added = t_add;
	item->time_visited = t_vis;
	item->time_modified = t_mod;
	item->toolbar_style = tb_style;
	item->accel_key = acc_key;
	item->accel_mods = acc_mod;

	if (default_f && BOOKMARK_ITEM_IS_FOLDER (item))
		li->default_bookmarks_root = item;

	if (autobm) 
		item->type = BM_AUTOBOOKMARKS;

	g_free (nick);
	g_free (smarturl);
	g_free (pixmap);
	g_free (owner);
}

/**
 * Start the import of a XBEL document, with a given filename and
 * the default bookmarks folder
 */
static BookmarkItem *
load_from_file (const char *file, BookmarkItem **default_bookmarks_root)
{
	xmlDocPtr doc;
	BookmarkItem *b = NULL;

	if (!(g_file_exists (file))) {
		/* no bookmarks */
		return NULL;
	}

	doc = xmlParseFile (file);	
	
	if (doc) {
		b = xbel_read (doc->root, default_bookmarks_root);
		xmlFreeDoc (doc);
		return b;
	} else {
		g_warning ("unable to parse bookmarks file: %s", file);
		return NULL;
	}
}

static BookmarkItem *
load_from_string (const char *data, BookmarkItem **default_bookmarks_root)
{
	xmlDocPtr doc;
	BookmarkItem *b = NULL;
	
	g_return_val_if_fail (data != NULL, NULL);

	doc = xmlParseMemory ((gchar *) data, strlen (data));	
	
	if (doc) {
		b = xbel_read (doc->root, default_bookmarks_root);
		xmlFreeDoc (doc);
		return b;
	} else {
		g_warning ("unable to parse XBEL bookmarks data: %s", data);
		return NULL;
	}
}

static BookmarkItem * 
xbel_read (xmlNodePtr item, BookmarkItem **default_bookmarks_root) 
{
	BookmarkLoaderInfo *li = bookmarks_io_loader_info_new ();
	BookmarkItem *result;
		
	li->root = xbel_read_node (li, item);
	
	bookmarks_io_resolve_loaded_alias (li);

	if (default_bookmarks_root != NULL) 
	{
		*default_bookmarks_root = li->default_bookmarks_root;
	}
	result = li->root;
	bookmarks_io_loader_info_free (li);
	return result;
}


/**
 * Recursive XBEL node import.
 */
static BookmarkItem *
xbel_read_node (BookmarkLoaderInfo *li, 
		xmlNodePtr node)
{ 
	BookmarkItem *item = NULL;
	xmlNodePtr node2;
        g_return_val_if_fail (node != NULL, NULL);
	
	if (!strcmp (node->name, "folder")
	    || !strcmp (node->name, "xbel")) {
		gchar *title = get_child_value_string (node, "title");
		if (!title) 
		{
			if (!strcmp (node->name, "xbel"))
				title = g_strdup (_("XBEL bookmarks root"));
			else
				title = g_strdup (_("Untitled folder"));
		}
		item = bookmarks_new_bookmark (BM_FOLDER, TRUE, title, NULL,
					       NULL, NULL, NULL);
		g_free (title);
                // Now apply the attributes from the XBEL
                xbel_read_populate_bookmark (li, node, item);
                for (node2 = node->childs; node2 != NULL;
		     node2 = node2->next) {
			BookmarkItem *b = xbel_read_node (li, node2);
			if (b) {
				item->list = g_list_append 
				       (item->list, b);
				b->parent = item;
			}
		}
	} else if (!strcmp (node->name, "bookmark")) {
		// Here, node is a <bookmark> element and item points
                // to the relevant parent folder in the bookmark tree.
                gchar *title = get_child_value_string (node, "title");
                gchar *url = xmlGetRawProp (node, "href");
		
		/* if title == NULL it may be because it was not specified
		   or because it is "". Let's try to check it */
		if (title == NULL) 
		{
			xmlNodePtr tn = get_child (node, "title");
			if (tn != NULL) /* it was present */
				title = g_strdup ("");
		}
		
		item = bookmarks_new_bookmark (BM_SITE, TRUE, title, url,
					       NULL, NULL, NULL);
                g_free (title); 
		xmlFree (url);
		xbel_read_populate_bookmark (li, node, item);
        } else if (!strcmp (node->name, "alias")) {
		gchar *refstr = xmlGetProp (node, "ref");
		guint refval = strtoul (refstr, NULL, 10);
		g_free (refstr);

		item = bookmarks_new_alias (NULL);
		li->unresolved_aliases = g_list_prepend 
			(li->unresolved_aliases, 
			 bookmarks_io_unresolved_alias_info_new (item,
								 refval));
		xbel_read_populate_bookmark (li, node, item);
        } else if (!strcmp (node->name, "separator")) {
		item  = bookmarks_new_bookmark (BM_SEPARATOR, TRUE, NULL, NULL,
						NULL, NULL, NULL);
	} else {
		// Unknown or title, info...
        }
	return item;
}


/***********************************/

/**
 * LibXML Utility Function: Iterate through all children of parent returning a
 * pointer to the first child which is called name.
 */
static xmlNodePtr
get_child (xmlNodePtr parent, const gchar *name)
{
        xmlNodePtr child;
        g_return_val_if_fail (parent != NULL, NULL);
        g_return_val_if_fail (sstrlen (name) > 0, NULL);
        for (child = parent->childs ; child != NULL ; child = child->next)
		if (!strcmp (child->name, name))
                        return child;
        return NULL;
}

/**
 * LibXML Utility Function: Get the text value of the first child of parent 
 * with the name name.
 */
static gchar *
get_child_value_string (xmlNodePtr parent, const gchar *name)
{
        xmlNodePtr child = get_child (parent, name);
        if (child) {
		gchar *v = xmlNodeGetContent (child);
		gchar *ret = NULL;
		if (v) ret = g_strdup (v);
		xmlFree (v);
		return ret;
	} else return NULL;
}

static gboolean
get_child_value_boolean (xmlNodePtr parent, const gchar *name)
{
	gchar *sval = get_child_value_string (parent, name);
	gboolean ret = (sval 
			&& (!g_strcasecmp (sval, "yes")
			    || !g_strcasecmp (sval, "true")))
		? TRUE : FALSE;
	g_free (sval);
	return ret;
}

static glong
get_child_value_long (xmlNodePtr parent, const gchar *name)
{
	gchar *sval = get_child_value_string (parent, name);
        glong ret = sval ? atol (sval) : 0;
	g_free (sval);
	return ret;
}

