/*
 * Copyright 2008 The Native Client Authors. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can
 * be found in the LICENSE file.
 */


// Portable interface for browser interaction - NPAPI implementation

#include "native_client/src/trusted/plugin/npapi/browser_impl_npapi.h"

#include <setjmp.h>
#include <stdio.h>
#include <string.h>

#include <map>

#include "native_client/src/include/checked_cast.h"
#include "native_client/src/include/nacl_elf.h"
#include "native_client/src/include/nacl_string.h"
#include "native_client/src/include/portability_io.h"
#include "native_client/src/shared/npruntime/nacl_npapi.h"
#include "native_client/src/trusted/plugin/npapi/plugin_npapi.h"
#include "native_client/src/trusted/plugin/npapi/scriptable_impl_npapi.h"
#include "native_client/src/trusted/plugin/srpc/utility.h"

using nacl::assert_cast;

namespace {

// Create an NPString from a C string.
bool CopyToNPString(const nacl::string& string, NPString* npstr) {
  // We have to create a copy of the string using NPN_MemAlloc, because the
  // browser sometimes will free the string using NPN_MemFree. Strdup and
  // friends use malloc, which would mix the memory allocators.
  // The copy does not include the terminating NUL, as the NPString UTF8Length
  // member does not include the NUL.
  uint32_t len = nacl::saturate_cast<uint32_t>(string.length());
  char* copy = reinterpret_cast<char*>(NPN_MemAlloc(len));
  if (copy == NULL) {
    PLUGIN_PRINTF(("NPN_MemAlloc failed in CopyToNPString.\n"));
    return false;
  }
  memcpy(copy, string.c_str(), len);
  // Populate the NPString.
  npstr->UTF8Characters = reinterpret_cast<NPUTF8*>(copy);
  npstr->UTF8Length = len;
  return true;
}

void FreeNPString(NPString* npstr) {
  void* str_bytes =
      reinterpret_cast<void*>(const_cast<NPUTF8*>(npstr->UTF8Characters));
  NPN_MemFree(str_bytes);
}

}  // namespace

namespace plugin {

uintptr_t BrowserImplNpapi::StringToIdentifier(const nacl::string& str) {
  return reinterpret_cast<uintptr_t>(NPN_GetStringIdentifier(str.c_str()));
}

nacl::string BrowserImplNpapi::IdentifierToString(uintptr_t ident) {
  NPIdentifier npident = reinterpret_cast<NPIdentifier>(ident);
  // For NPAPI there is no method to test that an identifier is "valid"
  // in the sense that it was created by either NPN_GetStringIdentifier or
  // NPN_GetIntIdentifier.  There is a only test for whether it was created by
  // the former.  Therefore, we may report invalid identifiers as integer
  // identifiers, or on platforms where integer identifiers are checked, we
  // may fail.  (No such platform is known at this point.)
  if (NPN_IdentifierIsString(npident)) {
    return reinterpret_cast<char*>(NPN_UTF8FromIdentifier(npident));
  } else {
    char buf[10];
    SNPRINTF(buf, sizeof buf, "%d", NPN_IntFromIdentifier(npident));
    return buf;
  }
}

bool BrowserImplNpapi::EvalString(InstanceIdentifier instance_id,
                                  const nacl::string& expression) {
  NPP instance = InstanceIdentifierToNPP(instance_id);
  NPVariant dummy_return;

  VOID_TO_NPVARIANT(dummy_return);

  NPString str;
  if (!CopyToNPString(expression, &str)) {
    return false;
  }

  do {
    NPObject* element_obj;
    if (NPERR_NO_ERROR !=
        NPN_GetValue(instance, NPNVPluginElementNPObject, &element_obj)) {
      break;
    }
    if (!NPN_Evaluate(instance, element_obj, &str, &dummy_return)) {
      break;
    }
  } while (0);

  // Free the dummy return from the evaluate.
  NPN_ReleaseVariantValue(&dummy_return);
  // Free the string copy we allocated in CopyToNPString.
  FreeNPString(&str);

  return true;
}

bool BrowserImplNpapi::Alert(InstanceIdentifier instance_id,
                             const nacl::string& text) {
  // Usually these messages are important enough to call attention to them.
  puts(text.c_str());

  NPObject* window;
  NPP npp = InstanceIdentifierToNPP(instance_id);
  if (NPN_GetValue(npp, NPNVWindowNPObject, &window) != NPERR_NO_ERROR) {
    return false;
  }

  NPVariant message;  // doesn't hold its own string data, so don't release
  STRINGN_TO_NPVARIANT(text.c_str(),
                       static_cast<uint32_t>(text.size()),
                       message);

  NPVariant result;
  VOID_TO_NPVARIANT(result);
  bool ok = NPN_Invoke(npp, window, NPN_GetStringIdentifier("alert"),
                       &message, 1, &result);
  if (ok) NPN_ReleaseVariantValue(&result);
  // TODO(adonovan): NPN_ReleaseObject(window) needed?
  return ok;
}

bool BrowserImplNpapi::GetOrigin(InstanceIdentifier instance_id,
                                 nacl::string* origin) {
  NPP npp = InstanceIdentifierToNPP(instance_id);
  NPObject* win_obj = NULL;
  NPVariant loc_value;
  NPVariant href_value;

  *origin = "";

  VOID_TO_NPVARIANT(loc_value);
  VOID_TO_NPVARIANT(href_value);

  // TODO(gregoryd): consider making this block a function returning origin.
  do {
    if (NPERR_NO_ERROR !=
        NPN_GetValue(npp, NPNVWindowNPObject, &win_obj)) {
        PLUGIN_PRINTF(("GetOrigin: No window object\n"));
        // no window; no URL as NaCl descriptors will be allowed
        break;
    }
    if (!NPN_GetProperty(npp,
                         win_obj,
                         PluginNpapi::kLocationIdent,
                         &loc_value)) {
        PLUGIN_PRINTF(("GetOrigin: no location property value\n"));
        break;
    }
    NPObject* loc_obj = NPVARIANT_TO_OBJECT(loc_value);

    if (!NPN_GetProperty(npp,
                         loc_obj,
                         PluginNpapi::kHrefIdent,
                         &href_value)) {
        PLUGIN_PRINTF(("GetOrigin: no href property value\n"));
        break;
    }
    *origin =
        nacl::string(NPVARIANT_TO_STRING(href_value).UTF8Characters,
                     NPVARIANT_TO_STRING(href_value).UTF8Length);
    PLUGIN_PRINTF(("GetOrigin: origin %s\n", origin->c_str()));
  } while (0);

  if (win_obj != NULL) {
    NPN_ReleaseObject(win_obj);
  }
  NPN_ReleaseVariantValue(&loc_value);
  NPN_ReleaseVariantValue(&href_value);

  return ("" != *origin);
}

// Creates a browser scriptable handle for a given portable handle.
ScriptableHandle* BrowserImplNpapi::NewScriptableHandle(
    PortableHandle* handle) {
  return ScriptableImplNpapi::New(handle);
}

NPP InstanceIdentifierToNPP(InstanceIdentifier id) {
  return reinterpret_cast<NPP>(assert_cast<intptr_t>(id));
}

InstanceIdentifier NPPToInstanceIdentifier(NPP npp) {
  return assert_cast<InstanceIdentifier>(reinterpret_cast<intptr_t>(npp));
}

}  // namespace plugin
