/*
 * Multicast receiver thread for OpenGate
 * 
 * Copyright (c) Egoboo Ltd. 1999-2000
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Open Gatekeeper
 *
 * The Initial Developer of the Original Code is Egoboo Ltd.
 *
 * $Log: MulticastThread.cxx,v $
 * Revision 1.9  2000/05/12 14:05:48  aunitt
 * Made sure we don't make unnecessary includes when not needed.
 *
 * Revision 1.8  2000/04/27 17:56:59  aunitt
 * Fixed Win32 multicast support.
 * Added thread name.
 * Changed to use new PWLib socket option command.
 *
 * Revision 1.7  2000/04/20 18:48:19  aunitt
 * Added parameter to signify that we are handling multicast requests.
 *
 * Revision 1.6  2000/04/12 13:31:12  aunitt
 * Changed to use new convenience functions in AddrUtils.
 *
 * Revision 1.5  2000/04/10 19:17:49  aunitt
 * Moved environment to new environment object to tidy things up and to make
 * adding new things easier.
 *
 * Revision 1.4  2000/04/05 15:19:20  aunitt
 * Added RasLog object.
 *
 * Revision 1.3  2000/02/27 20:46:39  aunitt
 * Fixed problems with replying to wrong RAS address
 *
 * Revision 1.2  2000/02/15 20:51:08  aunitt
 * Added support for using system log
 *
 * Revision 1.1.1.1  2000/02/01 22:25:40  aunitt
 * Initial revision
 *
 *
 */

#include <assert.h>
#include <ptlib.h>
#include <ptlib/svcproc.h>
#ifdef WIN32
#include <winsock.h>
#include <q931.h>               // To stop "ostream" ambiguous symbol error
#else
#include <netinet/in.h>
#endif
#include "MulticastThread.h"
#include "AddrUtils.h"
#include "RasServer.h"

static const int BufferSize = 4096;	// Size of network buffers
static const char * MulticastGroup = "224.0.1.41";

MulticastThread::MulticastThread( const Environ &   AkaEnviron,
                                        WORD        MultiPort,
                                        WORD        RASPort
                                )
	: PThread(1000, NoAutoDeleteThread, NormalPriority, "MulticastThread"),
	  MyEnviron(AkaEnviron), MultiSocket(MultiPort)
{
	RasServ = new RasServer( AkaEnviron, RASPort );
	Resume();
}

MulticastThread::~MulticastThread()
{
	delete RasServ;
}

BOOL MulticastThread::ReadRasReq( H225_RasMessage & Mesg )
// Task: to read a RAS message from the given socket
{
	PBYTEArray  Buffer(BufferSize);
	PPER_Stream ReadStream(Buffer);

	if ( MultiSocket.Read( ReadStream.GetPointer(), ReadStream.GetSize() ) )
	{
		return Mesg.Decode( ReadStream );
	}
	return FALSE;
}

BOOL MulticastThread::WriteRasReply( const H225_TransportAddress & ReplyTo,
                                           H225_RasMessage &       Mesg 
                                   )
// Task: to write a RAS message to the given socket
{
	assert( ReplyTo.GetTag() == H225_TransportAddress::e_ipAddress );
	const H225_TransportAddress_ipAddress & ReplyToIP = ReplyTo;
	
	PIPSocket::Address  ReplyToAddr;
	WORD                Port;
	PBYTEArray          Buffer(BufferSize);
	PPER_Stream         WriteStream(Buffer);

	AddrUtils::ConvertToIPAddress( ReplyToIP, ReplyToAddr, Port );
	Mesg.Encode( WriteStream );
	WriteStream.CompleteEncoding();
	MultiSocket.SetSendAddress( ReplyToAddr, Port );
	return MultiSocket.Write( WriteStream.GetPointer(), WriteStream.GetSize() );
}

void MulticastThread::Main()
{
	// Set up the information for the multicast group
	struct ip_mreq mreq;
	mreq.imr_multiaddr.s_addr = inet_addr(MulticastGroup);
#if defined(WIN32)	
    mreq.imr_interface.s_addr = MyEnviron.LocalAddr;
#else
	if ( MyEnviron.LocalAddr != INADDR_ANY )
	{
		mreq.imr_interface.s_addr = MyEnviron.LocalAddr;
	}
	else
	{
		// I believe using INADDR_ANY should work, but it doesn't seem to
		// so explicitly use our local address
		PIPSocket::Address AkaLocal("127.0.0.1");
		mreq.imr_interface.s_addr = AkaLocal;
	}
#endif

	if ( MultiSocket.Listen( MyEnviron.LocalAddr ) )
	{
		H225_RasMessage       Request;
		H225_RasMessage       Reply;
		H225_TransportAddress ReplyTo;

		// Join the multicast group
		if ( !MultiSocket.SetOption( IP_ADD_MEMBERSHIP,
		                             &mreq,
		                             sizeof(mreq),
		                             IPPROTO_IP
		                           )
		   )
		{		
			PSYSTEMLOG( Error, "Can't join multicast group" );
			MultiSocket.Close();
		}

		while ( MultiSocket.IsOpen() )
		{
			if ( ReadRasReq( Request ) )
			{
				if ( RasServ->HandleRasRequest( Request, true, Reply, ReplyTo )
				   )
				{
				    PSYSTEMLOG( Info, "Sending multicast RAS reply:" );
				    PSYSTEMLOG( Info, Reply );
					WriteRasReply( ReplyTo, Reply );
				}
			}
			else
			{
			    PSYSTEMLOG( Info, "Failed to read multicast mesg" );
			}
		}
	}
	else
	{
		// Listen failed
		PSYSTEMLOG( Error, "Multicast listen failed" );
	}
}

void MulticastThread::Close()
{
	// Leave the multicast group
	struct ip_mreq mreq;
	mreq.imr_multiaddr.s_addr = inet_addr(MulticastGroup);
	mreq.imr_interface.s_addr = MyEnviron.LocalAddr;
	MultiSocket.SetOption( IP_DROP_MEMBERSHIP, &mreq, sizeof(mreq), IPPROTO_IP );
	
	MultiSocket.Close();
}
