/*
 * OpenGatekeeper environment definition
 *
 * Copyright (c) Egoboo Ltd. 1999-2000
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Open Gatekeeper
 *
 * The Initial Developer of the Original Code is Egoboo Ltd.
 *
 * $Log: Environ.h,v $
 * Revision 1.9  2000/05/23 10:13:54  aunitt
 * Added support for statically defined gateway prefixes.
 *
 * Revision 1.8  2000/05/12 14:07:30  aunitt
 * Renamed PRasLog to Log.
 * Made sure we only use includes when necessary.
 *
 * Revision 1.7  2000/05/10 09:14:36  aunitt
 * Added setting for routing H.245
 * Removed truncated identifier warning in Windows debug build.
 *
 * Revision 1.6  2000/05/05 11:20:24  aunitt
 * Added bandwidth management.
 *
 * Revision 1.5  2000/04/27 17:59:53  aunitt
 * Changed to use real value for endpoint TTL, no longer debugging.
 *
 * Revision 1.4  2000/04/26 17:09:44  aunitt
 * Added initial support for registration time to live.
 *
 * Revision 1.3  2000/04/21 13:42:43  aunitt
 * Removed bogus warnings.
 * Thanks Microsoft!
 *
 * Revision 1.2  2000/04/20 18:49:42  aunitt
 * Added function to read list of neighbouring Gatekeepers.
 *
 * Revision 1.1  2000/04/10 19:23:11  aunitt
 * Environment object for OpenGatekeeper.
 *
 *
 */

#ifndef _ENVIRON_H
#define _ENVIRON_H

#include <ptlib.h>
#include <ptlib/sockets.h>
#include <ptlib/timeint.h>
#include <h225.h>
#if (_MSC_VER >= 1200)
#include <q931.h>
#include <h245.h>
#pragma warning( disable : 4786 ) // Disable about debug information being truncated
#pragma warning( disable : 4800 ) // remove performance warning about bool...
#pragma warning( push, 2 )
#endif
#include <vector>
#if (_MSC_VER >= 1200)  
#pragma warning( pop )
#endif
#include "EndpointTabl.h"
#include "CallTabl.h"
#include "Log.h"


typedef vector<PIPSocket::Address> AddressList;

class Environ
{
    public:
        Environ()           { Cfg.SetDefaultSection("System"); };
        virtual ~Environ()  {};

        bool IsGatekeeperRouted() const { return Cfg.GetBoolean( "IsGKRouted" ); }
        //Task: to return true iff we should be using gatekeeper routed calls

        bool RouteH245() const { return Cfg.GetBoolean( "Route H245" ); }
        // Task: to return true iff we should also route H.245

        AddressList GetNeighbours() const;
        //Task: to return a list of the IP addresses of other gatekeepers we know about

        void AddStaticPrefixes( H225_GatewayInfo &          Info,
                                const PIPSocket::Address &  Addr
                              ) const;
        // Task: to add any statically defined prefixes for the gateway at the given address...

        PTimeInterval GetEndpointTTL() const
        // Task: to return the time to live for an endpoint
        {
            PTimeInterval TTL( Cfg.GetString( "Endpoint TTL",
                                              "10:00"           // Ten minutes
                                            )
                             );
            return TTL;
        }

        unsigned MaxBandwidth() const
        // Task: to return the maximum bandwidth to be used, 0 for unlimited bandwidth
        { return static_cast<unsigned>(Cfg.GetInteger( "Max Bandwidth", 0 )); }

        unsigned MinBandwidth() const
        // Task: to return the minimum bandwidth allocated per call, work around for
        //       NetMeeting bug(?)
        { return static_cast<unsigned>(Cfg.GetInteger( "Min Call Bandwidth", 10000 )); }

        EndpointTable *             EPTable;
        CallTable *                 CTable;
        H225_GatekeeperIdentifier   MyId;
        H225_TransportAddress       MyCallAddr;
        PIPSocket::Address          LocalAddr;
        OpengateLog *               Log;

    protected:
        H225_ArrayOf_SupportedPrefix GetStaticPrefixes( const PIPSocket::Address & Addr ) const;
        // Task: to return the array of additional prefixes defined for the given gateway
        //       in the list of static prefixes

        PConfig Cfg;
};

#endif // _ENVIRON_H