/*
 * OpenGatekeeper environment definition
 *
 * Copyright (c) Egoboo Ltd. 1999-2000
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Open Gatekeeper
 *
 * The Initial Developer of the Original Code is Egoboo Ltd.
 *
 * $Log: Environ.cxx,v $
 * Revision 1.5  2001/08/26 12:48:11  aunitt
 * Updated to OpenH323 1.6
 *
 * Revision 1.4  2000/05/23 10:15:06  aunitt
 * Added support for statically defined gateway prefixes.
 *
 * Revision 1.3  2000/05/12 14:07:56  aunitt
 * Made sure we only do includes when necessary.
 *
 * Revision 1.2  2000/04/21 13:43:38  aunitt
 * Fixed compilation on Windows.
 *
 * Revision 1.1  2000/04/20 18:54:15  aunitt
 * Environment implementation module.
 *
 *
 */

#include <ptlib.h>
#include <ptlib/svcproc.h>
#if (_MSC_VER >= 1200)
#include <q931.h>               // To stop "ostream" ambiguous symbol error
#include <h245.h>
#endif
#include "Environ.h"


#if (_MSC_VER >= 1200)  
#pragma warning( disable : 4800 ) // remove performance warning about bool...
#endif 

AddressList Environ::GetNeighbours() const
//Task: to return a list of the IP addresses of other gatekeepers we know about
{
    // Read the list of neighbours from the configuration file in the section
    // Neighbours. Format is key name (used only to make readable), host name / addr
    AddressList Result;
    PStringList Keys = Cfg.GetKeys( "Neighbours" );

    for ( PINDEX i=0; i < Keys.GetSize(); ++i )
    {
        PIPSocket::Address Neighbour;
        if ( PIPSocket::GetHostAddress( Cfg.GetString( "Neighbours", Keys[i], "0.0.0.0" ),
                                        Neighbour
                                      )
           )
        {
            Result.push_back(Neighbour);
        }
    }

    return Result;
}

static H225_SupportedPrefix GetPrefix( const PString & PrefixStr )
// Task: to convert the given string into an H225 prefix (assuming voice protocol,
//       E.164 address)
{
    H225_SupportedPrefix Result;

    Result.m_prefix.SetTag( H225_AliasAddress::e_dialedDigits );
    PASN_IA5String & IA5 = (PASN_IA5String &)Result.m_prefix;
    IA5 = PrefixStr;

    return Result;
}

bool operator==( const PIPSocket::Address & a, const PIPSocket::Address & b )
// Equality operator for addresses.
// Probably should move this to PIPSocket::Address
{
    return ( ( a.Byte1() == b.Byte1() ) &&
             ( a.Byte2() == b.Byte2() ) &&
             ( a.Byte3() == b.Byte3() ) &&
             ( a.Byte4() == b.Byte4() )
           );
}

H225_ArrayOf_SupportedPrefix Environ::GetStaticPrefixes( const PIPSocket::Address & Addr ) const
// Task: to return the array of additional prefixes defined for the given gateway
//       in the list of static prefixes
{
    H225_ArrayOf_SupportedPrefix Result;
    PINDEX j=0;
    PStringList Keys = Cfg.GetKeys( "Prefixes" );

    for ( PINDEX i=0; i < Keys.GetSize(); ++i )
    {
        PIPSocket::Address KeyAddr;
        PIPSocket::GetHostAddress( Keys[i], KeyAddr );
        if ( Addr == KeyAddr )
        {
            // This key refers to this address...
            Result.SetSize(j+1);
            Result[j] = GetPrefix( Cfg.GetString( "Prefixes", Keys[i], "Error" ) );
            ++j;
        }
    }

    return Result;
}

void Environ::AddStaticPrefixes( H225_GatewayInfo &         Info,
                                 const PIPSocket::Address & Addr
                               ) const
// Task: to add any statically defined prefixes for the gateway at the given address...
{
    H225_ArrayOf_SupportedPrefix AdditionalPrefixes = GetStaticPrefixes( Addr );

    if ( AdditionalPrefixes.GetSize() > 0 )
    {
        // Add our voice protocol definition to the end of the existing definitions
        // (if any).
        // We may get 2 or more voice protocol definitions that way but that shouldn't
        // really matter...

        if ( !Info.HasOptionalField( H225_GatewayInfo::e_protocol ) )
            Info.IncludeOptionalField( H225_GatewayInfo::e_protocol );
        PINDEX NewPosition = Info.m_protocol.GetSize();
        Info.m_protocol.SetSize(NewPosition + 1);
        Info.m_protocol[NewPosition].SetTag( H225_SupportedProtocols::e_voice );
        H225_VoiceCaps & VoiceCaps = Info.m_protocol[NewPosition];
        VoiceCaps.IncludeOptionalField( H225_VoiceCaps::e_supportedPrefixes );
        VoiceCaps.m_supportedPrefixes = AdditionalPrefixes;
    }
}