// Copyright (c) 2000-2001 Brad Hughes <bhughes@trolltech.com>
//
// Use, modification and distribution is allowed without limitation,
// warranty, or liability of any kind.
//

#include "mainvisual.h"
#include "analyzeroptions.h"
#include "scopeoptions.h"
#include "topographoptions.h"
#include "spectroscopeoptions.h"
#include "colorbutton.h"
#include "configuration.h"
#include "constants.h"
#include "buffer.h"
#include "output.h"

#include <qtimer.h>
#include <qpainter.h>
#include <qevent.h>
#include <qapplication.h>
#include <qcheckbox.h>
#include <qradiobutton.h>
#include <qsettings.h>
#include <qspinbox.h>
#include <qpixmap.h>
#include <qimage.h>

#ifdef FFTW
#include <rfftw.h>
#include <fftw.h>
#endif // FFTW

#include <math.h>

// fast inlines
#include "inlines.h"

// Prefs classes
class ScopePrefs : public ScopeOptions, public Prefs
{
public:
    ScopePrefs( QWidget *parent, const char *name );
    virtual ~ScopePrefs();

    void init( QSettings &settings );
    void write( QSettings &settings );
};

ScopePrefs::ScopePrefs( QWidget *parent, const char *name )
    : ScopeOptions( parent, name )
{
}

ScopePrefs::~ScopePrefs()
{
}

void ScopePrefs::init( QSettings &settings )
{
    show();

    QString val;

    val = settings.readEntry("/MQ3/Scope/startColor", "green" );
    loColor->setColor( QColor( val ) );

    val = settings.readEntry("/MQ3/Scope/targetColor", "red" );
    hiColor->setColor( QColor( val ) );

    bool rband = settings.readBoolEntry( "/MQ3/Scope/enableRubberBand", true );
    rubberBand->setChecked( rband );

    val = settings.readEntry( "/MQ3/Scope/rubberBandSnapSpeed", "Normal" );
    if ( val == "Slow" )
	slowSnap->setChecked( true );
    else if ( val == "Fast" )
	fastSnap->setChecked( true );
    else
	normalSnap->setChecked( true );
}

void ScopePrefs::write( QSettings &settings )
{
    settings.writeEntry( "/MQ3/Scope/startColor", loColor->color().name() );
    settings.writeEntry( "/MQ3/Scope/targetColor", hiColor->color().name() );
    settings.writeEntry( "/MQ3/Scope/enableRubberBand", rubberBand->isChecked() );

    QString snapVal;
    if ( slowSnap->isChecked() )
	snapVal = "Slow";
    else if ( fastSnap->isChecked() )
	snapVal = "Fast";
    else
	snapVal = "Normal";
    settings.writeEntry( "/MQ3/Scope/rubberBandSnapSpeed", snapVal );
}

#ifdef FFTW
class AnalyzerPrefs : public AnalyzerOptions, public Prefs
{
public:
    AnalyzerPrefs( QWidget *parent, const char *name );
    virtual ~AnalyzerPrefs();

    void init( QSettings &settings );
    void write( QSettings &settings );
};

AnalyzerPrefs::AnalyzerPrefs( QWidget *parent, const char *name )
    : AnalyzerOptions( parent, name ), Prefs()
{
}

AnalyzerPrefs::~AnalyzerPrefs()
{
}

void AnalyzerPrefs::init( QSettings &settings )
{
    show();

    QString val;

    val = settings.readEntry("/MQ3/Analyzer/startColor", "green" );
    loColor->setColor( QColor( val ) );

    val = settings.readEntry("/MQ3/Analyzer/targetColor", "red" );
    hiColor->setColor( QColor( val ) );

    int bw = settings.readNumEntry( "/MQ3/Analyzer/barWidth", 4 );
    barWidth->setValue( bw );

    val = settings.readEntry( "/MQ3/Analyzer/fallOffSpeed", "Normal" );
    if ( val == "Slow" )
	slowFall->setChecked( true );
    else if ( val == "Fast" )
	fastFall->setChecked( true );
    else
	normalFall->setChecked( true );
}

void AnalyzerPrefs::write( QSettings &settings )
{
    settings.writeEntry( "/MQ3/Analyzer/startColor", loColor->color().name() );
    settings.writeEntry( "/MQ3/Analyzer/targetColor", hiColor->color().name() );
    settings.writeEntry( "/MQ3/Analyzer/barWidth", barWidth->value() );

    QString fallVal;
    if ( slowFall->isChecked() )
	fallVal = "Slow";
    else if ( fastFall->isChecked() )
	fallVal = "Fast";
    else
	fallVal = "Normal";
    settings.writeEntry( "/MQ3/Analyzer/fallOffSpeed", fallVal );
}

class TopographPrefs : public TopographOptions, public Prefs
{
public:
    TopographPrefs( QWidget *parent, const char *name );
    virtual ~TopographPrefs();

    void init( QSettings &settings );
    void write( QSettings &settings );
};

TopographPrefs::TopographPrefs( QWidget *parent, const char *name )
    : TopographOptions( parent, name )
{
}

TopographPrefs::~TopographPrefs()
{
}

void TopographPrefs::init( QSettings &settings )
{
    show();

    QString val;

    val = settings.readEntry( "/MQ3/Topograph/targetColor", "red" );
    hiColor->setColor( QColor( val ) );

    int bw = settings.readNumEntry( "/MQ3/Topograph/blockWidth", 2 );
    blockWidth->setValue( bw );

    val = settings.readEntry( "/MQ3/Topograph/fadeOutSpeed", "Normal" );
    if ( val == "Slow" )
	slowFade->setChecked( true );
    else if ( val == "Fast" )
	fastFade->setChecked( true );
    else
	normalFade->setChecked( true );
}

void TopographPrefs::write( QSettings &settings )
{
    settings.writeEntry( "/MQ3/Topograph/targetColor", hiColor->color().name() );
    settings.writeEntry( "/MQ3/Topograph/blockWidth", blockWidth->value() );

    QString fadeVal;
    if ( slowFade->isChecked() )
	fadeVal = "Slow";
    else if ( fastFade->isChecked() )
	fadeVal = "Fast";
    else
	fadeVal = "Normal";

    settings.writeEntry( "/MQ3/Topograph/fadeOutSpeed", fadeVal );
}

class SpectroscopePrefs : public SpectroscopeOptions, public Prefs
{
public:
    SpectroscopePrefs( QWidget *parent, const char *name );
    virtual ~SpectroscopePrefs();

    void init( QSettings &settings );
    void write( QSettings &settings );
};

SpectroscopePrefs::SpectroscopePrefs( QWidget *parent, const char *name )
    : SpectroscopeOptions( parent, name )
{
}

SpectroscopePrefs::~SpectroscopePrefs()
{
}

void SpectroscopePrefs::init( QSettings &settings )
{
    show();

    QString val;
    int num;

    val = settings.readEntry( "/MQ3/Spectroscope/leftColor", "red" );
    leftColor->setColor( QColor( val ) );

    val = settings.readEntry( "/MQ3/Spectroscope/rightColor", "blue" );
    rightColor->setColor( QColor( val ) );

    num = settings.readNumEntry( "/MQ3/Spectroscope/edges", 3 );
    edges->setValue( num );

    num = settings.readNumEntry( "/MQ3/Spectroscope/cushion", 3 );
    cushion->setValue( num );

    num = settings.readNumEntry( "/MQ3/Spectroscope/scale" ,3 );
    scale->setValue( num );
}

void SpectroscopePrefs::write( QSettings &settings )
{
    settings.writeEntry( "/MQ3/Spectroscope/leftColor", leftColor->color().name() );
    settings.writeEntry( "/MQ3/Spectroscope/rightColor", rightColor->color().name() );
    settings.writeEntry( "/MQ3/Spectroscope/edges", edges->value() );
    settings.writeEntry( "/MQ3/Spectroscope/cushion", cushion->value() );
    settings.writeEntry( "/MQ3/Spectroscope/scale", scale->value() );
}

#endif


MainVisual::MainVisual( QWidget *parent, const char *name )
    : QWidget( parent, name ), vis( 0 ), playing( FALSE ), fps( 20 )
{
    timer = new QTimer(this);
    connect(timer, SIGNAL(timeout()), this, SLOT(timeout()));
    timer->start(1000 / fps);
}

MainVisual::~MainVisual()
{
    delete vis;
    vis = 0;
}

void MainVisual::setVisual( const QString &visualname )
{
    VisualBase *newvis = 0;

    if ( visualname == "Mono Scope" )
	newvis = new MonoScope;
    else if (visualname == "Stereo Scope" )
	newvis = new StereoScope;
#ifdef FFTW
    else if ( visualname == "Mono Analyzer" )
	newvis = new MonoAnalyzer;
    else if ( visualname == "Stereo Analyzer" )
	newvis =  new StereoAnalyzer;
    else if ( visualname == "Mono Topograph" )
	newvis = new MonoTopograph;
    else if ( visualname == "Stereo Topograph" )
	newvis = new StereoTopograph;
    else if ( visualname == "Stereo Spectroscope" )
	newvis = new StereoSpectroscope;
    else if ( visualname == "Mono Spectroscope" )
	newvis = new MonoSpectroscope;
#endif // FFTW

    setVisual( newvis );
}

void MainVisual::setVisual( VisualBase *newvis )
{
    delete vis;

    vis = newvis;
    if ( vis )
	vis->resize( size() );

    // force an update
    timer->stop();
    timer->start( 1000 / fps );
}

void MainVisual::configChanged( QSettings &settings )
{
    QString newvis = settings.readEntry( "/MQ3/Visual/mode", "Mono Analyzer" );
    setVisual( newvis );

    fps = settings.readNumEntry("/MQ3/Visual/frameRate", 20);
    timer->stop();
    timer->start(1000 / fps);
    if ( vis )
	vis->configChanged( settings );
}

void MainVisual::prepare()
{
    nodes.setAutoDelete(TRUE);
    nodes.clear();
    nodes.setAutoDelete(FALSE);
}

void MainVisual::add(Buffer *b, unsigned long w, int c, int p)
{
    long len = b->nbytes, cnt;
    short *l = 0, *r = 0;

    len /= c;
    len /= (p / 8);
    if (len > 512)
	len = 512;
    cnt = len;

    if (c == 2) {
	l = new short[len];
	r = new short[len];

	if (p == 8)
	    stereo16_from_stereopcm8(l, r, b->data, cnt);
	else if (p == 16)
	    stereo16_from_stereopcm16(l, r, (short *) b->data, cnt);
    } else if (c == 1) {
	l = new short[len];

	if (p == 8)
	    mono16_from_monopcm8(l, b->data, cnt);
	else if (p == 16)
	    mono16_from_monopcm16(l, (short *) b->data, cnt);
    } else
	len = 0;

    nodes.append(new VisualNode(l, r, len, w));
}

void MainVisual::timeout()
{
    VisualNode *node = 0;

    if (playing && output()) {
	output()->mutex()->lock();
	long olat = output()->latency();
	long owrt = output()->written();
	output()->mutex()->unlock();

	long synctime = owrt < olat ? 0 : owrt - olat;

	mutex()->lock();
	VisualNode *prev = 0;
	while ((node = nodes.first())) {
	    if (node->offset > synctime)
		break;

	    delete prev;
	    nodes.removeFirst();
	    prev = node;
	}
	mutex()->unlock();
	node = prev;
    }

    bool stop = TRUE;
    if ( vis )
	stop = vis->process( node );
    delete node;

    if ( vis ) {
	QPainter p(&pixmap);
	vis->draw( &p, backgroundColor() );
    } else
	pixmap.fill( backgroundColor() );
    bitBlt(this, 0, 0, &pixmap);

    if (! playing && stop)
	timer->stop();
}

void MainVisual::paintEvent(QPaintEvent *)
{
    bitBlt(this, 0, 0, &pixmap);
}

void MainVisual::resizeEvent( QResizeEvent *event )
{
    pixmap.resize(event->size());
    pixmap.fill(backgroundColor());
    QWidget::resizeEvent( event );
    if ( vis )
	vis->resize( size() );
}

void MainVisual::customEvent(QCustomEvent *event)
{
    switch (event->type()) {
    case OutputEvent::Playing:
	playing = TRUE;
	// fall through intended

    case OutputEvent::Info:
    case OutputEvent::Buffering:
    case OutputEvent::Paused:
	if (! timer->isActive())
	    timer->start(1000 / fps);

	break;

    case OutputEvent::Stopped:
    case OutputEvent::Error:
	playing = FALSE;
	break;

    default:
	;
    }
}

QStringList MainVisual::visuals()
{
    QStringList list;
    list << "Stereo Scope";
    list << "Mono Scope";
#ifdef FFTW
    list << "Stereo Analyzer";
    list << "Mono Analyzer";
    list << "Stereo Topograph";
    list << "Mono Topograph";
    list << "Stereo Spectroscope";
    list << "Mono Spectroscope";
#endif // FFTW

    return list;
}

Prefs *MainVisual::createPrefs( const QString &visualname,
				QWidget *parent, const char *name )
{
    Prefs *prefs = 0;

    if ( visualname == "Mono Scope" || visualname == "Stereo Scope" )
	prefs = new ScopePrefs( parent, name );
#ifdef FFTW
    else if ( visualname == "Mono Analyzer" || visualname == "Stereo Analyzer" )
	prefs = new AnalyzerPrefs( parent, name );
    else if ( visualname == "Mono Topograph" || visualname == "Stereo Topograph" )
	prefs = new TopographPrefs( parent, name );
    else if ( visualname == "Mono Spectroscope" || visualname == "Stereo Spectroscope" )
	prefs = new SpectroscopePrefs( parent, name );
#endif // FFTW

    return prefs;
}

StereoScope::StereoScope()
    : rubberband( true ), falloff( 1.0 ), fps( 20 )
{
}

StereoScope::~StereoScope()
{
}

void StereoScope::resize( const QSize &newsize )
{
    size = newsize;

    uint os = magnitudes.size();
    magnitudes.resize( size.width() * 2 );
    for ( ; os < magnitudes.size(); os++ )
	magnitudes[os] = 0.0;
}

void StereoScope::configChanged( QSettings &settings )
{
    QString val;

    // need the framerate for the fall off speed
    fps = settings.readNumEntry("/MQ3/Visual/frameRate", 20);

    val = settings.readEntry("/MQ3/Scope/startColor");
    if (! val.isEmpty())
        startColor = QColor(val);
    else
	startColor = Qt::green;

    val = settings.readEntry("/MQ3/Scope/targetColor");
    if (! val.isEmpty())
	targetColor = QColor(val);
    else
	targetColor = Qt::red;

    rubberband = settings.readBoolEntry( "/MQ3/Scope/enableRubberBand", true );

    val = settings.readEntry( "/MQ3/Scope/fallOffSpeed", "Normal" );
    if ( val == "Slow" )
	falloff = .125;
    else if ( val == "Fast" )
	falloff = .5;
    else
	falloff = .25;

    falloff *= ( 80. / double( fps ) );

    resize( size );
}

bool StereoScope::process( VisualNode *node )
{
    bool allZero = TRUE;
    int i;
    long s, index, indexTo, step = 512 / size.width();
    double *magnitudesp = magnitudes.data();
    double valL, valR, tmpL, tmpR;

    if (node) {
	index = 0;
	for ( i = 0; i < size.width(); i++) {
	    indexTo = index + step;

	    if ( rubberband ) {
		valL = magnitudesp[ i ];
		valR = magnitudesp[ i + size.width() ];
		if (valL < 0.) {
		    valL += falloff;
		    if ( valL > 0. )
			valL = 0.;
		} else {
		    valL -= falloff;
		    if ( valL < 0. )
			valL = 0.;
		}
		if (valR < 0.) {
		    valR += falloff;
		    if ( valR > 0. )
			valR = 0.;
		} else {
		    valR -= falloff;
		    if ( valR < 0. )
			valR = 0.;
		}
	    } else
		valL = valR = 0.;

	    for (s = index; s < indexTo && s < node->length; s++) {
		tmpL = ( ( node->left ?
			   double( node->left[s] ) : 0.) *
			 double( size.height() / 4 ) ) / 32768.;
		tmpR = ( ( node->right ?
			   double( node->right[s]) : 0.) *
			 double( size.height() / 4 ) ) / 32768.;
		if (tmpL > 0)
		    valL = (tmpL > valL) ? tmpL : valL;
		else
		    valL = (tmpL < valL) ? tmpL : valL;
		if (tmpR > 0)
		    valR = (tmpR > valR) ? tmpR : valR;
		else
		    valR = (tmpR < valR) ? tmpR : valR;
	    }

	    if (valL != 0. || valR != 0.)
		allZero = FALSE;

	    magnitudesp[ i ] = valL;
	    magnitudesp[ i + size.width() ] = valR;

	    index = indexTo;
	}
    } else if (rubberband) {
	for ( i = 0; i < size.width(); i++) {
	    valL = magnitudesp[ i ];
	    if (valL < 0) {
		valL += 2;
		if (valL > 0.)
		    valL = 0.;
	    } else {
		valL -= 2;
		if (valL < 0.)
		    valL = 0.;
	    }

	    valR = magnitudesp[ i + size.width() ];
	    if (valR < 0.) {
		valR += falloff;
		if (valR > 0.)
		    valR = 0.;
	    } else {
		valR -= falloff;
		if (valR < 0.)
		    valR = 0.;
	    }

	    if (valL != 0. || valR != 0.)
		allZero = FALSE;

	    magnitudesp[ i ] = valL;
	    magnitudesp[ i + size.width() ] = valR;
	}
    } else {
	for ( i = 0; (unsigned) i < magnitudes.size(); i++ )
	    magnitudesp[ i ] = 0.;
    }

    return allZero;
}

void StereoScope::draw( QPainter *p, const QColor &back )
{
    double *magnitudesp = magnitudes.data();
    double r, g, b, per;

    p->fillRect(0, 0, size.width(), size.height(), back);
    for ( int i = 1; i < size.width(); i++ ) {
	// left
	per = double( magnitudesp[ i ] * 2 ) /
	      double( size.height() / 4 );
	if (per < 0.0)
	    per = -per;
	if (per > 1.0)
	    per = 1.0;
	else if (per < 0.0)
	    per = 0.0;

	r = startColor.red() + (targetColor.red() -
				startColor.red()) * (per * per);
	g = startColor.green() + (targetColor.green() -
				  startColor.green()) * (per * per);
	b = startColor.blue() + (targetColor.blue() -
				 startColor.blue()) * (per * per);

	if (r > 255.0)
	    r = 255.0;
	else if (r < 0.0)
	    r = 0;

	if (g > 255.0)
	    g = 255.0;
	else if (g < 0.0)
	    g = 0;

	if (b > 255.0)
	    b = 255.0;
	else if (b < 0.0)
	    b = 0;

	p->setPen( QColor( int(r), int(g), int(b) ) );
	p->drawLine( i - 1, ( size.height() / 4 ) + magnitudesp[ i - 1 ],
		     i, ( size.height() / 4 ) + magnitudesp[ i ] );

	// right
	per = double( magnitudesp[ i + size.width() ] * 2 ) /
	      double( size.height() / 4 );
	if (per < 0.0)
	    per = -per;
	if (per > 1.0)
	    per = 1.0;
	else if (per < 0.0)
	    per = 0.0;

	r = startColor.red() + (targetColor.red() -
				startColor.red()) * (per * per);
	g = startColor.green() + (targetColor.green() -
				  startColor.green()) * (per * per);
	b = startColor.blue() + (targetColor.blue() -
				 startColor.blue()) * (per * per);

	if (r > 255.0)
	    r = 255.0;
	else if (r < 0.0)
	    r = 0;

	if (g > 255.0)
	    g = 255.0;
	else if (g < 0.0)
	    g = 0;

	if (b > 255.0)
	    b = 255.0;
	else if (b < 0.0)
	    b = 0;

	p->setPen( QColor( int(r), int(g), int(b) ) );
	p->drawLine( i - 1, ( size.height() * 3 / 4 ) +
		     magnitudesp[ i + size.width() - 1 ],
		     i, ( size.height() * 3 / 4 ) + magnitudesp[ i + size.width() ] );
    }
}

MonoScope::MonoScope()
{
}

MonoScope::~MonoScope()
{
}

bool MonoScope::process( VisualNode *node )
{
    bool allZero = TRUE;
    int i;
    long s, index, indexTo, step = 512 / size.width();
    double *magnitudesp = magnitudes.data();
    double val, tmp;

    if (node) {
	index = 0;
	for ( i = 0; i < size.width(); i++) {
	    indexTo = index + step;

	    if ( rubberband ) {
		val = magnitudesp[ i ];
		if (val < 0.) {
		    val += falloff;
		    if ( val > 0. )
			val = 0.;
		} else {
		    val -= falloff;
		    if ( val < 0. )
			val = 0.;
		}
	    } else
		val = 0.;

	    for (s = index; s < indexTo && s < node->length; s++) {
		tmp = ( double( node->left[s] ) +
			(node->right ? double( node->right[s] ) : 0) *
			double( size.height() / 2 ) ) / 65536.;
		if (tmp > 0)
		    val = (tmp > val) ? tmp : val;
		else
		    val = (tmp < val) ? tmp : val;
	    }

	    if ( val != 0. )
		allZero = FALSE;
	    magnitudesp[ i ] = val;
	    index = indexTo;
	}
    } else if (rubberband) {
	for ( i = 0; i < size.width(); i++) {
	    val = magnitudesp[ i ];
	    if (val < 0) {
		val += 2;
		if (val > 0.)
		    val = 0.;
	    } else {
		val -= 2;
		if (val < 0.)
		    val = 0.;
	    }

	    if ( val != 0. )
		allZero = FALSE;
	    magnitudesp[ i ] = val;
	}
    } else {
	for ( i = 0; i < size.width(); i++ )
	    magnitudesp[ i ] = 0.;
    }

    return allZero;
}

void MonoScope::draw( QPainter *p, const QColor &back )
{
    double *magnitudesp = magnitudes.data();
    double r, g, b, per;

    p->fillRect( 0, 0, size.width(), size.height(), back );
    for ( int i = 1; i < size.width(); i++ ) {
	per = double( magnitudesp[ i ] ) /
	      double( size.height() / 4 );
	if (per < 0.0)
	    per = -per;
	if (per > 1.0)
	    per = 1.0;
	else if (per < 0.0)
	    per = 0.0;

	r = startColor.red() + (targetColor.red() -
				startColor.red()) * (per * per);
	g = startColor.green() + (targetColor.green() -
				  startColor.green()) * (per * per);
	b = startColor.blue() + (targetColor.blue() -
				 startColor.blue()) * (per * per);

	if (r > 255.0)
	    r = 255.0;
	else if (r < 0.0)
	    r = 0;

	if (g > 255.0)
	    g = 255.0;
	else if (g < 0.0)
	    g = 0;

	if (b > 255.0)
	    b = 255.0;
	else if (b < 0.0)
	    b = 0;

	p->setPen(QColor(int(r), int(g), int(b)));
	p->drawLine( i - 1, size.height() / 2 + magnitudesp[ i - 1 ],
		     i, size.height() / 2 + magnitudesp[ i ] );
    }
}


#ifdef FFTW

StereoAnalyzer::StereoAnalyzer()
    : scaleFactor( 1.0 ), falloff( 1.0 ), analyzerBarWidth( 4 ), fps( 20 )
{
    plan =  rfftw_create_plan(512, FFTW_REAL_TO_COMPLEX, FFTW_ESTIMATE);
}

StereoAnalyzer::~StereoAnalyzer()
{
    rfftw_destroy_plan(plan);
}

void StereoAnalyzer::resize( const QSize &newsize )
{
    size = newsize;

    scale.setMax(192, size.width() / analyzerBarWidth);

    rects.resize( scale.range() );
    int i = 0, w = 0;
    for (; (unsigned) i < rects.count(); i++, w += analyzerBarWidth)
	rects[i].setRect(w, size.height() / 2, analyzerBarWidth - 1, 1);

    int os = magnitudes.size();
    magnitudes.resize( scale.range() * 2 );
    for (; (unsigned) os < magnitudes.size(); os++)
	magnitudes[os] = 0.0;

    scaleFactor = double( size.height() / 2 ) / log( 512.0 );
}

void StereoAnalyzer::configChanged( QSettings &settings )
{
    QString val;

    // need the framerate for the fall off speed
    fps = settings.readNumEntry("/MQ3/Visual/frameRate", 20);

    val = settings.readEntry("/MQ3/Analyzer/startColor");
    if (! val.isEmpty())
        startColor = QColor(val);
    else
	startColor = Qt::green;

    val = settings.readEntry("/MQ3/Analyzer/targetColor");
    if (! val.isEmpty())
	targetColor = QColor(val);
    else
	targetColor = Qt::red;

    analyzerBarWidth = settings.readNumEntry( "/MQ3/Analyzer/barWidth", 4 );

    val = settings.readEntry( "/MQ3/Analyzer/fallOffSpeed", "Normal" );
    if ( val == "Slow" )
	falloff = .25;
    else if ( val == "Fast" )
	falloff = 1.;
    else
	falloff = .5;

    falloff *= ( 80. / double( fps ) );

    resize( size );
}

bool StereoAnalyzer::process( VisualNode *node )
{
    bool allZero = TRUE;
    uint i;
    long w = 0, index;
    QRect *rectsp = rects.data();
    double *magnitudesp = magnitudes.data();
    double magL, magR, tmp;

    if (node) {
	i = node->length;
	fast_real_set_from_short(lin, node->left, node->length);
	if (node->right)
	    fast_real_set_from_short(rin, node->right, node->length);
    } else
	i = 0;

    fast_reals_set(lin + i, rin + i, 0, 512 - i);

    rfftw_one(plan, lin, lout);
    rfftw_one(plan, rin, rout);

    index = 1;
    for (i = 0; i < rects.count(); i++, w += analyzerBarWidth) {
	magL = (log(lout[index] * lout[index] +
		    lout[512 - index] * lout[512 - index]) - 22.0) *
	       scaleFactor;
	magR = (log(rout[index] * rout[index] +
		    rout[512 - index] * rout[512 - index]) - 22.0) *
	       scaleFactor;

	if (magL > size.height() / 2)
	    magL = size.height() / 2;
	if (magL < magnitudesp[i]) {
	    tmp = magnitudesp[i] - falloff;
	    if ( tmp < magL )
		tmp = magL;
	    magL = tmp;
	}
	if (magL < 1.)
	    magL = 1.;

	if (magR > size.height() / 2)
	    magR = size.height() / 2;
	if (magR < magnitudesp[i + scale.range()]) {
	    tmp = magnitudesp[i + scale.range()] - falloff;
	    if ( tmp < magR )
		tmp = magR;
	    magR = tmp;
	}
	if (magR < 1.)
	    magR = 1.;

	if (magR != 1 || magL != 1)
	    allZero = FALSE;

	magnitudesp[i] = magL;
	magnitudesp[i + scale.range()] = magR;

	rectsp[i].setTop( size.height() / 2 - int( magL ) );
	rectsp[i].setBottom( size.height() / 2 + int( magR ) );

	index = scale[i];
    }

    return allZero;
}

void StereoAnalyzer::draw( QPainter *p, const QColor &back )
{
    QRect *rectsp = rects.data();
    double r, g, b, per;

    p->fillRect(0, 0, size.width(), size.height(), back);
    for (uint i = 0; i < rects.count(); i++) {
	per = double( rectsp[i].height() - 2 ) / double( size.height() );
	if (per > 1.0)
	    per = 1.0;
	else if (per < 0.0)
	    per = 0.0;

	r = startColor.red() + (targetColor.red() -
			    startColor.red()) * (per * per);
	g = startColor.green() + (targetColor.green() -
			      startColor.green()) * (per * per);
	b = startColor.blue() + (targetColor.blue() -
			     startColor.blue()) * (per * per);

	if (r > 255.0)
	    r = 255.0;
	else if (r < 0.0)
	    r = 0;

	if (g > 255.0)
	    g = 255.0;
	else if (g < 0.0)
	    g = 0;

	if (b > 255.0)
	    b = 255.0;
	else if (b < 0.0)
	    b = 0;

	p->fillRect(rectsp[i], QColor(int(r), int(g), int(b)));
    }
}


MonoAnalyzer::MonoAnalyzer()
{
}

MonoAnalyzer::~MonoAnalyzer()
{
}

bool MonoAnalyzer::process( VisualNode *node )
{
    bool allZero = TRUE;
    uint i;
    long w = 0, index;
    QRect *rectsp = rects.data();
    double *magnitudesp = magnitudes.data();
    double mag, tmp;

    if (node) {
	i = node->length;
	if (! node->right)
	    fast_real_set_from_short(lin, node->left,
				     node->length);
	else
	    fast_real_avg_from_shorts(lin, node->left, node->right,
				      node->length);
    } else
	i = 0;

    fast_real_set(lin + i, 0, 512 - i);

    rfftw_one(plan, lin, lout);

    index = 1;
    for (i = 0; i < rects.count(); i++, w += analyzerBarWidth) {
	mag = (log(lout[index] * lout[index] +
		   lout[512 - index] * lout[512 - index]) - 22.0) *
	      scaleFactor;

	if (mag > size.height() / 2)
	    mag = size.height() / 2;
	if (mag < magnitudesp[i]) {
	    tmp = magnitudesp[i] - falloff;
	    if ( tmp < mag )
		tmp = mag;
	    mag = tmp;
	}
	if (mag < 1)
	    mag = 1;

	if (mag != 1)
	    allZero = FALSE;

	magnitudesp[i] = mag;

	rectsp[i].setTop(size.height() * 3 / 4 - int( mag ) );
	rectsp[i].setBottom(size.height() * 3 / 4);

	index = scale[i];
    }

    return allZero;
}

void MonoAnalyzer::draw( QPainter *p, const QColor &back )
{
    QRect *rectsp = rects.data();
    double r, g, b, per;

    p->fillRect(0, 0, size.width(), size.height(), back);
    for (uint i = 0; i < rects.count(); i++) {
	per = double( rectsp[i].height() - 2 ) / double( size.height() / 2 );
	if (per > 1.0)
	    per = 1.0;
	else if (per < 0.0)
	    per = 0.0;

	r = startColor.red() + (targetColor.red() -
			    startColor.red()) * (per * per);
	g = startColor.green() + (targetColor.green() -
			      startColor.green()) * (per * per);
	b = startColor.blue() + (targetColor.blue() -
			     startColor.blue()) * (per * per);

	if (r > 255.0)
	    r = 255.0;
	else if (r < 0.0)
	    r = 0;

	if (g > 255.0)
	    g = 255.0;
	else if (g < 0.0)
	    g = 0;

	if (b > 255.0)
	    b = 255.0;
	else if (b < 0.0)
	    b = 0;

	p->fillRect(rectsp[i], QColor(int(r), int(g), int(b)));
    }
}

StereoTopograph::StereoTopograph()
    : clear( false ), scaleFactor( 1.0 ), falloff( 1.0 ), blockwidth( 2 ), fps( 20 )
{
    plan = rfftw_create_plan(512, FFTW_REAL_TO_COMPLEX, FFTW_ESTIMATE);
}

StereoTopograph::~StereoTopograph()
{
    rfftw_destroy_plan( plan );
}

void StereoTopograph::resize( const QSize &newsize )
{
    size = newsize;

    scale.setMax( 192, size.height() / blockwidth );

    uint os = magnitudes.size();
    magnitudes.resize( scale.range() * 2 );
    for ( ; os < magnitudes.size(); os++ )
	magnitudes[os] = 0.0;

    scaleFactor = 32. / log( 512. ) ;

    clear = true;
}

void StereoTopograph::configChanged( QSettings &settings )
{
    QString val;

    // need the framerate for the fall off speed
    fps = settings.readNumEntry("/MQ3/Visual/frameRate", 20);

    val = settings.readEntry("/MQ3/Topograph/targetColor", "green" );
    if (! val.isEmpty())
	targetColor = QColor(val);
    else
	targetColor = Qt::red;

    blockwidth = settings.readNumEntry( "/MQ3/Topograph/blockWidth", 2 );

    val = settings.readEntry( "/MQ3/Topograph/fallOffSpeed", "Normal" );
    if ( val == "Slow" )
	falloff = .25;
    else if ( val == "Fast" )
	falloff = 1.;
    else
	falloff = .5;

    falloff *= ( 80. / double( fps ) );

    resize( size );
}

bool StereoTopograph::process( VisualNode *node )
{
    bool allZero = TRUE;
    long index;
    int i;
    double *magnitudesp = magnitudes.data();
    double magL, magR;

    if (node) {
	i = node->length;
	fast_real_set_from_short(lin, node->left, node->length);
	if (node->right)
	    fast_real_set_from_short(rin, node->right, node->length);
    } else
	i = 0;

    fast_reals_set(lin + i, rin + i, 0, 512 - i);

    rfftw_one(plan, lin, lout);
    rfftw_one(plan, rin, rout);

    index = 1;
    for (i = 0; i < scale.range(); i++) {
	magL = (log(lout[index] * lout[index] +
		    lout[512 - index] * lout[512 - index]) - 22.0) *
	       scaleFactor;
	magR = (log(rout[index] * rout[index] +
		    rout[512 - index] * rout[512 - index]) - 22.0) *
	       scaleFactor;

	if (magL > 32.)
	    magL = 32.;
	if (magL < magnitudesp[i])
	    magL = magnitudesp[i] - falloff;
	if (magL < 0.)
	    magL = 0.;

	if (magR > 32.)
	    magR = 32.;
	if ( magR < magnitudesp[ i + scale.range() ] )
	    magR = magnitudesp[ i + scale.range() ] - falloff;
	if (magR < 0.)
	    magR = 0.;

	if (magL != 0. || magR != 0.)
	    allZero = FALSE;

	magnitudesp[ i ] = magL;
	magnitudesp[ i + scale.range() ] = magR;

	index = scale[i];
    }

    return allZero;
}

void StereoTopograph::draw( QPainter *p, const QColor &back )
{
    double *magnitudesp = magnitudes.data();
    double r, g, b, per;
    int y = ( ( size.height() + ( scale.range() * blockwidth ) ) / 2 ) - blockwidth;

    if ( ! clear ) {
	// scroll one block right for the left channel
	bitBlt(p->device(), 0, 0,
	       p->device(), blockwidth, 0,
	       ( size.width() / 2 ) - blockwidth, size.height());
	// scroll one block left for the right channel
	bitBlt(p->device(), ( size.width() / 2 ) + blockwidth, 0,
	       p->device(), size.width() / 2, 0, size.width() / 2, size.height());
    } else {
	p->fillRect( 0, 0, size.width(), size.height(), back );
	clear = false;
    }

    for ( int i = 1; i < scale.range(); i++ ) {
	// left
	per = double( magnitudesp[ i ] ) / 32. ;
	if (per < 0.0)
	    per = -per;
	if (per > 1.0)
	    per = 1.0;
	else if (per < 0.0)
	    per = 0.0;

	r = back.red() + (targetColor.red() - back.red()) * (per * per);
	g = back.green() + (targetColor.green() - back.green()) * (per * per);
	b = back.blue() + (targetColor.blue() - back.blue()) * (per * per);

	if (r > 255.0)
	    r = 255.0;
	else if (r < 0.0)
	    r = 0;

	if (g > 255.0)
	    g = 255.0;
	else if (g < 0.0)
	    g = 0;

	if (b > 255.0)
	    b = 255.0;
	else if (b < 0.0)
	    b = 0;

	p->setPen(QColor(int(r), int(g), int(b)));
	p->drawRect( ( size.width() / 2) - blockwidth, y, blockwidth, blockwidth );

	// right
	per = double( magnitudesp[ i + scale.range() ] ) / 32.;
	if (per < 0.0)
	    per = -per;
	if (per > 1.0)
	    per = 1.0;
	else if (per < 0.0)
	    per = 0.0;

	r = back.red() + (targetColor.red() - back.red()) * (per * per);
	g = back.green() + (targetColor.green() - back.green()) * (per * per);
	b = back.blue() + (targetColor.blue() - back.blue()) * (per * per);

	if (r > 255.0)
	    r = 255.0;
	else if (r < 0.0)
	    r = 0;

	if (g > 255.0)
	    g = 255.0;
	else if (g < 0.0)
	    g = 0;

	if (b > 255.0)
	    b = 255.0;
	else if (b < 0.0)
	    b = 0;

	p->setPen(QColor(int(r), int(g), int(b)));
	p->drawRect(  size.width() / 2, y, blockwidth, blockwidth );

	y -= blockwidth;
    }
}

MonoTopograph::MonoTopograph()
{
}

MonoTopograph::~MonoTopograph()
{
}

bool MonoTopograph::process( VisualNode *node )
{
    bool allZero = TRUE;
    long index;
    int i;
    double *magnitudesp = magnitudes.data();
    double mag;

    if (node) {
	i = node->length;
	if (! node->right)
	    fast_real_set_from_short(lin, node->left, node->length);
	else
	    fast_real_avg_from_shorts(lin, node->left, node->right, node->length);
    } else
	i = 0;

    fast_reals_set(lin + i, rin + i, 0, 512 - i);

    rfftw_one(plan, lin, lout);
    rfftw_one(plan, rin, rout);

    index = 1;
    for (i = 0; i < scale.range(); i++) {
	mag = (log(lout[index] * lout[index] +
		   lout[512 - index] * lout[512 - index]) - 22.0) *
	      scaleFactor;

	if (mag > 32.)
	    mag = 32.;
	if (mag < magnitudesp[i])
	    mag = magnitudesp[i] - falloff;
	if (mag < 0.)
	    mag = 0.;

	if ( mag != 0. )
	    allZero = FALSE;

	magnitudesp[ i ] = mag;

	index = scale[i];
    }

    return allZero;
}

void MonoTopograph::draw( QPainter *p, const QColor &back )
{
    double *magnitudesp = magnitudes.data();
    double r, g, b, per;
    int y = ( ( size.height() + ( scale.range() * blockwidth ) ) / 2 ) - blockwidth;

    if ( ! clear ) {
	// scroll one block right
	bitBlt(p->device(), 0, 0,
	       p->device(), blockwidth, 0, size.width() - blockwidth, size.height() );
    } else {
	p->fillRect( 0, 0, size.width(), size.height(), back );
	clear = false;
    }

    for ( int i = 1; i < scale.range(); i++ ) {
	per = double( magnitudesp[ i ] ) / 32. ;
	if (per < 0.0)
	    per = -per;
	if (per > 1.0)
	    per = 1.0;
	else if (per < 0.0)
	    per = 0.0;

	r = back.red() + (targetColor.red() - back.red()) * (per * per);
	g = back.green() + (targetColor.green() - back.green()) * (per * per);
	b = back.blue() + (targetColor.blue() - back.blue()) * (per * per);

	if (r > 255.0)
	    r = 255.0;
	else if (r < 0.0)
	    r = 0;

	if (g > 255.0)
	    g = 255.0;
	else if (g < 0.0)
	    g = 0;

	if (b > 255.0)
	    b = 255.0;
	else if (b < 0.0)
	    b = 0;

	p->setPen(QColor(int(r), int(g), int(b)));
	p->drawRect( size.width() - blockwidth, y, blockwidth, blockwidth );

	y -= blockwidth;
    }
}

StereoSpectroscope::StereoSpectroscope()
    : edges( 3.0 ), cushion( 3.0 ), scale( 3.0 )
{
    plan = fftw_create_plan( 512, FFTW_FORWARD, FFTW_ESTIMATE );
}

StereoSpectroscope::~StereoSpectroscope()
{
    fftw_destroy_plan( plan );
}

void StereoSpectroscope::resize( const QSize &newsize )
{
    size = newsize;

}

void StereoSpectroscope::configChanged( QSettings &settings )
{
    QString val;
    int num;

    val = settings.readEntry( "/MQ3/Spectroscope/leftColor", "red" );
    leftColor = QColor( val );

    val = settings.readEntry( "/MQ3/Spectroscope/rightColor", "blue" );
    rightColor = QColor( val );

    num = settings.readNumEntry( "/MQ3/Spectroscope/edges", 3 );
    edges = double( num );

    num = settings.readNumEntry( "/MQ3/Spectroscope/cushion", 3 );
    cushion = double( num );

    num = settings.readNumEntry( "/MQ3/Spectroscope/scale" ,3 );
    scale = double( num );
}

bool StereoSpectroscope::process( VisualNode *node )
{
    int i;
    if (node) {
	fast_complex_set_from_short(lin, node->left, node->length);
	if (node->right)
	    fast_complex_set_from_short(rin, node->right, node->length);

	i = node->length;
    } else
	i = 0;

    fast_complex_set(lin + i, rin + i, fftw_complex_from_real( 0 ), 512 - i);

    fftw_one( plan, lin, lout );
    fftw_one( plan, rin, rout );

    for( i = 0; i < 512; i++ ) {
	// magnitudes
	lmag[ i ] = sqrt( lout[ i ].re * lout[ i ].re +
			  lout[ i ].im * lout[ i ].im );
	rmag[ i ] = sqrt( rout[ i ].re * rout[ i ].re +
			  rout[ i ].im * rout[ i ].im );

	// angles
	lang[ i ] = atan( lout[ i ].im / lout[ i ].re );
	rang[ i ] = atan( rout[ i ].im / rout[ i ].re );

    }

    return ( node == 0 );
}

void StereoSpectroscope::draw( QPainter *p, const QColor &back )
{
    fftw_real x,y;
    int i;
    p->fillRect( 0, 0, size.width(), size.height(), back );

    // left
    p->setPen( leftColor );
    for( i = 0; i < 512; i++ ) {
	x = ( size.width() / 3 ) - cos( lang[ i ] * 4 ) *
	    ( log( lmag[ i ] ) - fabs( sin( ( edges / 2 ) *
					    ( lang[ i ] * 4 + angle ) ) *
				       cushion ) ) * scale;
	y = ( size.height() / 2 ) - sin( lang[ i ] * 4 ) *
	    ( log( lmag[ i ] ) - fabs( sin( ( edges / 2 ) *
					    ( lang[ i ] * 4 + angle ) ) *
				       cushion ) ) * scale;
	p->drawPoint( x, y );
    }

    // right
    p->setPen( rightColor );
    for( i = 0; i < 512; i++ ) {
	x = ( size.width() * 2 / 3 ) + cos( rang[ i ] * 4 ) *
	    ( log( rmag[ i ] ) - fabs( sin( ( edges / 2 ) *
					    rang[ i ] * 4 + angle ) *
				       cushion ) ) * scale;
	y = ( size.height() / 2 ) - sin( rang[ i ] * 4 ) *
	    ( log( rmag[ i ] ) - fabs( sin( ( edges / 2 ) *
					    rang[ i ] * 4 + angle ) *
				       cushion ) ) * scale;
	p->drawPoint( x, y );
    }
}

MonoSpectroscope::MonoSpectroscope()
{
}

MonoSpectroscope::~MonoSpectroscope()
{
}

bool MonoSpectroscope::process( VisualNode *node )
{
    int i;
    if (node) {
	i = node->length;
	if (! node->right)
	    fast_complex_set_from_short(lin, node->left,
					node->length);
	else
	    fast_complex_avg_from_shorts(lin, node->left, node->right,
					 node->length);
    } else
	i = 0;

    fast_complex_set(lin + i, fftw_complex_from_real( 0 ), 512 - i);

    fftw_one( plan, lin, lout );

    for( i = 0; i < 512; i++ ) {
	lmag[ i ] = sqrt( lout[ i ].re * lout[ i ].re +
			  lout[ i ].im * lout[ i ].im );
	lang[ i ] = atan( lout[ i ].im / lout[ i ].re );
    }

    return ( node == 0 );
}

void MonoSpectroscope::draw( QPainter *p, const QColor &back )
{
    fftw_real x,y;
    int i;
    p->fillRect( 0, 0, size.width(), size.height(), back );

    // left
    p->setPen( leftColor );
    for( i = 0; i < 512; i++ ) {
	x = ( size.width() / 2 ) - cos( lang[ i ] * 4 ) *
	    ( log( lmag[ i ] ) - fabs( sin( ( edges / 2 ) *
					    ( lang[ i ] * 4 + angle ) ) *
				       cushion ) ) * scale;
	y = ( size.height() / 2 ) - sin( lang[ i ] * 4 ) *
	    ( log( lmag[ i ] ) - fabs( sin( ( edges / 2 ) *
					    ( lang[ i ] * 4 + angle ) ) *
				       cushion ) ) * scale;
	p->drawPoint( x, y );
    }
}

#endif // FFTW
