// Copyright (c) 2000-2001 Brad Hughes <bhughes@trolltech.com>
//
// Use, modification and distribution is allowed without limitation,
// warranty, or liability of any kind.
//

#include "icons.h"
#include "constants.h"

#include <qpixmap.h>
#include <qimage.h>
#include <qdir.h>
#include <qstringlist.h>

#include <stdio.h>


static QString *theme = 0;
static QStringList* pathlist = 0;

static void initpathlist()
{
    if (! theme)
	theme = new QString("default");

    if (! pathlist) {
	pathlist = new QStringList;
	pathlist->append(INSTALLDIR + QString("/share/mq3/icons"));
	pathlist->append(QDir::homeDirPath() + "/.mq3/icons");
	pathlist->append( INSTALLDIR + QString( "/icons" ) );
    }
}


void IconLoader::addPath(const QString &path)
{
    initpathlist();
    pathlist->append(path);
}


void IconLoader::addPaths(const QStringList &paths)
{
    initpathlist();
    (*pathlist) += paths;
}


void IconLoader::removePath(const QString &path)
{
    initpathlist();

    // do not remove the default paths
    if (path == INSTALLDIR + QString("/share/mq3/icons") ||
	path == QDir::homeDirPath() + "/.mq3/icons")
	return;

    pathlist->remove(path);
}


void IconLoader::clearPaths()
{
    if (pathlist)
	delete pathlist;
    pathlist = 0;
    initpathlist();
}


QStringList IconLoader::paths(bool defaults)
{
    initpathlist();

    QStringList l = *pathlist;

    if (! defaults) {
	l.remove(INSTALLDIR + QString("/share/mq3/icons"));
	l.remove(QDir::homeDirPath() + "/.mq3/icons");
    }

    return l;
}


void IconLoader::setTheme(const QString &t)
{
    initpathlist();
    *::theme = t;
}


const QString &IconLoader::theme()
{
    initpathlist();
    return *::theme;
}


QStringList IconLoader::themes()
{
    QString path;
    QStringList tlist;

    initpathlist();
    QStringList::Iterator it = pathlist->begin();
    while (it != pathlist->end()) {
	path = *it++;

	QDir dir(path);
	if (dir.exists()) {
	    QStringList l = dir.entryList(QDir::Dirs);
	    l.remove(".");
	    l.remove("..");
	    tlist += l;
	}
    }

    return tlist;
}


QIconSet IconLoader::load(const QString &file)
{
    QString path;
    QPixmap normals, actives, disableds;
    QPixmap normall, activel, disabledl;

    initpathlist();

    QStringList::Iterator it = pathlist->begin();
    while (it != pathlist->end()) {
	path = *it++;

	normals.load(path + "/" + (*::theme) + "/small/" + file + ".png");
	if (! normals.isNull())
	    break;
    }

    // small
    QIconSet iconset(normals);
    actives.load(path + "/" + (*::theme) + "/small/" + file + "-active.png");
    disableds.load(path + "/" + (*::theme) + "/small/" + file + "-disabled.png");

    iconset.setPixmap(normals, QIconSet::Small, QIconSet::Normal);
    if (! actives.isNull())
	iconset.setPixmap(actives, QIconSet::Small, QIconSet::Active);
    if (! disableds.isNull())
	iconset.setPixmap(disableds, QIconSet::Small, QIconSet::Disabled);

    // large
    normall.load(path + "/" + (*::theme) + "/large/" + file + ".png");
    if (normall.isNull())
	normall.convertFromImage(normals.convertToImage().smoothScale(32, 32));

    activel.load(path + "/" + (*::theme) + "/large/" + file + "-active.png");
    disabledl.load(path + "/" + (*::theme) + "/large/" + file + "-disabled.png");

    iconset.setPixmap(normall, QIconSet::Large, QIconSet::Normal);
    if (! activel.isNull())
	iconset.setPixmap(activel, QIconSet::Large, QIconSet::Active);
    if (! disabledl.isNull())
	iconset.setPixmap(disabledl, QIconSet::Large, QIconSet::Disabled);

    return iconset;
}


void IconLoader::cleanup() {
    if (pathlist)
	delete pathlist;
    pathlist = 0;

    if (::theme)
	delete ::theme;
    ::theme = 0;
}
