/*
   CIPE - encrypted IP over UDP tunneling

   socks5.c - talk to a SOCKS5 server

   Copyright 1997 Olaf Titz <olaf@bigred.inka.de>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version
   2 of the License, or (at your option) any later version.
*/
/* $Id: socks5.c,v 1.2.2.1 1999/04/27 21:53:22 olaf Exp $ */

#include <stdlib.h>
#include <syslog.h>
#include <unistd.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include "ciped.h"

struct socksrq {
    char		ver;
    char		cmd;
    char		rsv;
    char		atyp;
    unsigned int	dstaddr __attribute__((packed));
    unsigned short	dstport __attribute__((packed));
};

const char * const socks_errlist[] = {
    "Succeeded",
    "Server failure",
    "Connection not allowed",
    "Network unreachable",
    "Host unreachable",
    "Connection refused",
    "TTL expired",
    "Command not supported",
    "Address type not supported"
};

static int readn(int fd, char *b, int n, const char *m)
{
    int p=0, e;
    while (p<n) {
	if ((e=read(fd, b+p, n-p))<=0) {
	    logerr(LOG_ERR, m);
	    return -1;
	}
	p+=e;
    }
    return n;
}

static int writen(int fd, const char *b, int n, const char *m)
{
    int p=0, e;
    while (p<n) {
	if ((e=write(fd, b+p, n-p))<0) {
	    logerr(LOG_ERR, m);
	    return e;
	}
	p+=e;
    }
    return n;
}

int socks5_open(struct sockaddr_in *so)
{
    int e;
    char *su=getenv("SOCKS5_USER");
    char *sp=getenv("SOCKS5_PASSWD");
    char buf[520];

    int fd=socket(AF_INET, SOCK_STREAM, 0);
    if (fd<0) {
	logerr(LOG_ERR, "socks5_open: socket"); return -1;
    }
    if (connect(fd, (struct sockaddr *)so, sizeof(*so))<0) {
	logerr(LOG_ERR, "socks5_open: connect");
	goto error;
    }
    if (su)
	e=writen(fd, "\005\002\000\002", 4, "socks5_open: winit");
    else
	e=writen(fd, "\005\001\000", 3, "socks5_open: winit");
    if (e<0)
	goto error;
    if (readn(fd, buf, 2, "socks5_open: rinit")<2)
	goto error;

    switch (buf[1]) {
    case '\000':
	break;
    case '\002':
    {
	unsigned char l;
	int lu, lp;

	if (!su) su="";
	if (!sp) sp="";
	lu=strlen(su); lu=MIN(lu, 255);
	lp=strlen(sp); lp=MIN(lp, 255);
	if ((writen(fd, "\001", 1, "socks5_open: wauth1")<0) ||
	    (l=lu, writen(fd, &l, 1, "socks5_open: wauth2")<0) ||
	    (writen(fd, su, lu, "socks5_open: wauth3")<0) ||
	    (l=lp, writen(fd, &l, 1, "socks5_open: wauth4")<0) ||
	    (writen(fd, sp, lp, "socks5_open: wauth5")<0))
	    goto error;
	if (readn(fd, buf, 2, "socks5_open: rauth")<0)
	    goto error;
	if (buf[1]!=0) {
	    logs(LOG_ERR, "socks5_open: authentication failed", NULL);
	    goto error;
	}
	break;
    }
    default:
	logs(LOG_ERR, "socks5_open: authentication required", NULL);
	goto error;
    }
    return fd;

 error:
    close(fd);
    return -1;
}

int socks5_cmd(int fd, int cmd, struct sockaddr_in *so)
{
    struct socksrq r;
    r.ver=5;
    r.cmd=cmd;
    r.rsv=0;
    r.atyp=1;
    r.dstaddr=so->sin_addr.s_addr;
    r.dstport=so->sin_port;
    if (writen(fd, (char *)&r, sizeof(r), "socks5_cmd: wcmd")<0)
	goto error;
    if (readn(fd, (char *)&r, sizeof(r), "socks5_cmd: rreply")<0)
	goto error;
    if (r.cmd==0) {
	if (r.atyp!=1) {
	    logd(LOG_ERR, "socks5_cmd: unknown address type %d", r.atyp);
	    goto error;
	}
	so->sin_addr.s_addr=r.dstaddr;
	so->sin_port=r.dstport;
	return fd;
    }
    logs(LOG_ERR, "socks5_cmd: server: %s",
         (r.cmd>=sizeof(socks_errlist)) ?
         "Unknown error" : socks_errlist[(int)r.cmd]);
 error:
    close(fd);
    return -1;
}
