#!/usr/bin/python -i

import sys
import xml.etree.ElementTree as etree
try:
    import urllib.request as urllib2
except ImportError:
    import urllib2
import json

#############################
# vuid_mapping.py script
#
# VUID Mapping Details
#  The Vulkan spec creation process automatically generates string-based unique IDs for each Valid Usage statement
#  For implicit VUs, the format is VUID-<func|struct>-[<param_name>]-<type>
#   func|struct is the name of the API function or structure that the VU is under
#   param_name is an optional entry with the name of the function or struct parameter
#   type is the type of implicit check, see table below for possible values
#
#  For explicit VUs, the format is VUID-<func|struct>-[<param_name>]-<uniqueid>
#   All fields are the same as implicit VUs except the last parameter is a globally unique integer ID instead of a string type
#
# The values below are used to map the strings into unique integers that are used for the unique enum values returned by debug callbacks
# Here's how the bits of the numerical unique ID map to the ID type and values
# 31:21 - 11 bits that map to unique value for the function/struct
# 20:1  - 20 bits that map to param-type combo for implicit VU and uniqueid for explicit VU
# 0     - 1 bit on for implicit VU or off for explicit VU
#
# For implicit VUs 20:1 is split into 20:9 for parameter and 8:1 for type
FUNC_STRUCT_SHIFT = 21
EXPLICIT_ID_SHIFT = 1
IMPLICIT_TYPE_SHIFT = 1
IMPLICIT_PARAM_SHIFT = 9
explicit_bit0 = 0x0 # All explicit IDs are even
implicit_bit0 = 0x1 # All implicit IDs are odd
# Implicit type values, shifted up by ID_SHIFT bits in final ID
implicit_type_map = {
'parameter'       : 0,
'requiredbitmask' : 1,
'zerobitmask'     : 2,
'parent'          : 3,
'commonparent'    : 4,
'sType'           : 5,
'pNext'           : 6,
'unique'          : 7,
'queuetype'       : 8,
'recording'       : 9,
'cmdpool'         : 10,
'renderpass'      : 11,
'bufferlevel'     : 12,
'arraylength'     : 13,
}
# Function/struct value mappings, shifted up FUNC_STRUCT_SHIFT bits in final ID
func_struct_id_map = {
'VkAcquireNextImageInfo' : 0,
'VkAllocationCallbacks' : 1,
'VkAndroidSurfaceCreateInfo' : 2,
'VkApplicationInfo' : 3,
'VkAttachmentDescription' : 4,
'VkAttachmentReference' : 5,
'VkBindBufferMemoryInfo' : 6,
'VkBindImageMemoryInfo' : 7,
'VkBindImageMemorySwapchainInfo' : 8,
'VkBindSparseInfo' : 9,
'VkBufferCreateInfo' : 10,
'VkBufferImageCopy' : 11,
'VkBufferMemoryBarrier' : 12,
'VkBufferViewCreateInfo' : 13,
'VkClearAttachment' : 14,
'VkClearDepthStencilValue' : 15,
'VkClearValue' : 16,
'VkCmdProcessCommandsInfoNVX' : 17,
'VkCmdReserveSpaceForCommandsInfoNVX' : 18,
'VkCommandBufferAllocateInfo' : 19,
'VkCommandBufferBeginInfo' : 20,
'VkCommandBufferInheritanceInfo' : 21,
'VkCommandPoolCreateInfo' : 22,
'VkComponentMapping' : 23,
'VkComputePipelineCreateInfo' : 24,
'VkCopyDescriptorSet' : 25,
'VkD3D12FenceSubmitInfo' : 26,
'VkDebugMarkerMarkerInfoEXT' : 27,
'VkDebugMarkerObjectNameInfoEXT' : 28,
'VkDebugMarkerObjectTagInfoEXT' : 29,
'VkDebugReportCallbackCreateInfoEXT' : 30,
'VkDedicatedAllocationBufferCreateInfoNV' : 31,
'VkDedicatedAllocationImageCreateInfoNV' : 32,
'VkDedicatedAllocationMemoryAllocateInfoNV' : 33,
'VkDescriptorBufferInfo' : 34,
'VkDescriptorImageInfo' : 35,
'VkDescriptorPoolCreateInfo' : 36,
'VkDescriptorPoolSize' : 37,
'VkDescriptorSetAllocateInfo' : 38,
'VkDescriptorSetLayoutBinding' : 39,
'VkDescriptorSetLayoutCreateInfo' : 40,
'VkDescriptorUpdateTemplateCreateInfo' : 41,
'VkDescriptorUpdateTemplateEntry' : 42,
'VkDeviceCreateInfo' : 43,
'VkDeviceEventInfoEXT' : 44,
'VkDeviceGeneratedCommandsFeaturesNVX' : 45,
'VkDeviceGeneratedCommandsLimitsNVX' : 46,
'VkDeviceGroupBindSparseInfo' : 47,
'VkDeviceGroupCommandBufferBeginInfo' : 48,
'VkDeviceGroupDeviceCreateInfo' : 49,
'VkDeviceGroupPresentInfo' : 50,
'VkDeviceGroupRenderPassBeginInfo' : 51,
'VkDeviceGroupSubmitInfo' : 52,
'VkDeviceGroupSwapchainCreateInfo' : 53,
'VkDeviceQueueCreateInfo' : 54,
'VkDispatchIndirectCommand' : 55,
'VkDisplayEventInfoEXT' : 56,
'VkDisplayModeCreateInfo' : 57,
'VkDisplayPowerInfoEXT' : 58,
'VkDisplayPresentInfo' : 59,
'VkDisplaySurfaceCreateInfo' : 60,
'VkDrawIndexedIndirectCommand' : 61,
'VkDrawIndirectCommand' : 62,
'VkEventCreateInfo' : 63,
'VkExportMemoryAllocateInfo' : 64,
'VkExportMemoryAllocateInfoNV' : 65,
'VkExportMemoryWin32HandleInfo' : 66,
'VkExportMemoryWin32HandleInfoNV' : 67,
'VkExportSemaphoreCreateInfo' : 68,
'VkExportSemaphoreWin32HandleInfo' : 69,
'VkExternalMemoryBufferCreateInfo' : 70,
'VkExternalMemoryImageCreateInfo' : 71,
'VkExternalMemoryImageCreateInfoNV' : 72,
'VkFenceCreateInfo' : 73,
'VkFramebufferCreateInfo' : 74,
'VkGraphicsPipelineCreateInfo' : 75,
'VkIOSSurfaceCreateInfoMVK' : 76,
'VkImageBlit' : 77,
'VkImageCopy' : 78,
'VkImageCreateInfo' : 79,
'VkImageMemoryBarrier' : 80,
'VkImageResolve' : 81,
'VkImageSubresource' : 82,
'VkImageSubresourceLayers' : 83,
'VkImageSubresourceRange' : 84,
'VkImageSwapchainCreateInfo' : 85,
'VkImageViewCreateInfo' : 86,
'VkImportMemoryFdInfo' : 87,
'VkImportMemoryWin32HandleInfo' : 88,
'VkImportMemoryWin32HandleInfoNV' : 89,
'VkImportSemaphoreFdInfo' : 90,
'VkImportSemaphoreWin32HandleInfo' : 91,
'VkIndirectCommandsLayoutCreateInfoNVX' : 92,
'VkIndirectCommandsLayoutTokenNVX' : 93,
'VkIndirectCommandsTokenNVX' : 94,
'VkInstanceCreateInfo' : 95,
'VkMacOSSurfaceCreateInfoMVK' : 96,
'VkMappedMemoryRange' : 97,
'VkMemoryAllocateFlagsInfo' : 98,
'VkMemoryAllocateInfo' : 99,
'VkMemoryBarrier' : 100,
'VkMirSurfaceCreateInfo' : 101,
'VkObjectTableCreateInfoNVX' : 102,
'VkObjectTableDescriptorSetEntryNVX' : 103,
'VkObjectTableEntryNVX' : 104,
'VkObjectTableIndexBufferEntryNVX' : 105,
'VkObjectTablePipelineEntryNVX' : 106,
'VkObjectTablePushConstantEntryNVX' : 107,
'VkObjectTableVertexBufferEntryNVX' : 108,
'VkPhysicalDeviceDiscardRectanglePropertiesEXT' : 109,
'VkPhysicalDeviceExternalBufferInfo' : 110,
'VkPhysicalDeviceExternalImageFormatInfo' : 111,
'VkPhysicalDeviceExternalSemaphoreInfo' : 112,
'VkPhysicalDeviceFeatures' : 113,
'VkPhysicalDeviceFeatures2' : 114,
'VkPhysicalDeviceImageFormatInfo2' : 115,
'VkPhysicalDeviceMultiviewFeatures' : 116,
'VkPhysicalDevicePushDescriptorProperties' : 117,
'VkPhysicalDeviceSparseImageFormatInfo2' : 118,
'VkPhysicalDeviceSurfaceInfo2' : 119,
'VkPipelineCacheCreateInfo' : 120,
'VkPipelineColorBlendAttachmentState' : 121,
'VkPipelineColorBlendStateCreateInfo' : 122,
'VkPipelineDepthStencilStateCreateInfo' : 123,
'VkPipelineDiscardRectangleStateCreateInfoEXT' : 124,
'VkPipelineDynamicStateCreateInfo' : 125,
'VkPipelineInputAssemblyStateCreateInfo' : 126,
'VkPipelineLayoutCreateInfo' : 127,
'VkPipelineMultisampleStateCreateInfo' : 128,
'VkPipelineRasterizationStateCreateInfo' : 129,
'VkPipelineRasterizationStateRasterizationOrderAMD' : 130,
'VkPipelineShaderStageCreateInfo' : 131,
'VkPipelineTessellationStateCreateInfo' : 132,
'VkPipelineVertexInputStateCreateInfo' : 133,
'VkPipelineViewportStateCreateInfo' : 134,
'VkPipelineViewportSwizzleStateCreateInfoNV' : 135,
'VkPipelineViewportWScalingStateCreateInfoNV' : 136,
'VkPresentInfo' : 137,
'VkPresentRegion' : 138,
'VkPresentRegions' : 139,
'VkPresentTimesInfoGOOGLE' : 140,
'VkPushConstantRange' : 141,
'VkQueryPoolCreateInfo' : 142,
'VkRectLayer' : 143,
'VkRenderPassBeginInfo' : 144,
'VkRenderPassCreateInfo' : 145,
'VkRenderPassMultiviewCreateInfo' : 146,
'VkSamplerCreateInfo' : 147,
'VkSemaphoreCreateInfo' : 148,
'VkShaderModuleCreateInfo' : 149,
'VkSparseBufferMemoryBindInfo' : 150,
'VkSparseImageMemoryBind' : 151,
'VkSparseImageMemoryBindInfo' : 152,
'VkSparseImageOpaqueMemoryBindInfo' : 153,
'VkSparseMemoryBind' : 154,
'VkSpecializationInfo' : 155,
'VkSpecializationMapEntry' : 156,
'VkStencilOpState' : 157,
'VkSubmitInfo' : 158,
'VkSubpassDependency' : 159,
'VkSubpassDescription' : 160,
'VkSurfaceCapabilities2EXT' : 161,
'VkSwapchainCounterCreateInfoEXT' : 162,
'VkSwapchainCreateInfo' : 163,
'VkValidationFlagsEXT' : 164,
'VkVertexInputAttributeDescription' : 165,
'VkVertexInputBindingDescription' : 166,
'VkViSurfaceCreateInfoNN' : 167,
'VkViewport' : 168,
'VkViewportSwizzleNV' : 169,
'VkWaylandSurfaceCreateInfo' : 170,
'VkWin32KeyedMutexAcquireReleaseInfo' : 171,
'VkWin32KeyedMutexAcquireReleaseInfoNV' : 172,
'VkWin32SurfaceCreateInfo' : 173,
'VkWriteDescriptorSet' : 174,
'VkXcbSurfaceCreateInfo' : 175,
'VkXlibSurfaceCreateInfo' : 176,
'vkAcquireNextImage2' : 177,
'vkAcquireNextImage' : 178,
'vkAcquireXlibDisplayEXT' : 179,
'vkAllocateCommandBuffers' : 180,
'vkAllocateDescriptorSets' : 181,
'vkAllocateMemory' : 182,
'vkBeginCommandBuffer' : 183,
'vkBindBufferMemory' : 184,
'vkBindBufferMemory2' : 185,
'vkBindImageMemory' : 186,
'vkBindImageMemory2' : 187,
'vkCmdBeginQuery' : 188,
'vkCmdBeginRenderPass' : 189,
'vkCmdBindDescriptorSets' : 190,
'vkCmdBindIndexBuffer' : 191,
'vkCmdBindPipeline' : 192,
'vkCmdBindVertexBuffers' : 193,
'vkCmdBlitImage' : 194,
'vkCmdClearAttachments' : 195,
'vkCmdClearColorImage' : 196,
'vkCmdClearDepthStencilImage' : 197,
'vkCmdCopyBuffer' : 198,
'vkCmdCopyBufferToImage' : 199,
'vkCmdCopyImage' : 200,
'vkCmdCopyImageToBuffer' : 201,
'vkCmdCopyQueryPoolResults' : 202,
'vkCmdDebugMarkerBeginEXT' : 203,
'vkCmdDebugMarkerEndEXT' : 204,
'vkCmdDebugMarkerInsertEXT' : 205,
'vkCmdDispatch' : 206,
'vkCmdDispatchBase' : 207,
'vkCmdDispatchIndirect' : 208,
'vkCmdDraw' : 209,
'vkCmdDrawIndexed' : 210,
'vkCmdDrawIndexedIndirect' : 211,
'vkCmdDrawIndexedIndirectCountAMD' : 212,
'vkCmdDrawIndirect' : 213,
'vkCmdDrawIndirectCountAMD' : 214,
'vkCmdEndQuery' : 215,
'vkCmdEndRenderPass' : 216,
'vkCmdExecuteCommands' : 217,
'vkCmdFillBuffer' : 218,
'vkCmdNextSubpass' : 219,
'vkCmdPipelineBarrier' : 220,
'vkCmdProcessCommandsNVX' : 221,
'vkCmdPushConstants' : 222,
'vkCmdPushDescriptorSet' : 223,
'vkCmdPushDescriptorSetWithTemplate' : 224,
'vkCmdReserveSpaceForCommandsNVX' : 225,
'vkCmdResetEvent' : 226,
'vkCmdResetQueryPool' : 227,
'vkCmdResolveImage' : 228,
'vkCmdSetBlendConstants' : 229,
'vkCmdSetDepthBias' : 230,
'vkCmdSetDepthBounds' : 231,
'vkCmdSetDeviceMask' : 232,
'vkCmdSetDiscardRectangleEXT' : 233,
'vkCmdSetEvent' : 234,
'vkCmdSetLineWidth' : 235,
'vkCmdSetScissor' : 236,
'vkCmdSetStencilCompareMask' : 237,
'vkCmdSetStencilReference' : 238,
'vkCmdSetStencilWriteMask' : 239,
'vkCmdSetViewport' : 240,
'vkCmdSetViewportWScalingNV' : 241,
'vkCmdUpdateBuffer' : 242,
'vkCmdWaitEvents' : 243,
'vkCmdWriteTimestamp' : 244,
'vkCreateAndroidSurface' : 245,
'vkCreateBuffer' : 246,
'vkCreateBufferView' : 247,
'vkCreateCommandPool' : 248,
'vkCreateComputePipelines' : 249,
'vkCreateDebugReportCallbackEXT' : 250,
'vkCreateDescriptorPool' : 251,
'vkCreateDescriptorSetLayout' : 252,
'vkCreateDescriptorUpdateTemplate' : 253,
'vkCreateDevice' : 254,
'vkCreateDisplayMode' : 255,
'vkCreateDisplayPlaneSurface' : 256,
'vkCreateEvent' : 257,
'vkCreateFence' : 258,
'vkCreateFramebuffer' : 259,
'vkCreateGraphicsPipelines' : 260,
'vkCreateIOSSurfaceMVK' : 261,
'vkCreateImage' : 262,
'vkCreateImageView' : 263,
'vkCreateIndirectCommandsLayoutNVX' : 264,
'vkCreateInstance' : 265,
'vkCreateMacOSSurfaceMVK' : 266,
'vkCreateMirSurface' : 267,
'vkCreateObjectTableNVX' : 268,
'vkCreatePipelineCache' : 269,
'vkCreatePipelineLayout' : 270,
'vkCreateQueryPool' : 271,
'vkCreateRenderPass' : 272,
'vkCreateSampler' : 273,
'vkCreateSemaphore' : 274,
'vkCreateShaderModule' : 275,
'vkCreateSharedSwapchains' : 276,
'vkCreateSwapchain' : 277,
'vkCreateViSurfaceNN' : 278,
'vkCreateWaylandSurface' : 279,
'vkCreateWin32Surface' : 280,
'vkCreateXcbSurface' : 281,
'vkCreateXlibSurface' : 282,
'vkDebugMarkerSetObjectNameEXT' : 283,
'vkDebugMarkerSetObjectTagEXT' : 284,
'vkDebugReportMessageEXT' : 285,
'vkDestroyBuffer' : 286,
'vkDestroyBufferView' : 287,
'vkDestroyCommandPool' : 288,
'vkDestroyDebugReportCallbackEXT' : 289,
'vkDestroyDescriptorPool' : 290,
'vkDestroyDescriptorSetLayout' : 291,
'vkDestroyDescriptorUpdateTemplate' : 292,
'vkDestroyDevice' : 293,
'vkDestroyEvent' : 294,
'vkDestroyFence' : 295,
'vkDestroyFramebuffer' : 296,
'vkDestroyImage' : 297,
'vkDestroyImageView' : 298,
'vkDestroyIndirectCommandsLayoutNVX' : 299,
'vkDestroyInstance' : 300,
'vkDestroyObjectTableNVX' : 301,
'vkDestroyPipeline' : 302,
'vkDestroyPipelineCache' : 303,
'vkDestroyPipelineLayout' : 304,
'vkDestroyQueryPool' : 305,
'vkDestroyRenderPass' : 306,
'vkDestroySampler' : 307,
'vkDestroySemaphore' : 308,
'vkDestroyShaderModule' : 309,
'vkDestroySurface' : 310,
'vkDestroySwapchain' : 311,
'vkDeviceWaitIdle' : 312,
'vkDisplayPowerControlEXT' : 313,
'vkEndCommandBuffer' : 314,
'vkEnumerateDeviceExtensionProperties' : 315,
'vkEnumerateDeviceLayerProperties' : 316,
'vkEnumerateInstanceExtensionProperties' : 317,
'vkEnumerateInstanceLayerProperties' : 318,
'vkEnumeratePhysicalDeviceGroups' : 319,
'vkEnumeratePhysicalDevices' : 320,
'vkFlushMappedMemoryRanges' : 321,
'vkFreeCommandBuffers' : 322,
'vkFreeDescriptorSets' : 323,
'vkFreeMemory' : 324,
'vkGetBufferMemoryRequirements' : 325,
'vkGetDeviceGroupPeerMemoryFeatures' : 326,
'vkGetDeviceGroupPresentCapabilities' : 327,
'vkGetDeviceGroupSurfacePresentModes' : 328,
'vkGetDeviceMemoryCommitment' : 329,
'vkGetDeviceProcAddr' : 330,
'vkGetDeviceQueue' : 331,
'vkGetDisplayModeProperties' : 332,
'vkGetDisplayPlaneCapabilities' : 333,
'vkGetDisplayPlaneSupportedDisplays' : 334,
'vkGetEventStatus' : 335,
'vkGetFenceStatus' : 336,
'vkGetImageMemoryRequirements' : 337,
'vkGetImageSparseMemoryRequirements' : 338,
'vkGetImageSubresourceLayout' : 339,
'vkGetInstanceProcAddr' : 340,
'vkGetMemoryFd' : 341,
'vkGetMemoryFdProperties' : 342,
'vkGetMemoryWin32Handle' : 343,
'vkGetMemoryWin32HandleNV' : 344,
'vkGetMemoryWin32HandleProperties' : 345,
'vkGetPastPresentationTimingGOOGLE' : 346,
'vkGetPhysicalDeviceDisplayPlaneProperties' : 347,
'vkGetPhysicalDeviceDisplayProperties' : 348,
'vkGetPhysicalDeviceExternalBufferProperties' : 349,
'vkGetPhysicalDeviceExternalImageFormatPropertiesNV' : 350,
'vkGetPhysicalDeviceExternalSemaphoreProperties' : 351,
'vkGetPhysicalDeviceFeatures' : 352,
'vkGetPhysicalDeviceFeatures2' : 353,
'vkGetPhysicalDeviceFormatProperties' : 354,
'vkGetPhysicalDeviceFormatProperties2' : 355,
'vkGetPhysicalDeviceGeneratedCommandsPropertiesNVX' : 356,
'vkGetPhysicalDeviceImageFormatProperties' : 357,
'vkGetPhysicalDeviceImageFormatProperties2' : 358,
'vkGetPhysicalDeviceMemoryProperties' : 359,
'vkGetPhysicalDeviceMemoryProperties2' : 360,
'vkGetPhysicalDeviceMirPresentationSupport' : 361,
'vkGetPhysicalDevicePresentRectangles' : 362,
'vkGetPhysicalDeviceProperties' : 363,
'vkGetPhysicalDeviceProperties2' : 364,
'vkGetPhysicalDeviceQueueFamilyProperties' : 365,
'vkGetPhysicalDeviceQueueFamilyProperties2' : 366,
'vkGetPhysicalDeviceSparseImageFormatProperties' : 367,
'vkGetPhysicalDeviceSparseImageFormatProperties2' : 368,
'vkGetPhysicalDeviceSurfaceCapabilities2EXT' : 369,
'vkGetPhysicalDeviceSurfaceCapabilities2' : 370,
'vkGetPhysicalDeviceSurfaceCapabilities' : 371,
'vkGetPhysicalDeviceSurfaceFormats2' : 372,
'vkGetPhysicalDeviceSurfaceFormats' : 373,
'vkGetPhysicalDeviceSurfacePresentModes' : 374,
'vkGetPhysicalDeviceSurfaceSupport' : 375,
'vkGetPhysicalDeviceWaylandPresentationSupport' : 376,
'vkGetPhysicalDeviceWin32PresentationSupport' : 377,
'vkGetPhysicalDeviceXcbPresentationSupport' : 378,
'vkGetPhysicalDeviceXlibPresentationSupport' : 379,
'vkGetPipelineCacheData' : 380,
'vkGetQueryPoolResults' : 381,
'vkGetRandROutputDisplayEXT' : 382,
'vkGetRefreshCycleDurationGOOGLE' : 383,
'vkGetRenderAreaGranularity' : 384,
'vkGetSemaphoreFd' : 385,
'vkGetSemaphoreWin32Handle' : 386,
'vkGetSwapchainCounterEXT' : 387,
'vkGetSwapchainImages' : 388,
'vkGetSwapchainStatus' : 389,
'vkImportSemaphoreFd' : 390,
'vkImportSemaphoreWin32Handle' : 391,
'vkInvalidateMappedMemoryRanges' : 392,
'vkMapMemory' : 393,
'vkMergePipelineCaches' : 394,
'vkQueueBindSparse' : 395,
'vkQueuePresent' : 396,
'vkQueueSubmit' : 397,
'vkQueueWaitIdle' : 398,
'vkRegisterDeviceEventEXT' : 399,
'vkRegisterDisplayEventEXT' : 400,
'vkRegisterObjectsNVX' : 401,
'vkReleaseDisplayEXT' : 402,
'vkResetCommandBuffer' : 403,
'vkResetCommandPool' : 404,
'vkResetDescriptorPool' : 405,
'vkResetEvent' : 406,
'vkResetFences' : 407,
'vkSetEvent' : 408,
'vkSetHdrMetadataEXT' : 409,
'vkTrimCommandPool' : 410,
'vkUnmapMemory' : 411,
'vkUnregisterObjectsNVX' : 412,
'vkUpdateDescriptorSetWithTemplate' : 413,
'vkUpdateDescriptorSets' : 414,
'vkWaitForFences' : 415,
'VkPhysicalDeviceProperties2' : 416,
'VkFormatProperties2' : 417,
'VkImageFormatProperties2' : 418,
'VkPhysicalDeviceMemoryProperties2' : 419,
'VkSurfaceCapabilities2' : 420,
'VkDeviceGroupPresentCapabilities' : 421,
'VkExternalBufferProperties' : 422,
'VkMemoryWin32HandleProperties' : 423,
'VkMemoryFdProperties' : 424,
'VkExternalSemaphoreProperties' : 425,
'VkQueueFamilyProperties2' : 426,
'VkSparseImageFormatProperties2' : 427,
'VkSurfaceFormat2' : 428,
'VkTextureLODGatherFormatPropertiesAMD' : 429,
'VkPhysicalDeviceMultiviewProperties' : 430,
'VkPhysicalDeviceGroupProperties' : 431,
'VkExternalImageFormatProperties' : 432,
'VkPhysicalDeviceIDProperties' : 433,
'VkPhysicalDeviceMultiviewPerViewAttributesPropertiesNVX' : 434,
'VkHdrMetadataEXT' : 435,
'VkExternalMemoryProperties' : 436,
'VkFormatProperties' : 437,
'VkImageFormatProperties' : 438,
'VkPhysicalDeviceLimits' : 439,
'VkQueueFamilyProperties' : 440,
'VkMemoryType' : 441,
'VkMemoryHeap' : 442,
'VkSparseImageFormatProperties' : 443,
'VkSurfaceCapabilities' : 444,
'VkDisplayProperties' : 445,
'VkDisplayPlaneCapabilities' : 446,
'VkSharedPresentSurfaceCapabilities' : 447,
'VkExternalImageFormatPropertiesNV' : 448,
'VkPhysicalDeviceBlendOperationAdvancedFeaturesEXT' : 449,
'VkPhysicalDeviceBlendOperationAdvancedPropertiesEXT' : 450,
'VkPhysicalDeviceSamplerFilterMinmaxPropertiesEXT' : 451,
'VkPipelineColorBlendAdvancedStateCreateInfoEXT' : 452,
'VkPipelineCoverageModulationStateCreateInfoNV' : 453,
'VkPipelineCoverageToColorStateCreateInfoNV' : 454,
'VkSamplerReductionModeCreateInfoEXT' : 455,
'VkPhysicalDeviceProperties' : 456,
'VkSurfaceFormat' : 457,
'VkExportFenceCreateInfo' : 458,
'VkPhysicalDeviceExternalFenceInfo' : 459,
'VkExternalFenceProperties' : 460,
'vkGetPhysicalDeviceExternalFenceProperties' : 461,
'VkImportFenceFdInfo' : 462,
'VkFenceGetFdInfo' : 463,
'vkImportFenceFd' : 464,
'vkGetFenceFd' : 465,
'VkImportFenceWin32HandleInfo' : 466,
'VkExportFenceWin32HandleInfo' : 467,
'VkFenceGetWin32HandleInfo' : 468,
'vkImportFenceWin32Handle' : 469,
'vkGetFenceWin32Handle' : 470,
'VkSemaphoreGetFdInfo' : 471,
'VkSemaphoreGetWin32HandleInfo' : 472,
'VkMemoryGetFdInfo' : 473,
'VkMemoryGetWin32HandleInfo' : 474,
'VkMemoryDedicatedRequirements' : 475,
'VkMemoryDedicatedAllocateInfo' : 476,
'VkBufferMemoryRequirementsInfo2' : 477,
'VkImageMemoryRequirementsInfo2' : 478,
'VkImageSparseMemoryRequirementsInfo2' : 479,
'VkMemoryRequirements2' : 480,
'VkSparseImageMemoryRequirements2' : 481,
'vkGetImageMemoryRequirements2' : 482,
'vkGetBufferMemoryRequirements2' : 483,
'vkGetImageSparseMemoryRequirements2' : 484,
'VkPhysicalDevice16BitStorageFeatures' : 485,
'VkPhysicalDeviceVariablePointerFeatures' : 486,
'VkSampleLocationsInfoEXT' : 487,
'VkRenderPassSampleLocationsBeginInfoEXT' : 488,
'VkPipelineSampleLocationsStateCreateInfoEXT' : 489,
'VkPhysicalDeviceSampleLocationsPropertiesEXT' : 490,
'VkMultisamplePropertiesEXT' : 491,
'vkGetPhysicalDeviceMultisamplePropertiesEXT' : 492,
'VkValidationCacheCreateInfoEXT' : 493,
'VkShaderModuleValidationCacheCreateInfoEXT' : 494,
'vkCreateValidationCacheEXT' : 495,
'vkGetValidationCacheDataEXT' : 496,
'vkCmdSetSampleLocationsEXT' : 497,
'vkDestroyValidationCacheEXT' : 498,
'vkMergeValidationCachesEXT' : 499,
'VkAttachmentSampleLocationsEXT' : 500,
'VkSubpassSampleLocationsEXT' : 501,
'VkPhysicalDevicePointClippingProperties' : 502,
'VkInputAttachmentAspectReference' : 503,
'VkRenderPassInputAttachmentAspectCreateInfo' : 504,
'VkImageViewUsageCreateInfo' : 505,
'VkPipelineTessellationDomainOriginStateCreateInfo' : 506,
'VkImageFormatListCreateInfo' : 507,
'VkSamplerYcbcrConversionCreateInfo' : 508,
'VkBindImagePlaneMemoryInfo' : 509,
'VkImagePlaneMemoryRequirementsInfo' : 510,
'vkCreateSamplerYcbcrConversion' : 511,
'VkBindBufferMemoryDeviceGroupInfo' : 512,
'VkBindImageMemoryDeviceGroupInfo' : 513,
'vkDestroySamplerYcbcrConversion' : 514,
'VkPhysicalDeviceSamplerYcbcrConversionFeatures' : 515,
'VkSamplerYcbcrConversionImageFormatProperties' : 516,
'VkSamplerYcbcrConversionInfo' : 517,
'VkDeviceQueueGlobalPriorityCreateInfoEXT' : 518,
'vkGetShaderInfoAMD' : 519,
'VkShaderStatisticsInfoAMD' : 520,
'VkImportMemoryHostPointerInfoEXT' : 521,
'VkMemoryHostPointerPropertiesEXT' : 522,
'VkPhysicalDeviceExternalMemoryHostPropertiesEXT' : 523,
'vkGetMemoryHostPointerPropertiesEXT' : 524,
'VkPhysicalDeviceConservativeRasterizationPropertiesEXT' : 525,
'VkPipelineRasterizationConservativeStateCreateInfoEXT' : 526,
'vkCmdWriteBufferMarkerAMD' : 527,
'VkDescriptorSetLayoutSupport' : 528,
'VkDeviceQueueInfo2' : 529,
'VkPhysicalDeviceMaintenance3Properties' : 530,
'VkPhysicalDeviceProtectedMemoryFeatures' : 531,
'VkPhysicalDeviceProtectedMemoryProperties' : 532,
'VkPhysicalDeviceShaderDrawParameterFeatures' : 533,
'VkPhysicalDeviceSubgroupProperties' : 534,
'VkProtectedSubmitInfo' : 535,
'vkEnumerateInstanceVersion' : 536,
'vkGetDescriptorSetLayoutSupport' : 537,
'vkGetDeviceQueue2' : 538,
'VkDebugUtilsObjectNameInfoEXT' : 539,
'VkDebugUtilsObjectTagInfoEXT' : 540,
'VkDebugUtilsLabelEXT' : 541,
'VkDebugUtilsMessengerCallbackDataEXT' : 542,
'VkDebugUtilsMessengerCreateInfoEXT' : 543,
'vkCreateDebugUtilsMessengerEXT' : 544,
'vkSubmitDebugUtilsMessageEXT' : 545,
'VkPhysicalDeviceVertexAttributeDivisorPropertiesEXT' : 546,
'VkPipelineVertexInputDivisorStateCreateInfoEXT' : 547,
### ADD New func/struct mappings above this line
}
# Mapping of params to unique IDs
implicit_param_map = {
'a' : 0,
'addressModeU' : 1,
'addressModeV' : 2,
'addressModeW' : 3,
'alphaBlendOp' : 4,
'alphaMode' : 5,
'aspectMask' : 6,
'attachmentCount' : 7,
'b' : 8,
'back' : 9,
'bindCount' : 10,
'bindInfoCount' : 11,
'bindingCount' : 12,
'buffer' : 13,
'bufferView' : 14,
'callback' : 15,
'colorBlendOp' : 16,
'colorWriteMask' : 17,
'commandBuffer' : 18,
'commandBufferCount' : 19,
'commandPool' : 20,
'compareOp' : 21,
'components' : 22,
'compositeAlpha' : 23,
'connection' : 24,
'contents' : 25,
'countBuffer' : 26,
'counter' : 27,
'createInfoCount' : 28,
'cullMode' : 29,
'dataSize' : 30,
'dependencyFlags' : 31,
'depthCompareOp' : 32,
'depthFailOp' : 33,
'descriptorCount' : 34,
'descriptorPool' : 35,
'descriptorSet' : 36,
'descriptorSetCount' : 37,
'descriptorSetLayout' : 38,
'descriptorType' : 39,
'descriptorUpdateEntryCount' : 40,
'descriptorUpdateTemplate' : 41,
'descriptorWriteCount' : 42,
'device' : 43,
'deviceEvent' : 44,
'disabledValidationCheckCount' : 45,
'discardRectangleCount' : 46,
'discardRectangleMode' : 47,
'display' : 48,
'displayEvent' : 49,
'displayMode' : 50,
'dpy' : 51,
'dstAccessMask' : 52,
'dstAlphaBlendFactor' : 53,
'dstBuffer' : 54,
'dstCache' : 55,
'dstColorBlendFactor' : 56,
'dstImage' : 57,
'dstImageLayout' : 58,
'dstSet' : 59,
'dstStageMask' : 60,
'dstSubresource' : 61,
'dynamicStateCount' : 62,
'event' : 63,
'eventCount' : 64,
'externalHandleType' : 65,
'faceMask' : 66,
'failOp' : 67,
'fence' : 68,
'fenceCount' : 69,
'filter' : 70,
'finalLayout' : 71,
'flags' : 72,
'format' : 73,
'framebuffer' : 74,
'front' : 75,
'frontFace' : 76,
'g' : 77,
'handleType' : 78,
'handleTypes' : 79,
'image' : 80,
'imageColorSpace' : 81,
'imageFormat' : 82,
'imageLayout' : 83,
'imageSharingMode' : 84,
'imageSubresource' : 85,
'imageType' : 86,
'imageUsage' : 87,
'imageView' : 88,
'indexType' : 89,
'indirectCommandsLayout' : 90,
'indirectCommandsTokenCount' : 91,
'initialLayout' : 92,
'inputRate' : 93,
'instance' : 94,
'layout' : 95,
'level' : 96,
'loadOp' : 97,
'magFilter' : 98,
'memory' : 99,
'memoryRangeCount' : 100,
'minFilter' : 101,
'mipmapMode' : 102,
'mode' : 103,
'modes' : 104,
'module' : 105,
'newLayout' : 106,
'objectCount' : 107,
'objectTable' : 108,
'objectType' : 109,
'oldLayout' : 110,
'oldSwapchain' : 111,
'pAcquireInfo' : 112,
'pAcquireKeys' : 113,
'pAcquireSyncs' : 114,
'pAcquireTimeoutMilliseconds' : 115,
'pAcquireTimeouts' : 116,
'pAllocateInfo' : 117,
'pAllocator' : 118,
'pApplicationInfo' : 119,
'pApplicationName' : 120,
'pAttachments' : 121,
'pAttributes' : 122,
'pBeginInfo' : 123,
'pBindInfo' : 124,
'pBindInfos' : 125,
'pBindings' : 126,
'pBinds' : 127,
'pBuffer' : 128,
'pBufferBinds' : 129,
'pBufferMemoryBarriers' : 130,
'pBuffers' : 131,
'pCallback' : 132,
'pCapabilities' : 133,
'pCode' : 134,
'pColor' : 135,
'pColorAttachments' : 136,
'pCommandBufferDeviceMasks' : 137,
'pCommandBuffers' : 138,
'pCommandPool' : 139,
'pCommittedMemoryInBytes' : 140,
'pCorrelationMasks' : 141,
'pCounterValue' : 142,
'pCreateInfo' : 143,
'pCreateInfos' : 144,
'pData' : 145,
'pDataSize' : 146,
'pDependencies' : 147,
'pDepthStencil' : 148,
'pDepthStencilAttachment' : 149,
'pDescriptorCopies' : 150,
'pDescriptorPool' : 151,
'pDescriptorSets' : 152,
'pDescriptorUpdateEntries' : 153,
'pDescriptorUpdateTemplate' : 154,
'pDescriptorWrites' : 155,
'pDevice' : 156,
'pDeviceEventInfo' : 157,
'pDeviceGroupPresentCapabilities' : 158,
'pDeviceIndices' : 159,
'pDeviceMasks' : 160,
'pDeviceRenderAreas' : 161,
'pDisabledValidationChecks' : 162,
'pDiscardRectangles' : 163,
'pDisplay' : 164,
'pDisplayCount' : 165,
'pDisplayEventInfo' : 166,
'pDisplayPowerInfo' : 167,
'pDisplayTimingProperties' : 168,
'pDisplays' : 169,
'pDynamicOffsets' : 170,
'pDynamicState' : 171,
'pDynamicStates' : 172,
'pEnabledFeatures' : 173,
'pEngineName' : 174,
'pEvent' : 175,
'pEvents' : 176,
'pExternalBufferInfo' : 177,
'pExternalBufferProperties' : 178,
'pExternalImageFormatProperties' : 179,
'pExternalSemaphoreInfo' : 180,
'pExternalSemaphoreProperties' : 181,
'pFd' : 182,
'pFeatures' : 183,
'pFence' : 184,
'pFences' : 185,
'pFormatInfo' : 186,
'pFormatProperties' : 187,
'pFramebuffer' : 188,
'pGranularity' : 189,
'pHandle' : 190,
'pImage' : 191,
'pImageBinds' : 192,
'pImageFormatInfo' : 193,
'pImageFormatProperties' : 194,
'pImageIndex' : 195,
'pImageIndices' : 196,
'pImageMemoryBarriers' : 197,
'pImageOpaqueBinds' : 198,
'pImportSemaphoreFdInfo' : 199,
'pImportSemaphoreWin32HandleInfo' : 200,
'pIndirectCommandsLayout' : 201,
'pIndirectCommandsTokens' : 202,
'pInitialData' : 203,
'pInputAssemblyState' : 204,
'pInputAttachments' : 205,
'pInstance' : 206,
'pLayerName' : 207,
'pLayerPrefix' : 208,
'pLayout' : 209,
'pLimits' : 210,
'pMarkerInfo' : 211,
'pMarkerName' : 212,
'pMemory' : 213,
'pMemoryBarriers' : 214,
'pMemoryFdProperties' : 215,
'pMemoryProperties' : 216,
'pMemoryRanges' : 217,
'pMemoryRequirements' : 218,
'pMemoryWin32HandleProperties' : 219,
'pMessage' : 220,
'pMetadata' : 221,
'pMode' : 222,
'pModes' : 223,
'pName' : 224,
'pNameInfo' : 225,
'pNext' : 226,
'pObjectEntryCounts' : 227,
'pObjectEntryTypes' : 228,
'pObjectEntryUsageFlags' : 229,
'pObjectIndices' : 230,
'pObjectName' : 231,
'pObjectTable' : 232,
'pOffsets' : 233,
'pPeerMemoryFeatures' : 234,
'pPhysicalDeviceCount' : 235,
'pPhysicalDeviceGroupCount' : 236,
'pPhysicalDeviceGroupProperties' : 237,
'pPhysicalDevices' : 238,
'pPipelineCache' : 239,
'pPipelineLayout' : 240,
'pPipelines' : 241,
'pPoolSizes' : 242,
'pPresentInfo' : 243,
'pPresentModeCount' : 244,
'pPresentModes' : 245,
'pPresentationTimingCount' : 246,
'pPresentationTimings' : 247,
'pPreserveAttachments' : 248,
'pProcessCommandsInfo' : 249,
'pProperties' : 250,
'pPropertyCount' : 251,
'pPushConstantRanges' : 252,
'pQueryPool' : 253,
'pQueue' : 254,
'pQueueCreateInfos' : 255,
'pQueueFamilyProperties' : 256,
'pQueueFamilyPropertyCount' : 257,
'pQueuePriorities' : 258,
'pRanges' : 259,
'pRasterizationState' : 260,
'pRectCount' : 261,
'pRectangles' : 262,
'pRects' : 263,
'pRegions' : 264,
'pReleaseKeys' : 265,
'pReleaseSyncs' : 266,
'pRenderPass' : 267,
'pRenderPassBegin' : 268,
'pReserveSpaceInfo' : 269,
'pResolveAttachments' : 270,
'pResults' : 271,
'pSFRRects' : 272,
'pSampleMask' : 273,
'pSampler' : 274,
'pScissors' : 275,
'pSemaphore' : 276,
'pSetLayout' : 277,
'pSetLayouts' : 278,
'pShaderModule' : 279,
'pSignalSemaphoreDeviceIndices' : 280,
'pSignalSemaphoreValues' : 281,
'pSignalSemaphores' : 282,
'pSparseMemoryRequirementCount' : 283,
'pSparseMemoryRequirements' : 284,
'pSpecializationInfo' : 285,
'pSrcCaches' : 286,
'pStages' : 287,
'pSubmits' : 288,
'pSubpasses' : 289,
'pSubresource' : 290,
'pSupported' : 291,
'pSurface' : 292,
'pSurfaceCapabilities' : 293,
'pSurfaceFormatCount' : 294,
'pSurfaceFormats' : 295,
'pSurfaceInfo' : 296,
'pSwapchain' : 297,
'pSwapchainImageCount' : 298,
'pSwapchainImages' : 299,
'pSwapchains' : 300,
'pTag' : 301,
'pTagInfo' : 302,
'pTimes' : 303,
'pTokens' : 304,
'pValues' : 305,
'pVertexAttributeDescriptions' : 306,
'pVertexBindingDescriptions' : 307,
'pVertexInputState' : 308,
'pView' : 309,
'pViewMasks' : 310,
'pViewOffsets' : 311,
'pWaitDstStageMask' : 312,
'pWaitSemaphoreDeviceIndices' : 313,
'pWaitSemaphoreValues' : 314,
'pWaitSemaphores' : 315,
'passOp' : 316,
'physicalDevice' : 317,
'pipeline' : 318,
'pipelineBindPoint' : 319,
'pipelineCache' : 320,
'pipelineLayout' : 321,
'pipelineStage' : 322,
'polygonMode' : 323,
'poolSizeCount' : 324,
'powerState' : 325,
'ppData' : 326,
'ppEnabledExtensionNames' : 327,
'ppEnabledLayerNames' : 328,
'ppObjectTableEntries' : 329,
'preTransform' : 330,
'presentMode' : 331,
'queryPool' : 332,
'queryType' : 333,
'queue' : 334,
'queueCount' : 335,
'queueCreateInfoCount' : 336,
'r' : 337,
'rangeCount' : 338,
'rasterizationOrder' : 339,
'rasterizationSamples' : 340,
'rectCount' : 341,
'regionCount' : 342,
'renderPass' : 343,
'sType' : 344,
'sampler' : 345,
'samples' : 346,
'scissorCount' : 347,
'semaphore' : 348,
'sequencesCountBuffer' : 349,
'sequencesIndexBuffer' : 350,
'shaderModule' : 351,
'sharingMode' : 352,
'size' : 353,
'srcAccessMask' : 354,
'srcAlphaBlendFactor' : 355,
'srcBuffer' : 356,
'srcCacheCount' : 357,
'srcColorBlendFactor' : 358,
'srcImage' : 359,
'srcImageLayout' : 360,
'srcSet' : 361,
'srcStageMask' : 362,
'srcSubresource' : 363,
'stage' : 364,
'stageCount' : 365,
'stageFlags' : 366,
'stageMask' : 367,
'stencilLoadOp' : 368,
'stencilStoreOp' : 369,
'storeOp' : 370,
'subpassCount' : 371,
'subresource' : 372,
'subresourceRange' : 373,
'surface' : 374,
'surfaceCounters' : 375,
'swapchain' : 376,
'swapchainCount' : 377,
'tagSize' : 378,
'targetCommandBuffer' : 379,
'templateType' : 380,
'tiling' : 381,
'tokenCount' : 382,
'tokenType' : 383,
'topology' : 384,
'transform' : 385,
'type' : 386,
'usage' : 387,
'viewType' : 388,
'viewportCount' : 389,
'w' : 390,
'window' : 391,
'x' : 392,
'y' : 393,
'z' : 394,
'externalMemoryFeatures' : 395,
'compatibleHandleTypes' : 396,
'exportFromImportedHandleTypes' : 397,
'linearTilingFeatures' : 398,
'optimalTilingFeatures' : 399,
'bufferFeatures' : 400,
'sampleCounts' : 401,
'framebufferColorSampleCounts' : 402,
'framebufferDepthSampleCounts' : 403,
'framebufferStencilSampleCounts' : 404,
'framebufferNoAttachmentsSampleCounts' : 405,
'sampledImageColorSampleCounts' : 406,
'sampledImageIntegerSampleCounts' : 407,
'sampledImageDepthSampleCounts' : 408,
'sampledImageStencilSampleCounts' : 409,
'storageImageSampleCounts' : 410,
'queueFlags' : 411,
'propertyFlags' : 412,
'supportedTransforms' : 413,
'currentTransform' : 414,
'supportedCompositeAlpha' : 415,
'supportedUsageFlags' : 416,
'supportedAlpha' : 417,
'sharedPresentSupportedUsageFlags' : 418,
'externalSemaphoreFeatures' : 419,
'supportedSurfaceCounters' : 420,
'blendOverlap' : 421,
'coverageModulationMode' : 422,
'coverageModulationTableCount' : 423,
'reductionMode' : 424,
'enabledLayerCount' : 425,
'enabledExtensionCount' : 426,
'waitSemaphoreCount' : 427,
'signalSemaphoreCount' : 428,
'bufferBindCount' : 429,
'imageOpaqueBindCount' : 430,
'imageBindCount' : 431,
'codeSize' : 432,
'initialDataSize' : 433,
'vertexBindingDescriptionCount' : 434,
'vertexAttributeDescriptionCount' : 435,
'setLayoutCount' : 436,
'pushConstantRangeCount' : 437,
'inputAttachmentCount' : 438,
'colorAttachmentCount' : 439,
'preserveAttachmentCount' : 440,
'dependencyCount' : 441,
'dynamicOffsetCount' : 442,
'rectangleCount' : 443,
'correlationMaskCount' : 444,
'acquireCount' : 445,
'releaseCount' : 446,
'deviceIndexCount' : 447,
'SFRRectCount' : 448,
'deviceRenderAreaCount' : 449,
'physicalDeviceCount' : 450,
'waitSemaphoreValuesCount' : 451,
'signalSemaphoreValuesCount' : 452,
'deviceType' : 453,
'colorSpace' : 454,
'pfnAllocation' : 455,
'pfnReallocation' : 556,
'pfnFree' : 457,
'blendConstants' : 458,
'displayName' : 459,
'pfnCallback' : 460,
'externalFenceFeatures' : 461,
'pInfo' : 462,
'pGetFdInfo' : 463,
'pGetWin32HandleInfo' : 464,
'pExternalFenceInfo' : 465,
'pExternalFenceProperties' : 466,
'pImportFenceProperties' : 467,
'pImportFenceFdInfo' : 468,
'pImportFenceWin32HandleInfo' : 469,
'basePipelineHandle' : 470,
'pImmutableSamplers' : 471,
'pTexelBufferView' : 472,
'sampleLocationsPerPixel' : 473,
'sampleLocationsCount' : 474,
'pSampleLocations' : 475,
'attachmentInitialSampleLocationsCount' : 476,
'pAttachmentInitialSampleLocations' : 477,
'postSubpassSampleLocationsCount' : 478,
'pSubpassSampleLocations' : 479,
'sampleLocationSampleCounts' : 480,
'pValidationCache' : 481,
'validationCache' : 482,
'sampleLocationsInfo' : 483,
'pSampleLocationsInfo' : 484,
'pMultisampleProperties' : 485,
'pointClippingBehavior' : 486,
'aspectReferenceCount' : 487,
'pAspectReferences' : 488,
'domainOrigin' : 489,
'ycbcrModel' : 490,
'ycbcrRange' : 491,
'xChromaOffset' : 492,
'yChromaOffset' : 493,
'chromaFilter' : 494,
'planeAspect' : 495,
'pYcbcrConversion' : 496,
'ycbcrConversion' : 497,
'pViewFormats' : 498,
'conversion' : 499,
'pPostSubpassSampleLocations' : 500,
'globalPriority' : 501,
'shaderStage' : 502,
'infoType' : 503,
'pInfoSize' : 504,
'shaderStageMask' : 505,
'pMemoryHostPointerProperties' : 506,
'pHostPointer' : 507,
'conservativeRasterizationMode' : 508,
'pViewports' : 509,
'pViewportWScalings' : 510,
'pSplitInstanceBindRegions' : 511,
'pApiVersion' : 512,
'pSupport' : 513,
'pQueueInfo' : 514,
'splitInstanceBindRegionCount' : 515,
'pLabelName' : 516,
'messageSeverity' : 517,
'messageType' : 518,
'pfnUserCallback' : 519,
'pMessenger' : 520,
'messageTypes' : 521,
'vertexBindingDivisorCount' : 522,
'pVertexBindingDivisors' : 523,
### ADD New implicit param mappings above this line
}

uniqueid_set = set() # store uniqueid to make sure we don't have duplicates

# Convert a string VUID into numerical value
#  See "VUID Mapping Details" comment above for more info
def convertVUID(vuid_string):
    """Convert a string-based VUID into a numerical value"""
    #func_struct_update = False
    #imp_param_update = False
    if vuid_string in ['', None]:
        return -1
    vuid_parts = vuid_string.split('-')
    # Alias core/KHR/KHX ids because not all VUIDs in the spec get updated at the same time
    if vuid_parts[1].endswith('KHR') or vuid_parts[1].endswith('KHX'):
        vuid_parts[1] = vuid_parts[1][:-3]
    if vuid_parts[1] not in func_struct_id_map:
        print ("ERROR: Missing func/struct map value for '%s'!" % (vuid_parts[1]))
        print (" TODO: Need to add mapping for this to end of func_struct_id_map")
        print ("   replace '### ADD New func/struct mappings above this line' line with \"'%s' : %d,\"" % (vuid_parts[1], len(func_struct_id_map)))
        func_struct_id_map[vuid_parts[1]] = len(func_struct_id_map)
        #func_struct_update = True
        sys.exit(1)
    uniqueid = func_struct_id_map[vuid_parts[1]] << FUNC_STRUCT_SHIFT
    if vuid_parts[-1].isdigit(): # explit VUID has int on the end
        explicit_id = int(vuid_parts[-1])
        # For explicit case, id is explicit_base + func/struct mapping + unique id
        uniqueid = uniqueid + (explicit_id << EXPLICIT_ID_SHIFT) + explicit_bit0
    else: # implicit case
        if vuid_parts[-1] not in implicit_type_map:
            print("ERROR: Missing mapping for implicit type '%s'!\nTODO: Please add new mapping." % (vuid_parts[-1]))
            sys.exit(1)
        else:
            param_id = 0 # Default when no param is available
            if vuid_parts[-2] != vuid_parts[1]: # we have a parameter
                if vuid_parts[-2] in implicit_param_map:
                    param_id = implicit_param_map[vuid_parts[-2]]
                else:
                    print ("ERROR: Missing param '%s' from implicit_param_map\n TODO: Please add new mapping." % (vuid_parts[-2]))
                    print ("   replace '### ADD New implicit param mappings above this line' line with \"'%s' : %d,\"" % (vuid_parts[-2], len(implicit_param_map)))
                    implicit_param_map[vuid_parts[-2]] = len(implicit_param_map)
                    #imp_param_update = True
                    sys.exit(1)
                uniqueid = uniqueid + (param_id << IMPLICIT_PARAM_SHIFT) + (implicit_type_map[vuid_parts[-1]] << IMPLICIT_TYPE_SHIFT) + implicit_bit0
            else: # No parameter so that field is 0
                uniqueid = uniqueid + (implicit_type_map[vuid_parts[-1]] << IMPLICIT_TYPE_SHIFT) + implicit_bit0
#    if uniqueid in uniqueid_set:
#        print ("ERROR: Uniqueid %d for string id %s is a duplicate!" % (uniqueid, vuid_string))
#        print (" TODO: Figure out what caused the dupe and fix it")
        #sys.exit()
    # print ("Storing uniqueid %d for unique string %s" % (uniqueid, vuid_string))
    uniqueid_set.add(uniqueid)
#    if func_struct_update:
#        print ("func_struct_id_map updated, here's new structure")
#        print ("func_struct_id_map = {")
#        fs_id = 0
#        for fs in sorted(func_struct_id_map):
#            print ("'%s' : %d," % (fs, fs_id))
#            fs_id = fs_id + 1
#        print ("### ADD New func/struct mappings above this line")
#        print ("}")
#    if imp_param_update:
#        print ("implicit_param_map updated, here's new structure")
#        print ("implicit_param_map = {")
#        ip_id = 0
#        for ip in sorted(implicit_param_map):
#            print ("'%s' : %d," % (ip, ip_id))
#            ip_id = ip_id + 1
#        print ("### ADD New implicit param mappings above this line")
#        print ("}")

    return uniqueid
