#ifndef _RHEOLEF_STREAM_H
#define _RHEOLEF_STREAM_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

/*Class:rheostream
NAME:  @code{irheostream}, @code{orheostream} - large data streams 
@clindex irheostream
@clindex orheostream
@clindex iorheo
@fiindex @file{.gz} gzip
@cindex RHEOPATH environment variable
@findex scatch
@findex has_suffix
@findex delete_suffix
@findex get_basename
@findex get_dirname
@findex get_full_name_from_rheo_path
@findex append_dir_to_rheo_path
@findex prepend_dir_to_rheo_path
@findex file_exists
@findex itos
@findex ftos
@toindex @code{gzip}
ABSTRACT:
  @noindent
  This class provides a stream interface for large
  data management.
  File decompresion is assumed using
  @code{gzip} and a recursive search in a directory list
  is provided for input.
  @example
     orheostream foo("NAME", "suffix");
  @end example
  @noindent
  is like
  @example
     ofstream foo("NAME.suffix").
  @end example
  @noindent
  However, if @var{NAME} does not end with @file{.suffix}, then
  @file{.suffix} is automatically added.
  By default, compression is performed on the fly with gzip, adding an additional @file{.gz} suffix.
  The @code{flush} action is nicely handled in compression mode:
  @example
     foo.flush();
  @end example
  This feature allows intermediate results to be available during long computations.
  The compression can be deactivated while opening a file by an optional argument:
  @example
     orheostream foo("NAME", "suffix", io::nogz);
  @end example
  An existing compressed file can be reopen in @code{append} mode: new results will
  be appended at the end of an existing file:
  @example
     orheostream foo("NAME", "suffix", io::app);
  @end example

  @noindent
  Conversely,
  @example
 	irheostream foo("NAME","suffix");
  @end example
  @noindent
  is like
  @example
	ifstream foo("NAME.suffix").
  @end example
  @noindent
  However, we look at a search path environment variable 
  @code{RHEOPATH} in order to find @var{NAME} while suffix is assumed.
  Moreover, @code{gzip} compressed files, ending
  with the @file{.gz} suffix is assumed, and decompression is done.

  Finally, a set of useful functions are provided.
DESCRIPTION:
  @noindent
  The following code:
  @example
	irheostream is("results", "data");
  @end example
  @noindent
  will recursively look for a @file{results[.data[.gz]]} file in the
  directory mentioned by the @code{RHEOPATH} environment variable.

  @noindent
  For instance, if you insert in our ".cshrc" something like:
  @example
   	setenv RHEOPATH ".:/home/dupont:/usr/local/math/demo"
  @end example
  @noindent
  the process will study the current directory @file{.}, then, if
  neither @file{square.data.gz} nor @file{square.data} exits, it scan
  all subdirectory of the current directory. Then, if file
  is not founded, it start recusively in @file{/home/dupond}
  and then in @file{/usr/local/math/demo}.

  @noindent
  File decompression is performed by using
  the @code{gzip} command, and data are pipe-lined 
  directly in memory.


  @noindent
  If the file start with @file{.} as @file{./square} or with a @file{/}
  as @file{/home/oscar/square}, no search occurs and @code{RHEOPATH}
  environment variable is not used.

  @noindent
  Also, if the environment variable @code{RHEOPATH} is not set, the
  default value is the current directory @file{.}.

  @noindent
  For output stream:
  @example
   	orheostream os("newresults", "data");
  @end example
  @noindent
  file compression is assumed, and "newresults.data.gz"
  will be created.

  @noindent
  File loading and storing are mentioned by a message,
  either:
  @example
 	! load "./results.data.gz"
  @end example
  @noindent
  or:
  @example
   	! file "./newresults.data.gz" created.
  @end example
  on the @code{clog} stream.
  By adding the following:
  @example
   	clog << noverbose;
  @end example
  @noindent
  you turn off these messages
 (@pxref{iorheo ialgorithm}).

AUTHORS:
    LMC-IMAG, 38041 Grenoble cedex 9, France
   | Pierre.Saramito@imag.fr
DATE:   31 october 1997
METHODS: @rheostream
End:
*/
#include "rheolef/compiler.h"
#include <boost/iostreams/filtering_stream.hpp>

namespace rheolef {

// input & output  modes:
namespace io {
  typedef enum {
    out  = 0,
    app  = 1, // append mode
    gz   = 0, // default is to use gzip
    nogz = 2  // force not using gzip
  } mode_type;
}
//<rheostream:
class irheostream : public boost::iostreams::filtering_stream<boost::iostreams::input> {
public:
    irheostream() : boost::iostreams::filtering_stream<boost::iostreams::input>() {}
    irheostream(const std::string& name, const std::string& suffix = std::string());
    virtual ~irheostream();
    void open  (const std::string& name, const std::string& suffix = std::string());
    void close();
protected:
    std::ifstream _ifs;
};
static const bool dont_gzip = false;
class orheostream : public boost::iostreams::filtering_stream<boost::iostreams::output> {
public:
    orheostream() : boost::iostreams::filtering_stream<boost::iostreams::output>() {}
    orheostream(const std::string& name, const std::string& suffix = std::string(),
	io::mode_type mode = io::out);
    virtual ~orheostream();
    void open  (const std::string& name, const std::string& suffix = std::string(),
	io::mode_type mode = io::out);
    void flush();
    void close();
    const std::string& filename() const { return _full_name; }
protected:
    void  _open_internal (io::mode_type mode);
    void _close_internal ();
// data:
    io::mode_type     _mode;
    std::string       _full_name;
};
std::string itos (std::string::size_type i);
std::string ftos (const Float& x);

// catch first occurrence of string in file
bool scatch (std::istream& in, const std::string& ch, bool full_match = true);

// has_suffix("toto.suffix", "suffix") -> true
bool has_suffix (const std::string& name, const std::string& suffix);

// "toto.suffix" --> "toto"
std::string delete_suffix (const std::string& name, const std::string& suffix);

// has_any_suffix("toto.any_suffix") -> true
bool has_any_suffix (const std::string& name);

// delete_any_suffix("toto.any_suffix") --> "toto"
std::string delete_any_suffix (const std::string& name);

// "/usr/local/dir/toto.suffix" --> "toto.suffix"
std::string get_basename (const std::string& name);

// "/usr/local/dir/toto.suffix" --> "/usr/local/dir"
std::string get_dirname (const std::string& name);

// "toto" --> "/usr/local/math/data/toto.suffix"
std::string get_full_name_from_rheo_path (const std::string& rootname, const std::string& suffix);

// "." + "../geodir" --> ".:../geodir"
void append_dir_to_rheo_path (const std::string& dir);

// "../geodir" + "." --> "../geodir:."
void prepend_dir_to_rheo_path (const std::string& dir);

bool file_exists (const std::string& filename);

// string to float
bool is_float (const std::string&);
Float to_float (const std::string&);

// in TMPDIR environment variable or "/tmp" by default
std::string get_tmpdir();
//>rheostream:

}// namespace rheolef
#endif // _RHEOLEF_STREAM_H
