/*
 *  Test program for the I²C implementation. (sig_i2c_bus.c/h).
 *
 * Copyright (C) 2008-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <stdbool.h>
#include <assert.h>
#include "sig_i2c_bus.h"
#include "glue-log.h"

const char *basedir = ".";

static struct {
	struct sig_i2c_bus *i2c;
} i2c_master;

static struct {
	struct sig_i2c_bus *i2c;
} i2c_slave;

/* just some data for testing. Doesn't have any meaning. */
static const unsigned char byte_buffer[16] = {
	0xac, 0x34, 0x53, 0x88, 0x99, 0xb2, 0x11, 0x80,
	0xff, 0x00, 0x00, 0x45, 0x83, 0x10, 0xF0, 0x0F
};

static unsigned int read_buffer_counter = 0;
static unsigned int ver_buffer_counter = 0;
static bool errors_seen = false;

static void
req_test_bytes(unsigned int count, unsigned char *buffer)
{
	while (count > 0) {
		*buffer = byte_buffer[read_buffer_counter];

		read_buffer_counter++;
		read_buffer_counter &= 0x0F;
		buffer++;
		count--;
	}
}

static void
ver_test_bytes(unsigned int count, const unsigned char *buffer)
{
	unsigned char exp;

	while (count > 0) {
		exp = byte_buffer[ver_buffer_counter];
		if (exp != *buffer) {
			faum_log(FAUM_LOG_WARNING, __FUNCTION__, "",
				"found %x, expected %x\n", *buffer, exp);
			errors_seen = true;
		}

		count--;
		ver_buffer_counter++;
		ver_buffer_counter &= 0x0F;
		buffer++;
	}
}

static unsigned char
req_test_byte(void)
{
	unsigned char ret = byte_buffer[read_buffer_counter];

	read_buffer_counter++;
	read_buffer_counter &= 0x0F;
	return ret;
}

static bool
ver_test_byte(unsigned char val)
{
	unsigned char exp = byte_buffer[ver_buffer_counter];
	bool ret = val == byte_buffer[ver_buffer_counter];

	ver_buffer_counter++;
	ver_buffer_counter &= 0x0F;

	if (! ret) {
		faum_log(FAUM_LOG_WARNING, __FUNCTION__, "",
			"found %x, expected %x\n", val, exp);
	}
	return ret;
}

static void
slave_stop_trans(void *_cpssp)
{
	/* do nothing :) */
}

static void 
slave_read_byte(void *_cpssp, unsigned char *val)
{
	*val = req_test_byte();

	faum_log(FAUM_LOG_INFO, __FUNCTION__, "",
		"slave read byte requested. Returning %x\n", *val);
}

static bool
slave_write_byte(void *_cpssp, unsigned char val)
{
	faum_log(FAUM_LOG_INFO, __FUNCTION__, "",
		"slave received byte %x\n", val);

	errors_seen |= (! ver_test_byte(val));

	return true;
}

static bool
slave_ack_addr(void *_cpssp, unsigned char addr)
{
	faum_log(FAUM_LOG_INFO, __FUNCTION__, "",
		"address %x sent to slave.\n", addr);

	return true;
}

static void
master_do_tests(void)
{
	unsigned char buf[32];
	unsigned int ret;
	bool bret;

	/* write tests */
	req_test_bytes(1, buf);
	ret = sig_i2c_bus_write_bytes(
		i2c_master.i2c, &i2c_master, 0x0a, 1, buf);
	assert(ret == 1);
	sig_i2c_bus_stop_transaction(i2c_master.i2c, &i2c_master);

	req_test_bytes(5, buf);
	ret = sig_i2c_bus_write_bytes(
		i2c_master.i2c, &i2c_master, 0xa0, 5, buf);
	assert(ret == 5);
	sig_i2c_bus_stop_transaction(i2c_master.i2c, &i2c_master);

	req_test_bytes(8, buf);
	ret = sig_i2c_bus_write_bytes(
		i2c_master.i2c, &i2c_master, 0x10, 8, buf);
	assert(ret == 8);
	sig_i2c_bus_stop_transaction(i2c_master.i2c, &i2c_master);

	/* read tests */
	bret = sig_i2c_bus_read_bytes(
		i2c_master.i2c, &i2c_master, 0x11, 1, buf);
	sig_i2c_bus_stop_transaction(i2c_master.i2c, &i2c_master);
	assert(bret);
	ver_test_bytes(1, buf);

	bret = sig_i2c_bus_read_bytes(
		i2c_master.i2c, &i2c_master, 0x11, 5, buf);
	sig_i2c_bus_stop_transaction(i2c_master.i2c, &i2c_master);
	assert(bret);
	ver_test_bytes(5, buf);

	bret = sig_i2c_bus_read_bytes(
		i2c_master.i2c, &i2c_master, 0x11, 8, buf);
	sig_i2c_bus_stop_transaction(i2c_master.i2c, &i2c_master);
	assert(bret);
	ver_test_bytes(8, buf);

	/* read after write tests */
	req_test_bytes(1, buf);
	ret = sig_i2c_bus_write_bytes(
		i2c_master.i2c, &i2c_master, 0xa0, 1, buf);
	assert(ret == 1);
	bret = sig_i2c_bus_read_bytes(
		i2c_master.i2c, &i2c_master, 0xa1, 1, buf);
	assert(bret);
	ver_test_bytes(1, buf);
	sig_i2c_bus_stop_transaction(i2c_master.i2c, &i2c_master);

	req_test_bytes(1, buf);
	ret = sig_i2c_bus_write_bytes(
		i2c_master.i2c, &i2c_master, 0xa0, 1, buf);
	assert(ret == 1);
	bret = sig_i2c_bus_read_bytes(
		i2c_master.i2c, &i2c_master, 0xa1, 8, buf);
	assert(bret);
	ver_test_bytes(8, buf);
	sig_i2c_bus_stop_transaction(i2c_master.i2c, &i2c_master);

	req_test_bytes(5, buf);
	ret = sig_i2c_bus_write_bytes(
		i2c_master.i2c, &i2c_master, 0xa0, 5, buf);
	assert(ret == 5);
	bret = sig_i2c_bus_read_bytes(
		i2c_master.i2c, &i2c_master, 0xa1, 8, buf);
	assert(bret);
	ver_test_bytes(8, buf);
	sig_i2c_bus_stop_transaction(i2c_master.i2c, &i2c_master);

	/* write after read tests */
	/* TODO */

#if 0 /* obsolete */
	/* write tests */
	val = req_test_byte();
	faum_log(FAUM_LOG_INFO, __FUNCTION__, "", 
		"writing %x to slave.\n", val);
	sig_i2c_bus_write_byte(i2c_master.i2c, &i2c_master, 0xa0, val);

	val = req_test_byte();
	faum_log(FAUM_LOG_INFO, __FUNCTION__, "", 
		"writing %x to slave.\n", val);
	sig_i2c_bus_write_byte(i2c_master.i2c, &i2c_master, 0xa1, val);

	val = req_test_byte();
	faum_log(FAUM_LOG_INFO, __FUNCTION__, "", 
		"writing %x to slave.\n", val);
	sig_i2c_bus_write_byte(i2c_master.i2c, &i2c_master, 0x11, val);

	/* read tests */
	faum_log(FAUM_LOG_INFO, __FUNCTION__, "",
		"starting read transaction.\n");
	sig_i2c_bus_read_byte(i2c_master.i2c, &i2c_master, 0xa0, &val);
	faum_log(FAUM_LOG_INFO, __FUNCTION__, "", 
		"read %x from slave.\n", val);
	errors_seen |= (! ver_test_byte(val));

	faum_log(FAUM_LOG_INFO, __FUNCTION__, "",
		"starting read transaction.\n");
	sig_i2c_bus_read_byte(i2c_master.i2c, &i2c_master, 0xa1, &val);
	faum_log(FAUM_LOG_INFO, __FUNCTION__, "", 
		"read %x from slave.\n", val);
	errors_seen |= (! ver_test_byte(val));

	faum_log(FAUM_LOG_INFO, __FUNCTION__, "",
		"starting read transaction.\n");
	sig_i2c_bus_read_byte(i2c_master.i2c, &i2c_master, 0x11, &val);
	faum_log(FAUM_LOG_INFO, __FUNCTION__, "", 
		"read %x from slave.\n", val);
	errors_seen |= (! ver_test_byte(val));
#endif /* obsolete */
}

static void
master_init(struct sig_i2c_bus *i2c)
{
	static struct sig_i2c_bus_funcs f = {
		.ack_addr = 0,
		.read_byte = 0,
		.write_byte = 0,
	};

	i2c_master.i2c = i2c;
	sig_i2c_bus_connect_cooked(i2c, &i2c_master, &f);
}

static void
slave_init(struct sig_i2c_bus *i2c)
{	
	static struct sig_i2c_bus_funcs f = {
		.ack_addr = slave_ack_addr,
		.read_byte = slave_read_byte,
		.write_byte = slave_write_byte,
		.stop_transaction = slave_stop_trans,
	};

	i2c_slave.i2c = i2c;
	sig_i2c_bus_connect_cooked(i2c, &i2c_slave, &f);
}

void
time_stop(void)
{
}

void
time_cont(void)
{
}

static void
system_create(const char *name, int nr)
{
	struct sig_i2c_bus *b = sig_i2c_bus_create(name);
	master_init(b);
	slave_init(b);
}


static void
system_destroy(const char *name, int nr)
{
	/* sig_i2c_bus_destroy(b); FIXME */
}

int
main(int argc, char **argv)
{
	system_create("i2c_test", 1);
	master_do_tests();
	system_destroy("i2c_test", 1);

	return errors_seen;
}
