/*
 *  Software MMU support
 *
 *  Copyright (c) 2005-2009 FAUmachine Team.
 *  Copyright (c) 2003 Fabrice Bellard
 *  Modified for FAUmachine by Volkmar Sieh
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301
 * USA
 */
#define DATA_SIZE (1 << SHIFT)

#if DATA_SIZE == 8
#define SUFFIX q
#define USUFFIX q
#define DATA_TYPE uint64_t
#elif DATA_SIZE == 4
#define SUFFIX l
#define USUFFIX l
#define DATA_TYPE uint32_t
#elif DATA_SIZE == 2
#define SUFFIX w
#define USUFFIX uw
#define DATA_TYPE uint16_t
#elif DATA_SIZE == 1
#define SUFFIX b
#define USUFFIX ub
#define DATA_TYPE uint8_t
#else
#error unsupported data size
#endif


/* (just for grep) io_readb io_readub io_readw io_readl */
static inline DATA_TYPE
glue(glue(io_read, SUFFIX), MMUSUFFIX)(struct cpssp *cpssp, paddr_t physaddr, unsigned int hash)
{
	DATA_TYPE res;

#ifdef CODE_ACCESS
#if SHIFT <= 2
	res = CHIP_(io_mem_code)[hash][SHIFT](cpssp, physaddr);
#else
#ifdef TARGET_WORDS_BIGENDIAN
	res = (uint64_t)CHIP_(io_mem_code)[hash][2](cpssp, physaddr) << 32;
	res |= CHIP_(io_mem_code)[hash][2](cpssp, physaddr + 4);
#else
	res = CHIP_(io_mem_code)[hash][2](cpssp, physaddr);
	res |= (uint64_t)CHIP_(io_mem_code)[hash][2](cpssp, physaddr + 4) << 32;
#endif
#endif /* SHIFT > 2 */
#else /* ! CODE_ACCESS */
#if SHIFT <= 2
	res = CHIP_(io_mem_read)[hash][SHIFT](cpssp, physaddr);
#else
#ifdef TARGET_WORDS_BIGENDIAN
	res = (uint64_t)CHIP_(io_mem_read)[hash][2](cpssp, physaddr) << 32;
	res |= CHIP_(io_mem_read)[hash][2](cpssp, physaddr + 4);
#else
	res = CHIP_(io_mem_read)[hash][2](cpssp, physaddr);
	res |= (uint64_t)CHIP_(io_mem_read)[hash][2](cpssp, physaddr + 4) << 32;
#endif
#endif /* SHIFT > 2 */
#endif /* ! CODE_ACCESS */
	return res;
}

/* (just for grep) slow_ldb_mmu slow_ldub_mmu slow_ldw_mmu slow_ldl_mmu */
static inline void
glue(glue(slow_ld, SUFFIX), MMUSUFFIX)(struct cpssp *cpssp, vaddr_t addr, int is_user, DATA_TYPE *valp, void *retaddr)
{
	DATA_TYPE res;
	int hash;
	vaddr_t tlb_addr;
	paddr_t physaddr;
	Haddr hostaddr;

	hash = (addr / CACHE_LINE_SIZE) & (CPU_TLB_SIZE - 1);
redo:
#ifdef CODE_ACCESS
	tlb_addr = cpssp->tlb_code[is_user][hash].address;
#else
	tlb_addr = cpssp->tlb_read[is_user][hash].address;
#endif
	if ((addr & CACHE_LINE_MASK)
			!= (tlb_addr & (CACHE_LINE_MASK | TLB_INVALID_MASK))) {
		/* The page is not in the TLB : fill it. */
#ifdef CODE_ACCESS
		CHIP_(tlb_fill)(cpssp, addr, 2, is_user, retaddr);
#else
		CHIP_(tlb_fill)(cpssp, addr, 0, is_user, retaddr);
#endif
		goto redo;
	}

	if (tlb_addr & ~CACHE_LINE_MASK) {
		/* I/O access. */
		cpssp->mem_write_pc = (unsigned long) retaddr;

#ifdef CODE_ACCESS
		physaddr = addr + cpssp->tlb_code[is_user][hash].phys_addend;
#else
		physaddr = addr + cpssp->tlb_read[is_user][hash].phys_addend;
#endif
		res = glue(glue(io_read, SUFFIX), MMUSUFFIX)(cpssp, physaddr,
				(tlb_addr >> IO_MEM_SHIFT) & (IO_MEM_NB_ENTRIES - 1));
	} else {
		/* ROM/RAM access. */
#ifdef CODE_ACCESS
		hostaddr = addr + cpssp->tlb_code[is_user][hash].host_addend;
#else
		hostaddr = addr + cpssp->tlb_read[is_user][hash].host_addend;
#endif
		res = glue(glue(ld, USUFFIX), _raw)(hostaddr);
	}

	*valp = res;
}

/* Just to make this clear (and grepable):
 * Through macro magic, this will become something like
 * __ldb_mmu
 * __ldub_mmu
 * __ldw_mmu
 * __ldl_mmu        */
void
glue(glue(CHIP_(__ld), SUFFIX), MMUSUFFIX)(struct cpssp *cpssp, vaddr_t addr, int is_user, DATA_TYPE *valp)
{
        if ((addr & (DATA_SIZE - 1)) == 0) {
		/* Aligned access. */
		glue(glue(slow_ld, SUFFIX), MMUSUFFIX)(cpssp, addr, is_user, valp, GETPC());
	} else {
		/* Unaligned access (might access two pages). */
		DATA_TYPE res;
		vaddr_t addr1;
		vaddr_t addr2;
		DATA_TYPE res1;
		DATA_TYPE res2;
		int shift;

		addr1 = addr & ~(DATA_SIZE - 1);
		addr2 = addr1 + DATA_SIZE;

		glue(glue(slow_ld, SUFFIX), MMUSUFFIX)(cpssp, addr1, is_user, &res1, GETPC());
		glue(glue(slow_ld, SUFFIX), MMUSUFFIX)(cpssp, addr2, is_user, &res2, GETPC());

		shift = (addr & (DATA_SIZE - 1)) * 8;
		res = (res1 >> shift) | (res2 << ((DATA_SIZE * 8) - shift));

		*valp = res;
	}
}

#ifndef CODE_ACCESS

/* (just for grep) io_writeb io_writeub io_writew io_writel */
static inline void
glue(io_write, SUFFIX)(struct cpssp *cpssp, paddr_t physaddr, DATA_TYPE val, unsigned int hash)
{
#if SHIFT <= 2
	CHIP_(io_mem_write)[hash][SHIFT](cpssp, physaddr, val);
#else
#ifdef TARGET_WORDS_BIGENDIAN
	CHIP_(io_mem_write)[hash][2](cpssp, physaddr, val >> 32);
	CHIP_(io_mem_write)[hash][2](cpssp, physaddr + 4, val);
#else
	CHIP_(io_mem_write)[hash][2](cpssp, physaddr, val);
	CHIP_(io_mem_write)[hash][2](cpssp, physaddr + 4, val >> 32);
#endif
#endif /* SHIFT > 2 */
}

/* (just for grep) slow_stb_mmu slow_stub_mmu slow_stw_mmu slow_stl_mmu */
static inline void
glue(glue(slow_st, SUFFIX), MMUSUFFIX)(struct cpssp *cpssp, vaddr_t addr, DATA_TYPE val, int is_user, void *retaddr)
{
	int hash;
	vaddr_t tlb_addr;
	paddr_t physaddr;
	Haddr hostaddr;

	hash = (addr / CACHE_LINE_SIZE) & (CPU_TLB_SIZE - 1);
redo:
	tlb_addr = cpssp->tlb_write[is_user][hash].address;
	if ((addr & CACHE_LINE_MASK)
			!= (tlb_addr & (CACHE_LINE_MASK | TLB_INVALID_MASK))) {
		/* The page is not in the TLB : fill it. */
		CHIP_(tlb_fill)(cpssp, addr, 1, is_user, retaddr);
		goto redo;
	}

	if (tlb_addr & ~CACHE_LINE_MASK) {
		/* I/O access. */
		cpssp->mem_write_pc = (unsigned long) retaddr;

		physaddr = addr + cpssp->tlb_write[is_user][hash].phys_addend;
		glue(io_write, SUFFIX)(cpssp, physaddr, val, (tlb_addr >> IO_MEM_SHIFT) & (IO_MEM_NB_ENTRIES - 1));
	} else {
		/* RAM access. */
		hostaddr = addr + cpssp->tlb_write[is_user][hash].host_addend;
		glue(glue(st, SUFFIX), _raw)(hostaddr, val);
	}
}

/* (just for grep) __stb_mmu __stub_mmu __stw_mmu __stl_mmu */
void
glue(glue(CHIP_(__st), SUFFIX), MMUSUFFIX)(struct cpssp *cpssp, vaddr_t addr, int is_user, DATA_TYPE val)
{
	if ((addr & (DATA_SIZE - 1)) == 0) {
		/* Aligned access. */
		glue(glue(slow_st, SUFFIX), MMUSUFFIX)(cpssp, addr, val, is_user, GETPC());
	} else {
		/* Unaligned access (might access two pages). */
		unsigned int i;

		for (i = 0; i < DATA_SIZE; i++) {
#ifdef TARGET_WORDS_BIGENDIAN
			glue(slow_stb, MMUSUFFIX)(cpssp, addr + i, val >> (((DATA_SIZE - 1) * 8) - (i * 8)), is_user, GETPC());
#else
			glue(slow_stb, MMUSUFFIX)(cpssp, addr + i, val >> (i * 8), is_user, GETPC());
#endif
		}
	}
}

#endif /* ! defined(CODE_ACCESS) */

#undef SHIFT
#undef DATA_TYPE
#undef SUFFIX
#undef USUFFIX
#undef DATA_SIZE
