# arch-tag: f82e5927-7118-40ff-b2ca-71279a6e68aa
# Copyright (C) 2004  David Allouche <david@allouche.net>
#                     Canonical Ltd.
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""PyArch exceptions
"""


class ExecProblem(RuntimeError):

    def __init__(self, proc):
        info = [ proc.command ]
        status = []
        if proc.signal is not None:
            status.append("got signal %d" % proc.signal)
        if proc.status is not None:
            status.append("exited with code %d" % proc.status)
        info.append(",".join(status))
        if proc.status is not None:
            codes = ' '.join(map(str, proc.expected))
            if len(proc.expected) == 1:
                info.append("(expected exit code %s)" % codes)
            else:
                info.append("(expected exit codes %s)" % codes)
        lines = [' '.join(info)]
        if proc.args: lines.append('argv: ' + ", ".join(map(repr, proc.args)))
        if proc.chdir: lines.append('chdir: ' + repr(proc.chdir))
        if proc.error is not None:
            error = proc.error.rstrip()
        else:
            error = None
        if error: lines.extend(("* error report", error))
        RuntimeError.__init__(self, '\n'.join(lines))
        self.proc = proc


class SourceTreeError(Exception):

    """Tried to create a source tree object from non-tree-root directory."""

    def __init__(self, dirname):
        message = "directory is not a tree-root: %r" % (dirname,)
        Exception.__init__(self, message)
        self.dirname = str(dirname)


class TreeRootError(Exception):

    """Could not find the Arch tree-root of a directory."""

    def __init__(self, dirname):
        message = "directory is not inside a project tree: %r" % (dirname,)
        Exception.__init__(self, message)
        self.dirname = str(dirname)


class TreeVersionError(Exception):

    """Arch source tree does not have a valid default version."""

    def __init__(self, tree, bad_version=None):
        if bad_version is None:
            message = "tree has no default version set: %r" % (tree,)
        else:
            message = "invalid default version for tree: tree=%r, version=%r" \
                      % (tree, bad_version)
        Exception.__init__(self, message)
        self.tree = tree
        self.bad_version = bad_version


class NamespaceError(Exception):

    """Invalid Arch namespace name, or incorrect kind of name.

    A value that does not make sense in the Arch namespace was
    provided, or the wrong kind of name was used, for example a
    revision where a patchlevel is expected.
    """

    def __init__(self, name, expected=None):
        if expected is None:
            message = "invalid name: %s" % (name,)
        else:
            message = "invalid %s: %s" % (expected, name)
        Exception.__init__(self, message)
        self.name = name
        self.expected = expected


class ArchiveNotRegistered(Exception):
    """Tried to access an unregistered archive.

    Namespace existence predicates, implementing `NamespaceObject.exists`,
    return a boolean value for the existence of a name in the Arch namespace.
    When the archive a name belongs to is not registered, existence cannot be
    decided, so this exception is raised.
    """

    def __init__(self, name):
        message = "archive not registered: %s" % (name,)
        Exception.__init__(self, message)
        self.name = name


class LocationNotRegistered(Exception):
    """Tried to unregister a location that is not registered.

    `ArchiveLocation.unregister` raises this exception if the location was not
    registered.
    """

    def __init__(self, url):
        message = "location not registered: %s" % (url,)
        Exception.__init__(self, message)
        self.url = url


class LocationAlreadyRegistered(Exception):
    """Tried to register a location that was already registered.

    `ArchiveLocation.register` raises this exception if the location was
    already registered.
    """

    def __init__(self, url):
        message = "location already registered: %s" % (url,)
        Exception.__init__(self, message)
        self.url = url


class MirrorLocationMismatch(Exception):
    """Tried to mirror to a location that contains a different archive.

    `ArchiveLocation.make_mirrorer` raises this exception if the target was a
    registered location for a different archive.
    """

    def __init__(self, source, target):
        message = ("source location %r contains archive %r,"
                   " but target location %r contains archive %r."
                   % (source, source.archive(), target, target.archive()))
        Exception.__init__(self, message)
        self.source = source
        self.target = target


class MetaInfoError(Exception):
    """The requested archive meta-info was not present.

    ArchiveLocation.meta_info raises this exception if the archive contained no
    meta-info for the specified key.
    """

    def __init__(self, url, key):
        message = "location %r has no meta-info %r" % (url, key)
        Exception.__init__(self, message)
        self.url = url
        self.key = key


class UntaggedFileError(Exception):
    """Tried to get the id-tag of file which has none.

    The id-tag of a file is an id which is set by an explicit id or a
    tagline. Implicit-style taglines ("tag:") and name-based ids do
    not count.
    """

    def __init__(self, name):
        message = "file has not explicit id nor tagline: %s" % (name,)
        Exception.__init__(self, message)
        self.name = name

class IllegalEscapeSequence(Exception):
    """Illegal syntax in a pika-escaped string.

    Pika-escaping is the syntax used to represent spaces, control
    characters and non-ASCII characters in file names with Arch.
    """

    def __init__(self, text):
        message = "illegal escape sequence in %r" % text
        Exception.__init__(self, message)
        self.text = text


class NoPristineFoundError(Exception):

    """Failed to find a pristine for the given revision.

    `WorkingTree.find_pristine` raises this exception if no pristine
    tree can be found for the given revision.
    """

    def __init__(self, tree, revision):
        message = "no pristine found for revision %s in tree %r" \
                  % (revision.fullname, str(tree))
        Exception.__init__(self, message)
        self.revision, self.tree = revision, tree


class MissingFileError(Exception):

    """A request file is not present in the given revision.

    `ArchSourceTree.file_find` raises this exception if the requested
    does not exist in the specified revision.
    """

    def __init__(self, tree, name, revision):
        message = "file %r from the tree %r is not present in revision %s" \
                  % (str(name), str(tree), revision)
        Exception.__init__(self, message)
        self.tree, self.name, self.revision = tree, name, revision


class ChangesetConflict(Exception):

    """A conflict occured when applying a changeset."""

    def __init__(self, tree, cset):
        message = "conflict when applying %r to %s" % (cset, tree)
        Exception.__init__(self, message)
        self.tree, self.changeset = tree, cset
