/***************************************************************************
                          csubtitles.cpp  -  description
                             -------------------
    begin                : di feb 4 2003
    copyright            : (C) 2003 by Tom Deblauwe
    email                : tom.deblauwe@pandora.be
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <qtextstream.h>
#include <qtextcodec.h>
#include <kmessagebox.h>

#include "csubtitles.h"

#include "csubtitle.h"

CSubtitles::CSubtitles() : QPtrList<CSubtitle>(){
  //subtitlesList->setAutoDelete(true);

}

CSubtitles::~CSubtitles(){
}

long CSubtitles::getMSecs(QTime timeObject)
{
   return ((timeObject.hour() * 60 * 60 + timeObject.minute() * 60 + timeObject.second()) * 1000) + timeObject.msec();
}

QTime CSubtitles::getQTimeFromMSecs(long totalmsecs)
{

  ldiv_t divresult;
  divresult = ldiv (totalmsecs, 1000);
  long hms = divresult.quot; //contains hours,minutes and seconds
  int msecs = divresult.rem;

  divresult = ldiv (hms, 60);
  long hm = divresult.quot; //contains hours and minutes
  int secs = divresult.rem;

  divresult = ldiv (hm, 60);
  long h = divresult.quot; //contains hours
  int mins = divresult.rem;

  int hours = (int) h;

//  printf("toQTime: hours %d mins %d secs %d msecs %d", hours, mins, secs, msecs);
  return QTime(hours, mins, secs, msecs);
}

CSubtitle* CSubtitles::getSubtitle(QTime curPos) const
{
    return getSubtitle(CSubtitles::getMSecs(curPos));
}

CSubtitle* CSubtitles::getSubtitle(long curPos) const
{

  QPtrListIterator<CSubtitle> it( *this );
  CSubtitle* sub;
  while ( (sub = it.current() ) != 0 )
  {
      ++it;
      if(curPos >= sub->getStartTime() && curPos <= sub->getEndTime() )
      {
        //this is the subtitle
        return sub;
      }
  }
  return 0;
}

int CSubtitles::getNrOfSubtitles() const
{
  return this->count();
}

long CSubtitles::getTotalOccupiedTime(void)
{
  if( this->count() != 0 )
  {
    return this->last()->getEndTime();
  }
  else
  {
    return 0;
  }

}

CSubtitle* CSubtitles::getFirstSubtitle(void)
{
  if( this->count() != 0 )
  {
    CSubtitle* sub;
    sub = this->first();
    return sub;
  }
  return 0;
}

CSubtitle* CSubtitles::getLastSubtitle(void)
{
  if( this->count() != 0 )
  {
    CSubtitle* sub;
    sub = this->last();
    return sub;
  }
  return 0;
}

QPtrListIterator<CSubtitle> CSubtitles::getIterator() const
{
  QPtrListIterator<CSubtitle> it( *this );
  return it;
}

//returns the endtime of the subtitle before the given subtitle

long CSubtitles::getEndTimeBefore(CSubtitle * currentSub) const
{
    QPtrListIterator<CSubtitle> it = getIterator();
    CSubtitle* sub;
    while ( (sub = it.current() ) != 0 )
    {
      ++it;
      if(*currentSub == *sub)
      {
        //get the previous sub

        --it;
        --it;
        if( (sub = it.current()) != 0 )
        {
          return sub->getEndTime();
        }
        else
        {
          return -1;
        }
        break;
      }
    }
    return 0;
}

//returns the starttime of the subtitle after the given subtitle

long CSubtitles::getStartTimeAfter(CSubtitle * currentSub) const
{

    QPtrListIterator<CSubtitle> it = getIterator();
    CSubtitle* sub;
    while ( (sub = it.current() ) != 0 )
    {
      ++it;
      if(*currentSub == *sub)
      {
        //try to get the next sub
        if( (sub = it.current()) != 0 )
        {
          return sub->getStartTime();
        }
        else
        {
          return 86400000;
        }

      }
    }
    return 86400000;
}

void CSubtitles::subDump( QTextStream & stream, SubFormat format) const
{
  QPtrListIterator<CSubtitle> it( *this );
  CSubtitle* sub;
  if(format == subFormatSRT)
  {
    int i = 0;
    while ( (sub = it.current() ) != 0 )
    {
        ++it;
        i++;
        stream << QString::number(i) << "\n";
        stream << sub->getStartTime_QTime().toString( QString ( "hh:mm:ss,zzz" ));
	stream << " --> ";
        stream << sub->getEndTime_QTime().toString( QString ( "hh:mm:ss,zzz" )) << "\n";
	stream << sub->getLine1() << "\n";
        if(sub->getLine2() != "")
        {
          stream << sub->getLine2() << "\n";
        }
        stream << "\n";
    }
  }
  if(format == subFormatSUB)
  {
      //save as sub format
  }
}

bool CSubtitles::stretchTime(long newTime, CSubtitle * stretchSub)
{
    //get starttime of sub that needs to be stretched from
    long subTime = stretchSub->getStartTime();

    //get amount of subs that are before 'sub'
    long subsbefore = 0;
    QPtrListIterator<CSubtitle> it( *this );
    CSubtitle *sub;
    while ( (sub = it.current() ) != 0 )
    {
        ++it;
        if(*sub == *stretchSub)
        {
            break;
        }
        subsbefore++;
    }
    //calculate amount after
    long subsafter = getNrOfSubtitles() - subsbefore - 1;  // -1 is to not calculate the stretchSub in
    if(newTime >= subsbefore)
    {
        //ok, newTime is larger than the amount of subs before stretchSub
        //this means we can shrink the existing subs in for sure
    }
    else
    {
        return false;
    }

    if(newTime <= 86399999 - subsafter)
    {
        //ok, newTime is small enough so we can have the amount of subs
        //after stretchSub fit in our maximum range
    }
    else
    {
        return false;
    }

	//we need to remember this value because we change it already in the shrink parts and we need it in the expand parts too!
	long stretchSubStartTime = 0;
	stretchSubStartTime = stretchSub->getStartTime();


    if(newTime < subTime)
    {
        //we stretch to the left on a timescale
        //shrink left part, then expand right part
        //   |---|----|     or     |------|
        //   |-|------|          |--------|

        // shrink left part
        //   |---|
        //   |-|
        if(subsbefore != 0)
        {
            long startTimeLast_orig = stretchSub->getStartTime();
            long startTimeFirst_orig = this->getFirstSubtitle()->getStartTime();
            long startTime_new = newTime;
            double ratio = (double) (startTime_new - startTimeFirst_orig)  / (double) (startTimeLast_orig - startTimeFirst_orig);
	        long counter = 0;
	        it.toFirst();
            while ( (sub = it.current() ) != 0 )
            {
                ++it;
                long newStartTime = long( ( (double) (sub->getStartTime() - startTimeFirst_orig) )*ratio + 0.5);
		        long newEndTime = long( ((double) (sub->getEndTime() - startTimeFirst_orig) )*ratio + 0.5);
                sub->setStartTime( newStartTime + startTimeFirst_orig);
                sub->setEndTime( newEndTime + startTimeFirst_orig );
		        counter++;
                if(counter == subsbefore + 1)  // +1 because the stretchSub must also be processed
		        {
                    //no need to go further, we shrinked  all subs before stretchsub including stretchsub
		            break;
                }
            }
        }

        // expand right part to the left
        //       |----|
        //     |------|
        if(subsafter != 0)
        {
            long startTimeLast_orig = this->getLastSubtitle()->getStartTime();
            long startTimeFirst_orig = stretchSubStartTime;
            long startTime_new = newTime;
            double ratio = (double) (startTimeLast_orig - startTime_new)  / (double) (startTimeLast_orig - startTimeFirst_orig);
            //if it is not the first sub we like to stretch,
            // we do not want the stretchSub to be stretched again
            //because it is already stretched in 'shrink left part'
            int notFirstsubCorrection = 0;
            if(subsbefore != 0)
            {
                notFirstsubCorrection = 1;
            }

            long counter = 0;
            it.toFirst();
            while ( (sub = it.current() ) != 0 )
            {
                ++it;
                counter++;
                if(counter > subsbefore + notFirstsubCorrection)  //we skip the first subs which are already shrinked
		        {
                    long newStartTime = long( ( (double) (sub->getStartTime() -startTimeFirst_orig) )*ratio + 0.5);
		            long newEndTime = long(( (double) (sub->getEndTime() - startTimeFirst_orig) )*ratio + 0.5);
                    sub->setStartTime( newStartTime + startTime_new );
                    sub->setEndTime( newEndTime + startTime_new );
		        }
            }
        }

    }
    else
    {
        //we stretch to the right on a timescale
        //shrink right part, then expand left part
        //   |---|----|     or     |------|
        //   |------|-|            |--------|

        //shrink right part
		//       |----|
		//          |-|
        if(subsafter != 0)
		{
            long startTimeLast_orig = this->getLastSubtitle()->getStartTime();
            long startTimeFirst_orig = stretchSub->getStartTime();
            long startTime_new = newTime;
            double ratio = (double) (startTimeLast_orig - startTime_new )  / (double) (startTimeLast_orig - startTimeFirst_orig);
	        long counter = 0;
	        it.toLast();
            while ( (sub = it.current() ) != 0 )
            {
                --it;
                long newStartTime = long( ( (double) (startTimeLast_orig - sub->getStartTime() ) )*ratio + 0.5);
		        long newEndTime = long( ((double) (startTimeLast_orig - sub->getEndTime()) )*ratio + 0.5);
                sub->setStartTime( startTimeLast_orig - newStartTime);
                sub->setEndTime( startTimeLast_orig - newEndTime );
		        counter++;
                if(counter == subsafter + 1)  // +1 because the stretchSub must also be processed
		        {
                    //no need to go further, we shrinked  all subs after stretchsub(including stretchsub)
		            break;
                }
            }
		}

		//expand left part
		//   |---|
		//   |------|
        if(subsbefore != 0)
        {
            long startTimeLast_orig = stretchSubStartTime;
            long startTimeFirst_orig = this->getFirstSubtitle()->getStartTime();
            long startTime_new = newTime;
            double ratio = (double) (startTime_new - startTimeFirst_orig)  / (double) (startTimeLast_orig - startTimeFirst_orig);
            //if it is the last sub we like to stretch,
            // we the stretchSub to be stretched
            //because it is not yet stretched in 'shrink right part'
            int lastsubCorrection = 0;
            if(subsafter == 0)
            {
                lastsubCorrection = 1;
            }

            long counter = 0;
            it.toFirst();
            while ( (sub = it.current() ) != 0 )
            {
                ++it;
                counter++;

                if(counter <= subsbefore + lastsubCorrection)  //we skip the last subs which are already shrinked
		{
		    long newStartTime = long( ( (double) (sub->getStartTime() - startTimeFirst_orig) )*ratio + 0.5);
		    long newEndTime = long(( (double) (sub->getEndTime() - startTimeFirst_orig) )*ratio + 0.5);
                    sub->setStartTime( newStartTime + startTimeFirst_orig );
                    sub->setEndTime( newEndTime + startTimeFirst_orig );
		}
            }
        }

    }
    return true;

}

bool CSubtitles::addSubtitle(CSubtitle *newSubtitle){

  QPtrListIterator<CSubtitle> it( *this );
  CSubtitle *sub;
  while ( (sub = it.current() ) != 0 )
  {
      ++it;
      if(newSubtitle->getStartTime() >= sub->getStartTime()
        && newSubtitle->getStartTime() <= sub->getEndTime()
      )
      {
        //new:---------------|xxxxxxx|-
        //old:--|xxx|------|xxxxx|-----
        //the start of the new subtitle lies between the beginning and end of an existing one
	/*cout << "geval 1 - ";
        cout << (newSubtitle->getStartTime()).toString().latin1();
        cout << (sub->getStartTime()).toString().latin1();
        cout << (newSubtitle->getStartTime()).toString().latin1();
        cout << (sub->getEndTime()).toString().latin1();  */
        return false;
      }
      if(newSubtitle->getEndTime() >= sub->getStartTime()
        && newSubtitle->getEndTime() <= sub->getEndTime()
      )
      {
        //new:--------|xxxxxxxx|-------
        //old:--|xxx|------|xxxxx|-----
        //the end of the new subtitle lies between the beginning and end of an existing one
	//        cout << "case 2";
        return false;
      }
      if(newSubtitle->getStartTime() < sub->getStartTime()
        && newSubtitle->getEndTime() > sub->getEndTime()
      )
      {
        //new:--------|xxxxxxxxxxxx|---
        //old:--|xxx|------|xxxxx|-----
        //the new subtitles includes a whole subtitle
        //        cout << "case 3";
        return false;
      }
  }
  this->inSort( newSubtitle );
//  this->sort();
  return true;
}

bool CSubtitles::addSubtitleWithWrongTiming(CSubtitle *newSubtitle){

  QPtrListIterator<CSubtitle> it( *this );
  CSubtitle *sub;
  while ( (sub = it.current() ) != 0 )
  {
      ++it;
      if(newSubtitle->getStartTime() >= sub->getStartTime()
        && newSubtitle->getStartTime() <= sub->getEndTime()
      )
      {
        //new:---------------|xxxxxxx|-
        //old:--|xxx|------|xxxxx|-----
        //the start of the new subtitle lies between the beginning and end of an existing one

		//find the starttime of the next sub:
		long next = getStartTimeAfter(sub);

		//the new starttime should be just after this sub, that is, if the next sub is not interfering
		if(next != sub->getEndTime() + 1)
        {
            //check if endtime of newSubtitle will not be smaller than starttime of newSubtitle!
			//example: new: --------|XX|---
			//         old: ------|XXXXXX|-
            if(newSubtitle->getEndTime() < sub->getEndTime() + 1)
			{
			    //the endtime is smaller than the starttime
                long length = newSubtitle->getEndTime() - newSubtitle->getStartTime();
				if(sub->getEndTime() + 1 + length < next)
				{
                    //move endtime and starttime
					newSubtitle->setEndTime(sub->getEndTime() + 1 + length);
                    newSubtitle->setStartTime(sub->getEndTime() + 1);
				}
				else
				{
                    newSubtitle->setEndTime(next - 1);
                    newSubtitle->setStartTime(sub->getEndTime() + 1);
				}
			}
			else
			{
			    //ok, move the starttime so it fits
			    newSubtitle->setStartTime(sub->getEndTime() + 1);
			}
		}
		else
		{
            //there is a sub interfering, we can't set the starttime immediately after this sub, so don't add this sub!
			return false;
		}

      }
      if(newSubtitle->getEndTime() >= sub->getStartTime()
        && newSubtitle->getEndTime() <= sub->getEndTime()
      )
      {
        //new:--------|xxxxxxxx|-------
        //old:--|xxx|------|xxxxx|-----
        //the end of the new subtitle lies between the beginning and end of an existing one

		long previous = getEndTimeBefore(sub);

        if(previous != sub->getStartTime() - 1)
		{
            //new endtime of newSubtitle will never be smaller than starttime of newSubtitle because
			//the above if-statement takes care of this
            newSubtitle->setEndTime(sub->getStartTime() - 1);
		}
		else
		{
		    //we can't change the endtime of the new sub to just before the sub who's interfering
            return false;
		}

      }
      if(newSubtitle->getStartTime() < sub->getStartTime()
        && newSubtitle->getEndTime() > sub->getEndTime()
      )
      {
        //new:--------|xxxxxxxxxxxx|---
        //old:--|xxx|------|xxxxx|-----
        //the new subtitle includes a whole subtitle
        //btw, this situation can not happen here(resolved in the first if-statement):
		//new: ------|xxx|-----
		//old: ------|xxx|-----

        //determine on what side we should add it
		long right = newSubtitle->getEndTime() - sub->getEndTime();
		long left =  sub->getStartTime() - newSubtitle->getStartTime();
		if(right >= left)
		{
		    //shrink to right
            long next = getStartTimeAfter(sub);
			if(next != sub->getEndTime() + 1)
			{
			    newSubtitle->setStartTime(sub->getEndTime() + 1);
			}
			else
			{
                //can't set this to the right, try to the left
				long previous = getEndTimeBefore(sub);
				if(previous != sub->getStartTime() - 1)
				{
                    //ok we set it on the left
					newSubtitle->setEndTime(sub->getStartTime() - 1);
				}
				else
				{
                    //we can't set it on the left and on the right
                    return false;
				}
			}
		}
        else
		{
            //shrink to the left
    		long previous = getEndTimeBefore(sub);
			if(previous != sub->getStartTime() - 1)
			{
                //ok we set it on the left
				newSubtitle->setEndTime(sub->getStartTime() - 1);
			}
			else
			{
			    //can't set this to the left, try to the right
				long next = getStartTimeAfter(sub);
				if(next != sub->getEndTime() + 1)
				{
				    //ok set it on the right
					newSubtitle->setStartTime(sub->getEndTime() + 1);
				}
				else
				{
                    return false;
				}
			}
		}
      }
  }
  this->inSort( newSubtitle );
//  this->sort();
  return true;
}


int CSubtitles::compareItems(QCollection::Item item1, QCollection::Item item2)
{
  //printf("----sort--------");
  CSubtitle* sub1 = (CSubtitle*) item1;
  CSubtitle* sub2 = (CSubtitle*) item2;
  if(*sub1 == *sub2)
  {
    return 0;
  }
  if(*sub2 < *sub1)
  {
    return 1;
  }
  return -1;
}

bool CSubtitles::removeSubtitle(CSubtitle *removeSub)
{
    bool ad = autoDelete();
    setAutoDelete(true);
    return this->removeRef(removeSub);
    setAutoDelete(ad);
}

bool CSubtitles::insertSubtitles( CSubtitles* newSubs, long insertTime)
{
    //We expect the CSubtitle objects to already be unique, so we don't make a (deep) copy prior to adding them
    //we also expect that insertTime is not on a existing subtitle
    
    //move the existing subtitles after insertTime to the right
    if(getNrOfSubtitles() != 0)
    {
	long moveAmount = newSubs->getTotalOccupiedTime();
    
	//optimize for the most common cases(insert before first and insert after last)
	if(insertTime < getFirstSubtitle()->getStartTime())
	{
	    //move everything
	    moveSubtitles(getFirstSubtitle()->getStartTime() + moveAmount, getFirstSubtitle(), getLastSubtitle());
	}
	else if(insertTime > getLastSubtitle()->getEndTime())
	{
	    //move nothing
	}
	else
	{
	    //search sub
	    QPtrListIterator<CSubtitle> it( *this );
	    CSubtitle* sub;
	    while ( (sub = it.current() ) != 0 )
	    {
		if(sub->getEndTime() > insertTime)
		{
		    //we have found the sub, move this sub and all subsequent subs
		    moveSubtitles(sub->getStartTime() + moveAmount, sub, getLastSubtitle());
		    break;
		}
		++it;
	    }
	}
    }
    
    //insert subtitles at the specified position
    QPtrListIterator<CSubtitle> insertIt( *newSubs );
    CSubtitle* insertSub;
    while ( (insertSub = insertIt.current() ) != 0 )
    {
	insertSub->setEndTime(insertSub->getEndTime() + insertTime);
	insertSub->setStartTime(insertSub->getStartTime() + insertTime);
	insert(0, insertSub);	
	++insertIt;
    }
    this->sort();
    
    return true;
}


//move function to move subtitles - note: no error checking, the GUI takes care of checking wether the input is valid

void CSubtitles::moveSubtitles( long newStartTime, CSubtitle * firstSub, CSubtitle * lastSub )
{
    long moveSecs = newStartTime - firstSub->getStartTime();

    bool change = false;
    QPtrListIterator<CSubtitle> it( *this );
    CSubtitle* sub;
    while ( (sub = it.current() ) != 0 )
    {
        //if we are at the first sub, we must move it and the following
        if(!change && *sub == *firstSub)
        {
            change = true;
        }
        //we move the subs
        if(change)
        {
            sub->setStartTime(sub->getStartTime() + moveSecs);
            sub->setEndTime(sub->getEndTime() + moveSecs);
        }
        //if we have moved the lastSub then we can stop the while loop
        if(change && *sub == *lastSub)
        {
            break;
        }
        ++it;
    }

}

void CSubtitles::dumpSubs()
{
  printf("dump of subtitles\n\n");

  QPtrListIterator<CSubtitle> it( *this );
  CSubtitle* sub;
  int i = 0;
  while ( (sub = it.current() ) != 0 )
  {
      ++it;
      i++;
      printf("%s", QString::number(i).latin1());
      printf(" - ");
      printf("%s", sub->getStartTime_QTime().toString( QString ( "hh:mm:ss,zzz" )).latin1());
      printf(" --> ");
      printf("%s", sub->getEndTime_QTime().toString( QString ( "hh:mm:ss,zzz" )).latin1());
      printf(" --- ");
      printf("%s", sub->getLine1().latin1());
      printf(" --- ");
      printf("%s", sub->getLine2().latin1());
      printf("\n");
  }
}
