#ifndef CONFIG_DIALOG_H
#define CONFIG_DIALOG_H

#include <qasciidict.h>
#include <qdict.h>
#include <qhbox.h>
#include <qkeysequence.h>
#include <qlineedit.h>
#include <qpushbutton.h>
#include <qstringlist.h>
#include <qvbox.h>

#include "config_file.h"

class QApplication;
class QCheckBox;
class QComboBox;
class QGrid;
class QHButtonGroup;
class QHGroupBox;
class QLabel;
class QListBox;
class QListView;
class QScrollView;
class QSlider;
class QSpinBox;
class QTextEdit;
class QVGroupBox;
class QVButtonGroup;

/**
    Kontrolka umoliwiajaca wybranie skrtu klawiszowego
**/
class HotKey : public QLineEdit
{
	public:
		HotKey(QWidget *parent =0, const char* name =0);
		/**
		  Pobiera skrt klawiszowy
		**/
		QString getShortCutString() const;

		/**
		  Pobiera skrt klawiszowy
		**/
		QKeySequence getShortCut() const;

		/**
		  Ustawia skrt klawiszowy
		**/
		void setShortCut(const QString& shortcut);

		/**
		  Ustawia skrt klawiszowy
		**/
		void setShortCut(const QKeySequence& shortcut);

		/**
		 Zwraca skrt klawiszowy jaki zosta nacinity przy zdarzeniu QKeyEvent
		**/
		static QString keyEventToString(QKeyEvent *e);

		/**
		 Pobiera skrt klawiszowy, z pliku konfiguracyjnego
		 z grupy "groupname", o polu "name"
		**/
		static QKeySequence shortCutFromFile(const QString& groupname, const QString &name);

		/**
		 Sprawdza czy skrt nacinity przy zdarzeniu QKeyEvent zgadza si
		 ze skrtem klawiszowym, z pliku konfiguracyjnego, o polu "name"
		**/
		static bool shortCut(QKeyEvent *e, const QString& groupname, const QString &name);
	protected:
		virtual void keyPressEvent(QKeyEvent *e);
		virtual void keyReleaseEvent(QKeyEvent *e);
};
/**
    Kontrolka do wyboru koloru
**/
class ColorButton : public QPushButton
{
	Q_OBJECT
	private:
		QColor actualcolor;
	public:
		/**
			Konstruktor obiektu w ktrym ustawia sie kolor "color",
			ktry ma rodzica "parent" oraz nazywa sie "name"
		**/
		ColorButton(const QColor &color, QWidget *parent =0, const char* name =0);

		/**
			Pobiera kolor z przycisku
		**/
		const QColor &color() const;

	private slots:
		void onClick();
	public slots:
		/**
			Ustawia kolor na przycisku
		**/
		void setColor(const QColor &color);
	signals:
		/**
			Sygna jest emitowany przy zmianie koloru, przesya on nazw komponentu oraz nowy kolor
		**/
		void changed(const char *name, const QColor& color);
};

class SelectFont : public QHBox
{
	Q_OBJECT
	public:
		SelectFont(const char *text, const QFont &def_val, QWidget *parent = 0,  const char *name = 0, const char *tip = 0);
		const QFont &font() const;
	public slots:
		void setFont(const QFont &font);
	private slots:
		void onClick();
	signals:
		/**
			Sygna jest emitowany przy zmianie czcionki, przesya on nazw komponentu oraz now czcionk
		**/
		void changed(const char *name, const QFont& font);
	private:
		QFont currentFont;
		QLineEdit *fontEdit;
};

class LayoutHelper;
class QResizeEvent;

class SelectPaths : public QHBox
{
	Q_OBJECT
	protected:
		QStringList releaseList;
		QListBox *pathListBox;
		QPushButton *pb_ok, *pb_cancel;
		QPushButton *pb_add, *pb_change, *pb_remove;
		QLineEdit *pathEdit;
		QPushButton *pb_choose;
		LayoutHelper *layoutHelper;

		virtual void resizeEvent(QResizeEvent *);

	protected slots:
		void okButton();
		void cancelButton();
		void addPath();
		void replacePath();
		void deletePath();
		void choosePath();
		void closeEvent(QCloseEvent *e);
		void keyPressEvent(QKeyEvent *e);

	public:
		SelectPaths(QWidget *parent =0, const char* name =0);
		~SelectPaths();
		/**
			Pobiera list cieek katalogw
		**/
		const QStringList &getPathList() const;

	public slots:
		/**
			Ustawia cieki z katalogami
		**/
		void setPathList(const QStringList& list);

	signals:

		/**
			Sygna jest emitowany przy zamkniciu okna przez "ok",
			przesya on zmienion list cieek katalogw
	    **/
		void changed(const QStringList& paths);
};

enum UserLevel {
	Beginner,
	Advanced,
	Expert
};

/**
	Okno z konfiguracj Kadu
**/
class ConfigDialog : public QVBox
{
	Q_OBJECT

	private:

		enum RegisteredControlType
		{
			CONFIG_CHECKBOX,
			CONFIG_COLORBUTTON,
			CONFIG_COMBOBOX,
			CONFIG_GRID,
			CONFIG_HBOX,
			CONFIG_HGROUPBOX,
			CONFIG_HRADIOGROUP,
			CONFIG_HOTKEYEDIT,
			CONFIG_LABEL,
			CONFIG_LINEEDIT,
			CONFIG_LINEEDIT2,
			CONFIG_TEXTEDIT,
			CONFIG_LISTBOX,
			CONFIG_LISTVIEW,
			CONFIG_PUSHBUTTON,
			CONFIG_RADIOGROUP,
			CONFIG_SELECTPATHS,
			CONFIG_SLIDER,
			CONFIG_SPINBOX,
			CONFIG_TAB,
			CONFIG_VBOX,
			CONFIG_VGROUPBOX,
			CONFIG_VRADIOGROUP,
			CONFIG_SELECTFONT,
			CONFIG_NULL
		};

		struct ElementConnections
		{
			ElementConnections(const char *signal, const QObject *receiver, const char *slot);
			ElementConnections();
			const char *signal;
			const QObject* receiver;
			const char *slot;
			bool operator== (const ElementConnections& r) const;
		};
		struct RegisteredControl
		{
			RegisteredControlType type;
			RegisteredControl(RegisteredControlType t = CONFIG_NULL,
				const char *groupname = 0,
				const char *parent = 0,
				const char *caption = 0,
				const char *name = 0,
				const UserLevel userLevelRequired = Beginner);
			const char *parent;
			const char *name;
			const char *caption;
			const char *group;
			const char *entry;
			QString defaultS;
			const char *tip;

			//dodatkowe dane
			QValueList<QVariant> additionalParams;
			QValueListIterator<RegisteredControl> parentControl;
			QWidget *widget;

			//w przypadku kontrolek skadajcych si z kilku mniejszych,
			// wskanik na rodzica, w przypadku pojedynczej kontrolki == widget
			QWidget *entireWidget;

			int nrOfControls;
			UserLevel userLevelRequired;

			ConfigFile* config;
			QValueList <ElementConnections> ConnectedSlots;
		};

		static QAsciiDict<QValueList <ElementConnections> > SlotsOnCreateTab;
		static QAsciiDict<QValueList <ElementConnections> > SlotsOnCloseTab;
		static QAsciiDict<QValueList <ElementConnections> > SlotsOnApplyTab;
		static QApplication* appHandle;

		static QAsciiDict<QValueList <RegisteredControl> > Tabs;
		static QValueList<const char *> TabNames;
		static QAsciiDict<int> TabSizes;
		static QDict<QSignal> tabChangesIn, tabChangesOut;//jakie lepsze nazwy by si przyday ;)


		QVGroupBox* box;
		/**
		    Dodaje kontrolk do listy
		**/
		static bool addControl(const char *groupname, ConfigDialog::RegisteredControl& control);

		void createWidget(QValueListIterator <RegisteredControl> control);
		void createTabAndWidgets(const char *tab);

	public:
		/**
		    Konstruktor
		**/

		ConfigDialog(QApplication *application, QWidget *parent = 0, const char *name = 0);
		~ConfigDialog();
		/**
		    Pokazuje okno konfiguracji
		**/
		static void showConfigDialog(QApplication* application);

		/**
		    Dodaje kontrolk do zakadki "groupname",
			Rodzicem kontrolki jest kontrolka "parent".
			Ustawia text kontrolki na "caption".
			Warto kontrolki jest zapisana do pliku konfiguracyjnego "config" w postaci
			-------------
			[groupname]
			entry= value {defaultS}
			-------------

			Domyln wartoci kontrolki przy pierwszym uruchomieniu programu
			jest "defaultS".
			Podpowied kontrolki ustawiona jest na "tip".
			Nazwa kontrolki ustawiona jest na "name".
		**/
 		static void addCheckBox(
				const char *groupname, const char *parent, const char *caption,
			    const char *entry, const bool defaultS = false, const char *tip = 0,
				const char *name = 0,
				UserLevel userLevelRequired = Beginner);
		static void addCheckBox(ConfigFile* config,
				const char *groupname, const char *parent, const char *caption,
			    const char *entry, const bool defaultS = false, const char *tip = 0,
				const char *name = 0,
				UserLevel userLevelRequired = Beginner);

		/**
		    Dodaje kontrolk do zakadki "groupname",
			Rodzicem kontrolki jest kontrolka "parent".
			Ustawia text kontrolki na "caption".
			Kolor przycisku jest ustawiony na "color"
			Podpowied kontrolki ustawiona jest na "tip".
			Nazwa kontrolki ustawiona jest na "name".
		**/
		static void addColorButton(
				const char *groupname, const char *parent, const char *caption,
				const char *entry, const QColor& color, const char *tip = 0,
				const char *name = 0,
				UserLevel userLevelRequired = Beginner);
		static void addColorButton(ConfigFile *config,
				const char *groupname, const char *parent, const char *caption,
				const char *entry, const QColor& color, const char *tip = 0,
				const char *name = 0,
				UserLevel userLevelRequired = Beginner);

		/**
		    Dodaje kontrolk do zakadki "groupname",
			Rodzicem kontrolki jest kontrolka "parent".
			Ustawia text kontrolki na "caption".
			Podpowied kontrolki ustawiona jest na "tip".
			Nazwa kontrolki ustawiona jest na "name".
		**/
		static void addComboBox(
				const char *groupname, const char *parent, const char *caption,
			    const char *tip = 0, const char *name = 0,
				UserLevel userLevelRequired = Beginner);

		/**
		    Dodaje kontrolk do zakadki "groupname",
			Rodzicem kontrolki jest kontrolka "parent".
			Ustawia text kontrolki na "caption".

			Warto kontrolki jest zapisana do pliku konfiguracyjnego "config" w postaci
			-------------
			[groupname]
			entry= value {defaultS}
			-------------
			Lista opcji do wyboru to "options", ktrym w pliku konfiguracyjnym odpowiadaj "values".
			Podpowied kontrolki ustawiona jest na "tip".
			Nazwa kontrolki ustawiona jest na "name".
		**/
		static void addComboBox(
				const char *groupname, const char *parent, const char *caption,
				const char *entry, const QStringList &options, const QStringList &values,
				const QString &defaultS = QString::null, const char *tip = 0, const char *name = 0,
				UserLevel userLevelRequired = Beginner);
		static void addComboBox(ConfigFile* config,
				const char *groupname, const char *parent, const char *caption,
				const char *entry, const QStringList &options, const QStringList &values,
				const QString &defaultS = QString::null, const char *tip = 0, const char *name = 0,
				UserLevel userLevelRequired = Beginner);

		/**
		    Dodaje kontrolk do zakadki "groupname",
			Rodzicem kontrolki jest kontrolka "parent".
			Ustawia text kontrolki na "caption".
			Ilo kolumn w kontrolce ustwiona jest na nrColumns
			Nazwa kontrolki ustawiona jest na "name".
		**/

		static void addGrid(
				const char *groupname, const char *parent, const char *caption,
				const int nrColumns = 3, const char *name = 0,
				UserLevel userLevelRequired = Beginner);

		/**
		    Dodaje kontrolk do zakadki "groupname",
			Rodzicem kontrolki jest kontrolka "parent".
			Ustawia text kontrolki na "caption".
			Nazwa kontrolki ustawiona jest na "name".
		**/

		static void addHBox(
				const char *groupname, const char *parent, const char *caption,
				const char *name = 0,
				UserLevel userLevelRequired = Beginner);

		/**
		    Dodaje kontrolk do zakadki "groupname",
			Rodzicem kontrolki jest kontrolka "parent".
			Ustawia text kontrolki na "caption".
			Nazwa kontrolki ustawiona jest na "name".
		**/

		static void addHGroupBox(
				const char *groupname, const char *parent, const char *caption,
				const char *name = 0,
				UserLevel userLevelRequired = Beginner);


		/**
		    Dodaje kontrolk do zakadki "groupname",
			Rodzicem kontrolki jest kontrolka "parent".
			Ustawia text kontrolki na "caption".
			Warto kontrolki jest zapisana do pliku konfiguracyjnego "config" w postaci
			-------------
			[groupname]
			entry= value {defaultS}
			-------------

			Domyln wartoci kontrolki przy pierwszym uruchomieniu programu
			jest "defaultS".
			Podpowied kontrolki ustawiona jest na "tip".
			Nazwa kontrolki ustawiona jest na "name".
		**/

		static void addHotKeyEdit(
				const char *groupname, const char *parent, const char *caption,
			    const char *entry, const QString& defaultS = QString::null, const char *tip = 0,
				const char *name = 0,
				UserLevel userLevelRequired = Beginner);
		static void addHotKeyEdit(ConfigFile* config,
				const char *groupname, const char *parent, const char *caption,
			    const char *entry, const QString& defaultS = QString::null, const char *tip = 0,
				const char *name = 0,
				UserLevel userLevelRequired = Beginner);

		/**
		    Dodaje kontrolk do zakadki "groupname",
			Rodzicem kontrolki jest kontrolka "parent".
			Ustawia text kontrolki na "caption".

			Warto kontrolki jest zapisana do pliku konfiguracyjnego "config" w postaci
			-------------
			[groupname]
			entry= value {defaultS}
			-------------
			Lista opcji do wyboru to "options", ktrym w pliku konfiguracyjnym odpowiadaj "values".
			Podpowied kontrolki ustawiona jest na "tip".
			Nazwa kontrolki ustawiona jest na "name".
		**/
		static void addHRadioGroup(
				const char *groupname, const char *parent, const char *caption,
				const char *entry, const QStringList &options, const QStringList &values,
				const QString &defaultS = QString::null, const char *tip = 0, const char *name = 0,
				UserLevel userLevelRequired = Beginner);
		static void addHRadioGroup(ConfigFile* config,
				const char *groupname, const char *parent, const char *caption,
				const char *entry, const QStringList &options, const QStringList &values,
				const QString &defaultS = QString::null, const char *tip = 0, const char *name = 0,
				UserLevel userLevelRequired = Beginner);

		/**
		    Dodaje kontrolk do zakadki "groupname",
			Rodzicem kontrolki jest kontrolka "parent".
			Ustawia text kontrolki na "caption".
			Nazwa kontrolki ustawiona jest na "name".
		**/

		static void addLabel(
				const char *groupname, const char *parent, const char *caption,
				const char *name = 0,
				UserLevel userLevelRequired = Beginner);

		/**
		    Dodaje kontrolk do zakadki "groupname",
			Rodzicem kontrolki jest kontrolka "parent".
			Ustawia text kontrolki na "caption".
			Warto kontrolki jest zapisana do pliku konfiguracyjnego "config" w postaci
			-------------
			[groupname]
			entry= value {defaultS}
			-------------

			Domyln wartoci kontrolki przy pierwszym uruchomieniu programu
			jest "defaultS".
			Podpowied kontrolki ustawiona jest na "tip".
			Nazwa kontrolki ustawiona jest na "name".
		**/

		static void addLineEdit(
				const char *groupname, const char *parent, const char *caption,
			    const char *entry, const QString& defaultS = QString::null, const char *tip = 0,
				const char *name = 0,
				UserLevel userLevelRequired = Beginner);
		static void addLineEdit(ConfigFile* config,
				const char *groupname, const char *parent, const char *caption,
			    const char *entry, const QString& defaultS = QString::null, const char *tip = 0,
				const char *name = 0,
				UserLevel userLevelRequired = Beginner);


		/**
		    Dodaje kontrolk do zakadki "groupname",
			Rodzicem kontrolki jest kontrolka "parent".
			Ustawia text kontrolki na "caption".
			Domyln wartoci kontrolki ustawion jest na "defaultS".
			Podpowied kontrolki ustawiona jest na "tip".
			Nazwa kontrolki ustawiona jest na "name".
		**/

		static void addLineEdit2(
				const char *groupname, const char *parent, const char *caption,
			    const QString& defaultS = QString::null, const char *tip = 0, const char *name = 0,
				UserLevel userLevelRequired = Beginner);

		/**
		    Dodaje kontrolk do zakadki "groupname",
			Rodzicem kontrolki jest kontrolka "parent".
			Ustawia text kontrolki na "caption".
			Warto kontrolki jest zapisana do pliku konfiguracyjnego "config" w postaci
			-------------
			[groupname]
			entry= value {defaultS}
			-------------

			Domyln wartoci kontrolki przy pierwszym uruchomieniu programu
			jest "defaultS".
			Podpowied kontrolki ustawiona jest na "tip".
			Nazwa kontrolki ustawiona jest na "name".
		**/

		static void addTextEdit(
				const char *groupname, const char *parent, const char *caption,
			    const char *entry, const QString& defaultS = QString::null, const char *tip = 0,
				const char *name = 0,
				UserLevel userLevelRequired = Beginner);
		static void addTextEdit(ConfigFile* config,
				const char *groupname, const char *parent, const char *caption,
			    const char *entry, const QString& defaultS = QString::null, const char *tip = 0,
				const char *name = 0,
				UserLevel userLevelRequired = Beginner);


		/**
		    Dodaje kontrolk do zakadki "groupname",
			Rodzicem kontrolki jest kontrolka "parent".
			Ustawia text kontrolki na "caption".
			Podpowied kontrolki ustawiona jest na "tip".
			Nazwa kontrolki ustawiona jest na "name".
		**/

		static void addListBox(
				const char *groupname, const char *parent, const char *caption,
				const char *tip = 0, const char *name = 0,
				UserLevel userLevelRequired = Beginner);

		/**
		    Dodaje kontrolk do zakadki "groupname",
			Rodzicem kontrolki jest kontrolka "parent".
			Ustawia text kontrolki na "caption".
			Podpowied kontrolki ustawiona jest na "tip".
			Nazwa kontrolki ustawiona jest na "name".
		**/

		static void addListView(
				const char *groupname, const char *parent, const char *caption,
				const char *tip = 0, const char *name = 0,
				UserLevel userLevelRequired = Beginner);

		/**
		    Dodaje kontrolk do zakadki "groupname",
			Rodzicem kontrolki jest kontrolka "parent".
			Ustawia text kontrolki na "caption".
			Podpowied kontrolki ustawiona jest na "tip".
			Nazwa kontrolki ustawiona jest na "name".
			@param iconname nazwa ikony z zestawu lub cieka do pliku
		**/

		static void addPushButton(
				const char *groupname, const char *parent, const char *caption,
			    const QString& iconFilename = QString::null, const char *tip = 0, const char *name = 0,
				UserLevel userLevelRequired = Beginner);

		/**
		    Dodaje kontrolk do zakadki "groupname",
			Rodzicem kontrolki jest kontrolka "parent".
			Ustawia text kontrolki na "caption".

			Warto kontrolki jest zapisana do pliku konfiguracyjnego "config" w postaci
			-------------
			[groupname]
			entry= value {defaultS}
			-------------
			Lista opcji do wyboru to "options", ktrym w pliku konfiguracyjnym odpowiadaj "values".
			Podpowied kontrolki ustawiona jest na "tip".
			Nazwa kontrolki ustawiona jest na "name".
		**/
		static void addRadioGroup(
				const char *groupname, const char *parent, const char *caption,
				const char *entry, const QStringList &options, const QStringList &values,
				int strips, Orientation orientation,
				const QString &defaultS = QString::null, const char *tip = 0, const char *name = 0,
				UserLevel userLevelRequired = Beginner);
		static void addRadioGroup(ConfigFile* config,
				const char *groupname, const char *parent, const char *caption,
				const char *entry, const QStringList &options, const QStringList &values,
				int strips, Orientation orientation,
				const QString &defaultS = QString::null, const char *tip = 0, const char *name = 0,
				UserLevel userLevelRequired = Beginner);

		/**
		    Dodaje kontrolk do zakadki "groupname",
			Rodzicem kontrolki jest kontrolka "parent".
			Ustawia tekst kontrolki na "caption".
			Warto kontrolki jest zapisana do pliku konfiguracyjnego "config" w postaci
			-------------
			[groupname]
			entry= value {defaultS}
			-------------

			Domyln wartoci kontrolki przy pierwszym uruchomieniu programu
			jest "defaultS".
			Podpowied kontrolki ustawiona jest na "tip".
			Nazwa kontrolki ustawiona jest na "name".
		**/

		static void addSelectFont(
				const char *groupname, const char *parent, const char *caption,
				const char *entry, const QString& defaultS = QString::null, const char *tip = 0,
				const char *name = 0,
				UserLevel userLevelRequired = Beginner);
		static void addSelectFont(ConfigFile *config,
				const char *groupname, const char *parent, const char *caption,
				const char *entry, const QString& defaultS = QString::null, const char *tip = 0,
				const char *name = 0,
				UserLevel userLevelRequired = Beginner);

		/**
		    Dodaje kontrolk do zakadki "groupname",
			Rodzicem kontrolki jest kontrolka "parent".
			Ustawia text kontrolki na "caption".
			Nazwa kontrolki ustawiona jest na "name".
		**/

		static void addSelectPaths(
				const char *groupname, const char *parent, const char *caption,
				const char *name = 0,
				UserLevel userLevelRequired = Beginner);

		/**
		    Dodaje kontrolk do zakadki "groupname",
			Rodzicem kontrolki jest kontrolka "parent".
			Ustawia text kontrolki na "caption".
			Warto kontrolki jest zapisana do pliku konfiguracyjnego "config" w postaci
			-------------
			[groupname]
			entry= value {defaultS}
			-------------
			Minimalna warto kontrolki ustawiona jest na "minValue", maksymalna
			na "maxValue", krok jest ustawiony na "pageStep"
			Domyln wartoci kontrolki przy pierwszym uruchomieniu programu
			jest "value"
			Podpowied kontrolki ustawiona jest na "tip".
			Nazwa kontrolki ustawiona jest na "name".
		**/

		static void addSlider(
				const char *groupname, const char *parent, const char *caption,
			    const char *entry, const int minValue = 0, const int maxValue = 100,
			    const int pageStep = 1, const int value = 50, const char *tip = 0,
				const char *name = 0,
				UserLevel userLevelRequired = Beginner);
		static void addSlider(ConfigFile* config,
				const char *groupname, const char *parent, const char *caption,
			    const char *entry, const int minValue = 0, const int maxValue = 100,
			    const int pageStep = 1, const int value = 50, const char *tip = 0,
				const char *name = 0,
				UserLevel userLevelRequired = Beginner);


		/**
		    Dodaje kontrolk do zakadki "groupname",
			Rodzicem kontrolki jest kontrolka "parent".
			Ustawia text kontrolki na "caption".
			Warto kontrolki jest zapisana do pliku konfiguracyjnego "config" w postaci
			-------------
			[groupname]
			entry= value {defaultS}
			-------------
			Minimalna warto kontrolki ustawiona jest na "minValue", maksymalna
			na "maxValue", krok jest ustawiony na "step"
			Domyln wartoci kontrolki przy pierwszym uruchomieniu programu
			jest "value"
			Podpowied kontrolki ustawiona jest na "tip".
			Nazwa kontrolki ustawiona jest na "name".
		**/

		static void addSpinBox(
				const char *groupname, const char *parent, const char *caption,
			    const char *entry, const int minValue = 0, const int maxValue = 100,
				const int step = 1, const int value = 50, const char *tip = 0,
				const char *name = 0,
				UserLevel userLevelRequired = Beginner);
		static void addSpinBox(ConfigFile* config,
				const char *groupname, const char *parent, const char *caption,
			    const char *entry, const int minValue = 0, const int maxValue = 100,
				const int step = 1, const int value = 50, const char *tip = 0,
				const char *name = 0,
				UserLevel userLevelRequired = Beginner);

		/**
		    Dodaje zakadk o nazwie "caption"
		    @param iconname nazwa ikony z zestawu lub cieka do pliku
		**/
		static void addTab(const char *caption, const QString& iconFilename = QString::null,
			UserLevel userLevelRequired = Beginner);

		/**
		    Dodaje kontrolk do zakadki "groupname",
			Rodzicem kontrolki jest kontrolka "parent".
			Ustawia text kontrolki na "caption".
			Nazwa kontrolki ustawiona jest na "name".
		**/

		static void addVBox(
				const char *groupname, const char *parent, const char *caption,
				const char *name = 0,
				UserLevel userLevelRequired = Beginner);

		/**
		    Dodaje kontrolk do zakadki "groupname",
			Rodzicem kontrolki jest kontrolka "parent".
			Ustawia text kontrolki na "caption".
			Nazwa kontrolki ustawiona jest na "name".
		**/

		static void addVGroupBox(
				const char *groupname, const char *parent, const char *caption,
				const char *name = 0,
				UserLevel userLevelRequired = Beginner);

		/**
		    Dodaje kontrolk do zakadki "groupname",
			Rodzicem kontrolki jest kontrolka "parent".
			Ustawia text kontrolki na "caption".

			Warto kontrolki jest zapisana do pliku konfiguracyjnego "config" w postaci
			-------------
			[groupname]
			entry= value {defaultS}
			-------------
			Lista opcji do wyboru to "options", ktrym w pliku konfiguracyjnym odpowiadaj "values".
			Podpowied kontrolki ustawiona jest na "tip".
			Nazwa kontrolki ustawiona jest na "name".
		**/
		static void addVRadioGroup(
				const char *groupname, const char *parent, const char *caption,
				const char *entry, const QStringList &options, const QStringList &values,
				const QString &defaultS = QString::null, const char *tip = 0, const char *name = 0,
				UserLevel userLevelRequired = Beginner);
		static void addVRadioGroup(ConfigFile* config,
				const char *groupname, const char *parent, const char *caption,
				const char *entry, const QStringList &options, const QStringList &values,
				const QString &defaultS = QString::null, const char *tip = 0, const char *name = 0,
				UserLevel userLevelRequired = Beginner);

		/**
		    Pobiera wskanik do kontrolki ButtonGroup(groupname, caption, name)
			jeeli okno konfiguracji jest zamknite zwraca NULL
		**/
		static QButtonGroup* getButtonGroup(const char *groupname, const char *caption, const char *name = 0);

		/**
		    Pobiera wskanik do kontrolki CheckBox(groupname, caption, name)
			jeeli okno konfiguracji jest zamknite zwraca NULL
		**/
		static QCheckBox*   getCheckBox(const char *groupname, const char *caption, const char *name = 0);

		/**
		    Pobiera wskanik do kontrolki ColorButton(groupname, caption, name)
			jeeli okno konfiguracji jest zamknite zwraca NULL
		**/
		static ColorButton* getColorButton(const char *groupname, const char *caption, const char *name = 0);

		/**
		    Pobiera wskanik do kontrolki ComboBox(groupname, caption, name)
			jeeli okno konfiguracji jest zamknite zwraca NULL
		**/
		static QComboBox*   getComboBox(const char *groupname, const char *caption, const char *name = 0);

		/**
		    Pobiera wskanik do kontrolki Grid(groupname, caption, name)
			jeeli okno konfiguracji jest zamknite zwraca NULL
		**/
		static QGrid*       getGrid(const char *groupname, const char *caption, const char *name = 0);

		/**
		    Pobiera wskanik do kontrolki HBox(groupname, caption, name)
			jeeli okno konfiguracji jest zamknite zwraca NULL
		**/
		static QHBox* 	    getHBox(const char *groupname, const char *caption, const char *name = 0);

		/**
		    Pobiera wskanik do kontrolki HGroupBox(groupname, caption, name)
			jeeli okno konfiguracji jest zamknite zwraca NULL
		**/
		static QHGroupBox*  getHGroupBox(const char *groupname, const char *caption, const char *name = 0);

		/**
		    Pobiera wskanik do kontrolki HButtonGroup(groupname, caption, name)
			jeeli okno konfiguracji jest zamknite zwraca NULL
		**/
		static QHButtonGroup* getHButtonGroup(const char *groupname, const char *caption, const char *name = 0);

		/**
		    Pobiera wskanik do kontrolki HotKeyEdit(groupname, caption, name)
			jeeli okno konfiguracji jest zamknite zwraca NULL
		**/
		static HotKey* 	    getHotKeyEdit(const char *groupname, const char *caption, const char *name = 0);

		/**
		    Pobiera wskanik do kontrolki LineEdit(groupname, caption, name)
			jeeli okno konfiguracji jest zamknite zwraca NULL
		**/
		static QLineEdit*   getLineEdit(const char *groupname, const char *caption, const char *name = 0);

		/**
		    Pobiera wskanik do kontrolki TextEdit(groupname, caption, name)
			jeeli okno konfiguracji jest zamknite zwraca NULL
		**/
		static QTextEdit*   getTextEdit(const char *groupname, const char *caption, const char *name = 0);

		/**
		    Pobiera wskanik do kontrolki Label(groupname, caption, name)
			jeeli okno konfiguracji jest zamknite zwraca NULL
		**/
		static QLabel*      getLabel(const char *groupname, const char *caption, const char *name = 0);

		/**
		    Pobiera wskanik do kontrolki ListBox(groupname, caption, name)
			jeeli okno konfiguracji jest zamknite zwraca NULL
		**/
		static QListBox*    getListBox(const char *groupname, const char *caption, const char *name = 0);

		/**
		    Pobiera wskanik do kontrolki ListView(groupname, caption, name)
			jeeli okno konfiguracji jest zamknite zwraca NULL
		**/
		static QListView*   getListView(const char *groupname, const char *caption, const char *name = 0);

		/**
		    Pobiera wskanik do kontrolki PushButton(groupname, caption, name)
			jeeli okno konfiguracji jest zamknite zwraca NULL
		**/
		static QPushButton* getPushButton(const char *groupname, const char *caption, const char *name = 0);

		/**
		    Pobiera wskanik do kontrolki SelectFont(groupname, caption, name)
			jeeli okno konfiguracji jest zamknite zwraca NULL
		**/
		static SelectFont* getSelectFont(const char *groupname, const char *caption, const char *name = 0);

		/**
		    Pobiera wskanik do kontrolki SelectPaths(groupname, caption, name)
			jeeli okno konfiguracji jest zamknite zwraca NULL
		**/
		static SelectPaths* getSelectPaths(const char *groupname, const char *caption, const char *name = 0);

		/**
		    Pobiera wskanik do kontrolki Slider(groupname, caption, name)
			jeeli okno konfiguracji jest zamknite zwraca NULL
		**/
		static QSlider*     getSlider(const char *groupname, const char *caption, const char *name = 0);

		/**
		    Pobiera wskanik do kontrolki SpinBox(groupname, caption, name)
			jeeli okno konfiguracji jest zamknite zwraca NULL
		**/
		static QSpinBox*    getSpinBox(const char *groupname, const char *caption, const char *name = 0);

		/**
		    Pobiera wskanik do kontrolki VBox(groupname, caption, name)
			jeeli okno konfiguracji jest zamknite zwraca NULL
		**/
		static QVBox*  	    getVBox(const char *groupname, const char *caption, const char *name = 0);

		/**
		    Pobiera wskanik do kontrolki VGroupBox(groupname, caption, name)
			jeeli okno konfiguracji jest zamknite zwraca NULL
		**/
		static QVGroupBox*  getVGroupBox(const char *groupname, const char *caption, const char *name = 0);

		/**
		    Pobiera wskanik do kontrolki VButtonGroup(groupname, caption, name)
			jeeli okno konfiguracji jest zamknite zwraca NULL
		**/
		static QVButtonGroup* getVButtonGroup(const char *groupname, const char *caption, const char *name = 0);

		/**
		    W przypadku kontrolki zawierajcej kilka mniejszych (z czego jedn z nich zwraca jedna z funkcji getXXX)
			zwraca wskanik do niej, w przeciwnym wypadku zwraca t (pojedyncz) kontrolk
			jeeli okno konfiguracji jest zamknite zwraca NULL
			UWAGA: nie wolno rzutowa zwracanego wskanika na adn klas dziedziczc pod QWidget!
			(np. addSpinBox dodaje QHBox zawierajc QLabel i QSpinBox, getSpinBox() zwrci SpinBoksa, getEntireWidget QHBoksa())
		**/
		static QWidget* getEntireWidget(const char *groupname, const char *caption, const char *name = 0);

		/**
		    Pobiera wskanik do kontrolki Widget(groupname, caption, name)
			jeeli okno konfiguracji jest zamknite zwraca NULL
		**/
		static QWidget* widget(const char *groupname, const char *caption, const char *name = 0);

		/**
			zwraca informacj czy kontrolka (groupname, caption, name) jest w konfiguracji
			przy okazji przekazujc jej iterator jeeli control != 0
		**/
		static bool controlExists(const char *groupname, const char *caption, const char *name = 0, QValueListConstIterator<RegisteredControl> *control = 0);

		/**
			zwraca informacj czy kontrolka (groupname, caption, name) jest w konfiguracji
			przy okazji przekazujc jej iterator jeeli control != 0
		**/
		static bool controlExists(const char *groupname, const char *caption, const char *name, QValueListIterator<RegisteredControl> *control);

		/**
		    Wyszukuje pozycj zakadki "name",
			zwraca przez parametry jej iterator oraz informacj o znalezieniu jeeli found != 0
		**/
		static void tab(const char *name, QValueListConstIterator<RegisteredControl> &ret, bool *found = 0);

		/**
		    Wyszukuje pozycj zakadki "name",
			zwraca przez parametry jej iterator oraz informacj o znalezieniu jeeli found != 0
		**/
		static void tab(const char *name, QValueListIterator<RegisteredControl> &ret, bool *found = 0);

		/**
			Podcza slot "slot" obiektu "receiver"
			pod sygna "signal" kontrolki (groupname, caption, name)
		**/
		static void connectSlot(const char *groupname, const char *caption, const char* signal,
			    const QObject* receiver, const char* slot,const char *name = 0);

		/**
			Odcza slot "slot" obiektu "receiver"
			od sygnau "signal" kontrolki (groupname, caption, name)
		**/
		static void disconnectSlot(const char *groupname, const char *caption, const char* signal,
			    const QObject* receiver, const char* slot,const char *name = 0);

		/**
			Rejestruje slot "slot" obiektu "receiver",
			ktry jest wywoywany przytworzeniu zakladki
		**/
		static void registerSlotOnCreateTab(const char *tab, const QObject* receiver, const char* slot);


		/**
			Wyrejestrowuje slot "slot" obiektu "receiver",
			ktry jest wywoywany przytworzeniu zakladki
		**/

		static void unregisterSlotOnCreateTab(const char *tab, const QObject* receiver, const char* slot);


		/**
			Rejestruje slot "slot" obiektu "receiver",
			ktry jest wywoywany przy zamykaniu okna konfiguracji
		**/
		static void registerSlotOnCloseTab(const char *tab, const QObject* receiver, const char* slot);

		/**
			Wyrejestrowuje slot "slot" obiektu "receiver",
			ktry jest wywoywany przy zamykaniu okna konfiguracji
		**/
		static void unregisterSlotOnCloseTab(const char *tab, const QObject* receiver, const char* slot);

		/**
			Rejestruje slot "slot" obiektu "receiver",
			ktry jest wywoywany przy zapisywaniu konfiguracji
		**/
		static void registerSlotOnApplyTab(const char *tab, const QObject* receiver, const char* slot);

		/**
			Wyrejestrowuje slot "slot" obiektu "receiver",
			ktry jest wywoywany przy zapisywaniu konfiguracji
		**/
		static void unregisterSlotOnApplyTab(const char *tab, const QObject* receiver, const char* slot);

		/**
			Rejestruje 2 sloty obiektu "receiver", ktre s
			wywoywane przy zmianie zakadki "name"
			slotIn - przy zmianie aktywnej zakadki na "name" (moe by == 0)
			slotOut - przy zmianie aktywnej zakadki z "name" na inn (moe by == 0)
		**/
		static void registerSlotsOnTabChange(const char *name, const QObject *receiver, const char *slotIn, const char *slotOut);

		/**
			Wyrejestrowuje 2 sloty obiektu "receiver", ktre s
			wywoywane przy zmianie zakadki "name"
			slotIn - przy zmianie aktywnej zakadki na "name" (moe by == 0)
			slotOut - przy zmianie aktywnej zakadki z "name" na inn (moe by == 0)
		**/
		static void unregisterSlotsOnTabChange(const char *name, const QObject *receiver, const char *slotIn, const char *slotOut);

		/**
		    Usuwa kontrolk z zakadki "groupname", o etykiecie "caption" i nazwie "name".
		**/
		static void removeControl(const char *groupname, const char *caption, const char *name = 0);

		/**
		    Usuwa zakadk o nazwie caption
		**/
		static bool removeTab(const char *caption);

		/**
			Zwraca prawd jeeli zakadka znajduje si w konfiguracji
		**/
		static bool tabExists(const char *caption);

		static bool dialogOpened();
		static void closeDialog();

		//uywa tylko w wyjtkowych sytuacjach
		static ConfigDialog *configdialog;

	protected:
		UserLevel userLevel;
		QListBox* listBox;
		QScrollView* view;
		QPushButton* okButton;
		QPushButton* applyButton;
		QPushButton* cancelButton;
		static QCString currentTab;
		virtual void keyPressEvent(QKeyEvent *e);
		void updateUserLevel(const QValueList<ConfigDialog::RegisteredControl> &tab);



	signals:

		void createTab();
		/**
		    Sygna jest emitowany przy otwieraniu okna
		**/
		void create();

		/**
		    Sygna jest emitowany przy zamykaniu okna
		**/
		void destroy();

		/**
		   Sygna jest emitowany przy zapisywaniu konfiguracji
		**/
		void apply();

	protected slots:
		/**
		    Zapisuje wartoci kontrolek do pliku
		**/

		void updateConfig();

		/**
		    Zapisuje wartoci kontrolek do pliku i zamyka okno
		**/

		void updateAndCloseConfig();

		/**
		    Zmienia zakadk na "name"
		**/
		void changeTab(const QString& name);

		/**
		    Ukrywa kontrolki o poziomie mniejszym od newUserLevel
		**/
		void changeUserLevel(int newUserLevel);
};
#endif
