/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qmap.h>
#include <qlineedit.h>
#include <qprocess.h>

#include "config_file.h"
#include "config_dialog.h"
#include "debug.h"
#include "history.h"
#include "misc.h"
#include "exec_notify.h"

extern "C" int exec_notify_init()
{
	kdebugf();
	exec_notify = new ExecNotify(NULL, "exec_notify");
	kdebugf2();
	return 0;
}

extern "C" void exec_notify_close()
{
	kdebugf();
	delete exec_notify;
	exec_notify = NULL;
	kdebugf2();
}

ExecNotify::ExecNotify(QObject *parent, const char *name) : Notifier(parent, name)
{
	kdebugf();

	ConfigDialog::addTab(QT_TRANSLATE_NOOP("@default", "Exec Notify"));
	config_file.addVariable("Exec Notify", "NewChatCmd", "Xdialog --msgbox \"%protocol %id %ids %action\" 10 100");
	config_file.addVariable("Exec Notify", "NewMessageCmd", "Xdialog --msgbox \"%protocol %id %ids %action\" 10 100");
	config_file.addVariable("Exec Notify", "ConnErrorCmd", "Xdialog --msgbox \"%protocol %action\" 10 100");
	config_file.addVariable("Exec Notify", "toAvailableCmd", "Xdialog --msgbox \"%protocol %id %action\" 10 100");
	config_file.addVariable("Exec Notify", "toBusyCmd", "Xdialog --msgbox \"%protocol %id %action\" 10 100");
	config_file.addVariable("Exec Notify", "toInvisibleCmd", "Xdialog --msgbox \"%protocol %id %action\" 10 100");
	config_file.addVariable("Exec Notify", "toNotAvailableCmd", "Xdialog --msgbox \"%protocol %id %action\" 10 100");
	config_file.addVariable("Exec Notify", "StatusChangedCmd", "Xdialog --msgbox \"%protocol %id %action %oldstatus -> %status\" 10 100");

	ConfigDialog::addLineEdit("Exec Notify", "Exec Notify", QT_TRANSLATE_NOOP("@default", "New chat command"), "NewChatCmd", QString::null, 0, 0, Advanced);
	ConfigDialog::addLineEdit("Exec Notify", "Exec Notify", QT_TRANSLATE_NOOP("@default", "New message command"), "NewMessageCmd", QString::null, 0, 0, Advanced);
	ConfigDialog::addLineEdit("Exec Notify", "Exec Notify", QT_TRANSLATE_NOOP("@default", "Connection error command"), "ConnErrorCmd", QString::null, 0, 0, Advanced);
	ConfigDialog::addLineEdit("Exec Notify", "Exec Notify", QT_TRANSLATE_NOOP("@default", "Status change to \"available\" command"), "toAvailableCmd", QString::null, 0, 0, Advanced);
	ConfigDialog::addLineEdit("Exec Notify", "Exec Notify", QT_TRANSLATE_NOOP("@default", "Status change to \"busy\" command"), "toBusyCmd", QString::null, 0, 0, Advanced);
	ConfigDialog::addLineEdit("Exec Notify", "Exec Notify", QT_TRANSLATE_NOOP("@default", "Status change to \"invisible\" command"), "toInvisibleCmd", QString::null, 0, 0, Advanced);
	ConfigDialog::addLineEdit("Exec Notify", "Exec Notify", QT_TRANSLATE_NOOP("@default", "Status change to \"not available\" command"), "toNotAvailableCmd", QString::null, 0, 0, Advanced);
	ConfigDialog::addLineEdit("Exec Notify", "Exec Notify", QT_TRANSLATE_NOOP("@default", "Status changed command"), "StatusChangedCmd", QString::null, 0, 0, Advanced);

	QMap<QString, QString> s;
	s["NewChat"] = SLOT(newChat(Protocol *, UserListElements, const QString &, time_t));
	s["NewMessage"] = SLOT(newMessage(Protocol *, UserListElements, const QString &, time_t, bool &));
	s["ConnError"] = SLOT(connectionError(Protocol *, const QString &));
	s["toAvailable"] = SLOT(userChangedStatusToAvailable(const QString &, UserListElement));
	s["toBusy"] = SLOT(userChangedStatusToBusy(const QString &, UserListElement));
	s["toInvisible"] = SLOT(userChangedStatusToInvisible(const QString &, UserListElement));
	s["toNotAvailable"] = SLOT(userChangedStatusToNotAvailable(const QString &, UserListElement));
	s["StatusChanged"] = SLOT(userStatusChanged(UserListElement, QString, const UserStatus &));

	config_file.addVariable("Notify", "NewChat_Exec", false);
	config_file.addVariable("Notify", "NewMessage_Exec", false);
	config_file.addVariable("Notify", "ConnError_Exec", false);
	config_file.addVariable("Notify", "ChangingStatus_Exec", false);
	config_file.addVariable("Notify", "toAvailable_Exec", false);
	config_file.addVariable("Notify", "toBusy_Exec", false);
	config_file.addVariable("Notify", "toInvisible_Exec", false);
	config_file.addVariable("Notify", "toNotAvailable_Exec", false);
	config_file.addVariable("Notify", "Message_Exec", false);

	notify->registerNotifier(QT_TRANSLATE_NOOP("@default", "Exec"), this, s);

//	ConfigDialog::registerSlotOnCreateTab("Exec", this, SLOT(onCreateConfigDialog()));
//	ConfigDialog::registerSlotOnApplyTab("Exec", this, SLOT(onApplyConfigDialog()));
//	ConfigDialog::registerSlotOnCloseTab("Exec", this, SLOT(onCloseConfigDialog()));

	kdebugf2();
}

ExecNotify::~ExecNotify()
{
	kdebugf();

	ConfigDialog::removeControl("Exec Notify", "New chat command");
	ConfigDialog::removeControl("Exec Notify", "New message command");
	ConfigDialog::removeControl("Exec Notify", "Connection error command");
	ConfigDialog::removeControl("Exec Notify", "Status change to \"available\" command");
	ConfigDialog::removeControl("Exec Notify", "Status change to \"busy\" command");
	ConfigDialog::removeControl("Exec Notify", "Status change to \"invisible\" command");
	ConfigDialog::removeControl("Exec Notify", "Status change to \"not available\" command");
	ConfigDialog::removeControl("Exec Notify", "Status changed command");

	notify->unregisterNotifier("Exec");

//	ConfigDialog::unregisterSlotOnCreateTab("Exec", this, SLOT(onCreateConfigDialog()));
//	ConfigDialog::unregisterSlotOnApplyTab("Exec", this, SLOT(onApplyConfigDialog()));
//	ConfigDialog::unregisterSlotOnCloseTab("Exec", this, SLOT(onCloseConfigDialog()));

	ConfigDialog::removeTab("Exec Notify");

	kdebugf2();
}

/*void ExecNotify::onCreateConfigDialog()
{
	kdebugf();

	kdebugf2();
}

void ExecNotify::onCloseConfigDialog()
{
	kdebugf();

	kdebugf2();
}

void ExecNotify::onApplyConfigDialog()
{
	kdebugf();

	kdebugf2();
}*/

void ExecNotify::newChat(Protocol *protocol, UserListElements senders, const QString &msg, time_t /*t*/)
{
	kdebugf();
	QString syntax = config_file.readEntry("Exec Notify", "NewChatCmd");
	if (syntax.isEmpty())
	{
		kdebugf2();
		return;
	}
	QStringList s = HistoryManager::mySplit(' ', syntax);
	FOREACH(it, s)
	{
		if ((*it).contains("%action"))
			(*it).replace("%action", "NewChat");
		if ((*it).contains("%protocol"))
			(*it).replace("%protocol", protocol->protocolID());
		if ((*it).contains("%ids"))
		{
			QStringList sndrs;
			CONST_FOREACH(sndr, senders)
				sndrs.append((*sndr).ID(protocol->protocolID()));
			(*it).replace("%ids", sndrs.join(","));
		}
		if ((*it).contains("%id"))
			(*it).replace("%id", senders[0].ID(protocol->protocolID()));
	}
	run(s, msg);

	kdebugf2();
}

void ExecNotify::newMessage(Protocol *protocol, UserListElements senders, const QString &msg, time_t /*t*/, bool &/*grab*/)
{
	kdebugf();

	QString syntax = config_file.readEntry("Exec Notify", "NewMessageCmd");
	if (syntax.isEmpty())
	{
		kdebugf2();
		return;
	}
	QStringList s = HistoryManager::mySplit(' ', syntax);
	FOREACH(it, s)
	{
		if ((*it).contains("%action"))
			(*it).replace("%action", "NewMessage");
		if ((*it).contains("%protocol"))
			(*it).replace("%protocol", protocol->protocolID());
		if ((*it).contains("%ids"))
		{
			QStringList sndrs;
			CONST_FOREACH(sndr, senders)
				sndrs.append((*sndr).ID(protocol->protocolID()));
			(*it).replace("%ids", sndrs.join(","));
		}
		if ((*it).contains("%id"))
			(*it).replace("%id", senders[0].ID(protocol->protocolID()));
	}
	run(s, msg);

	kdebugf2();
}

void ExecNotify::connectionError(Protocol *protocol, const QString &message)
{
	kdebugf();

	QString syntax = config_file.readEntry("Exec Notify", "ConnErrorCmd");
	if (syntax.isEmpty())
	{
		kdebugf2();
		return;
	}
	QStringList s = HistoryManager::mySplit(' ', syntax);
	FOREACH(it, s)
	{
		if ((*it).contains("%action"))
			(*it).replace("%action", "ConnError");
		if ((*it).contains("%protocol"))
			(*it).replace("%protocol", protocol->protocolID());
	}
	run(s, message);

	kdebugf2();
}

void ExecNotify::userStatusChanged(UserListElement ule, QString protocolName, const UserStatus &oldStatus)
{
	kdebugf();

	QString syntax = config_file.readEntry("Exec Notify", "StatusChangedCmd");
	if (syntax.isEmpty())
	{
		kdebugf2();
		return;
	}
	QStringList s = HistoryManager::mySplit(' ', syntax);
	FOREACH(it, s)
	{
		if ((*it).contains("%action"))
			(*it).replace("%action", "StatusChanged");
		if ((*it).contains("%protocol"))
			(*it).replace("%protocol", protocolName);
		if ((*it).contains("%id"))
			(*it).replace("%id", ule.ID(protocolName));
		if ((*it).contains("%status"))
			(*it).replace("%status", ule.status(protocolName).toString());
		if ((*it).contains("%oldstatus"))
			(*it).replace("%oldstatus", oldStatus.toString());
	}
	run(s, QString::null);

	kdebugf2();
}

void ExecNotify::userChangedStatusToAvailable(const QString &protocolName, UserListElement ule)
{
	kdebugf();

	QString syntax = config_file.readEntry("Exec Notify", "toAvailableCmd");
	if (syntax.isEmpty())
	{
		kdebugf2();
		return;
	}
	QStringList s = HistoryManager::mySplit(' ', syntax);
	FOREACH(it, s)
	{
		if ((*it).contains("%action"))
			(*it).replace("%action", "toAvailable");
		if ((*it).contains("%protocol"))
			(*it).replace("%protocol", protocolName);
		if ((*it).contains("%id"))
			(*it).replace("%id", ule.ID(protocolName));
	}
	run(s, QString::null);

	kdebugf2();
}

void ExecNotify::userChangedStatusToBusy(const QString &protocolName, UserListElement ule)
{
	kdebugf();

	QString syntax = config_file.readEntry("Exec Notify", "toBusyCmd");
	if (syntax.isEmpty())
	{
		kdebugf2();
		return;
	}
	QStringList s = HistoryManager::mySplit(' ', syntax);
	FOREACH(it, s)
	{
		if ((*it).contains("%action"))
			(*it).replace("%action", "toBusy");
		if ((*it).contains("%protocol"))
			(*it).replace("%protocol", protocolName);
		if ((*it).contains("%id"))
			(*it).replace("%id", ule.ID(protocolName));
	}
	run(s, QString::null);

	kdebugf2();
}

void ExecNotify::userChangedStatusToInvisible(const QString &protocolName, UserListElement ule)
{
	kdebugf();

	QString syntax = config_file.readEntry("Exec Notify", "toInvisibleCmd");
	if (syntax.isEmpty())
	{
		kdebugf2();
		return;
	}
	QStringList s = HistoryManager::mySplit(' ', syntax);
	FOREACH(it, s)
	{
		if ((*it).contains("%action"))
			(*it).replace("%action", "toInvisible");
		if ((*it).contains("%protocol"))
			(*it).replace("%protocol", protocolName);
		if ((*it).contains("%id"))
			(*it).replace("%id", ule.ID(protocolName));
	}
	run(s, QString::null);

	kdebugf2();
}

void ExecNotify::userChangedStatusToNotAvailable(const QString &protocolName, UserListElement ule)
{
	kdebugf();

	QString syntax = config_file.readEntry("Exec Notify", "toNotAvailableCmd");
	if (syntax.isEmpty())
	{
		kdebugf2();
		return;
	}
	QStringList s = HistoryManager::mySplit(' ', syntax);
	FOREACH(it, s)
	{
		if ((*it).contains("%action"))
			(*it).replace("%action", "toNotAvailable");
		if ((*it).contains("%protocol"))
			(*it).replace("%protocol", protocolName);
		if ((*it).contains("%id"))
			(*it).replace("%id", ule.ID(protocolName));
	}
	run(s, QString::null);

	kdebugf2();
}

void ExecNotify::externalEvent(const QString &/*notifyType*/, const QString &/*msg*/, const UserListElements &/*ules*/)
{
}

void ExecNotify::run(const QStringList &args, const QString &stdin)
{
#ifdef DEBUG_ENABLED
	CONST_FOREACH(arg, args)
		kdebugm(KDEBUG_INFO, "arg: '%s'\n", (*arg).local8Bit().data());
	kdebugm(KDEBUG_INFO, "stdin: %s\n", stdin.local8Bit().data());
#endif
	QProcess *p = new QProcess(args);
	connect(p, SIGNAL(processExited()), p, SLOT(deleteLater()));
	p->launch(stdin.local8Bit());
}

ExecNotify *exec_notify = NULL;
