/*
 * rfb_protocol.h - declaration of class rfbConnection, an implementation of the RFB-protocol, used by VNC servers
 *
 * iTALC
 * Copyright (c) 2004-2005 Tobias Doerffel <tobias@doerffel.de>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program (see COPYING); if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 */

#ifndef _RFB_PROTOCOL_H
#define _RFB_PROTOCOL_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qobject.h>
#include <qstring.h>
#include <qimage.h>

#ifdef HAVE_ARPA_INET_H
#include <arpa/inet.h>
#endif

#ifdef HAVE_LIBZ
#include <zlib.h>
#endif

#ifdef HAVE_LIBJPEG
namespace jpeglib
{
	extern "C"
	{		// otherwise jpeg-lib doesn't work with C++ source...
		#include <jpeglib.h>
	}
}
#endif

#include "italc_rfb_ext.h"
#include "rsa_crypt_lib.h"



extern int endian_test;

#define Swap16IfLE(s) (*(char *)&endian_test ? ((((s) & 0xff) << 8) | (((s) >> 8) & 0xff)) : (s))

#define Swap32IfLE(l) \
    (*(char *)&endian_test ? ((((l) & 0xff000000) >> 24) |	\
			     (((l) & 0x00ff0000) >> 8)  |	\
			     (((l) & 0x0000ff00) << 8)  |	\
			     (((l) & 0x000000ff) << 24))  : (l))



const int VNC_SERVER_PORT_OFFSET = 5900;
const int MAX_ENCODINGS  = 10;


const unsigned int RFBP_MAJOR_VERSION = 3;
const unsigned int RFBP_MINOR_VERSION = 3;

const unsigned int RFBP_VERSION_MESSAGE_LEN = 12;


const int RFB_CONNECTION_FAILED = 0;
const int RFB_NO_AUTHENTICATION = 1;
const int RFB_VNC_AUTHENTICATION = 2;

const int RFB_AUTH_OK = 0;
const int RFB_AUTH_FAILED = 1;
const int RFB_AUTH_TOO_MANY = 2;



/*****************************************************************************
 *
 * Message types
 *
 *****************************************************************************/

// server -> client

const Q_UINT8 RFB_FRAMEBUFFER_UPDATE = 0;
const Q_UINT8 RFB_SET_COLORMAP_ENTRIES = 1;
const Q_UINT8 RFB_BELL = 2;
const Q_UINT8 RFB_SERVER_CUT_TEXT = 3;



// client -> server

const Q_UINT8 RFB_SET_PIXEL_FORMAT = 0;
const Q_UINT8 RFB_FIX_COLORMAP_ENTRIES = 1; // not currently supported
const Q_UINT8 RFB_SET_ENCODINGS = 2;
const Q_UINT8 RFB_FRAMEBUFFER_UPDATE_REQUEST = 3;
const Q_UINT8 RFB_KEY_EVENT = 4;
const Q_UINT8 RFB_POINTER_EVENT = 5;
const Q_UINT8 RFB_CLIENT_CUT_TEXT = 6;



/*****************************************************************************
 *
 * Encoding types (we do not support all of them, because corre-encoding
 * and tight-encoding are the most important (and best) encodings...
 * if all doesn't work, we fall back to raw-encoding...
 *
 *****************************************************************************/

const Q_UINT32 RFB_ENCODIG_RAW = 0;
const Q_UINT32 RFB_ENCODING_COPYRECT = 1;
const Q_UINT32 RFB_ENCODIG_RRE = 2;		// not supported yet
const Q_UINT32 RFB_ENCODING_CORRE = 4;
const Q_UINT32 RFB_ENCODING_HEXTILE = 5;	// not supported yet
const Q_UINT32 RFB_ENCODING_ZLIB = 6;		// not supported yet
const Q_UINT32 RFB_ENCODING_TIGHT = 7;
const Q_UINT32 RFB_ENCODING_ZLIBHEX = 8;	// not supported yet

/*
 * Special encoding numbers:
 *   0xFFFFFF00 .. 0xFFFFFF0F -- encoding-specific compression levels;
 *   0xFFFFFF10 .. 0xFFFFFF1F -- mouse cursor shape data;
 *   0xFFFFFF20 .. 0xFFFFFF2F -- various protocol extensions;
 *   0xFFFFFF30 .. 0xFFFFFFDF -- not allocated yet;
 *   0xFFFFFFE0 .. 0xFFFFFFEF -- quality level for JPEG compressor;
 *   0xFFFFFFF0 .. 0xFFFFFFFF -- cross-encoding compression levels.
 */

const Q_UINT32 RFB_ENCODING_COMPRESS_LEVEL_0 = 0xFFFFFF00;
const Q_UINT32 RFB_ENCODING_X_CURSOR = 0xFFFFFF10;
const Q_UINT32 RFB_ENCODING_RICH_CURSOR = 0xFFFFFF11;
const Q_UINT32 RFB_ENCODING_POINTER_POS = 0xFFFFFF18;
const Q_UINT32 RFB_ENCODING_LAST_RECT = 0xFFFFFF20;
const Q_UINT32 RFB_ENCODING_QUALITY_LEVEL_0 = 0xFFFFFFE0;





struct s_rfb_pixel_format
{
	Q_UINT8 bits_per_pixel;		// 8,16,32 only
	Q_UINT8 depth;			// 8 to 32
	Q_UINT8 big_endian;		// True if multi-byte pixels are interpreted as big endian, or if
					// single-bit-per-pixel has most significant bit of the byte corresponding to
					// first (leftmost) pixel. Of course this is meaningless for 8 bits/pix
	Q_UINT8 true_color;		// If false then we need a "colour map" to convert pixels to RGB.
					// If true, xxxMax and xxxShift specify bits used for red, green and blue

// the following fields are only meaningful if true_color is true

	Q_UINT16 red_max;		// maximum red value (= 2^n - 1 where n is the number of bits used for red).
					// Note this value is always in big endian order.
	Q_UINT16 green_max;		// similar for green
	Q_UINT16 blue_max;		// and blue
	Q_UINT8 red_shift;		// number of shifts needed to get the red value in a pixel to the least
					// significant bit. To find the red value from a given pixel, do the following:
					//	1) Swap pixel value according to big_endian (e.g. if big_endian is false
					//	   and host byte order is big endian, then swap).
					//	2) Shift right by red_shift.
					// 	3) AND with red_max (in host byte order).
					// 4) You now have the red value between 0 and red_max.
	Q_UINT8 green_shift;		// similar for green
	Q_UINT8 blue_shift;		// and blue

	Q_UINT8 pad1;
	Q_UINT8 pad2;
} ;



struct s_rfb_client_init_msg
{
	Q_UINT8 shared;
} ;



struct s_rfb_server_init_msg
{
	Q_UINT16 framebuffer_width;
	Q_UINT16 framebuffer_height;
	s_rfb_pixel_format format;	// the server's preferred pixel format
	Q_UINT32 name_length;
	// followed by char name[nameLength]
} ;


struct s_rfb_set_pixel_format_msg
{
	Q_UINT8 type;			// always RFB_SET_PIXEL_FORMAT
	Q_UINT8 pad1;
	Q_UINT16 pad2;
	s_rfb_pixel_format format;
} ;



struct s_rfb_set_encodings_msg
{
	Q_UINT8 type;			// always RFB_SET_ENCODINGS
	Q_UINT8 pad;
	Q_UINT16 n_encodings;
    // followed by nEncodings * Q_UINT32 encoding types
} ;



struct s_rfb_framebuffer_update_request_msg
{
	Q_UINT8 type;			// always RFB_FRAMEBUFFER_UPDATE_REQUEST
	Q_UINT8 incremental;
	Q_UINT16 x;
	Q_UINT16 y;
	Q_UINT16 w;
	Q_UINT16 h;
} ;



struct s_rfb_pointer_event_msg
{
	Q_UINT8 type;			// always RFB_POINTER_EVENT
	Q_UINT8 button_mask;		// bits 0-7 are buttons 1-8, 0=up, 1=down
	Q_UINT16 x;
	Q_UINT16 y;
} ;

const Q_UINT8 RFB_BUTTON1_MASK = 1;
const Q_UINT8 RFB_BUTTON2_MASK = 2;
const Q_UINT8 RFB_BUTTON3_MASK = 4;



struct s_rfb_key_event_msg
{
	Q_UINT8 type;			// always RFB_KEY_EVENT
	Q_UINT8 down;			// true if down (press), false if up
	Q_UINT16 pad;
	Q_UINT32 key;			// key is specified as an X keysym
} ;








/*****************************************************************************
 *
 * Server -> client message definitions
 *
 *****************************************************************************/


/*-----------------------------------------------------------------------------
 * FramebufferUpdate - a block of rectangles to be copied to the framebuffer.
 *
 * This message consists of a header giving the number of rectangles of pixel
 * data followed by the rectangles themselves.  The header is padded so that
 * together with the type byte it is an exact multiple of 4 bytes (to help
 * with alignment of 32-bit pixels):
 */

struct s_rfb_framebuffer_update_msg
{
	Q_UINT8 type;			// always RFB_FRAMEBUFFER_UPDATE
	Q_UINT8 pad;
	Q_UINT16 n_rects;
    // followed by n_rects rectangles
} ;



/*
 * Each rectangle of pixel data consists of a header describing the position
 * and size of the rectangle and a type word describing the encoding of the
 * pixel data, followed finally by the pixel data.  Note that if the client has
 * not sent a SetEncodings message then it will only receive raw pixel data.
 * Also note again that this structure is a multiple of 4 bytes.
 */
struct s_rfb_rectangle
{
	Q_UINT16 x;
	Q_UINT16 y;
	Q_UINT16 w;
	Q_UINT16 h;
} ;


struct s_rfb_framebuffer_update_rect_header
{
	s_rfb_rectangle r;
	Q_UINT32 encoding;		// one of the encoding types
} ;



/*- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 * Raw Encoding.  Pixels are sent in top-to-bottom scanline order,
 * left-to-right within a scanline with no padding in between.
 */


/*- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 * CopyRect Encoding.  The pixels are specified simply by the x and y position
 * of the source rectangle.
 */

struct s_rfb_copy_rect
{
	Q_UINT16 src_x;
	Q_UINT16 src_y;
} ;


/*- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 * RRE - Rise-and-Run-length Encoding.  We have an rfb_RRE_header structure
 * giving the number of subrectangles following.  Finally the data follows in
 * the form [<bgpixel><subrect><subrect>...] where each <subrect> is
 * [<pixel><rfbRectangle>].
 */

struct s_rfb_RRE_header
{
	Q_UINT32 n_subrects;
} ;


/*- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 * CoRRE - Compact RRE Encoding.  We have an s_rfb_RRE_header structure giving
 * the number of subrectangles following.  Finally the data follows in the form
 * [<bgpixel><subrect><subrect>...] where each <subrect> is
 * [<pixel><s_rfb_CoRRE_rectangle>].  This means that
 * the whole rectangle must be at most 255x255 pixels.
 */

struct s_rfb_CoRRE_rectangle
{
	Q_UINT8 x;
	Q_UINT8 y;
	Q_UINT8 w;
	Q_UINT8 h;
} ;




/*- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 * ``Tight'' Encoding.
 */

const Q_UINT8 RFB_TIGHT_EXPLICIT_FILTER = 0x04;
const Q_UINT8 RFB_TIGHT_FILL = 0x08;
const Q_UINT8 RFB_TIGHT_JPEG = 0x09;
const Q_UINT8 RFB_TIGHT_MAX_SUBENCODING = 0x09;



// Filters to improve compression efficiency
const Q_UINT8 RFB_TIGHT_FILTER_COPY = 0x00;
const Q_UINT8 RFB_TIGHT_FILTER_PALETTE = 0x01;
const Q_UINT8 RFB_TIGHT_FILTER_GRADIENT = 0x02;


/*-----------------------------------------------------------------------------
 * SetColourMapEntries - these messages are only sent if the pixel
 * format uses a "colour map" (i.e. trueColour false) and the client has not
 * fixed the entire colour map using FixColourMapEntries.  In addition they
 * will only start being sent after the client has sent its first
 * FramebufferUpdateRequest.  So if the client always tells the server to use
 * trueColour then it never needs to process this type of message.
 */

struct s_rfb_set_colormap_entries_msg
{
	Q_UINT8 type;				// always RFB_SET_COLORMAP_ENTRIES
	Q_UINT8 pad;
	Q_UINT16 first_color;
	Q_UINT16 n_colors;
	// Followed by n_colors * 3 * Q_UINT16
	//   r1, g1, b1, r2, g2, b2, r3, g3, b3, ..., rn, bn, gn

} ;



/*-----------------------------------------------------------------------------
 * Bell - ring a bell on the client if it has one.
 */

struct s_rfb_bell_msg
{
	Q_UINT8 type;			// always RFB_BELL
} ;



struct s_rfb_server_cut_text_msg
{
	Q_UINT8 type;                 // always RFB_SERVER_CUT_TEXT */
	Q_UINT8 pad1;
	Q_UINT16 pad2;
	Q_UINT32 length;
	// followed by char text[length]
} ;


/*-----------------------------------------------------------------------------
 * Union of all server->client messages.
 */

union u_rfb_server_to_client_msg
{
	Q_UINT8 type;
	s_rfb_framebuffer_update_msg fu;
	s_rfb_set_colormap_entries_msg scme;
	s_rfb_bell_msg b;
	s_rfb_server_cut_text_msg sct;
} ;




const unsigned int INTERNAL_READ_BUF_SIZE = 4096;



class rfbConnection : public QObject
{
	Q_OBJECT
public:
	static void initializeAuthentication( void );

	rfbConnection( const QString & _host_ip );
//	rfbConnection( const rfbConnection & _rc );
	~rfbConnection();

	bool openConnection( void );
	bool connected( void ) const
	{
		return( m_connected );
	}
	bool resetConnection( const QString & _new_host_ip = "" );
	void closeConnection( void );

	QImage & clientScreen( void )
	{
		return( m_scaledClientScreen );
	}
	const QImage & unscaledScreen( void ) const
	{
		return( m_clientScreen );
	}
	void setImageSize( const QSize & _s )
	{
		m_destImgSize = _s;
	}

	bool handleServerMessages( bool _send_screen_update );
	bool sendFramebufferUpdateRequest( void );
	bool sendIncrementalFramebufferUpdateRequest( void );
	bool sendFramebufferUpdateRequest( Q_UINT16 _x, Q_UINT16 _y, Q_UINT16 _w, Q_UINT16 _h, bool _incremental );
	bool sendPointerEvent( Q_UINT16 _x, Q_UINT16 _y, Q_UINT16 _button_mask );
	bool sendKeyEvent( Q_UINT32 key, bool down );
	bool sendGetUserRequest( void );
	const QString & user( void )
	{
		sendGetUserRequest();
		return( m_user );
	}
	const QString & ip( void ) const
	{
		return( m_hostIP );
	}
	bool execCmds( const QString & _cmd );
	bool startDemo( const QString & _master_ip, bool _full_screen = FALSE );
	bool stopDemo( void );
	bool lockDisplay( void );
	bool unlockDisplay( void );
	bool sendMessage( const QString & _msg );
	bool postFile( const QString & _fname );
	bool getFile( const QString & _nfilter );
	bool inviteForSupport( const QString & _user_name );


private:

	bool readFromServer( char * _out, unsigned int _n );
	bool writeToServer( const char * _buf, unsigned int _n );
	bool dataFromServer( void );

	long readCompactLen( void );


	bool handleRRE( Q_UINT16 rx, Q_UINT16 ry, Q_UINT16 rw, Q_UINT16 rh );

	bool handleCoRRE( Q_UINT16 rx, Q_UINT16 ry, Q_UINT16 rw, Q_UINT16 rh );

	bool handleTight( Q_UINT16 rx, Q_UINT16 ry, Q_UINT16 rw, Q_UINT16 rh );
	Q_UINT8 initFilterCopy( Q_UINT16 rw, Q_UINT16 rh );
	Q_UINT8 initFilterPalette( Q_UINT16 rw, Q_UINT16 rh );
	Q_UINT8 initFilterGradient( Q_UINT16 rw, Q_UINT16 rh );
	void filterCopy( Q_UINT16 num_rows, Q_UINT32 * dest_buffer );
	void filterPalette( Q_UINT16 num_rows, Q_UINT32 * dest_buffer );
	void filterGradient( Q_UINT16 num_rows, Q_UINT32 * dest_buffer );
	bool decompressJpegRect( Q_UINT16 x, Q_UINT16 y, Q_UINT16 w, Q_UINT16 h );
	//void jpegSetSrcManager (j_decompress_ptr cinfo, Q_UINT8 *compressed_data, int compressed_len);


	static bool stringToIPAddr( const QString & _host, in_addr_t * _addr );
	static int connectToTCPAddr( in_addr_t _host, int _port );



	bool m_connected;
	QString m_hostIP;
	QString m_user;

	bool m_gotInvitationResult;
	bool m_invitationOk;

	int m_sock;

	s_rfb_server_init_msg m_si;

	QImage m_clientScreen;
	QImage m_scaledClientScreen;
	QSize m_destImgSize;

	static const s_rfb_pixel_format s_localDisplayFormat;
#ifdef RSA_CRYPT_AVAILABLE
	static RSA * s_privateKey;
#endif
	// Note that the CoRRE encoding uses this m_buffer and assumes it is big enough
	// to hold 255 * 255 * 32 bits -> 260100 bytes.  640*480 = 307200 bytes.
	// Hextile also assumes it is big enough to hold 16 * 16 * 32 bits.
	// Tight encoding assumes BUFFER_SIZE is at least 16384 bytes. */
	#define BUFFER_SIZE (640*480)
	char m_buffer[BUFFER_SIZE];


	char m_buf[INTERNAL_READ_BUF_SIZE];
	char * m_bufOutPtr;
	unsigned int m_buffered;



	// Variables for the ``tight'' encoding implementation.
#ifdef HAVE_LIBZ
	// Separate m_buffer for compressed data.
	#define ZLIB_BUFFER_SIZE 512
	char m_zlibBuffer[ZLIB_BUFFER_SIZE];

	// Four independent compression streams for zlib library.
	z_stream m_zlibStream[4];
	bool m_zlibStreamActive[4];

	// Filter stuff. Should be initialized by filter initialization code.
	Q_UINT16 m_rectWidth, m_rectColors;
	char m_tightPalette[256*4];
	Q_UINT8 m_tightPrevRow[2048*3*sizeof(Q_UINT16)];
#ifdef HAVE_LIBJPEG
	// JPEG decoder state.
	jpeglib::jpeg_source_mgr m_jpegSrcManager;
#endif
#endif
} ;


#endif
