/*************************************************************************
 * 
 * irmp3 - Multimedia Audio Jukebox for Linux
 * http://irmp3.sourceforge.net
 *
 * $Source: /cvsroot/irmp3/irmp3/src/irmp3d/mod_timer.c,v $ -- timer functions
 * $Id: mod_timer.c,v 1.11 2004/02/17 20:40:36 boucman Exp $
 *
 * Copyright (C) by Andreas Neuhaus
 *
 * Please contact the current maintainer, Jeremy Rosen <jeremy.rosen@enst-bretagne.fr>
 * for information and support regarding irmp3.
 *
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <time.h>

#include "config.h"
#include "irmp3tools.h"
#include "irmp3log.h"
#include "irmp3mod.h"
#include "mod_timer.h"


/*************************************************************************
 * GLOBALS
 */
time_t mod_timer_sleeptimeout = 0;
time_t mod_timer_sleepfadeout = 0;


/*************************************************************************
 * MODULE INFO
 */
mod_t mod_timer = {
	"mod_timer",
	NULL,			// deinit
	NULL,			// reload
	&blank_fd,		// watchfd
	NULL,			// poll
	mod_timer_update,	// update
	mod_timer_message,	// message
	NULL,			// SIGCHLD handler
	NULL,
	NULL,			// avoid warning
};


/*************************************************************************
 * UPDATE
 */
void mod_timer_update (void)
{
	static time_t lastupdate;
	time_t now;
	int stepsleft;
	mod_message_t *answer;

	if (!mod_timer_sleeptimeout)
		return;

	now = time(NULL);
	if (now >= lastupdate+10) {
		lastupdate = now;
		log_printf(LOG_DEBUG, "mod_timer_update(): checking timers\n");

		// check if sleep timer expired
		if (mod_timer_sleeptimeout <= now) {
			log_printf(LOG_NORMAL, "Sleep timer expired\n");
			mod_timer_sleeptimeout = 0;
			mod_timer_sleepfadeout = 0;
			answer = mod_query(MSGTYPE_INPUT, "stop");
			free_message(answer);
			sleep(10);
			answer = mod_query(MSGTYPE_INPUT, "mixer default");
			free_message(answer);
			mod_sendmsg(MSGTYPE_EVENT, "sleep expired");
			return;
		}

		// check if volume decrease should be done
		if (now < mod_timer_sleepfadeout)
			return;

		stepsleft = (mod_timer_sleeptimeout - now) / 10;
		log_printf(LOG_VERBOSE, "Sleep timer decreasing volume (%d steps left)\n", stepsleft);
		mod_sendmsgf(MSGTYPE_INPUT, "mixer volume -%d%%", stepsleft ? (100 / stepsleft) : 100);
		mod_sendmsgf(MSGTYPE_EVENT, "sleep decreased %d",stepsleft);
	}
}


/*************************************************************************
 * RECEIVE MESSAGE
 */
void mod_timer_message (int msgtype, char *msg,const char __attribute__((unused))*sender)
{
	char *c1, *c2, *c3;
	int sleeptime, fadetime;

	// handle input messages
	if (msgtype == MSGTYPE_INPUT) {
		c1 = strtok(msg, " \t");
		if(!c1) return;

		// process sleep command
		if ( !strcasecmp(c1, "sleep")) {
			c2 = strtok(NULL, " \t");
			c3 = strtok(NULL, "");
			sleeptime = c2 ? atoi(c2) : 0;
			fadetime = c3 ? atoi(c3) : 0;
			if (sleeptime) {
				mod_timer_sleeptimeout = time(NULL) + sleeptime * 60;
				mod_timer_sleepfadeout = mod_timer_sleeptimeout - fadetime * 60;
				mod_sendmsgf(MSGTYPE_EVENT, "sleep %d %d", sleeptime, fadetime);
				log_printf(LOG_NORMAL, "Sleep timer set to %d mins, %d mins fadeout\n", sleeptime, fadetime);
			} else {
				mod_timer_sleeptimeout = 0;
				mod_timer_sleepfadeout = 0;
				mod_sendmsg(MSGTYPE_EVENT, "sleep 0");
				log_printf(LOG_NORMAL, "Sleep timer off\n");
			}
		}
	}
}




/*************************************************************************
 * EOF
 */
