#   Only One Close Button - Epiphany Extension
#   Copyright (C) 2006  Stefan Stuhr
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License along
#   with this program; if not, write to the Free Software Foundation, Inc.,
#   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.


import gtk
import epiphany

EGG_TB_MODEL_NAME_KNOWN = 1 << 2

class OnlyOneCloseButton:
	tab_popup_xml = '''
		<ui>
		  <popup name="EphyNotebookPopup" action="NotebookPopupAction">
		    <separator/>
		    <menuitem name="TabCloseENP" action="FileCloseTab"/>
		    <separator/>
		  </popup>
		</ui>'''

	def __init__(self):
		self.win_data = {}

		# Make the toolbar button available in the editor
		shell = epiphany.ephy_shell_get_default()
		model = shell.get_toolbars_model(False)
		name_flags = model.get_name_flags("FileCloseTab")
		model.set_name_flags("FileCloseTab", name_flags | EGG_TB_MODEL_NAME_KNOWN)

	# Not supported by Epiphany - but may be supported in the future
	def finalize_extension(self):
		# Remove the toolbar button from the available buttons in the editor
		shell = epiphany.ephy_shell_get_default()
		model = shell.get_toolbars_model(False)
		name_flags = model.get_name_flags("FileCloseTab")
		model.set_name_flags("FileCloseTab", name_flags & ~EGG_TB_MODEL_NAME_KNOWN)

	def attach_window(self, window):
		data = self.win_data[window] = {"tab_data": {}}

		ui_manager = window.get_ui_manager()
		merge_id = ui_manager.add_ui_from_string(self.tab_popup_xml)
		data["ui_merge_id"] = merge_id

		notebook = window.get_notebook()

		initially_homogeneous = notebook.get_property("homogeneous")
		data["initially_homogeneous"] = initially_homogeneous
		notebook.set_property("homogeneous", True)

	def detach_window(self, window):
		data = self.win_data.pop(window)

		ui_manager = window.get_ui_manager()

		merge_id = data["ui_merge_id"]
		ui_manager.remove_ui(merge_id)
		ui_manager.ensure_update()

		notebook = window.get_notebook()

		initially_homogeneous = data["initially_homogeneous"]
		notebook.set_property("homogeneous", initially_homogeneous)

	def hbox_style_set_cb(self, hbox, previous_style, window, tab):
		data = self.win_data[window]
		tab_data = data["tab_data"][tab]

		tab_data["tab_initial_width"] = hbox.get_size_request()[0]
		hbox.set_size_request(16, -1)

	def attach_tab(self, window, tab):
		data = self.win_data[window]
		tab_data = data["tab_data"][tab] = {}

		notebook = window.get_notebook()
		widget = notebook.get_tab_label(tab)

		if isinstance(widget, gtk.HBox):
			children = widget.get_children()

			tab_data["tab_initial_width"] = widget.get_size_request()[0]
			widget.set_size_request(16, -1)

			handler_id = widget.connect("style-set", self.hbox_style_set_cb, window, tab)
			tab_data["hbox_style_set_hid"] = handler_id

			if isinstance(children[-1], gtk.Button):
				children[-1].hide()

			notebook.child_set(tab, "tab-expand", True)

	def detach_tab(self, window, tab):
		data = self.win_data[window]
		tab_data = data["tab_data"].pop(tab)

		notebook = window.get_notebook()
		widget = notebook.get_tab_label(tab)

		if tab.parent == notebook:
			notebook.child_set(tab, "tab-expand", False)

		if isinstance(widget, gtk.HBox):
			children = widget.get_children()

			handler_id = tab_data["hbox_style_set_hid"]
			widget.disconnect(handler_id)

			widget.set_size_request(tab_data["tab_initial_width"], -1)

			if isinstance(children[-1], gtk.Button):
				children[-1].show()


oocb_extension = OnlyOneCloseButton()

attach_window = oocb_extension.attach_window
detach_window = oocb_extension.detach_window

attach_tab = oocb_extension.attach_tab
detach_tab = oocb_extension.detach_tab
