// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#ifndef __VoidPtrArray__
#define __VoidPtrArray__

#include <assert.h>

namespace Puma {

class VoidPtrArray {
protected:
  static const long default_init_size;
  static const long default_increment;

protected:
  VoidPtrArray (long is = default_init_size, long incr = default_increment);
  VoidPtrArray (const VoidPtrArray &array);
  VoidPtrArray &operator =(const VoidPtrArray &array);
  ~VoidPtrArray ();

  void append (const void *item);
  void insert (long index, const void *item);
  void prepend (const void *item);
  void remove (long index);
  void reset ();
  void *&get (long index);
  void *&operator[] (long index);
  void *fetch (long index) const;
  void *&lookup (long index) const;
  long length () const;

private:
  void** data;
  long size;
  long count;
  long increment;
      
  void check (long wanted);
};


inline VoidPtrArray::VoidPtrArray (long is, long incr) {
  count     = 0; 
  size      = 0; //is;
  increment = incr;
  data      = 0; //new void*[size];
}


inline VoidPtrArray::VoidPtrArray (const VoidPtrArray &array) {
  count = 0;
  size  = array.size;
  if (size) {
    data  = new void*[size];
    for (int pos = 0; pos < array.length (); pos++)
      append (array.lookup (pos));
  }
  else
    data = 0;
  increment = array.increment;
}


inline VoidPtrArray &VoidPtrArray::operator =(const VoidPtrArray &array) {
  if (data)
    delete[] data;
  count = 0;
  size  = array.size;
  if (size) {
    data  = new void*[size];
    for (int pos = 0; pos < array.length (); pos++)
      append (array.lookup (pos));
  }
  else
    data = 0;
  increment = array.increment;
  return *this;
}


inline VoidPtrArray::~VoidPtrArray () {
  if (data)
    delete[] data;
}


inline void VoidPtrArray::append (const void *item) {
  check (count);
  data[count++] = (void*)item;
}


inline void VoidPtrArray::prepend (const void *item) {
  insert (0, item);
}


inline void VoidPtrArray::insert (long index, const void *item) {
  check (count);
  for (int pos = count; pos > index; pos--)
    data[pos] = data[pos - 1];
  data[index] = (void*)item;
  count++;
}


inline void *&VoidPtrArray::get (long index) {
  check (index);
  if (index >= count)
    count = index + 1;
  return data[index];
}


inline void *&VoidPtrArray::operator[] (long index) {
  return get (index);
}


inline void *VoidPtrArray::fetch (long index) const {
  return data[index];
}


inline long VoidPtrArray::length () const {
  return count;
}


inline void VoidPtrArray::check (long wanted) {
  if (wanted >= size) {
    void** new_data;

    while (wanted >= size) {
      size += increment;
      increment*=2;
    }

    new_data = new void*[size];
    if (data) {
      for (int pos = 0; pos < count; pos++)
        new_data[pos] = data[pos];
      delete[] data;
    }
    data = new_data;
  }
}
   

inline void VoidPtrArray::remove (long index) {
  if (index < count && count > 0) {
    for (int pos = index; pos < count - 1; pos++)
      data[pos] = data[pos + 1];
    count--;
  }
}


inline void VoidPtrArray::reset () {
//  if (data)
//    delete[] data;
  count     = 0; 
//  size      = default_init_size;
//  increment = default_increment;
//  data      = new void*[size];
}


inline void *&VoidPtrArray::lookup (long index) const {
   assert(index >= 0 && index < count);
//  if (index >= count) index = count - 1; 
//  if (index < 0)      index = 0;
   return data[index];
}


} // namespace Puma

#endif /* __VoidPtrArray__ */
