###################################################################################################
# _zreferableitem.py
#
# $Id: _zreferableitem.py,v 1.5 2004/02/24 19:57:21 dnordmann Exp $
# $Name:  $
# $Author: dnordmann $
# $Revision: 1.5 $
#
# Implementation of classes ZReferableItem (see below).
# 
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
###################################################################################################

# Imports.
from Globals import HTML, HTMLFile
import copy
import time
import urllib
# Product Imports.
import _globals
import _objattrs

# ---------------------------------------------------------------------------------------------
#  getMedlineLink:
# ---------------------------------------------------------------------------------------------
def getMedlineLink(url):
  try:
    http_prefix = 'http://'
    if url.find(http_prefix) == 0:
      url = url[len(http_prefix):]
    url = 'http://www.ncbi.nlm.nih.gov/entrez/query.fcgi?cmd=Retrieve&db=PubMed&list_uids=%i&dopt=Abstract'%int(url)
  except:
    pass
  return url

# ---------------------------------------------------------------------------------------------
#  isMailLink:
# ---------------------------------------------------------------------------------------------
def isMailLink(url):
  return url.lower().find('mailto:') == 0

# ---------------------------------------------------------------------------------------------
#  isInternalLink:
# ---------------------------------------------------------------------------------------------
def isInternalLink(url):
  return type(url) is type('') and len(url)>=3 and url[:2]=='{$' and url[-1]=='}'

# ---------------------------------------------------------------------------------------------
#  absolute_home:
# ---------------------------------------------------------------------------------------------
def absolute_home(ob):
  s = ''
  while ob is not None:
    try:
      if ob.meta_type.find('ZMS')<0:
        if len(s) > 0:
          s = '/' + s
        s = ob.id + s
      ob = ob.aq_parent
    except: 
      break
  return s


###################################################################################################
###################################################################################################
###
###   class ZReferableItem
###
###################################################################################################
###################################################################################################
class ZReferableItem: 

  # Management Permissions.
  # -----------------------
  __authorPermissions__ = (
		'manage_RefForm',
		)
  __ac_permissions__=(
		('ZMS Author', __authorPermissions__),
		)

  # Management Interface.
  # ---------------------
  manage_RefForm = HTMLFile('dtml/zmslinkelement/manage_refform', globals())
  browse_objs = HTMLFile('dtml/zmslinkelement/browse_objs', globals()) 

  # ---------------------------------------------------------------------------------------------
  #  ZReferableItem.getRefObjPath:
  # ---------------------------------------------------------------------------------------------
  def getRefObjPath(self, ob):
    ref = ''
    if ob is not None:
      path = ob.relative_obj_path()
      clientIds = absolute_home(ob).split('/')
      thisIds = absolute_home(self).split('/')
      if clientIds[-1] != thisIds[-1]:
        if len(clientIds) <= len(thisIds):
          path = clientIds[-1] + '@' + path
        else:
          while len(clientIds) > 0 and \
                len(thisIds) > 0 and \
                clientIds[0] == thisIds[0]:
            del thisIds[0]
            del clientIds[0]
          s = ''
          for clientId in clientIds:
            if len(s) > 0: s = s + '/'
            s += clientId
          path = s + '@' + path
      ref = '{$' + path + '}'
    return ref


  """
  ###############################################################################################
  ###  
  ###  Links FROM other objects.
  ### 
  ###############################################################################################
  """

  # ---------------------------------------------------------------------------------------------
  #  ZReferableItem.getRefByObjs:
  #
  #  Returns references BY other objects.
  # ---------------------------------------------------------------------------------------------
  def getRefByObjs(self, REQUEST={}):
    ref_by = []
    if _objattrs.hasobjattr(self,'ref_by'):
      ref_by = getattr(self,'ref_by',[])
      ref_by = copy.deepcopy(ref_by)
    return ref_by


  # ---------------------------------------------------------------------------------------------
  #  ZReferableItem.synchronizeRefByObjs:
  #
  #  Synchronizes references BY other objects.
  # ---------------------------------------------------------------------------------------------
  def synchronizeRefByObjs(self, strict=1):
    key = 'ref_by'
    v = getattr(self,key,[])
    ref_by = []
    for i in v:
      ob = self.getLinkObj(i)
      if ob is not None and \
         ob.meta_type[:3] == 'ZMS':
        ob_path = self.getRefObjPath(ob)
        if not ob_path in ref_by:
          ref_by.append(ob_path)
    if strict or len(ref_by) < len(v):
      setattr(self,'ref_by',ref_by)


  # ---------------------------------------------------------------------------------------------
  #  ZReferableItem.clearRegisteredRefObjs:
  #
  #  Clears registered referencing objects.
  # ---------------------------------------------------------------------------------------------
  def clearRegisteredRefObjs(self, REQUEST):
    ref_by = []
    setattr(self,'ref_by',ref_by)

    
  # ---------------------------------------------------------------------------------------------
  #  ZReferableItem.registerRefObj:
  #
  #  Registers referencing object.
  # ---------------------------------------------------------------------------------------------
  def registerRefObj(self, ob, REQUEST, onchange=1):
    #++ if _globals.debug: print "[%s.registerRefObj]: %s(%s)"%(self.meta_type,ob.id,ob.meta_type)
    ref = self.getRefObjPath(ob)
    ref_by = self.getRefByObjs(REQUEST)
    if not ref in ref_by:
      ref_by.append(ref)
    ##### Set Property ####
    setattr(self,'ref_by',ref_by)
    ##### VersionManager ####
    if onchange: self.onChangeObj(REQUEST)


  # ---------------------------------------------------------------------------------------------
  #  ZReferableItem.unregisterRefObj:
  #
  #  Unregisters referencing object.
  # ---------------------------------------------------------------------------------------------
  def unregisterRefObj(self, ob, REQUEST, onchange=1):
    #++ if _globals.debug: print "[%s.unregisterRefObj]: %s(%s)"%(self.meta_type,ob.id,ob.meta_type)
    prefix = '{$'
    homeId = ob.getHome().id
    if homeId != self.getHome().id:
      prefix += homeId + '@'
    suffix = ob.id + '}'
    ref_by = self.getRefByObjs(REQUEST)
    for url in ref_by:
      if len(url) >= len(prefix+suffix) and \
         url[:len(prefix)] == prefix and \
         url[-len(suffix):] == suffix:
        del ref_by[ref_by.index(url)]
    ##### Set Property ####
    setattr(self,'ref_by',ref_by)
    ##### VersionManager ####
    if onchange: self.onChangeObj(REQUEST)


  # ---------------------------------------------------------------------------------------------
  #  ZReferableItem.refreshRefObj:
  #
  #  Refreshs referencing object.
  # ---------------------------------------------------------------------------------------------
  def refreshRefObj(self, ob, REQUEST):
    #++ if _globals.debug: print "[%s.refreshRefObj]: %s(%s) -> %s(%s)"%(self.meta_type,self.id,self.meta_type,ob.id,ob.meta_type)
    self.unregisterRefObj(ob,REQUEST,onchange=0)
    self.registerRefObj(ob,REQUEST,onchange=0)

    
  """
  ###############################################################################################
  ###  
  ###  References TO other objects.
  ### 
  ###############################################################################################
  """

  # ---------------------------------------------------------------------------------------------
  #  ZReferableItem.getRefToObjs:
  #
  #  Returns list of references TO other objects.
  # ---------------------------------------------------------------------------------------------
  def getRefToObjs(self, REQUEST):
    ref_to =  []
    for key in self.getObjAttrs().keys():
      obj_attr = self.getObjAttr(key)
      datatype = obj_attr['datatype_key']
      if datatype == _globals.DT_URL:
        ref = self.getObjProperty(key,REQUEST)
        if not ref in ref_to:
          ref_to.append(ref)
    return ref_to


  # ---------------------------------------------------------------------------------------------
  #  ZReferableItem.synchronizeRefToObjs:
  #
  #  Synchronizes references TO other objects.
  # ---------------------------------------------------------------------------------------------
  def synchronizeRefToObjs(self):
    #++ if _globals.debug: print "[%s.synchronizeRefToObjs]:"%(self.meta_type)
    for key in self.getObjAttrs().keys():
      obj_attr = self.getObjAttr(key)
      datatype = obj_attr['datatype_key']
      if datatype == _globals.DT_URL:
        for s_lang in self.getLangIds():
          req = {'lang':s_lang,'preview':'preview'}
          ref = self.getObjProperty(key,req)
          ref_obj = self.getLinkObj(ref)
          if ref_obj is not None:
            ref_obj.refreshRefObj(self,req)


  # ---------------------------------------------------------------------------------------------
  #  ZReferableItem.refreshRefToObj:
  #
  #  Refreshs reference TO given destination.
  # ---------------------------------------------------------------------------------------------
  def refreshRefToObj(self, dest_obj, REQUEST):
    #++ if _globals.debug: print "[%s.refreshRefToObj]: %s -> dest_obj=%s(%s)"%(self.meta_type,self.id,dest_obj.absolute_url(),dest_obj.meta_type)
    ref_to =  []
    for key in self.getObjAttrs().keys():
      obj_attr = self.getObjAttr(key)
      datatype = obj_attr['datatype_key']
      if datatype == _globals.DT_URL:
        ref = self.getObjProperty(key,REQUEST)
        ref_obj = self.getLinkObj(ref)
        if ref_obj is not None:
          if dest_obj.id == ref_obj.id:
            ##### Set Property ####
            dest_obj_path = self.getRefObjPath(dest_obj)
            self.setObjProperty(key,dest_obj_path,REQUEST['lang'],1)


  """
  ###############################################################################################
  ###  
  ###  Process Events 
  ### 
  ###############################################################################################
  """

  # ---------------------------------------------------------------------------------------------
  #  ZReferableItem.onMoveRefObj:
  #
  #  This method is executed after an object has been moved.
  # ---------------------------------------------------------------------------------------------
  def onMoveRefObj(self, REQUEST, deep=0):
    #++ if _globals.debug: print "[%s.onMoveRefObj]: %s"%(self.meta_type,self.id)

    ##### Update references TO other objects ####
    for ref in self.getRefToObjs(REQUEST):
      ob = self.getLinkObj(ref)
      if ob is not None:
        ob.refreshRefObj(self,REQUEST)

    ##### Update references FROM other objects ####
    for ref in self.getRefByObjs(REQUEST):
      ob = self.getLinkObj(ref)
      if ob is not None:
        ob.refreshRefToObj(self,REQUEST)


  # ---------------------------------------------------------------------------------------------
  #  ZReferableItem.onCopyRefObj:
  #
  #  This method is executed after an object has been copied.
  # ---------------------------------------------------------------------------------------------
  def onCopyRefObj(self, REQUEST, deep=0):
    #++ if _globals.debug: print "[%s.onCopyRefObj]: %s"%(self.meta_type,self.id)

    ##### Register copy in references TO other objects ####
    for ref in self.getRefToObjs(REQUEST):
      ob = self.getLinkObj(ref)
      if ob is not None:
        ob.registerRefObj(self,REQUEST)

    ##### Clear references FROM other objects ####
    self.clearRegisteredRefObjs(REQUEST)


  # ---------------------------------------------------------------------------------------------
  #  ZReferableItem.onRollbackRefObj:
  #
  #  This method is executed before an object is rolled-back.
  # ---------------------------------------------------------------------------------------------
  def onRollbackRefObj(self, REQUEST):
    #++ if _globals.debug: print "[%s.onRollbackRefObj]: %s"%(self.meta_type,self.id)
  
    ##### Unregister references TO other objects from work-version ####
    req = {'lang':REQUEST['lang'],'preview':'preview'}
    for ref in self.getRefToObjs(req):
      ob = self.getLinkObj(ref)
      if ob is not None:
        ob.unregisterRefObj(self,req)
        
    ##### Register references TO other objects from live-version ####
    req = {'lang':REQUEST['lang'],'preview':''}
    for ref in self.getRefToObjs(req):
      ob = self.getLinkObj(ref)
      if ob is not None:
        ob.registerRefObj(self,req)


  """
  ###############################################################################################
  ###  
  ###  Resolve Links 
  ### 
  ###############################################################################################
  """

  # ---------------------------------------------------------------------------------------------
  #  ZReferableItem.getLinkObj:
  #
  #  Resolves internal/external links and returns Object.
  # ---------------------------------------------------------------------------------------------
  def getLinkObj(self, url, REQUEST={}):
    ob = None
    if isInternalLink(url):
      if url.find('{$__') != 0:
        docElmnt = None
        path = url[2:-1]
        i = path.find('@')
        if i > 0:
          clientIds = path[:i].split('/')
          path = path[i+1:]
          thisHome = self.getHome()
          clientHome = None
          if thisHome.aq_parent.id == clientIds[-1]:
            clientHome = thisHome
            for j in range(len(clientIds)):
              if clientHome.aq_parent.id == clientIds[-(j+1)]:
                clientHome = clientHome.aq_parent
          elif hasattr(thisHome,clientIds[0]):
            clientHome = thisHome
            for j in range(len(clientIds)):
              clientHome = getattr(clientHome,clientIds[j],None)
          if clientHome is not None:
            obs = clientHome.objectValues(['ZMS'])
            if obs:
              docElmnt = obs[0]
        else:
          docElmnt = self.getDocumentElement()
        if docElmnt is not None:
          ob = docElmnt.findObjId(path,REQUEST)
      if ob is None:
        _globals.writeError(self,'[getLinkObj]: Invalid internal url %s'%url)
    return ob

  # ---------------------------------------------------------------------------------------------
  #  ZReferableItem.getLinkUrl:
  #
  #  Resolves internal/external links and returns URL.
  # ---------------------------------------------------------------------------------------------
  def getLinkUrl(self, url, REQUEST={}): 
    if isInternalLink(url):
      ob = self.getLinkObj(url,REQUEST)
      url = None
      if ob is not None:
        if ob.meta_type == 'ZMSFile':
          file = ob.getFile(REQUEST)
          if file is not None:
            url = file.getHref(REQUEST)
        else:
          url = ob.getHref2IndexHtml(REQUEST)
    return url

  # ---------------------------------------------------------------------------------------------
  #  ZReferableItem.getLinkHtml:
  #
  #  Resolves internal/external links and returns Html.
  # ---------------------------------------------------------------------------------------------
  def getLinkHtml(self, url, html='<a href="%s">&raquo;</a>', REQUEST={}):
    s = ''
    ob = self.getLinkObj(url,REQUEST)
    if ob is not None:
      if ob.isActive(REQUEST) and \
         ob.isVisible(REQUEST):
        url = ob.getHref2IndexHtml(REQUEST)
        s = html%url
    return s

###################################################################################################
